/*****************************************************************************
 *                                  R K 4 4
 *****************************************************************************
 *
 *   PROGRAM ID:        RK44.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This subroutine is a Runge-Kutta-Merson ODE solver of order 4.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void xODE44( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h, float *w[] );

static void xRKStep( int n, float y[], float dy[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float ytemp[], float ysav[],
                     float dysav[], float *w[] );

static void xRK4b( int n, float y[], float dy[], float x, float h,
                   float yout[], float k1[], float k2[], float k3[],
                   float k4[], float k5[] );

static void xRK4a( int n, float yout[], float k2[] );



/* Here's the entry point for this module */
void rk44( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (16*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/

int i;
float **w;

  w= matrix( 1, 16, 1, n );
  f= fxn;
  gkmax= nPts +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
      gszDataFile[i]= szFile[i];

  xODE44( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
  free_matrix( w, 1, 16, 1, n );
}

static void xODE44( int n, float ystart[], float y[], float yscal[],
                   float dy[], float x1, float x2, float eps,
                   float h, float *w[] )
{
/*
   This subroutine also drives three other ode solvers:
               1) a Runge-Kutta-Gill solver
               2) Runge-Kutta-Fehlberg solver
               3) Bulirsch-Stoer (modified midpoint) solver

   All of them will solve a system of first order ordinary
   differential equations.  The RK solvers are good, all-
   purpose routines.  The BS solver can be faster and more
   accurate, but is less robust.

   The difference between the Runge-Kutta methods is the
   way that the local error is estimated.  Gill uses a full-,
   half-step technique and makes (3n-1) derivative evaluations
   per step. Fehlberg uses two methods of different orders
   which just happen to have the same intermediate vectors;
   it only uses (n+2) derivative evaluations per step.  Merson
   uses two methods of identical order which have common inter-
   mediates and uses (n+1) derivative evals per step.  These
   error estimates are critical to adaptive step-size strategies.

   ystart is the vector of dependent variables, on input it
         contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h1 is the estimate of the step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy)  where the derivatives are returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/

const int maxstp= 10000;
const float tiny= 1.e-30;

float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i, nstp;
int bDone= 0;
FILE *fp= NULL;

  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );



  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      enStep= nstp;  /* signal next step to external process */
      f( n, x, y, dy);

      for( i=1 ; i<=n ; i++ )
          yscal[i]= fabs(y[i]) +fabs(h*dy[i]) +tiny;

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }

      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )    h= x2-x;
      xRKStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3], &w[3] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
             ystart[i]= y[i];

          if( fp && gkmax )
          {
              kount= kount + 1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                 fprintf( fp, "  %e", y[i] );
          }
          if( fp )
              fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Stepsize underflow in RK44()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Maximum iterations in RK44()" );
  }
}



static void xRKStep( int n, float y[], float dy[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float ytemp[], float ysav[],
                    float dysav[], float *w[] )
{
const float pgrow= -0.2;
const float pshrnk= -0.25;
const float safety= 0.75;
const float errcon= 3.2e-4;

int bDone= 0;
int i;
float h= htry;
float xsav= x[0];
float errmax;


  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy[i];
  }


  while ( !bDone )
  {
      float temp;

  // Use two fourth order methods; note that most of the calculation
  // for xRK4a is bDone in xRK4b, thus the abbreviated argument list
      x[0]= xsav +h;
      xRK4b( n, ysav, dysav, xsav, h, ytemp,
            w[1], w[2], w[3], w[4], w[5] );
      if( x[0]==xsav )
          nrerror( "Stepsize underflow in RK44()" );

      xRK4a( n, ytemp, w[2] );
      errmax= 0.0;


      // Put error estimate ---> ytemp
      for( i=1 ; i<=n ; i++ )
      {
          ytemp[i]= (y[i] -ytemp[i]) / 5.0;
          temp= fabs( ytemp[i] / yscal[i] );
          errmax= (errmax > temp)? errmax : temp;
      }
      errmax= errmax / eps;

      // Try new step-size if we didn't meet eps
      if( errmax>1 )
          h= safety * h * pow(errmax,pshrnk);
      else
          bDone= 1;
  }

  hdid[0]= h;

  if( errmax > errcon )
      hnext[0]= safety * h * pow(errmax,pgrow);
  else
      hnext[0]= 4.0 * h;

  // Make correction to get fifth order accuracy
  for( i=1 ; i<=n ; i++ )
      y[i]= y[i] + ytemp[i];
}



static void xRK4b( int n, float y[], float dy[], float x, float h,
                  float yout[], float k1[], float k2[], float k3[],
                  float k4[], float k5[] )
{
int i;
float xh;

  for( i=1 ; i<=n ; i++ )
  {
      k1[i]= h*dy[i];
      yout[i]= y[i] + k1[i]/3.0;
  }

  xh= x +h/3.0;
  f( n, xh, yout, k2 );

  for( i=1 ; i<=n ; i++ )
  {
      k2[i]= h*k2[i];
      yout[i]= y[i] + (k1[i]+k2[i])/6.0;
  }

  xh= x + h/3.0;
  f( n, xh, yout, k3 );


  for( i=1 ; i<=n ; i++ )
  {
      k3[i]= h*k3[i];
      yout[i]= y[i] +(k1[i]+3.0*k3[i])/8.0;
  }

  xh= x + h/2.0;
  f( n, xh, yout, k4 );

  for( i=1 ; i<=n ; i++ )
  {
      k4[i]= h*k4[i];
      yout[i]= y[i] + k1[i]/2.0 - 3.0*k3[i]/2.0 + 2.0*k4[i];
  }

  xh= x + h;
  f( n, xh, yout, k5 );

  // Note that I'm saving the yout for k4 as k2
  for( i=1 ; i<=n ; i++ )
  {
      k5[i]= h*k5[i];
      k2[i]= yout[i];
      yout[i]= y[i] + k1[i]/6.0 + 2.0*k4[i]/3.0 + k5[i]/6.0;
  }
}



static void xRK4a( int n, float yout[], float k2[] )
{
int i;

  // All I do here is retrieve a calculation from xRK4b!!
  for( i=1 ; i<=n ; i++ )
      yout[i]= k2[i];
}

