/*****************************************************************************
 *                                  R K 4 5
 *****************************************************************************
 *
 *   PROGRAM ID:        RK45.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This subroutine is a Runge-Kutta-Fehlberg ODE solver of orders 4 & 5.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void xODE45( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h, float *w[] );

static void xRKStep( int n, float y[], float dy[], float yscal[],
                     float x[], float htry, float eps, float *hdid,
                     float *hnext, float ytemp[], float ysav[],
                     float dysav[], float *w[] );

static void xRK5( int n, float y[], float dy[], float x, float h,
                  float yout[], float k1[], float k2[], float k3[],
                  float k4[], float k5[], float k6[] );

static void xRK4( int n, float y[], float dy[], float x, float h,
                  float yout[], float k1[], float k2[], float k3[],
                  float k4[], float k5[], float k6[] );



/* Here's the entry point for the module */
void rk45( int n, DERIV fxn, float ystart[], float x1, float x2,
           float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (16*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
int i;
float **w;

  w= matrix( 1, 16, 1, n );
  f= fxn;
  gkmax= nPts +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
      gszDataFile[i]= szFile[i];

  xODE45( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
  free_matrix( w, 1, 16, 1, n );
}


static void xODE45( int n, float ystart[], float y[], float yscal[],
                   float dy[], float x1, float x2, float eps,
                   float h, float *w[] )
{
/*
   This subroutine also drives three other ode solvers:
               1) a Runge-Kutta-Gill solver
               2) Runge-Kutta-Fehlberg solver
               3) Bulirsch-Stoer (modified midpoint) solver

   All of them will solve a system of first order ordinary
   differential equations.  The RK solvers are good, all-
   purpose routines.  The BS solver can be faster and more
   accurate, but is less robust.

   The difference between the Runge-Kutta methods is the
   way that the local error is estimated.  Gill uses a full-,
   half-step technique and makes (3n-1) derivative evaluations
   per step. Fehlberg uses two methods of different orders
   which just happen to have the same intermediate vectors;
   it only uses (n+2) derivative evaluations per step.  Merson
   uses two methods of identical order which have common inter-
   mediates and uses (n+1) derivative evals per step.  These
   error estimates are critical to adaptive step-size strategies.

   ystart is the vector of dependent variables, on input it
         contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h1 is the estimate of the step size
   hmin is the minimum allowed step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy)  where the derivatives are returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/
const unsigned long maxstp= 10000;
unsigned long nstp;
const float tiny= 1.e-30;

float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i;
int bDone= 0;
FILE *fp= NULL;


  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );



  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      enStep= nstp;  // signal next step to external process
      ///****/       printf("RK23: n=%d, t=%e,\n\t x=(%e %e %e %e)\n", n,x,y[1],y[2],y[3],y[4] );
      f( n, x, y, dy);

      for( i=1 ; i<=n ; i++ )
          yscal[i]= fabs(y[i]) +fabs(h*dy[i]) +tiny;

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }

      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )    h= x2-x;
      xRKStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3], &w[3] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
              ystart[i]= y[i];

          if( fp && gkmax )
          {
              kount= kount + 1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
          }
          if( fp )
              fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Stepsize underflow in RK45()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Maximum iterations in RK45()" );
  }
}



static void xRKStep( int n, float y[], float dy[], float yscal[],
                     float x[], float htry, float eps, float *hdid,
                     float *hnext, float ytemp[], float ysav[],
                     float dysav[], float *w[] )
{
const float pgrow= -0.2;
const float pshrnk= -0.25;
const float safety= 0.8;
const float errcon= 3.2e-4;

int bDone= 0;
int i;
float h= htry;
float xsav= x[0];
float errmax;


  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy[i];
  }


  while ( !bDone )
  {
      float temp;

      // Use a fifth and a fourth order method
      x[0]= xsav +h;
      xRK5( n, ysav, dysav, xsav, h, y,
            w[1], w[2], w[3], w[4], w[5], w[6] );
      if( x[0]==xsav )
          nrerror( "Stepsize underflow in RK45()" );
      xRK4( n, ysav, dysav, xsav, h, ytemp,
           w[1], w[2], w[3], w[4], w[5], w[6] );
      errmax= 0.0;

      // Put error estimate ---> ytemp
      for( i=1 ; i<=n ; i++ )
      {
          ytemp[i]= ( y[i] -ytemp[i] );
          temp= fabs( ytemp[i] / yscal[i] );
          errmax= (errmax > temp)? errmax : temp;
      }
      errmax= errmax / eps;

      // Try new step-size if we didn't meet eps
      if( errmax>1 )
          h= safety * h * pow(errmax,pshrnk);
      else
          bDone= 1;
  }

  hdid[0]= h;

  if( errmax > errcon )
      hnext[0]= safety * h * pow(errmax,pgrow);
  else
      hnext[0]= 4.0 * h;
}


static void xRK5( int n, float y[], float dy[], float x, float h,
                  float yout[], float k1[], float k2[], float k3[],
                  float k4[], float k5[], float k6[] )
{
int i;
float xh;

  for( i=1 ; i<=n ; i++ )
  {
      k1[i]= h*dy[i];
      yout[i]= y[i] + k1[i]/4.0;
  }

  xh= x + h/4.0;
  f( n, xh, yout, k2 );

  for( i=1 ; i<=n ; i++ )
  {
      k2[i]= h*k2[i];
      yout[i]= y[i] + 3.0*( k1[i]+3.0*k2[i] )/32.0;
  }

  xh= x + 3.0*h/8.0;
  f( n, xh, yout, k3 );

  for( i=1 ; i<=n ; i++ )
  {
      k3[i]= h*k3[i];
      yout[i]= y[i] + ( 1932.0*k1[i]-7200.0*k2[i]+7296.0*k3[i] )/2197.0;
  }

  xh= x + 12.0*h/13.0;
  f( n, xh, yout, k4 );


  for( i=1 ; i<=n ; i++ )
  {
      k4[i]= h*k4[i];
      yout[i]= y[i] + 439.0*k1[i]/216.0 - 8.0*k2[i]
                    + 3680.0*k3[i]/513.0 - 845.0*k4[i]/4104.0;
  }

  xh= x + h;
  f( n, xh, yout, k5 );


  for( i=1 ; i<=n ; i++ )
  {
      k5[i]= h*k5[i];
      yout[i]= y[i] - 8.0*k1[i]/27.0 + 2.0*k2[i] - 3544.0*k3[i]/2565.0
                    + 1859.0*k4[i]/4104.0 - 11.0*k5[i]/40.0;
  }

  xh= x + h/2.0;
  f( n, xh, yout, k6 );

  for( i=1 ; i<=n ; i++ )
  {
      k6[i]= h*k6[i];
      yout[i]= y[i] + 16.0*k1[i]/135.0 + 6656.0*k3[i]/12825.0
                    + 28561.0*k4[i]/56430.0 - 9.0*k5[i]/50.0
                    + 2.0*k6[i]/55.0;
  }

}




static void xRK4( int n, float y[], float dy[], float x, float h,
                  float yout[], float k1[], float k2[], float k3[],
                  float k4[], float k5[], float k6[] )
{
int i;
float xh;

  for( i=1 ; i<=n ; i++ )
  {
      yout[i]= y[i] + 25.0*k1[i]/216.0 + 1408.0*k3[i]/2565.0
                    + 2197.0*k4[i]/4104.0 - k5[i]/5.0;
  }
}

