/*****************************************************************************
 *                                  S T I F F
 *****************************************************************************
 *
 *   PROGRAM ID:        STIFF.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is an ODE solver for stiff systems of equations.  It is
 *      mainly due to an article in Computers in Physics, May/Jun 1991,
 *      p.89.  The article was by two of the authors of Numerical
 *      Recipes....Their routine was based on the following excellent
 *      article by LF Shampine:
 *
 *           ACM TOMS, vol 8 no 2, (June 1982), p. 93
 *
 *      I have adapted it to look very much like the Runge-Kutta solvers.
 *      I also eliminated the need (ability?) for the calling application
 *      to supply a subroutine to calculate the Jacobian -- finite
 *      differencing is used instead.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>



/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static float **df_dy;
static float *df_dx;
static float **a;
static float **g;
static int   *indx;


static void xJacobi( int n, float x, float y[], float dy_dx[],
                     float dysav[], float *df_dy[], float df_dx[] );

static void xODEStiff( int n, float ystart[], float y[], float yscal[],
                       float dy_dx[], float x1, float x2, float eps,
                       float h, float *w[] );

static void xKRStep( int n, float y[], float dy_dx[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float err[], float ysav[],
                     float dysav[] );



/* Here's the entry point for this module */
void stiff( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (6*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
int i;
float **w;

  w= matrix( 1, 6, 1, n );
  df_dy= matrix( 1, n, 1, n );
  a= matrix( 1, n, 1, n );
  g= matrix( 1, 4, 1, n );
  df_dx= vector( 1, n );
  indx= ivector( 1, n );

  f= fxn;
  gkmax= abs(nPts) +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
      gszDataFile[i]= szFile[i];

  xODEStiff( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );

  free_matrix( w, 1, 6, 1, n );
  free_matrix( df_dy, 1, n, 1, n );
  free_matrix( a, 1, n, 1, n );
  free_matrix( g, 1, 4, 1, n );
  free_vector( df_dx, 1, n );
  free_ivector( indx, 1, n );
}



static void xODEStiff( int n, float ystart[], float y[], float yscal[],
                   float dy_dx[], float x1, float x2, float eps,
                   float h, float *w[] )
{
/*
   ystart is the vector of dependent variables, on input it
         contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h1 is the estimate of the step size
   hmin is the minimum allowed step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy_dx)  where the derivatives are returned in dy_dx.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/
const int maxstp= 10000;
const float tiny= 1.e-30;

float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i, nstp;
int bDone= 0;
FILE *fp= NULL;

  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );


  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      enStep= nstp;  // signal next step to external process
      f( n, x, y, dy_dx);

      for( i=1 ; i<=n ; i++ )
      {
          // yscal[i]= fabs(y[i]) +fabs(h*dy_dx[i]) +tiny;   // non-stiff scaling
          yscal[i]= max( fabs(y[i]), 1.0 );                // stiff scaling
      }

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }

      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )    h= x2-x;
      xKRStep( n, y, dy_dx, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
              ystart[i]= y[i];

          if( fp && gkmax )
          {
              kount= kount + 1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
          }
          if( fp )
              fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Stepsize underflow in STIFF()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Maximum iterations in STIFF()" );
  }
}





static void xKRStep( int n, float y[], float dy_dx[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float err[], float ysav[],
                    float dysav[] )
{
const float PGROW=-0.25, GROW=1.5, PSHRNK=-0.333, SHRNK=0.5,
            SAFETY=0.9,  ERRCON=0.1296;
const int   MAXTRY= 40;

int bDone= 0;
int i, j, jtry;
float det, errmax, errtemp;
float xsav= x[0];
float h= htry;

/*
THESE ARE THE KAPS-RENTROP PARAMETERS. THEY ARE NOT RATIONAL LIKE
THOSE OF SHAMPINE WHICH I'LL USE

const float GAM=0.231, A21=2.0, A31=4.52470820736, A32=4.1635287886,
            C21=-5.07167533877, C31=6.02015272865, C32=0.159750684673,
            C41=-1.856343618677, C42=-8.50538085819, C43=-2.08407513602,
            B1=3.95750374663, B2=4.62489238836, B3=0.617477263873,
            B4=1.282612945268, E1=-2.30215540292, E2=-3.07363448539,
            E3=0.873280801802, E4=1.282612945268, C1X=0.231,
            C2X=-0.39629667752E-1, C3X=0.550778939579, C4X=-0.5535098457E-1,
            A2X=0.462, A3X=0.880208333333;
*/
const float GAM= 1./2,

            A21= 2.0,       A31= 48./25,   A32= 6./25,

            C21= -8.0,      C31= 372./25,  C32= 12./5,

            C41= -112./125, C42= -54./125, C43= -2./5,

            B1= 19./9,      B2= 1./2,      B3= 25./108,  B4= 125./108,

            E1= 17./54,     E2= 7./36,     E3= 0.0,      E4= 125./108,

            C1X= 1./2,      C2X= -3./2,    C3X= 121./50, C4X= 29./250,

            A2X= 1.0,       A3X= 3./5;



  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy_dx[i];
  }

  xJacobi( n, x[0], y, dy_dx, dysav, df_dy, df_dx );

  for( jtry=1 ; jtry<=MAXTRY && !bDone ; jtry++ )
  {
      for( i=1 ; i<=n ; i++ )
      {
          for( j=1 ; j<=n ; j++ )
              a[i][j]= -df_dy[i][j];
          a[i][i] += 1.0 / (h*GAM);
      }
      ludcmp( n, a, indx, &det );


      for( i=1 ; i<=n ; i++ )
          g[1][i]= dysav[i] + h*C1X*df_dx[i];
      lubksb( n, a, indx, g[1] );


      for( i=1 ; i<=n ; i++ )
          y[i]= ysav[i] + A21*g[1][i];
      x[0]= xsav + h*A2X;
      f( n, x[0], y, dy_dx );


      for( i=1 ; i<=n ; i++ )
          g[2][i]= dy_dx[i] + h*C2X*df_dx[i] + C21*g[1][i]/h;
      lubksb( n, a, indx, g[2] );


      for( i=1 ; i<=n ; i++ )
          y[i]= ysav[i] + A31*g[1][i] + A32*g[2][i];
      x[0]= xsav + h*A3X;
      f( n, x[0], y, dy_dx );


      for( i=1 ; i<=n ; i++ )
          g[3][i]= dy_dx[i] + h*C3X*df_dx[i] + ( C31*g[1][i]+C32*g[2][i] )/h;
      lubksb( n, a, indx, g[3] );


      for( i=1 ; i<=n ; i++ )
          g[4][i]= dy_dx[i] + h*C4X*df_dx[i]
                            + ( C41*g[1][i]+C42*g[2][i]+C43*g[3][i] )/h;
      lubksb( n, a, indx, g[4] );


      for( i=1 ; i<=n ; i++ )
      {
          y[i]= ysav[i] + B1*g[1][i] + B2*g[2][i] + B3*g[3][i] + B4*g[4][i];
          err[i]= E1*g[1][i] + E2*g[2][i] + E3*g[3][i] + E4*g[4][i];
      }
      x[0]= xsav + h;


      if( x[0]==xsav )
          nrerror( "Stepsize underflow in STIFF()" );



      for( errmax=0.0, i=1 ; i<=n ; i++ )
      {
          errtemp= fabs( err[i] / yscal[i] );
          errmax= ( errtemp>errmax ) ? errtemp : errmax;
      }
      errmax /= eps;


      if( errmax <= 1.0 )
      {
          hdid[0]= h;
          if( errmax >= ERRCON )
              hnext[0]= SAFETY * h * pow( errmax, PGROW );
          else
              hnext[0]= h * GROW;
          bDone= 1;
      }
      else
      {
          hnext[0]= SAFETY * h * pow( errmax, PSHRNK );

          if( hnext[0] > h*SHRNK )
              hnext[0]= h*SHRNK;
          h= hnext[0];
      }
  }

  if( !bDone )
      nrerror( "Maximum iterations in STIFF()" );
}




static void xJacobi( int n, float x, float y[],
                    float dy_dx[], float dysav[], float *df_dy[], float df_dx[] )
{
/*
   If dy/dx= f(x,y), then we define the return values of
   the Jacobian by

      1)  df_dy[i][j]= df[i] / dy[j]
      2)  df_dx[j]= df[j] / dx
*/

int i, j;
float h;
float eps= G_EPS;

  for( j=1 ; j<=n ; j++ )
  {
      h= fabs( y[j]*eps );
      if( h<eps )   h= eps;
      y[j] += h;
      f( n, x, y, dy_dx );
      for( i=1 ; i<=n ; i++ )
          df_dy[i][j]= ( dy_dx[i]-dysav[i] )/h;
      y[j] -= h;


      h= fabs( x*eps );
      if( h<eps )   h= eps;
      x += h;
      f( n, x, y, dy_dx );
      df_dx[j]= ( dy_dx[j]-dysav[j] )/h;
      x -= h;
  }

}
