/*****************************************************************************
 *                                  T A I N D O U B
 *****************************************************************************
 *
 *   PROGRAM ID:        TAINDOUB.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      And here is a stiff ODE solver from H Tain-Min, BIT (1983), p.118.
 *      Unlike other stiff methods, this one uses no Jacobian!! Instead
 *      he uses a basic trapezoidal scheme plus iteration. The article is
 *      rather vague about how the iteration parameters are controlled, so
 *      the routine is my best shot at it.
 *
 *      This routine incorporates a standard one-step & two-half-step
 *      stepsize scheme.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>



/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void xODE23( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h1, float *w[] );

static void xRKStep( int n, float y[], float dy[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float ytemp[], float ysav[],
                     float dysav[], float k0[], float kn[], float ktemp[] );




/* Here's the entry point for this module */
void tainmin( int n, DERIV fxn, float ystart[], float x1, float x2,
              float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (9*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
int i;
float **w;

  w= matrix( 1, 9, 1, n );
  f= fxn;
  gkmax= nPts +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
      gszDataFile[i]= szFile[i];

  xODE23( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
  free_matrix( w, 1, 9, 1, n );
}


static void xODE23( int n, float ystart[], float y[], float yscal[],
                   float dy[], float x1, float x2, float eps,
                   float h, float *w[] )
{
/*
   ystart is the vector of dependent variables, on input it
         contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h is the estimate of the step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy)  where the derivatives are returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/
const unsigned long maxstp= 123456;
unsigned long nstp;
const float tiny= 1.e-30;

/****/float xlast= -1;
float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i;
int bDone= 0;
FILE *fp= NULL;

  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );



  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      enStep= nstp;  // signal next step to external process
      f( n, x, y, dy);

      for( i=1 ; i<=n ; i++ )
          //yscal[i]= fabs(y[i]) +fabs(h*dy_dx[i]) +tiny;   // non-stiff scaling
          yscal[i]= max( fabs(y[i]), 1.0 );                // stiff scaling

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }

      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )
          h= x2-x;

      xRKStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3], w[4], w[5], w[6] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
              ystart[i]= y[i];

          if( fp && gkmax )
          {
              kount= kount + 1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
          }
          if( fp )
             fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Stepsize underflow in Taindoub()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Maximum iterations in Taindoub()" );
  }
}



static void xRKStep( int n, float y[], float dy[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float ytemp[], float ysav[],
                    float dysav[], float k0[], float kn[], float ytemp2[] )
{
const float pgrow= -0.333;
const float pshrnk= -0.5;
const float safety= 0.9;
const float errcon= 8.0e-3;
//    const float e= G_EPS;
const float e= 0.05;

int bDone= 0, bInnerDone;
int loops, maxloops= 500;
int i;
float htemp, h= htry;
float xtemp, xsav= x[0];
float errmax, ktemp;

/*********/printf( "Start xrkstep with h= %e \n", h );

  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy[i];
  }


  while( !bDone )
  {
      float temp;

      // Use a forward step....
      x[0]= xsav +h;
      if( x[0]==xsav )
          nrerror( "Stepsize underflow in Taindoub()" );
      bInnerDone= 0;
      for( i=1 ; i<=n ; i++ )
          k0[i]= kn[i]= h*dysav[i];

      loops= 0;
      while( !bInnerDone  && loops<maxloops )
      {
          for( i=1 ; i<=n ; i++ )
              y[i]= ysav[i] +( k0[i]+kn[i] )/2.0;
          f( n, x[0], y, dy );

          errmax= 0.0;
          for( i=1 ; i<=n ; i++ )
          {
              ktemp= kn[i];
              kn[i]= (e*h*dy[i] +( k0[i]+kn[i] )/2 ) / (1+e);

              errmax += fabs( (kn[i]-ktemp)/kn[i] );
          }
          errmax /= n;

//*********/printf( "kn= (%e, %e, %e) \n", kn[1], kn[2], kn[3] );
//*********/printf( "kn-1= (%e, %e, %e),  eps=%e \n", ktemp[1], ktemp[2], ktemp[3], eps );
//*********/printf( "y= (%e, %e, %e) \n", y[1], y[2], y[3] );
//*********/printf( "dy= (%e, %e, %e) \n\n", dy[1], dy[2], dy[3] );


          if( errmax < 1e-7 )
          //if( errmax < G_EPS )
          //if( errmax < eps )
              bInnerDone= 1;
          else
              loops++;
      }

      // ....and a double step.
      htemp= h/2;
      xtemp= xsav;

      // #1
      xtemp += htemp;
      bInnerDone= 0;
      for( i=1 ; i<=n ; i++ )
          k0[i]= kn[i]= htemp*dysav[i];

      loops= 0;
      while( !bInnerDone  && loops<maxloops )
      {
          for( i=1 ; i<=n ; i++ )
              ytemp[i]= ysav[i] +( k0[i]+kn[i] )/2.0;
          f( n, xtemp, ytemp, dy );

          errmax= 0.0;
          for( i=1 ; i<=n ; i++ )
          {
              ktemp= kn[i];
              kn[i]= (e*htemp*dy[i] +( k0[i]+kn[i] )/2 ) / (1+e);

              errmax += fabs( (kn[i]-ktemp)/kn[i] );
          }
          errmax /= n;


          if( errmax < 1e-7 )
              bInnerDone= 1;
          else
              loops++;
      }

      // #2
      xtemp += htemp;
      bInnerDone= 0;
      for( i=1 ; i<=n ; i++ )
          k0[i]= kn[i]= htemp*dy[i];

      loops= 0;
      while( !bInnerDone  && loops<maxloops )
      {
          for( i=1 ; i<=n ; i++ )
              ytemp2[i]= ytemp[i] +( k0[i]+kn[i] )/2.0;
          f( n, xtemp, ytemp2, dy );

          errmax= 0.0;
          for( i=1 ; i<=n ; i++ )
          {
              ktemp= kn[i];
              kn[i]= (e*htemp*dy[i] +( k0[i]+kn[i] )/2 ) / (1+e);

              errmax += fabs( (kn[i]-ktemp)/kn[i] );
          }
          errmax /= n;


          if( errmax < 1e-7 )
              bInnerDone= 1;
          else
              loops++;
      }

      // We now have the estimate for the full step in y[],
      // and the estimate to the double step in ytemp2[].

      // Error control is handled by comparing ysav to ytemp.

      // Put error estimate ---> ytemp */
      errmax= 0.0;
      for( i=1 ; i<=n ; i++ )
      {
          ytemp[i]= ytemp2[i] -y[i];
          temp= fabs( ytemp[i] / yscal[i] );
          errmax= (errmax > temp)? errmax : temp;

          // form 4th order approx
          y[i]= ytemp2[i] +ytemp[i]/3;
      }
      errmax= errmax / eps;


      // Try new step-size if we didn't meet eps
      if( errmax>1 )
          h= safety * h * pow(errmax,pshrnk);
      else
          bDone= 1;
  }

  hdid[0]= h;

  hnext[0]= safety * h * pow(errmax,pgrow);
  if( hnext[0] > 2*h )
      hnext[0]= 2*h;

/*********/printf( "Exit xrkstep with h= %e \n\n", h );
}

