/*****************************************************************************
 *                                  U S M A N I
 *****************************************************************************
 *
 *   PROGRAM ID:        USMANI.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      Here is a stiff ODE solver due to:
 *          RA Usmani & RP Agarwal, Comp and Math with Appls (1985), p.1183
 *      as described in:
 *          DM Kondrat & DIB Jacques, Internat J Comp Math (1992), p.117
 *
 *      This method is A-stable, 3rd order, 1-step. It is actually 2-step, but
 *      uses an interation scheme like Tain-Min's.
 *
 *      In addition, Kondrat & Jacques present a method that is A-stable, 4th
 *      order, 2-step (3-step with iteration).
 *
 *      This routine incorporates a standard one-step & two-half-step
 *      stepsize scheme.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>



/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void xODE23( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h1, float *w[] );

static void xRKStep( int n, float y[], float dy[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float ynext[], float ysav[],
                    float dysav[], float dynext[], float dylast[], float ylast[] );




/* Here's the entry point for this module */
void usmani( int n, DERIV fxn, float ystart[], float x1, float x2,
             float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (9*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
int i;
float **w;

  w= matrix( 1, 9, 1, n );
  f= fxn;
  gkmax= nPts +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
      gszDataFile[i]= szFile[i];

  xODE23( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
  free_matrix( w, 1, 9, 1, n );
}


static void xODE23( int n, float ystart[], float y[], float yscal[],
                   float dy[], float x1, float x2, float eps,
                   float h, float *w[] )
{
/*
   ystart is the vector of dependent variables, on input it
         contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h is the estimate of the step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy)  where the derivatives are returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/
const unsigned long maxstp= 123456;
unsigned long nstp;
const float tiny= 1.e-30;

/****/float xlast= -1;
float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i;
int bDone= 0;
FILE *fp= NULL;

  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );



  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      enStep= nstp;  /* signal next step to external process */
      f( n, x, y, dy);

      for( i=1 ; i<=n ; i++ )
          //yscal[i]= fabs(y[i]) +fabs(h*dy_dx[i]) +tiny;   // non-stiff scaling
          yscal[i]= max( fabs(y[i]), 1.0 );                // stiff scaling

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }

      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )
          h= x2-x;

      xRKStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3], w[4], w[5], w[6] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
              ystart[i]= y[i];

          if( fp && gkmax )
          {
              kount= kount + 1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
          }
          if( fp )
             fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Stepsize underflow in usmani()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Maximum iterations in usmani()" );
  }
}



static void xRKStep( int n, float y[], float dy[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float ynext[], float ysav[],
                    float dysav[], float dynext[], float dylast[], float ylast[] )
{
const float pgrow= -0.25;
const float pshrnk= -0.333;
const float safety= 0.9;

int bDone= 0, bInnerDone;
int loops, maxloops= 500;
int i;
float htemp, h= htry;
float xtemp, xsav= x[0];
float errmax, temp;


  printf( "Start xrkstep with h= %e \n", h ); // *********
  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy[i];
  }


  while( !bDone )
  {

  //  Use a double step....
  //        htemp= h/2;
  //  Use a full forward step....
      htemp= h;
      x[0]= xsav;

    // #1
      bInnerDone= 0;
      x[0] += htemp;
      for( i=1 ; i<=n ; i++ )
      {
          ylast[i]= ysav[i];
          dylast[i]= dysav[i];

          y[i]= ylast[i] +htemp*dylast[i]; // Use Euler as 1st guess
      }

      loops= 0;
      while( !bInnerDone  && loops<maxloops )
      {
          f( n, x[0], y, dy );

          for( i=1 ; i<=n ; i++ )
              ynext[i]= 5*ylast[i] -4*y[i] +htemp*( 2*dylast[i]+4*dy[i] );
          f( n, x[0]+htemp, ynext, dynext );

          errmax= 0.0;
          for( i=1 ; i<=n ; i++ )
          {
              temp= y[i];
              y[i]= ylast[i] +htemp/12*( 5*dylast[i]+8*dy[i]-dynext[i] );

              errmax += fabs( 1 - y[i]/temp );
          }
          errmax /= n;


          if( errmax < 1e-7 )
              bInnerDone= 1;
          else
              loops++;
      }

  printf( "\tForward step OK\n" ); // *********

/*
  printf( "\tFirst half step OK\n" ); // *********
    // #2
      bInnerDone= 0;
      x[0] += htemp;
      for( i=1 ; i<=n ; i++ )
      {
          ylast[i]= y[i];
          dylast[i]= dy[i];

          y[i]= ynext[i];  // use extrapolation from last step
      }

      loops= 0;
      while( !bInnerDone  && loops<maxloops )
      {
          f( n, x[0], y, dy );

          for( i=1 ; i<=n ; i++ )
              ynext[i]= 5*ylast[i] -4*y[i] +htemp*( 2*dylast[i]+4*dy[i] );
          f( n, x[0]+htemp, ynext, dynext );

          errmax= 0.0;
          for( i=1 ; i<=n ; i++ )
          {
              temp= y[i];
              y[i]= ylast[i] +htemp/12*( 5*dylast[i]+8*dy[i]-dynext[i] );

              errmax += fabs( 1 - y[i]/temp );
          }
          errmax /= n;


          if( errmax < 1e-7 )
              bInnerDone= 1;
          else
              loops++;
      }

  printf( "\tSecond half step OK\n" ); // *********
// ....and a single step --Backwards!
*/

// ....and a Backward step.
      htemp= xsav -x[0];
      bInnerDone= 0;
      x[0] += htemp;
      for( i=1 ; i<=n ; i++ )
      {
          ylast[i]= y[i];
          dylast[i]= dy[i];

          y[i]= ylast[i] +htemp*dylast[i]; // Use Euler as 1st guess
      }

      loops= 0;
      while( !bInnerDone  && loops<maxloops )
      {
          f( n, x[0], y, dy );

          for( i=1 ; i<=n ; i++ )
              ynext[i]= 5*ylast[i] -4*y[i] +htemp*( 2*dylast[i]+4*dy[i] );
          f( n, x[0]+htemp, ynext, dynext );

          errmax= 0.0;
          for( i=1 ; i<=n ; i++ )
          {
              temp= y[i];
              y[i]= ylast[i] +htemp/12*( 5*dylast[i]+8*dy[i]-dynext[i] );

              errmax += fabs( 1 - y[i]/temp );
          }
          errmax /= n;


          if( errmax < 1e-7 )
              bInnerDone= 1;
          else
              loops++;
      }

  printf( "\tFull reverse step OK\n" ); // *********

   // We now have the estimate for the double step in ylast[].

   // Error control is estimated by the difference between ysav[] & y[].
      errmax= 0.0;
      for( i=1 ; i<=n ; i++ )
      {
          temp= ysav[i] -y[i];
          temp= fabs( temp / yscal[i] );
          errmax= (errmax > temp)? errmax : temp;

          // put our guess into y[] for return to caller
          y[i]= ylast[i];
      }
      errmax= errmax / eps;


   // Try new step-size if we didn't meet eps
      if( errmax>1 )
          h= safety * h * pow(errmax,pshrnk);
      else
          bDone= 1;

      if( xsav == xsav+h )
          nrerror( "Stepsize underflow in usmani()" );
  }

  hdid[0]= h;

  if( errmax < 1e-30 )
      hnext[0]= 4*h;
  else
      hnext[0]= safety * h * pow(errmax,pgrow);

  if( hnext[0] > 4*h )
      hnext[0]= 4*h;

  printf( "Exit xrkstep with h= %e \n\n", h ); // *********
}

