/*****************************************************************************
 *                                  B A N D B
 *****************************************************************************
 *
 *   PROGRAM ID:        BANDB.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is the "nearly" Conjugate Gradient method of Borwein & Barzilai,
 *      IMA Journal of Numerical Analysis (1988), p. 141. It was not designed
 *      as a CG method, but as an improved version of the (classical) gradient
 *      method.
 *
 *      Its strength is that is uses no line search at all!!  What's more,
 *      it is very stingy on storage.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

// locally global variables
static OPTFXN F;
static GRADF grad;
static int nLoops;

// local functions and macros
#define TINY   1.0e-10
static void opt( int n, float *x, float tol, float *g, float *dg, float *dx );
static void gradF( int n, float *x, float F0, float *g );



/*
    ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int bandb( int n, float *x, OPTFXN FXN, GRADF DFXN, float tol )
/*
   This subroutine is just a dummy to allocate the (3*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
float **w;

  w= matrix( 1, 3, 1, n );
  F= FXN;
  if( DFXN )
      grad= DFXN;
  else
      grad= gradF;

  if( tol <= 0.0 )  tol= 1.0e-6;
  opt( n, x, tol, w[1], w[2], w[3] );
  free_matrix( w, 1, 3, 1, n );

  return( nLoops );
}





static void opt( int n, float *x, float tol, float *g, float *dg, float *dx )
/*
----------------------VARIABLE KEY-------------------------
   n          length of independent (vector) variable

   x          independent variable-- on input it contains
              the initial guess;  on output, the solution

   F          objective function (to be supplied by user!!)

   F0,F1      stores previous values of F

   g          stores value of gradF call

   dg         change in g for two most recent iterations

   b          scale factor applied to g

   dx         change in x for two most recent iterations
              for Barzilia & Barwein it is always = -b*g

---------------------------------------------------------------------
*/
{
int i, nLoopMax;
int bConverged;
float F0, F1, b, gg, dxdg, dgdg, dxdx, gdg, gabs;

// initializations
  nLoopMax= max( 4*n, 1000 );

  b= 1;
  F0= F( n, x );
  F1= F0;
  grad( n, x, F0, g );

  for( gg= 0.0, i=1 ; i<=n ; i++ )
  {
      dg[i]= g[i];
      gg += g[i]*g[i];
  }
  bConverged= 0;
  if( gg<= TINY )   bConverged= 2;



// main loop
  for( nLoops=1 ; bConverged<2 && nLoops<=nLoopMax ; nLoops++ )
  {
      printf( "%5d  F=%16.8e\r", nLoops, F0 );

      gabs= -sqrt( 1+gg);
      for( i=1 ; i<=n ; i++ )
      {
           dx[i]= b*g[i]/gabs;
           x[i] += dx[i];
      }
      F0= F( n, x );


///****/ printf( "bandb: F= %e\n", F0 );
///****/ printf( "\t  g= (%e %e)\n", g[1], g[2] );
///****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );

      // Exit test(s)
      if( 2*fabs(F0-F1) <= tol*(fabs(F0)+fabs(F1)+TINY) )
          bConverged++;
      else if( 2*fabs(F0-F1) <= tol*(2+fabs(F0)+fabs(F1)+TINY) )
          bConverged++;
      else if( gg<= TINY )
          bConverged= 2;
      else
      {
          bConverged= 0;

// oh well, keep going
          F1= F0;
          grad( n, x, F0, g );

// calculate dot products
          gg= gdg= dgdg= dxdx= dxdg= 0;
          for( i=1 ; i<=n ; i++ )
          {
               dg[i]= g[i] -dg[i];  // recall dg contains last g

               gg   += g[i]*g[i];
               gdg  += g[i]*dg[i];
               dgdg += dg[i]*dg[i];
               dxdx += dx[i]*dx[i];
               dxdg += dx[i]*dg[i];
          }

// Here are the Borwein-Barzilai betas, apparently they are equally good
          b= dxdg / dgdg;
//            b= dxdx / dxdg;


// Dump current g --> dg
          for( i=1 ; i<=n ; i++ )
               dg[i]= g[i];
      }

  }

  if( bConverged < 2 )
      nLoops= -nLoops;
}





static void gradF( int n, float *x, float F0, float *g )
{
int i;
float a, t, e= G_EPS;

  for ( i=1 ; i<=n ; i++ )
  {
      a= fabs( x[i] );
      if ( a< 1.0 )  a= 1.0;
      a= a*e;
      t= x[i];
      x[i]= t+a;
      g[i]= ( F(n,x) -F0 ) / a;
      x[i]= t;
  }
}
