/*****************************************************************************
 *                                  B F G S
 *****************************************************************************
 *
 *   PROGRAM ID:        BFGS.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is the Broyden-Fletcher-Goldfarb-Shanno quasi-newton
 *      optimization method.  It was discovered by all four authors
 *      more or less simultaneously in 1970.  It is recognized as
 *      the best of the multivariable optimization methods.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

// locally global variables
static OPTFXN F;
static GRADF grad;
static int nLoops;


// local functions and macros
#define TINY   1.0e-10
static void opt( int n, float x[], float tol, float g[], float s[],
                 float dg[], float Hdg[], float Hg[], float *H[] );
static void gradF( int n, float x[], float F0, float g[] );



/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int bfgs( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol )
/*
   This subroutine is just a dummy to allocate the n(n+5)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
float **w;

  w= matrix( 1, n+5, 1, n );
  F= FXN;
  if( DFXN )
      grad= DFXN;
  else
      grad= gradF;

  if ( tol <= 0.0 )
       tol= 1.0e-6;

  opt( n, x, tol, w[1], w[2], w[3], w[4], w[5], &w[5] );
  free_matrix( w, 1, n+5, 1, n );

  return( nLoops );
}




static void opt( int n, float x[], float tol, float g[], float s[],
                 float dg[], float Hdg[], float Hg[], float *H[] )

/* ----------------------VARIABLE KEY-----------------------------
   n          length of independent (vector) variable

   x          independent variable-- on input it contains
              the initial guess;  on output, the solution

   F          objective function (to be supplied by user!!)

   F0..F3     stores previous values of F

   g          stores value of gradF call

   s          the search direction

   dg         the change in g between steps

   t          scale factor applied to s

   H          "iteration matrix" --approx inverse of the Jacobian

   Hg, Hdg    the vector dot products   H.g, H.dg

   sg, sdg,   the scalar dot products   s.g, s.dg,
   dgHdg,                               dg.H.dg, gHdg
   gHdg

------------------------------------------------------------------
*/
{
int i, k, nMain, bRestart, bConverged;
float F0, F1, F2, F3, ss, sg, sdg, dgHdg, gHdg;


// initializations
  F1= 0.0;
  F0= F( n, x );
  grad( n, x, F0, g );
  nLoops= 0;
  nMain= 0;
  bRestart= 1;
  bConverged= 0;


// main loop
  while (  !bConverged && ( nMain<5 ) )
  {
      printf( "%5d  F=%16.8e\r", nLoops, F0 );

      if ( bRestart )
      {
          bRestart= 0;
          nMain++;
          ss= 0.0;
          for ( k=1 ; k<=n ; k++ )
          {
              for ( i=k ; i<=n ; i++ )
                  H[i][k]= H[k][i]= 0.0;
              H[k][k]= 1.0;
              s[k]= -g[k];
              dg[k]= g[k];
              ss += s[k]*s[k];
          }
      }


// These next couple of lines do a lot of crunching!
///****/ printf( "bfgs: F= %e\n", F0 );
///****/ printf( "\t  g= (%e %e)\n", g[1], g[2] );
///****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );
///****/ printf( "\t  s= (%e %e)\n", s[1], s[2] );
///****/ printf( "\t dg= (%e %e)\n", dg[1], dg[2] );
      F1= F0;
//        linmin2( n, x, F, grad, s, &F0 );
      linmin( n, x, F, s, &F0 );
      grad( n, x, F0, g );
      for ( i=1 ; i<=n ; i++ )
          dg[i]= g[i] -dg[i];



      if ( fabs(F1-F0) <= 0.5*tol*(fabs(F1)+fabs(F0)+TINY) )
          bConverged= 1;
      else
      {

// calculate dot products
      sg= sdg= gHdg= dgHdg= 0.0;
      for ( i=1 ; i<=n ; i++ )
      {
          Hg[i] = 0.0;
          Hdg[i]= 0.0;
          for ( k=1 ; k<=n ; k++ )
          {
              Hg[i]  += H[k][i]*g[k];
              Hdg[i] += H[k][i]*dg[k];
          }
          sg  += s[i]*g[i];
          sdg += s[i]*dg[i];
          gHdg  += g[i]*Hdg[i];
          dgHdg += dg[i]*Hdg[i];
      }
///****/ printf( "\t Hg= (%e %e)\n", Hg[1], Hg[2] );
///****/ printf( "\t Hdg= (%e %e)\n", Hdg[1], Hdg[2] );
///****/ printf( "\t  sg= %e, sdg= %e, gHdg= %e, dgHdg= %e\n", sg,sdg,gHdg,dgHdg );
///****/ printf( "\t H= (%e %e)\n\t    (%e %e)\n", H[1][1], H[1][2],H[2][1],H[2][2]  );


//   update H and calc new s utilizing the symmetry of H and
//   the long form of H-update to avoid extra loop for s= -H.g

      for ( i=1 ; i<=n ; i++ )
      {
          for ( k=i ; k<=n ; k++ )
          {
              H[k][i] += ( 1.0+dgHdg/sdg ) *s[i]*s[k] /sdg
                       - ( s[i]*Hdg[k]+Hdg[i]*s[k] ) /sdg;
              H[i][k]= H[k][i];
          }
          s[i]= -Hg[i] -(1.0+dgHdg/sdg)*s[i]*sg/sdg
                 +( s[i]*gHdg+Hdg[i]*sg )/sdg;
      }

      for ( i=1 ; i<=n ; i++ )
          dg[i]= g[i];

      nLoops++;
      if ( nLoops%(20*n*n) == 0 )   bRestart= 1;
      }
  }


  if( !bConverged )
      nLoops= -nLoops;
}







static void gradF( int n, float x[], float F0, float g[] )
{
int i;
float a, t, e= G_EPS;

  for ( i=1 ; i<=n ; i++ )
  {
      a= fabs( x[i] );
      if ( a< 1.0 )  a= 1.0;
      a= a*e;
      t= x[i];
      x[i]= t+a;
      g[i]= ( F(n,x) -F0 ) / a;
      x[i]= t;
  }
}

