/*****************************************************************************
 *                                  B R A N I N
 *****************************************************************************
 *
 *   PROGRAM ID:        BRANIN.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      The following is a homotopy method based on the soln of an ODE.
 *      The ODE is essentially one suggested by FH Brannin in 1972:
 *
 *                dx/dt =  -g
 *
 *
 *      This is basically a continuous version of Cauchy's gradient
 *      method.  My contribution is to add a small inertial term to
 *      the equation:
 *
 *                dx/dt =  -g + b.xlast
 *
 *
 *      Where b < 1, is a small constant that controls how much inertia
 *      the equations carry from step to step.  What we then have is
 *      a continous version of the Conjugate Gradient method.
 *
 *      The x-vector is of length 2n+1.  x[0] contains the value for b.
 *      x[1] thru x[n] contain the initial guess at the soln.
 *      x[n+1] thru x[n+n] contain the initial velocity.
 *
 *      On exit the velocity is zero, and x[1] thru x[n] is the soln.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

// locally global variables
static OPTFXN F;
static GRADF grad;
static int nLoops;
static float *dxo;  // alias, contains velocity info
static float b;
// extern unsigned long enStep;  // communicates with Runge-Kutta routine


// local functions and macros
#define TINY   1.0e-10
static void opt( int n, float x[], float tol, float g[], float s[],
                 float dg[], float Hdg[], float Hg[], float dx[], float *H[] );
static void gradF( int n, float x[], float F0, float g[] );
static void derivs( int n, float t, float x[], float dx[] );
static int Comp( int n, float v[], float vo[], float tol );
static int Diff( int n, float v[], float dv[], float tol );



/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int branin( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol )
{
int   k, nOldStepCnt, bConverged= 0;
float Fo, Fn,          // Used to test for convergence
      tstart= 0.0,
      tend= 0.5,       // This parameter controls convergence;
                       //   just keep taking it halfway to 1.0 -- due to reparameterization
                       //   t=1.0 represents 100% accuracy and infinite time!

      etol= 1e-4,      // a terminal value problem requires sloppy tolerance
      h= 0.0001;       // this is only a guess right now


  dxo= &x[n+1]; // extract initial velocity into
  b= x[0];      // extract inertia parameter

  F= FXN;
  if( DFXN )
      grad= DFXN;
  else
      grad= gradF;


  Fn= F(n,x);
  enStep= 0;
  nLoops= 0;

  while( !bConverged )
  {
      printf( "%5d\r", nLoops );

      nLoops++;
      Fo= Fn;
      nOldStepCnt= enStep;


      rk23( 2*n, derivs, x, tstart, tend, etol, h, 0, 0 );
//      stiff( 2*n, derivs, x, tstart, tend, etol, h, 0, 0 );
//      rk45( n, derivs, x, tstart, tend, etol, h, 0, 0 );
      h= (tend-tstart) / (enStep-nOldStepCnt);
      tstart= tend;
      tend= (1+3*tend)/4; // go a quarter of the way to 1.0

///      tend= 0.1; // I also tried integrating from 0.0 --> 0.1 over and over
                    // it worked with Numerical Recipes' convergence criteria
                    // but took too many F-evals.

      Fn= F(n,x);

/****/printf( "Brannin conv Control: Fn=%e, Fo=%e, #Steps= %lu\n", Fn, Fo, enStep );

//      if( 2.0*fabs(Fn-Fo) <= tol*(fabs(Fn)+fabs(Fo))+TINY )// ||  gg < tol  )
//      if( Fn < tol )
      if( Comp(0,&Fn,&Fo,tol) )
          bConverged= 1;
  }




//  rk23( n, derivs, x, 0, tend, etol, h, 0, 0 );
//  rk45( n, derivs, x, 0, tend, etol, h, 0, 0 );

  return( nLoops );
}

static int Comp( int n, float v[], float vo[], float tol )
{
int i;
float mv, mvo, mdv;  // magnitudes (modulii)

  for( mv=mvo=mdv=0, i=1 ; i<=n ; i++ )
  {
      mv += v[i]*v[i];
      mvo += vo[i]*vo[i];
      mdv += (v[i]-vo[i]) * (v[i]-vo[i]);
  }

  if( n==0 ) // scalar flag
  {
      mv= v[0]*v[0];
      mvo= vo[0]*vo[0];
      mdv= (v[0]-vo[0]) * (v[0]-vo[0]);
  }
  mv= sqrt(mv);
  mvo=sqrt(mvo);
  mdv= sqrt(mdv);

  return( 2*mdv < (2+mv+mvo)*tol );
}

static int Diff( int n, float v[], float dv[], float tol )
{
int i;
float mv, mdv;  // magnitudes (modulii)

  for( mv=mdv=0, i=1 ; i<=n ; i++ )
  {
      mv += v[i]*v[i];
      mdv += dv[i]*dv[i];
  }

  if( n==0 ) // scalar flag
  {
      mv= v[0]*v[0];
      mdv= dv[0]*dv[0];
  }
  mv= sqrt(mv);
  mdv= sqrt(mdv);

  return( mdv < (1+mv)*tol );
}








static void gradF( int n, float x[], float F0, float g[] )
{
int i;
float a, t, e= G_EPS;

  for( i=1 ; i<=n ; i++ )
  {
      a= fabs( x[i] );
      if( a< 1.0 )
          a= 1.0;
      a= a*e;
      t= x[i];
      x[i]= t+a;
      g[i]= ( F(n,x) -F0 ) / a;
      x[i]= t;
  }
}


static void derivs( int n, float t, float x[], float dx[] )
{
float F0;
int   i;
static long nStep= -1;

  F0= F(n,x);
  gradF( n, x, F0, dx );

  for( i=1 ; i<=n ; i++ )
      dx[i]= b*dxo[i] -dx[i]/(1-2*t+t*t);

  if( nStep != enStep )   // enStep is an external used by the Runge-Kutta routine
  {                       // to communicate when it starts a new step.
      nStep= enStep;
      for( i=1 ; i<=n ; i++ )
          dxo[i]= dx[i];
  }
}
