/*****************************************************************************
 *                                  C G P R
 *****************************************************************************
 *
 *   PROGRAM ID:        CGPR.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is the the Polak-Ribeire variant of the Conjugate
 *      Gradient method.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

// locally global variables
static OPTFXN F;
static GRADF grad;
static int nLoops;


// local functions and macros
#define TINY   1.0e-10
static void opt( int n, float x[], float tol,
                 float s[], float g[], float ga[] );
static void gradF( int n, float x[], float F0, float g[] );





/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int cgpr( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol )
/*
   This subroutine is just a dummy to allocate the (3*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
float **w;

  w= matrix( 1, 3, 1, n );
  F= FXN;
  if( DFXN )
      grad= DFXN;
  else
      grad= gradF;


  if ( tol <= 0.0 )  tol= 1.0e-6;
  opt( n, x, tol, w[1], w[2], w[3] );
  free_matrix( w, 1, 3, 1, n );

  return( nLoops );
}








static void opt( int n, float x[], float tol,
                 float s[], float g[], float ga[] )
/*
----------------------VARIABLE KEY-------------------------
   n          length of independent (vector) variable

   x          independent variable-- on input it contains
              the initial guess;  on output, the solution

   F          objective function (to be supplied by user!!)

   F0,F1      stores previous values of F

   g          stores value of gradF call

   s          the search direction

   ga         g during last step

   b          s(new) is 1-part g and b-parts s(old)

---------------------------------------------------------------------
*/
{
int i, nLoopMax, bConverged;
float F0, F1, b, gg, gag, gaga, gabs, gconv;

// initializations
    nLoopMax= 4*n;
    if ( nLoopMax < 200 )    nLoopMax= 200;

    F0= F( n, x );
    F1= F0;
    grad( n, x, F0, g );

    gg= 0.0;
    for ( i=1 ; i<=n ; i++ )
    {
        ga[i]= g[i];
        gg += g[i]*g[i];
    }

    gabs= sqrt(1+gg);
    for ( i=1 ; i<=n ; i++ )
        s[i]= -g[i]/gabs;

    bConverged= 0;
    gconv= max(TINY,tol);
    if ( gg<= gconv )   bConverged= 2;



// main loop
    for ( nLoops=1 ; bConverged<2 && nLoops<=nLoopMax ; nLoops++ )
    {
        printf( "%5d  F=%16.8e\r", nLoops, F0 );

        //linmin4( n, x, F, g, s, &F0 );
        linmin( n, x, F, s, &F0 );

///****/ printf( "cgpr: F= %e\n", F0 );
///****/ printf( "\t  g= (%e %e)\n", g[1], g[2] );
///****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );

        // Exit test(s)
        if( 2*fabs(F0-F1) <= tol*(fabs(F0)+fabs(F1)+TINY) )
            bConverged++;
        else if( 2*fabs(F0-F1) <= tol*(2+fabs(F0)+fabs(F1)+TINY) )
            bConverged++;
        else if( gg<= TINY )
            bConverged= 2;
        else
        {
            bConverged= 0;

        //if ( //2.0*fabs(F0-F1) <= tol*(fabs(F0)+fabs(F1)+TINY) ||
        //     nLoops>n  ||  gg<=gconv  )
        //    bConverged= 1;
        //else
        //{

// oh well, keep going
            F1= F0;
            grad( n, x, F0, g );

// calculate dot products
            gaga= gg;
            gg=0.0;
            gag= 0.0;

            for ( i=1 ; i<=n ; i++ )
            {
                gg  += g[i]*g[i];
                gag += ga[i]*g[i];
            }

// Here's the Fletcher-Reeves beta
//          b= gg/gaga;

// Here's the Polak-Ribeire beta
            b= (gg-gag) / gaga;

// And here's a suggestion due to Powell
// to improve either method.
            b= max(b,0);

// calc new s and dump g---> ga
            for ( i=1 ; i<=n ; i++ )
            {
                s[i]= b*s[i] -g[i];
                ga[i]= g[i];
            }
        }
    }

    if ( nLoops >= nLoopMax )
        nLoops= -nLoops;
}





static void gradF( int n, float x[], float F0, float g[] )
{
int i;
float a, t, e= G_EPS;

    for ( i=1 ; i<=n ; i++ )
    {
        a= fabs( x[i] );
        if ( a< 1.0 )  a= 1.0;
        a= a*e;
        t= x[i];
        x[i]= t+a;
        g[i]= ( F(n,x) -F0 ) / a;
        x[i]= t;
    }
}



