/*****************************************************************************
 *                                  L I N M I N
 *****************************************************************************
 *
 *   PROGRAM ID:        LINMIN.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This routine does line searches, which are an integral part of most
 *      multi-dimensional optimizations.
 *
 *      This line search is from Numerical Recipes.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


/* locally global variables */
static OPTFXN F;


/* local macros & functions */
#define Func(t)        F1dim(n,x,s,t)
#define SHFT(a,b,c,d)  (a)=(b),(b)=(c),(c)=(d)
#define SIGN(a,b)      ((b)> 0.0 ? fabs(a) : -fabs(a))

static float F1dim( int n, float x[], float s[], float t )
{
      int k;
      float a;

      for ( k=1 ; k<=n ; k++ )
          x[k] += t*s[k];
      a= F( n, x );

      for ( k=1 ; k<=n ; k++ )
          x[k] -= t*s[k];

      return ( a );
}

static void MNBRAK( int n, float x[], float s[], float *ax, float *bx,
                    float *cx, float *Fa, float *Fb, float *Fc )
/*
   This subroutine brackets the values of x which minimize of
   Func(x).  On return, (ax,bx,cx) are values which bracket the
   minimum.

   Notice that I have created an artificial one dimensional
   function by using global function pointers and macros.
*/
{
    int   irc= 0;
    float dum, r, q, u, ulim, Fu;
    float gold= 1.618034,
          glimit= 100.0,
          tiny= 1.0e-20;


    Fa[0]= Func(ax[0]);
    Fb[0]= Func(bx[0]);

/* put in descending order Fa> Fb */
    if ( Fb[0]> Fa[0] )
    {
        SHFT(dum,ax[0],bx[0],dum);
        SHFT(dum,Fb[0],Fa[0],dum);
    }

/* first guess for cx */
    cx[0]= bx[0] +gold*(bx[0]-ax[0]);
    Fc[0]= Func(cx[0]);


/* main loop */
    while ( Fb[0] >= Fc[0] )
    {

/*
   parabolic extapolation of u from ax,bx,cx
   tiny's job is  to prevent division by zero
*/
        r= (bx[0]-ax[0])*(Fb[0]-Fc[0]);
        q= (bx[0]-cx[0])*(Fb[0]-Fa[0]);
        u= (bx[0])-((bx[0]-cx[0])*q-(bx[0]-ax[0])*r) /
            (2.0*SIGN(max(fabs(q-r),tiny),q-r));

/* ulim is limit on extrapolation step size */
        ulim= (bx[0]) +glimit*(cx[0]-bx[0]);

/* u is between bx and cx */
        if ( (bx[0]-u)*(u-cx[0])> 0.0 )
        {
            Fu= Func(u);
/* if it's a minimum we are done */
            if ( Fu< Fc[0] )
            {
                SHFT(ax[0],bx[0],u,u);
                SHFT(Fa[0],Fb[0],Fu,Fu);
                irc= 1;
            }
            else if ( Fu> Fb[0])
            {
/* ok, it's a minimum between ax and u; weez still done! */
                cx[0]= u;
                Fc[0]= Fu;
                irc= 1;
            }
            else
            {
/* so the parabolic fit was no use.  set u by default */
                u= (cx[0]) +gold*(cx[0]-bx[0]);
                Fu= Func(u);
            }

        }
        else if ( (cx[0]-u)*(u-ulim)> 0.0 )
        {
/* parabolic fit is between cx and allowed limit */
            Fu= Func(u);
            if ( Fu< Fc[0] )
            {
                SHFT(bx[0],cx[0],u,cx[0] +gold*(cx[0]-bx[0]));
                SHFT(Fb[0],Fc[0],Fu,Func(u));
            }
        }
        else if ( (u-ulim)*(ulim-cx[0])>= 0.0 )
        {
/* limit parabolic u to its max */
            u= ulim;
            Fu= Func(u);
        }
        else
        {
/* reject parabolic u, go with default */
            u= (cx[0]) +gold*(cx[0]-bx[0]);
            Fu= Func(u);
        }

/* updates all around */
        if ( !irc )
        {
            SHFT(ax[0],bx[0],cx[0],u);
            SHFT(Fa[0],Fb[0],Fc[0],Fu);
        }

    }
}

static float BRENT( int n, float x[], float s[],
                    float ax, float bx, float cx,
                    float tol, float *xmin )
/*
   Given a bracketing triplet of values, (ax,bx,cx) such that
   bx is between ax and cx, and F1dim(bx) is less than both
   F1dim(ax) and F1dim(cx), this routine isolates the minimum
   to a fractional precision of about tol using Brent's method.

   It also returns the function value at the minimum.
*/
{
    int   iter, itmax=100;
    float cgold= 0.381966, zeps= 1.0e-10, e= 0.0;
    float a,b,d,etemp,Fu,Fv,Fw,Fz,p,q,r,tol1,tol2,u,v,w,z,xm;

/* initial stuff */
    a= min(ax,cx);
    b= max(ax,cx);
    xm= 0.5*(a+b);

    z=w=v=    bx;
    Fz=Fw=Fv= Func(z);
    tol1= tol*fabs(z) +zeps;
    tol2= 2.0*tol1;


/* main loop */
    for ( iter=1 ;  iter<=itmax && fabs(z-xm) >= tol2-0.5*(b-a) ; iter++ )
    {
/* trial parabolic fit */
        if ( fabs(e)> tol1 )
        {
            r= (z-w) * (Fz-Fv);
            q= (z-v) * (Fz-Fw);
            p= (z-v)*q -(z-w)*r;
            q= 2.0 * (q-r);
            if ( q> 0.0 )  p= -p;
            q= fabs(q);
            etemp= e;
            e= d;

/* is this parabolic step ok? */
            if ( fabs(p)>= fabs(0.5*q*etemp) || p<= q*(a-z) || p>= q*(b-z) )
            {
                e= ( z >= xm ? a-z : b-z );
                d= e * cgold;
            }
            else
            {
/* it was ok, keep going */
                d= p/q;
                u= z +d;

/* can we skip the Golden Section reduction of interval? */
                if ( u-a < tol2 || b-u < tol2 )
                    d= SIGN(tol1,xm-z);
            }

        }
        else
        {
            e= ( z >= xm ? a-z : b-z );
            d= e * cgold;
        }




/*
   we arrive here with d computed from parabolic fit or
   else from Golden Section
*/
        if ( fabs(d) >= tol1 )
            u= z +d;
        else
            u= z +SIGN(tol1,d);


/* the only Function call in the loop! */
        Fu= Func(u);

/* what do we do with Fu now that we have it? */
        if ( Fu <= Fz )
        {
            if ( u >= z )
                a= z;
            else
                b= z;

            SHFT(v,w,z,u);
            SHFT(Fv,Fw,Fz,Fu);
        }
        else
        {
            if ( u < z )
                a= u;
            else
                b= u;

            if ( Fu<=Fw || w==z )
            {
                v= w;
                w= u;
                Fv= Fw;
                Fw= Fu;
            }
            else if ( Fu<=Fv || v==z || v==w )
            {
                v= u;
                Fv= Fu;
            }
        }

        xm= 0.5*(a+b);
        tol1= tol*fabs(z) +zeps;
        tol2= 2.0*tol1;
    }

    if ( iter>= itmax )
        nrerror( "Maximum iterations in BRENT()" );

    xmin[0]= z;
    return ( Fz );
}





/* *********** HERE IS THE ENTRY POINT FOR THIS MODULE ************* */

void linmin( int n, float x[], OPTFXN FXN, float s[], float *F0 )
/*
   This subroutine starts at the point x and moves in the
   search direction s so as to minimize the objective function
   F.  On return, x is updated to x+dx which minimizes F,
   s is set to the actual step (dx), and F0 is F(x+dx).
*/
{
    int   k;
    float a= -1e-5,
          b= 1.,
          c= 2.,
          abc,
          Fa, Fb, Fc,
          e= G_EPS;

    Fa= *F0;
    F= FXN;

    MNBRAK( n, x, s, &a, &b, &c, &Fa, &Fb, &Fc);
    F0[0]= BRENT( n, x, s, a, b, c, e, &abc);


    for ( k=1 ; k<=n ; k++ )
    {
        s[k]= abc *s[k];
        x[k]= x[k] +s[k];
    }
}


