/*****************************************************************************
 *                                  M U K A I
 *****************************************************************************
 *
 *   PROGRAM ID:        MUKAI.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is the the Conjugate Gradient method of:
 *        H Mukai, Math Prog (1978), p.298
 *
 *      It uses no line search. There is even a form that uses no gradients.
 *
 *      Mukai used Fletcher-Reeves conjugate gradients as the underlying
 *      method; I use Polak-Ribiere plus a modification due to MJD Powell.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

// locally global variables
static OPTFXN F;
static GRADF grad;
static int nLoops;
static float fTiny=   G_EPS*G_EPS;
static float fTiny2=  G_EPS*G_EPS*G_EPS*G_EPS;

// local functions and macros
static void opt( int n, float *x, float tol, float *g, float *ga, float *xe, float *s );
static void gradF( int n, float *x, float F0, float *g );



/*
    ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int mukai( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol )
/*
   This subroutine is just a dummy to allocate the (3*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
    float **w;
    w= matrix( 1, 4, 1, n );
    F= FXN;
    if( DFXN )
        grad= DFXN;
    else
        grad= gradF;

    if( tol <= 0.0 )
        tol= G_EPS;
    opt( n, x, tol, w[1], w[2], w[3], w[4] );
    free_matrix( w, 1, 4, 1, n );

    return( nLoops );
}





static void opt( int n, float *x, float tol, float *g, float *ga, float *xe, float *s )
/*
----------------------VARIABLE KEY-------------------------
   n          length of independent (vector) variable

   x          independent variable-- on input it contains
              the initial guess;  on output, the solution

   xe         temporary x value

   F          objective function (to be supplied by user!!)

   F0,F1      stores previous values of F

   g          stores value of gradF call

   ga         stores value of previous gradF call

   b          mixing factor used to det the next s

   s          search direction

---------------------------------------------------------------------
*/
{
int i, nLoopMax;
int bConverged;
float F0, F1, b, gg, gaga, ss, xx, xabs, sabs, gag;
float dx, e= G_EPS, F1e, F2e;

// initializations
    nLoopMax= max( 4*n, 1000 );

    b= 1;
    F0= F( n, x );
    F1= F0;
    grad( n, x, F0, g );

    for( gg= xx= 0.0, i=1 ; i<=n ; i++ )
    {
        s[i]= -g[i];
        ga[i]= g[i];
        gg += g[i]*g[i];
        xx += x[i]*x[i];
    }
    ss= gg;
    gaga= gg;
    xabs= sqrt(1+xx);
    sabs= sqrt(ss);
    if( sabs < fTiny )
        sabs= fTiny;

    bConverged= 0;
    if( gg<= fTiny2 )
        bConverged= 2;



// main loop
    for( nLoops=1 ; bConverged<2 && nLoops<=nLoopMax ; nLoops++ )
    {
        printf( "%5d  F=%16.8e\r", nLoops, F0 );

        dx= e*xabs/sabs;
        if( dx < e )
            dx= e;

        for( i=1 ; i<=n ; i++ )
        {
            xe[i]= x[i] + dx*s[i];
        }
        F1e= F( n, xe );
        for( i=1 ; i<=n ; i++ )
        {
            xe[i] += dx*s[i];
        }
        F2e= F( n, xe );


        // Calculate new x[] & F
        for( i=1 ; i<=n ; i++ )
        {
            if( fabs(F2e-F1e-F1e+F0) > fTiny2 )
            {
                // Mukai model
                x[i] += dx*s[i]*(F0-F1e) / (F2e-F1e-F1e+F0);
            }
            else
            {
                // something to keep us going
                x[i] += dx*s[i];
                //x[i] += s[i];
            }
        }
        F0= F( n, x );



        // Exit test(s)
        // Test 1
        if( gg<= fTiny2 )
            bConverged= 2;
        // Test 2a
        // else if( 2*fabs(F0-F1) <= tol*(fabs(F0)+fabs(F1)+fTiny) )
        //    bConverged++;
        // Test 2b
        else if( 2*fabs(F0-F1) <= tol*(fabs(F0)+fabs(F1)+2)*tol )
            bConverged++;
        // Not converged, oh well, keep going
        else
        {
            bConverged= 0;

            F1= F0;
            grad( n, x, F0, g );

            // calculate dot products
            for( gg=gag=0, i=1 ; i<=n ; i++ )
            {
                 gag += ga[i]*g[i];
                 gg  += g[i]*g[i];
            }

            // Fletcher-Reeves beta
            // b= gg / gaga;
            // Polak-Ribierre beta (this is more robust)
            b= (gg-gag) / gaga;

            // Use Powell's b-restriction  (this is MUCH more robust)
            if( b < 0.0 )
                b= 0.0;

// Update s[] & gaga
            gaga= gg;
            for( xx=ss=0.0, i=1 ; i<=n ; i++ )
            {
                ga[i]= g[i];
                s[i]= b*s[i] -g[i];
                ss += s[i]*s[i];
                xx += x[i]*x[i];
            }
            xabs= sqrt(1+xx);
            sabs= sqrt(ss);
            if( sabs < fTiny )
                sabs= fTiny;
        }

    }

    if( bConverged < 2 )
        nLoops= -nLoops;
}





static void gradF( int n, float *x, float F0, float *g )
{
    int i;
    float a, t, e= G_EPS;

    for ( i=1 ; i<=n ; i++ )
    {
        a= fabs( x[i] );
        if ( a< 1.0 )  a= 1.0;
        a= a*e;
        t= x[i];
        x[i]= t+a;
        g[i]= ( F(n,x) -F0 ) / a;
        x[i]= t;
    }
}
