/*****************************************************************************
 *                                  O P T T E S T
 *****************************************************************************
 *
 *   PROGRAM ID:        OPTTEST.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is a program to test the various opt solvers.  The test problems
 *      themselves are located in a separate file.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *                      Must link with the  <TSTFXNS.C>  file.
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



# include <string.h>
# include <collins.h>

/*
   When the DEF_EL_TSTCNT flag is turned on it puts the following
   declaration into the source:       long elTstCnt
   otherwise it uses an extern definition.
*/
#define DEF_EL_TSTCNT 1
# include <tstfxns.h>


#if defined( DOUBLE_PRECISION )
    #define  SCANFMT  " %le"
#else
    #define  SCANFMT  " %e"
#endif



// **** FIXED DIMENSION PROBLEMS ****

//void  Freud_Roth(int n, float *x, float *f);
// This fxn is due to Freudenstein and Roth.
// It is a fixed dim fxn, n=2.
// The solns are:
//       f.f= 0             at x= (5, 4)
//       f.f= 48.9842....   at x= (11.41..., -0.8968...)
//
// The recommended starting point is:
//       x= (0.5, -2)


//void  Powell2d(int n, float *x, float *f);
// This fxn is due to MJD Powell.
// It is a fixed dim fxn, n=2.
// The solns are:
//       f.f= 0             at x= (1.098...e-5, 9.106...)
//
// The recommended starting point is:
//       x= (0, 1)


//void  HelicalValley(int n, float *x, float *f);
// I do not know who suggested this fxn.
// It is a fixed dim fxn, n=3.
// The solns are:
//       f.f= 0             at x= (1, 0, 0)
//
// The recommended starting point is:
//       x= (-1, 0, 0)


//void  GulfRnD(int n, float *x, float *f);
// This fxn is due to the Research and Development Dept at Gulf Oil.
// It is a fixed dim fxn, n=3.
// The solns are:
//       f.f= 0             at x= (50, 25, 1.5)
//
// The recommended starting point is:
//       x= (5, 2.5, 0.15)


//void  Box3d(int n, float *x, float *f);
// I do not know who suggested this fxn. (Box of Box-Keller fame?)
// It is a fixed dim fxn, n=3.
// The solns are:
//       f.f= 0             at x= (1, 10, 1)
//                     and  at x= (10, 1, -1)
//                     and  at x= (b, b, 0 ), b= arbitrary
//
// The recommended starting point is:
//       x= (0, 10, 20)


//void  MJDPowell4d(int n, float *x, float *f);
// This fxn is due to MJD Powell.
// It is a fixed dim fxn, n=4.
// The solns are:
//       f.f= 0             at x= (0, 0, 0, 0)
//
// The recommended starting point is:
//       x= (3, -1, 0, 1)



// **** VARIABLE DIMENSION PROBLEMS ****

//void  Rosenbrock(int n, float *x, float *f);
// This fxn is due to HH Rosenbrock.
// n must be a multiple of 2 (ie even).
// The solns are:
//       f.f= 0             at x= (1, 1, ... , 1)
//
// The recommended starting point is:
//       x= (-1.2, 1, -1.2, 1, ... , -1.2, 1)


//void  Powell(int n, float *x, float *f);
// This fxn is due to MJD Powell.
// n must be a multiple of 4.
// The solns are:
//       f.f= 0             at x= (0, 0, ... , 0)
//
// The recommended starting point is:
//       x= (3, -1, 0, 1, ... , 3, -1, 0, 1)


//void  TrigFxn(int n, float *x, float *f);
// I do not know who suggested this fxn.
// There are no restrictions on n.
// The solns are:
//       f.f= 0             at x= (0, 0, ... , 0)
//
// The recommended starting point is:
//       x= (1/n, 1/n, ... , 1/n)


//void  Brown(int n, float *x, float *f);
// I do not know who suggested this fxn. (Brown is a pretty common name.)
// There are no restrictions on n.
// The solns are:
//       f.f= 0             at x= (a, a, ... , a), where a satisfies:  n*a^n -(n+1)*a^(n-1) +1= 0
//                                                                        a= 1 in particular
//       f.f= 1             at x= (0, 0, ... , 0, n+1)
//
// The recommended starting point is:
//       x= (1/2, 1/2, ... , 1/2)


//void  More(int n, float *x, float *f);
// This fxn is due to JJ More', BS Garbow, & KE Hillstrom. It was in an
// article in ACM Trans. on Math. Software, Vol. 7, No. 1, March 1981.
// This article was the source for most of the functions in this module.
//
// There are no restrictions on n.
// The solns are:
//       f.f= 0             at x= (-1, -1, ... , -1))
//
// The recommended starting point is:
//       x= (1, 1, ... , 1)


//typedef  float (*OPTFXN)(int n, float *x);
//typedef  void (*NLFXNS)(int n, float *x, float *f);

typedef int (*PFINT)(int n, float x[], OPTFXN F, GRADF gradF, float tol );
static NLFXNS fxn;
static float f[65];

static float fSquared( int n, float x[] )
{
  fxn( n, x, f );

  for( f[0]=0 ; n>0 ; n-- )
      f[0] += f[n]*f[n];

  return( f[0] );
}


int main( int nArgc, char *pszArgv[] )
{
char  ch;
short  i,j,k;
int   n, iter;
float e;
float xo[1+32], x[1+32];   //the 0-component is used for parameter passing,
                           //otherwise a multiple of 4
                           //to meet most restrictive test fxn
                           //requirement for n-values (Powell).
char szStartFormat[]= "Enter  G  to start %s on %s, S to skip:  ";
char szDoneFormat[]= "%s is done, after %d iterations"
                     " &  %ld F-evals\n\tx= (";

char szEndMsg[]= "Enter C  to return to menu\n";
char szScanfFormat[]= SCANFMT;

char szFormat1[]= "%16.8e ";
char szFormat2[]= ")\n\tF= %16.8e\n";
char szConvPromptFormat[]= "Enter convergence tolerance for %s:  ";


char *pszTestFunction[]= { "",
                           "Freud_Roth",
                           "Powell2d",
                           "HelicalValley",
                           "GulfRnD",
                           "Box3d",
                           "Powell4d",
                           "Rosenbrock",
                           "MJDPowell",
                           "TrigFxn",
                           "Brown",
                           "More'",
                           ""  };
char *pszOptMethod[]= { "",
                        //"branin",
                        "ssr1",
                        "zirilli",
                        "bandb",
                        "bfgs",
                        "cgpr",
                        "marqsr1",
                        "mukai",
                        "trustbfgs",
                        "powell",
                        "qndav",
                        "nandm",
                        "homopt",
                        ""   };

NLFXNS pfTestFunction[]= { NULL,
                           Freud_Roth,
                           Powell2d,
                           HelicalValley,
                           GulfRnD,
                           Box3d,
                           Powell4d,
                           Rosenbrock,
                           MJDPowell,
                           TrigFxn,
                           Brown,
                           More,
                           NULL  };
PFINT pfOptMethod[]= { NULL,
                       //branin,
                       ssr1,
                       zirilli,
                       bandb,
                       bfgs,
                       cgpr,
                       marqsr1,
                       mukai,
                       trustbfgs,
                       powell,
                       qndav,
                       nandm,
                       homopt,
                       NULL   };


///****/printf( "sizeof(float)= %d\n", sizeof(float) );
///****/printf( "sizeof(double)= %d\n", sizeof(double) );
//  fclose( stderr );


START:
  printf( "\x1B[2J" ); //printf( "\n\n\n\n\n\n\n\n\n\n" ); //system( "cls" );
  printf( "Select a test function:\n"
           "\t1) %s\n"
           "\t2) %s\n"
           "\t3) %s\n"
           "\t4) %s\n"
           "\t5) %s\n"
           "\t6) %s\n"
           "\t7) %s\n"
           "\t8) %s\n"
           "\t9) %s\n"
           "\t0) %s\n"
           "\ta) %s\n\n"
           "\tx) To Exit\n\n",
           pszTestFunction[1],
           pszTestFunction[2],
           pszTestFunction[3],
           pszTestFunction[4],
           pszTestFunction[5],
           pszTestFunction[6],
           pszTestFunction[7],
           pszTestFunction[8],
           pszTestFunction[9],
           pszTestFunction[10],
           pszTestFunction[11]   );


  scanf( " %c", &ch );
  //printf( "\x1B[2J" ); //printf( "\n\n\n\n\n\n\n\n\n\n" ); //system( "cls" );


  switch( ch )
  {
      //  let's start with Freud_Roth
      case '1':  i= 1;
        n= 2;
        xo[1]= 0.5;  xo[2]= 2.0;
        break;


      //  next, Powell2d
      case '2':  i= 2;
        n= 2;
        xo[1]= 0.0;  xo[2]= 1.0;
        break;


      //  next, HelicalValley
      case '3':  i= 3;
        n= 3;
        xo[1]= -1.0;  xo[2]= 0.0;  xo[3]= 0.0;
        break;


      //  next, GulfRnD
      case '4':  i= 4;
        n= 3;
        xo[1]= -1.0;  xo[2]= 0.0;  xo[3]= 0.0;
        break;


      //  next, Box3d
      case '5':  i= 5;
        n= 3;
        xo[1]= 0.0;  xo[2]= 10.0;  xo[3]= 20.0;
        break;


      //  next, Powell4d
      case '6':  i= 6;
        n= 4;
        xo[1]= 3.0;  xo[2]= -1.0;  xo[3]= 0.0;  xo[4]= 1.0;
        break;


      //  next, Rosenbrock
      case '7':  i= 7;
        printf( "Enter Number of Dimensions (must be multiple of 2):  " );
        scanf( " %d", &n );
        if( n<2 )
            n= 2;
        n= 2 * (n/2);
        for( k=1 ; k<n ; k+=2 )
        {
            xo[k]=  -1.2;
            xo[k+1]= 1.0;
        }
        break;


      //  next, Powell
      case '8':  i= 8;
        printf( "Enter Number of Dimensions (must be multiple of 4):  " );
        scanf( " %d", &n );
        if( n<4 )
            n= 4;
        n= 4 * (n/4);
        for( k=1 ; k<n ; k+=4 )
        {
            xo[k]=    3.0;
            xo[k+1]= -1.0;
            xo[k+2]=  0.0;
            xo[k+3]= 1.0;
        }
        break;

      //  next, TrigFxn
      case '9':  i= 9;
        printf( "Enter Number of Dimensions:  " );
        scanf( " %d", &n );
        if( n<2 )
            n= 2;
        for( k=1 ; k<=n ; k++ )
        {
            xo[k]=    1.0/n;
        }
        break;

      //  next, Brown
      case '0':  i= 10;
        printf( "Enter Number of Dimensions:  " );
        scanf( " %d", &n );
        if( n<2 )
            n= 2;
        for( k=1 ; k<=n ; k++ )
        {
            xo[k]=    0.5;
        }
        break;


      //  next, Broyden
      case 'A':
      case 'a':  i= 11;
        printf( "Enter Number of Dimensions:  " );
        scanf( " %d", &n );
        if( n<2 )
            n= 2;
        for( k=1 ; k<=n ; k++ )
        {
            xo[k]=    1.0;
        }
        break;




      case 'X':
      case 'x':
        return( 0 );
        break;

      default:  i= 100;
        printf( "Don't understand your choice" );
        break;
  }  // end of switch


  if( i < 90 )
  {
      fxn= pfTestFunction[i];
      printf( szConvPromptFormat, pszTestFunction[i] );
      scanf( szScanfFormat, &e );
  }


  for( j=1 ; i<90 && pszOptMethod[j][0] ; j++ )
  {
      printf( szStartFormat, pszOptMethod[j], pszTestFunction[i] );

      while( (ch=getchar())!='G' && ch!='g' && ch!='S' && ch!='s' );
      if( ch=='G' || ch=='g' )
      {
          if( strncmp("branin", pszOptMethod[j],6) == 0 )
          {
              printf( "\tEnter an inertia parameter between 0 and 1:  " );
              scanf( szScanfFormat, &x[0] );

              printf( "\tDo you wish to enter initial velocity? (Y or N):  " );
              while( (ch=getchar())!='Y' && ch!='y' && ch!='N' && ch!='n' );
              if( ch=='Y' || ch=='y' )
              {
                  printf( "\tEnter an initial velocity vector(%d values):  ", n );
                  for( k=1 ; k<=n ; k++ )
                      scanf( szScanfFormat, &x[n+k] );
              }
              else
              {
                  for( k=1 ; k<=n ; k++ )
                      x[n+k]= 0;
              }
          }

          if( strncmp("zirilli", pszOptMethod[j],7) == 0 )
          {
              printf( "\tEnter a viscosity parameter:  " );
              scanf( szScanfFormat, &x[2*n+1] );
              printf( "\tEnter a positive inertia parameter:  " );
              scanf( szScanfFormat, &x[2*n+2] );

              printf( "\tDo you wish to enter initial velocity? (Y or N):  " );
              while( (ch=getchar())!='Y' && ch!='y' && ch!='N' && ch!='n' );
              if( ch=='Y' || ch=='y' )
              {
                  printf( "\tEnter an initial velocity vector(%d values):  ", n );
                  for( k=1 ; k<=n ; k++ )
                      scanf( szScanfFormat, &x[n+k] );
              }
              else
              {
                  for( k=1 ; k<=n ; k++ )
                      x[n+k]= 0;
              }
          }

AGAIN:
          if( strncmp("homo", pszOptMethod[j],4) == 0 )  //homotopy method?
          {
              printf( "\t  Enter number of homotopy steps  " );
              scanf( SCANFMT, &x[0] );
          }

          // Initialize x[] and reset fxn counter
          elTstCnt= 0;
          for( k=1 ; k<=n ; k++ )
              x[k]= xo[k];

          iter= pfOptMethod[j]( n, x, fSquared, 0, e );
          printf( szDoneFormat, pszOptMethod[j], iter, elTstCnt );
          for( k=1 ; k<=n ; k++ )
              printf( szFormat1, x[k] );
          printf( szFormat2, fSquared(n,x) );

          if( strncmp("homo", pszOptMethod[j],4) == 0 )  //homotopy method?
          {
              printf( "\tWould you like to try more steps? (Y/N)  " );
              while( (ch=getchar())!='Y' && ch!='y' && ch!='N' && ch!='n' );
              if( ch=='Y' || ch=='y' )
                  goto AGAIN;
          }

      }
  }


  printf( szEndMsg );
  while( (ch=getchar())!='C' && ch!='c' );
  goto START;
}



