/*****************************************************************************
 *                                  P O W E L L
 *****************************************************************************
 *
 *   PROGRAM ID:        POWELL.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This subroutine is my translation of Powell's famous direction set
 *      search.  This method does not use derivative info, the presence of
 *      the derivative in the argument list is purely cosmetic (to make all
 *      of the optimization routines look the same) and is ignored.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

// locally global variables
static OPTFXN F;
static int nLoops;


// local functions and macros
#define TINY   1.0e-10
static void opt( int n, float x[], float tol, float s[],
                 float xi[], float xe[], float *M[] );




/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int powell( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol )
/*
   This subroutine is just a dummy to allocate the n(n+4)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
    float **w;
    w= matrix( 1, n+4, 1, n );
    F= FXN;


    if( tol <= 0.0 )
        tol= 1.0e-6;
    opt( n, x, tol, w[1], w[2], w[3], &w[3] );
    free_matrix( w, 1, n+4, 1, n );

    return( nLoops );
}





static void opt( int n, float x[], float tol, float s[],
                 float xi[], float xe[], float *M[] )
/*
------------------------VARIABLE KEY-----------------------

   x   the most current value of the soln vector

   xi  x at the start of a minor iteration

   xe  extrapolated x:  xe= x +(x-xi)

   s   current search direction
        (on input s(1) contains a tolerance specification)

   M   matrix of search directions

-----------------------------------------------------------
*/
{
    int i, j, k, nLoopMax, ibig, bConverged, bRestart;
    float f0, fx, fxe, del;

// initial stuff
    bConverged= 0;
    nLoopMax= 100*n*n;
    bRestart= 1;


// main loop, each pass through is a 'minor iteration'
    for( nLoops=1 ; !bConverged && nLoops<=nLoopMax ; nLoops++ )
    {
        printf( "%5d  F=%16.8e\r", nLoops, f0 );

        if( bRestart )
        {
            bRestart= 0;
            bConverged= 0;
            f0= F( n, x );
            for( k=1 ; k<=n ; k++ )
            {
                for( i=1 ; i<k ; i++ )
                    M[i][k]= M[k][i]= 0.0;
                M[k][k]= 1.0;
                xi[k]= x[k];
            }
        }


        fx= f0;
        ibig= 0;
        del= 0.0;
        for( i=1 ; i<=n ; i++ )
        {
// pull next search direction from M
            for( j=1 ; j<=n ; j++ )
                s[j]= M[j][i];

//  put latest F-value into fxe
            fxe= f0;
            linmin( n, x, F, s, &f0 );

//  was change in F the best of this iteration?
            if( fabs(fxe-f0) > del )
            {
                del= fabs( fxe-f0 );
                ibig= i;
            }
        }


///****/ printf( "powell: fx=%e, f0=%e\n", fx, f0 );
///****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );

//  exit tests
//        if( f0 <= tol )
        if( 2.0*fabs(fx-f0) <= tol*(fabs(fx)+fabs(f0))+TINY  && nLoops>2*n )
            bConverged= 1;

//  The following statement yields quadratic convergence, but it's not
//  necessary and on general non-linear functions might slow convergence!
//        else if( nLoops%(n+1) == 0 )
//            bRestart= 1;
//
//  The following statement is my attempt to get the best of both worlds:
//  Powell's heuristic method and (nearly) quadratic convergence.
//        else if( nLoops%(2*n+1) == 0 )
//            bRestart= 1;
        else
        {

/*
   now check make x-xi a candidate search direction
   extrapolate along this search direction
   reset xi
*/
            for( j=1 ; j<=n ; j++ )
            {
                s[j]= x[j] -xi[j];
                xe[j]= x[j] +s[j];
                xi[i]= x[j];
            }
            fxe= F( n, xe );


//   is s direction all played-out?
            if( fxe < fx )
            {

//   or perhaps decrease during last iteration was not due to any
//   single direction, or there is a substantial 2nd derivative
//   and we're near the bottom

                if( 2.0*(fx-2.0*f0+fxe)*(fx-f0-del)*(fx-f0-del) < del*(fx-fxe)*(fx-fxe) )
                {

//   since it passed our tests, accept s and throw out the
//   direction that gave us the largest decrease, since this
//   is a major component of s now.

                    for( j=1 ; j<=n ; j++ )
                        M[j][ibig]= s[j];
                }
            }
        }

    }


    if( !bConverged )
        nLoops= -nLoops;
}


