/*****************************************************************************
 *                                  Z I R I L L I
 *****************************************************************************
 *
 *   PROGRAM ID:        ZIRILLI.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      The following is a homotopy method based on the soln of an ODE.
 *      The ODE is the one suggested by Zirilli et al in 1982:
 *
 *                 m.d(dx/dt)/dt + b.dx/dt =  -g,   b= viscosity parameter
 *                                                  m= inertia (mass) parameter
 *
 *      Which is the analog of Newton's law of motion.  Unfortunately, this
 *      method is second order, which doubles the size of the vectors
 *      required to specify the problem.  And if matrices are required by
 *      your solution method, they are quadrupoled in size!  (Zirilli uses
 *      an implicit method that DOES make use of matrices).
 *
 *      The x-vector is of length 2n+3:
 *          x[1] thru x[n] contain the initial guess at the soln.
 *          x[n+1] thru x[n+n] contain the initial velocity.
 *          x[2n+1] contains the viscosity parameter.
 *          x[2n+2] contains the inertial parameter.
 *
 *      On exit the velocity is zero, and x[1] thru x[n] is the soln.
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of a subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

/* locally global variables */
static OPTFXN F;
static GRADF grad;
static int nLoops=0;
static float *g;  /* used for gradient */
static float b;
static float m;
//extern unsigned long enStep;  /* communicates with Runge-Kutta routine */


/* local functions and macros */
#define TINY   1.0e-10
static void gradF( int n, float x[], float F0, float g[] );
static void derivs( int n2, float t, float x[], float dx[] );
static int Comp( int n, float v[], float vo[], float tol );


/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/
int zirilli( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol )
{
int   k, nOldStepCnt, bConverged= 0;
float Fo, Fn, gg,      /* Used to test for convergence  */
      tstart= 0.0,
      tend= 0.5,       /* This parameter controls convergence; */
//                       /*   just keep taking it halfway to 1.0 -- due to reparameterization */
//                       /*   t=1.0 represents 100% accuracy and infinite time!               */

      etol= 1e-3,      /* a terminal value problem requires sloppy tolerance */
      h= 0.0001;       /* this is only a guess right now */
float *glast;

  g= vector(1,n);
  glast= vector(1,n);
  for( k=1 ; k<=n ; k++ )
      g[k]= 0;

  b= x[2*n+1];          /* extract viscosity parameter */
  m= x[2*n+2];          /* extract inertia parameter */


  F= FXN;
  if( DFXN )
      grad= DFXN;
  else
      grad= gradF;

  Fn= F(n,x);
  enStep= 0;

  while( !bConverged )
  {
      nLoops++;
      printf( "%5d\r", nLoops );
      Fo= Fn;
      for( gg=0, k=1 ; k<=n ; k++ )
      {
          gg= g[k]*g[k];
          glast[k]= g[k];
      }
      /****/printf( "Zirilli conv Control: F=%e, gg=%e\n", Fn, gg );

      nOldStepCnt= enStep;

//      stiff( 2*n, derivs, x, tstart, tend, etol, h, 0, 0 );
      rk23( 2*n, derivs, x, tstart, tend, etol, h, 0, 0 );
//      rk45( n, derivs, x, tstart, tend, etol, h, 0, 0 );

      h= (tend-tstart) / (enStep-nOldStepCnt);
      tstart= tend;

      tend= (1+3*tend)/4; // go a quarter of the way to 1.0
//      tend += tend;  // make t bigger, before re-parameterization was introduced
///      tend= 0.1; // I also tried integrating from 0.0 --> 0.1 over and over
                    // it worked with Numerical Recipes' convergence criteria
                    // but took too many F-evals.

      Fn= F(n,x);


//      if( 2.0*fabs(Fn-Fo) <= tol*(fabs(Fn)+fabs(Fo))+TINY ) // Num Recipes' criteria
      if( Comp(0,&Fn,&Fo,tol) || Comp(n,g,glast,tol) )// || Comp(n,x,xlast,tol) )
          bConverged= 1;
  }

  free_vector( g, 1, n );
  free_vector( glast, 1, n );
  return( nLoops );
}

static int Comp( int n, float v[], float vo[], float tol )
{
int i;
float mv, mvo, mdv;  // magnitudes (modulii)

  for( mv=mvo=mdv=0, i=1 ; i<=n ; i++ )
  {
      mv += v[i]*v[i];
      mvo += vo[i]*vo[i];
      mdv += (v[i]-vo[i]) * (v[i]-vo[i]);
  }

  if( n==0 ) // scalar flag
  {
      mv= v[0]*v[0];
      mvo= vo[0]*vo[0];
      mdv= (v[0]-vo[0]) * (v[0]-vo[0]);
  }
  mv= sqrt(mv);
  mvo=sqrt(mvo);
  mdv= sqrt(mdv);

  return( 2*mdv < (2+mv+mvo)*tol );
}

static int Diff( int n, float v[], float dv[], float tol )
{
int i;
float mv, mdv;  // magnitudes (modulii)

  for( mv=mdv=0, i=1 ; i<=n ; i++ )
  {
      mv += v[i]*v[i];
      mdv += dv[i]*dv[i];
  }

  if( n==0 ) // scalar flag
  {
      mv= v[0]*v[0];
      mdv= dv[0]*dv[0];
  }
  mv= sqrt(mv);
  mdv= sqrt(mdv);

  return( mdv < (1+mv)*tol );
}




static void gradF( int n, float x[], float F0, float g[] )
{
int i;
float a, t, e= G_EPS;

  for( i=1 ; i<=n ; i++ )
  {
      a= fabs( x[i] );
      if( a< 1.0 )
          a= 1.0;
      a= a*e;
      t= x[i];
      x[i]= t+a;
      g[i]= ( F(n,x) -F0 ) / a;
      x[i]= t;
  }
}


static void derivs( int n2, float t, float x[], float dx[] )
{
float F0, s;
int   i, n;
static long nStep= -1;

  n= n2/2;

  F0= F(n,x);
  grad( n, x, F0, g );

  s= 1 -t;
  for( i=1 ; i<=n ; i++ )
  {
//      dx[i]= x[i+n];
//      dx[i+n]= -b*x[i+n] - g[i];

      // The above equations had scaling problems, so I changed the independent
      // variable from t which ranged (0,inf.)  to  p= t/(1+t) which ranges (0,1). In
      // order to accomplish this, the parameter  s=(1-t) is used.
      dx[i]= x[i+n];
      dx[i+n]= ( (2*s-b/m)*x[i+n] - g[i]/(m*s*s) ) / (s*s);
  }
}
