/*****************************************************************************
 *                                  F F T
 *****************************************************************************
 *
 *   PROGRAM ID:        FFT.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              January 8, 1995
 *
 *   DESCRIPTION:
 *
 *          This subroutine computes a conventional Fast Fourier Transform.
 *          The source for this is mainly from:
 *
 *               JGG Dobbe, Dr Dobbs Journal, February 1995, p.150
 *
 *          An interesting feature of the algorithm is that the shuffle (or
 *          butterfly) is performed BEFORE any trig operations, and all trig
 *          functions are calculated incrementally from a table of starting
 *          values. This keeps the size of the auxilliary sine/cosine arrays
 *          to a length of log(n) versus n for other implementations.
 *          The auxillary arrays below are good for  n<=1073741824.
 *
 *          Although I have declared everything as float, in practice, I would:
 *             use float       for       2 < n <= 16384
 *             use double      for   16384 < n <= 1048576
 *             use long double for 1048576 < n <= 1073741824.
 *
 *
 *          I have modified the parameters to match my FHT module:
 *               n =  length of input vector (must be integer power of 2)
 *               f =  complex vector to be transformed; consists of F(t)
 *                    evaluated at n equally spaced points
 *                    The index on f is assumed to run from 0..n-1
 *
 *               dir= direction of transform ( forward = +1,
 *                                             reverse = -1 )
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: void
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>




// local prototypes
static unsigned int ShuffleIndex( unsigned int i, int WordLength );
static void Shuffle2Arr( int n, COMPLEX f[], int bitlength );
static int log2( WORD n );



void fft( int n, COMPLEX f[], int dir )
{
COMPLEX  temp, Q, ONE= {1.0, 0.0};
int   pwr, Section, AngleCounter, FlyDistance, FlyCount, index1, index2;
float c, s, scale;
float CosArray[]=
{
  // cos(2pi/N) for N= 2, 4, 8, ... 1073741824
  -1.0000000000000000,  0.0000000000000000,  0.7071067811865475,  0.9238795325112868,
   0.9807852804032305,  0.9951847266721969,  0.9987954562051724,  0.9996988186962042,
   0.9999247018391445,  0.9999811752826011,  0.9999952938095762,  0.9999988234517019,
   0.9999997058628822,  0.9999999264657179,  0.9999999816164293,  0.9999999954041073,
   0.9999999988510268,  0.9999999997127567,  0.9999999999281892,  0.9999999999820473,
   0.9999999999955118,  0.9999999999988780,  0.9999999999997195,  0.9999999999999299,
   0.9999999999999825,  0.9999999999999956,  0.9999999999999989,  0.9999999999999997,
   0.9999999999999999,  1.0000000000000000
};
float SinArray[]=
{ // sin(2pi/N) for N= 2, 4, 8, ... 1073741824
   0.0000000000000000,  1.0000000000000000,  0.7071067811865475,  0.3826834323650898,
   0.1950903220161283,  0.0980171403295606,  0.0490676743274180,  0.0245412285229123,
   0.0122715382857199,  0.0061358846491545,  0.0030679567629660,  0.0015339801862848,
   0.0007669903187427,  0.0003834951875714,  0.0001917475973107,  0.0000958737990960,
   0.0000479368996031,  0.0000239684498084,  0.0000119842249051,  0.0000059921124526,
   0.0000029960562263,  0.0000014980281132,  0.0000007490140566,  0.0000003745070283,
   0.0000001872535141,  0.0000000936267571,  0.0000000468133785,  0.0000000234066893,
   0.0000000117033446,  0.0000000058516723
};



  // Initialize
  pwr= log2(n);
  Shuffle2Arr( n, f, pwr );
  scale= 1.0;
  if( dir > 0 )
      scale= -1.0;

  Section= 1;
  AngleCounter= 0;
  while( Section < n )
  {
    FlyDistance= 2 * Section;
    c= CosArray[AngleCounter];
    s= SinArray[AngleCounter] * scale;
    for( FlyCount=0, Q=ONE ; FlyCount < Section ; FlyCount++ )
    {
        index1= FlyCount;
        do
        {
          index2= index1 +Section;

          temp.r= 1.0*Q.r*f[index2].r - 1.0*Q.i*f[index2].i;
          temp.i= 1.0*Q.r*f[index2].r + 1.0*Q.i*f[index2].i;

          f[index2].r= f[index1].r -temp.r;
          f[index1].r= f[index1].r +temp.r;
          f[index2].i= f[index1].i -temp.i;
          f[index1].i= f[index1].i +temp.i;

          index1 += FlyDistance;

        } while(index1 <= (n-1) );

        temp.r= Q.r;
        Q.r= Q.r*c -Q.i*s;
        Q.i= Q.i*c -temp.r*s;
    } // end of for(FlyCount)

    Section *= 2;
    AngleCounter++;
  }


  // inverse needs normalization
  if( dir < 0 )
  {
      scale= 1.0 / n;
      for( index1=0 ; index1 < n ; index1++ )
      {
          f[index1].r *= scale;
          f[index1].i *= scale;
      }
  }

}



static int log2( WORD n )
{
int i;

  for( i=1 ; n>1U ; i++ )
      n >>= 1;
  return( i-1 );
}

static unsigned int ShuffleIndex( unsigned int i, int WordLength )
{
unsigned int  NewIndex= 0;
unsigned char BitNr;

  for( BitNr=0 ; BitNr < WordLength ; BitNr++ )
  {
      NewIndex <<= 1;
      if( i&1 )
          NewIndex++;
      i >>= 1;
  }

  return( NewIndex );
}

static void Shuffle2Arr( int n, COMPLEX f[], int bitlength )
{
COMPLEX temp;
unsigned int  IndexOld, IndexNew;

  for( IndexOld=0 ; IndexOld < n ; IndexOld++ )
  {
      IndexNew= ShuffleIndex( IndexOld, bitlength );
      if( IndexNew > IndexOld )
      {
          temp= f[IndexOld];
          f[IndexOld]= f[IndexNew];
          f[IndexNew]= temp;
      }
  }
}

