/*****************************************************************************
 *                                  F H T
 *****************************************************************************
 *
 *   PROGRAM ID:        FHT.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *          This subroutine computes the Fast Hartley Transform very much
 *          like an FFT subroutine, but it uses only real quantities since
 *          in place of EXP(i*w*t)= COS(w*t) +i*SIN(w*t)
 *          it uses     CAS(w*t)=   COS(w*t) +  SIN(w*t)
 *
 *               n =  length of input vector (must be integer power of 2)
 *               f =  real vector to be transformed; consists of F(t)
 *                    evaluated at N equally spaced points
 *                    The index on f is assumed to run from 0..N-1
 *
 *               dir= direction of transform ( forward = +1,
 *                                             reverse = -1 )
 *
 *          You can reconstruct the function using the following sum:
 *              F(t)= f[k].cas(2.pi.t.k/L), where  0 <= k <= N-1
 *
 *          Of course to minimize aliasing, you should change the sum
 *          slightly:
 *              F(t)= f[k].cas(2.pi.t.k/L), where  0 <= k <= N/2-1
 *                  + f[j].cas(2.pi.t.(j-N)/L), where  N/2 <= j <= N-1
 *
 *          This code is my translation of the Pascal routines given by
 *          Mark O'Neill in an article in BYTE, April 1988, p. 293.  These
 *          in turn were O'Neill's implementation of the routines of
 *          Ronald N Bracewell. Bracewell's routines were an application of
 *          FFT ideas to the (CAS) transform of Hartley.
 *
 *          In his article, O'Neill makes statements about the FHT using only
 *          half the computer resources of the FFT. I'm not sure if that is
 *          true, but the code is a lot cleaner than an FFT since no complex
 *          arithmetic operations are required.
 *
 *
 *          If you absolutely require Fourier coefficients, a[k], you can recover
 *          them from the Hartley coeffs, h[k],  as follows:
 *               Real(a[k]) = h[k] + h[N-k]
 *               Imag(a[k]) = h[k] - h[N-k]
 *
 *          Notice that h[0]= h[N], this means that a[0] cannot be complex. In
 *          general, you cannot model complex-valued data in a straightforward
 *          manner using the Hartley transform. In fact, it was developed based
 *          on exactly the opposite observation -- usually you wish to analyze
 *          real-valued signals, and must force-fit them into the complex domain
 *          required by Fourier methods.
 *
 *          Convolution of the Hartley transform works just like the convolution
 *          of the Fourier transform:
 *               F(t)*G(t) = N * f[k] * g[k], k=0..N-1
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: void
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>



/* globals */
static int fa, ta;
static float *sne, *csn;

/* local prototypes */
static void transf( WORD ind, WORD i1, WORD i2, WORD i3, float *f[], WORD n );
static WORD modify( int m, WORD si, WORD sf, WORD i );
static WORD  perm( WORD ii, int l2n );
static int log2( WORD n );


void fht( WORD n, float f[], int dir )
{
WORD si, sf, sect, i, j, k, inc, ind;
int m, l2n;
float scale, a, b;
float *ff[3], *ftemp;
double t, cosa, sina, cosb, sinb;

   ftemp= vector( 0, n-1 );
   sne= vector( 1, n );
   csn= vector( 1, n );

// Initialize
   l2n= log2(n);
   fa= 1;
   ta= 2;
   m= 1;
   b= 2 *M_PI /n;

   ff[fa]= &ftemp[-1];  // ff is a unit offset matrix,
   ff[ta]= &f[-1];      // that's the reason for the -1s


   sina= sin(0);  sinb= sin(b);
   cosa= cos(0);  cosb= cos(b);
   for( i=1 ; i<=n ; i++ )
   {
       csn[i]= cosa;
       sne[i]= sina;
       t=    cosa*cosb - sina*sinb;
       sina= sina*cosb + cosa*sinb;
       cosa= t;
       ff[fa][perm(i,l2n)]= f[i-1];
   }

// This is the way O'Neill calculated the sin & cos vectors -- not good.
// The loop above uses just 2 multiplies and 1 add in place of a trig fxn call!
/*
   for( a=0, i=1 ; i<=n ; i++ )
   {
       csn[i]= cos(a);
       sne[i]= sin(a);
       a += b;
       ff[fa][perm(i,l2n)]= f[i-1];
   }
*/

// here's the actual transform
   for( i=1 ; i<= l2n ; i++ )
   {
      j= 1;
      sect= 1;
      inc= n / (m+m);


      do
      {
         ind= 1;
         si= m*sect +1;
         sf= m*sect +m;

         for( k=1 ; k<=m ; k++ )
         {
            transf( ind, j, j+m, modify(m, si, sf, j+m), ff, n );
            ind += inc;
            j++;
         }

         j += m;
         sect += 2;
      } while( j <= n );

      m += m;
      ta= 3 -ta;
      fa= 3 -fa;
//****/printf( ". " );
   }

// ouput the results
   scale= 1.0;
   if( dir==1 )
       scale /= n;

   for( i=1 ; i<=n ; i++ )
       f[i-1]= scale *ff[fa][i];

   free_vector( ftemp, 0, n-1 );
   free_vector( sne, 1, n );
   free_vector( csn, 1, n );
}




static void transf( WORD ind, WORD i1, WORD i2, WORD i3, float *f[], WORD n )
{
      f[ta][i1]= f[fa][i1] +f[fa][i2]*csn[ind] +f[fa][i3]*sne[ind];
      ind += n/2;
      f[ta][i2]= f[fa][i1] +f[fa][i2]*csn[ind] +f[fa][i3]*sne[ind];
}




static WORD modify( int m, WORD si, WORD sf, WORD i )
{
WORD iRc;

   if( si==i || m<3 )
      iRc= i;
   else
      iRc= si +sf +1 -i;

   return( iRc );
}




static WORD  perm( WORD i, int l2n )
{
int k;
WORD j, s;

   i--;
   j= 0;
   for( k=1 ; k<=l2n ; k++ )
   {
      s= i/2;
      j= j +j -s -s +i;
      i= s;
   }

   return( j+1 );
}



static int log2( WORD n )
{
int i;

  for( i=1 ; n>1U ; i++ )
      n >>= 1;
  return( i-1 );
}

