/****************************************************************
* FILE:	warp.c
* DESC:	This program lets you specify a file to warp, then
*		prompts you for control lines. It uses the lines to
*		warp the underlying image, going a step at a time,
*		tweening to the target and optionally saving the
*		sequence as numbered PCX files.
* 
* HISTORY:	Created	 7/06/1994
* LAST CHANGED: 7/06/1994
* 
*	Copyright (c) 1994 by Scott Anderson
*
****************************************************************/

/* ----------------------INCLUDES----------------------------- */

#include <conio.h>
#include <stdio.h>
#include <io.h>
#include <dos.h>			/* for the mouse */
#include <math.h>
#include <graph.h>
#include <memory.h>
#include <string.h>

#include "define.h"

/* ----------------------DEFINES------------------------------ */

#define WARP_TWEENS		2

/* ----------------------PROTOTYPES--------------------------- */

int		tweenWarp(PICTURE *src);

/* ----------------------EXTERNALS---------------------------- */

/**** line routines ****/
extern void		editLines(PICTURE *pic, LINE_LIST *lineList);
extern void		createLines(PICTURE *pic, LINE_LIST *lineList);

/**** warping and morphing routines ****/
extern int		sumLines(PICTURE *picture, COLOR *color,
					LINE *origline, POINT *warp, LINE *warpline);
extern int		setLength(LINE *line);

/**** i/o routines ****/
extern LINE_LIST *loadLines(char *filename, char *extension);
extern void      saveLines(char *filename,
						LINE_LIST *lineList, char *extension);

/**** external variables ****/ 
extern char		*OutFilename;
/* set from last picture loaded */
extern int		Xmin, Ymin, Xmax, Ymax;

/* ----------------------GLOBAL DATA-------------------------- */

PICTURE *Src;			/* source picture pointer */

LINE SrcLine[MAX_LINES];
LINE DstLine[MAX_LINES];

int		Tweens;
int		StartLine;
int		NumLines;

/*****************************************************************
* FUNC: main (int argc, char *argv[])
* 
* DESC: Read in a filename to warp
*****************************************************************/

main (int argc, char *argv[])
{
	int		segment;
	LINE_LIST *lineSrcList;
	LINE_LIST *lineDstList;
	char	answer;

	/* load the pcx file if one is given */
	if ((2 > argc) || (argc > 4)) {
		printf("Usage: warp <file> [<steps> [<output>]]\n\n");
		printf("Where: <file>   is the source PCX filename\n");
		printf("       <steps>  is the optional sequence size\n");
		printf("                (the max is %d, the default is %d)\n",
							   	MAX_TWEENS, WARP_TWEENS);
		printf("       <output> is the optional output filename\n");
		printf("                (defaults to no output)\n\n");
		printf("Note:  The output filename can be at most %d characters long.\n",
								MAX_NAME_SIZE);
		printf("       The PCX extension is added automatically, so don't\n");
		printf("       include it in the filename.\n");
		printf("       Morph only accepts PCX files with %d X %d resolution\n",
								MAX_WIDE, MAX_TALL);
		printf("       and %d colors.\n", COLORS);
		exit(0);
	}
	if (argc > 2) {
		Tweens = clip (atoi(argv[2]), 1, MAX_TWEENS);
		if (argc > 3)
			OutFilename = argv[3];
	}
	else
		Tweens = WARP_TWEENS;

	printf("Loading the file %s\n", argv[1]);
	Src = loadPicture(argv[1]);
	if (Src == NULL)
		quit(MEMORY_ERR, "");

	lineSrcList = loadLines(argv[1], EXT_LINE1);
	lineDstList = loadLines(argv[1], EXT_LINE2);
	if (lineSrcList->number != 0) {
		if (lineAsk(argv[1]) == 'N') {
			createLines(Src, lineSrcList);
			lineDstList->number = 0;
		}
		else 
			editLines(Src, lineSrcList);
	}
	else
		createLines(Src, lineSrcList);
		
	TargFlag = 1;	/* For the screen intro message */
	NumLines = lineSrcList->number;
	if (NumLines) {
		/* Make sure the number of lines match */
		if (lineDstList->number !=  NumLines) {	
			/* didn't match, so copy the source lines */
			lineDstList->number = NumLines;
			for (segment = 0; segment < NumLines; segment++) 
				lineDstList->line[segment]
						= lineSrcList->line[segment];
		}
		editLines(Src, lineDstList);
		saveLines(argv[1], lineSrcList, EXT_LINE1);
		saveLines(argv[1], lineDstList, EXT_LINE2);
		beep();

		for (segment = 0; segment < NumLines; segment++) {
			DstLine[segment].p[0]=lineDstList->line[segment].p[0];
			DstLine[segment].p[1]=lineDstList->line[segment].p[1];
			setLength(&DstLine[segment]);
			SrcLine[segment].p[0]=lineSrcList->line[segment].p[0];
			SrcLine[segment].p[1]=lineSrcList->line[segment].p[1];
			setLength(&SrcLine[segment]);
		}
		tweenWarp(Src);
	}

	setTextMode();
}

/*****************************************************************
* FUNC: int	tweenWarp(PICTURE *src)
* 
* DESC: In-between the warping lines to the final warped image.
*		Display the warped image as you go.
*****************************************************************/

int
tweenWarp(PICTURE *src)
{
	POINT warp;
	COLOR scolor;
	LINE warpLine[MAX_LINES];
	int tween, line, p;

	setGraphicsMode();
	displayPicture(src);
	
	if (Tweens > 1) {
		saveScreen(&src->pal);
		Tweens--;
	}
	/* src is already on screen, now tween to the target */
	for (tween = 1; tween <= Tweens; tween++) {
		/* Tween the control lines used to warp the images */
		for (line = 0; line < NumLines; line++) {
			/* tween each point of each line */
			for (p = 0; p < 2; p++) {
				warpLine[line].p[p].x = SrcLine[line].p[p].x +
						((DstLine[line].p[p].x
						- SrcLine[line].p[p].x) * tween) /Tweens;
				warpLine[line].p[p].y = SrcLine[line].p[p].y +
						((DstLine[line].p[p].y
						- SrcLine[line].p[p].y) * tween) /Tweens;
			}
			setLength(&warpLine[line]);
		}
		/* Go through the screen and get warped source pixels */
		for (warp.y = Ymin; warp.y <= Ymax; warp.y++)	{
			for (warp.x = Xmin; warp.x <= Xmax; warp.x++)	{	
				_setcolor (sumLines(src, &scolor, SrcLine,
											&warp, warpLine));
				_setpixel (warp.x, warp.y);
			}
		}
		if (!OutFilename) {	/* no output file name */
			beep();
			waitForKey();	/* so pause to enjoy the pictures */
		}
		else
			saveScreen(&src->pal);
	}
}


