/* Compile this program using the 'Large' model */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <string.h>
#include "extmath.h"

/* Return the smallest factor, up to 65533, of large number X */
/* Return 0 if factor not found */
u_long FindFactor(u_long X[], int N)
{
    u_long D, DCheck;

    /* Ignore any leading zeros */
    N = ExtNZLen(X, N);

    /* Return trivial cases */
    if (N == 0)              return 0;
    if (N == 1 && X[0] == 1) return 1;
    if ((X[0] & 1) == 0)     return 2;

    /* Check for factor <= 23 */
    DCheck = ExtMod1(0, (u_long)3*5*7*11*13*17*19*23, X, N);
    if (DCheck % 3  == 0)  return 3;
    if (DCheck % 5  == 0)  return 5;
    if (DCheck % 7  == 0)  return 7;
    if (DCheck % 11 == 0)  return 11;
    if (DCheck % 13 == 0)  return 13;
    if (DCheck % 17 == 0)  return 17;
    if (DCheck % 19 == 0)  return 19;
    if (DCheck % 23 == 0)  return 23;

    /* Check for factor <= 253 */
    for (D = 29;  D <= 245;  D += 12)
    {
        DCheck = ExtMod1(0, D * (D + 2) * (D + 6) * (D + 8), X, N);
        if (DCheck % D       == 0)  return D;
        if (DCheck % (D + 2) == 0)  return D + 2;
        if (DCheck % (D + 6) == 0)  return D + 6;
        if (DCheck % (D + 8) == 0)  return D + 8;
    }

    /* Check for factor <= 65533 */
    for (;  D <= 65531;  D += 6)
    {
        DCheck = ExtMod1(0, D * (D + 2), X, N);
        if (DCheck % D       == 0)  return D;
        if (DCheck % (D + 2) == 0)  return D + 2;
    }

    return 0;
}



/* Test a large number X for being prime.  Print comments to screen */
int CheckPrimality(u_long *X, int N, u_long Count)
{
    static u_long One = 1;
    int i, rflag, np, pshift;
    u_long Fact, DCheck, *M1, *P, *Z;

    printf("\nPossible prime at count %ld\n", Count);

    /* M1 = X - 1 */
    M1 = ExtMAlloc(N);
    ExtMov(X, N, M1, N);
    ExtDec(M1, N);

    /* Calculate power P = (X - 1) / (2**pshift) */
    P  = ExtMAlloc(N);
    np = N;
    ExtMov(M1, N, P, np);
    for (pshift = 0;  P[0] == 0;  pshift += 32)
    {
        np--;
        ExtMov(P + 1, np, P, np);
    }
    DCheck = P[0];
    while((DCheck & 1) == 0)
    {
        DCheck >>= 1;
        pshift++;
    }
    ExtShiftR(0, pshift & 31, P, np);

    /* Z is working space */
    Z  = ExtMAlloc(N);

    /* Check whether value is prime */
    for (DCheck = 3;  ;  DCheck += 2)
    {
        if ((Fact = FindFactor(&DCheck, 1)) == DCheck  ||  Fact == 0)
        {
            printf("   Primality check: Raise %ld to power mod X --- ",
                   DCheck);

            /* Raise DCheck to the power P mod X */
            ExtPowerMod(&DCheck, 1, P, np, X, N, Z);

            if (ExtCmp(&One, 1, Z, N) != 0) break;

            /* if Z = 1, not certain */
            printf("probably prime\n");
        }
    }

    rflag = 0;
    for (;  pshift > 0;  pshift--)
    {
        /* if Z = X - 1, X is, with a very high probability, prime  */
        if (ExtCmp(M1, N, Z, N) == 0)
        {
            rflag = 1;
            break;
        }

        /* Square Z mod X*/
        ExtMulMod(Z, N, Z, N, X, N, Z);
    }
    if (rflag)
        printf("PRIME!\n");
    else
        printf("composite\n");

    free(Z);
    free(P);
    free(M1);

    return rflag;
}



void PrimeSearch(u_long far *X, int N)
{
    static u_long Two = 2;
    int    j;
    u_long Factor, Count;

    /* Make sure X is odd */
    if ((X[0] & 1) == 0)
        ExtDec(X, N);

    printf("Searching for a prime number");
    Count = 1;
    while ((Factor = FindFactor(X, N)) != 0 ||
           CheckPrimality(X, N, Count) == 0)
    {
        if (N == 1 && Factor == X[0]) break;
        printf(".");
        ExtSub(0, &Two, 1, X, N);
        Count++;
    }
    printf("\nPrime number found after testing %ld numbers!!\n", Count);
}



void PrintNum(FILE *ofile, u_long *X, int N)
{
    u_long *Xdec;
    int i, i0, Ndec;

    Ndec = NeededForDec(N);
    Xdec = ExtMAlloc(Ndec);
    ExtMov(X, N, Xdec, Ndec);
    ExtToDec(Xdec, Ndec);
    Ndec = ExtNZLen(Xdec, Ndec);

    fprintf(ofile, "PRIME NUMBER: \n");
    i = Ndec - 1;
    fprintf(ofile, "%ld", Xdec[i]);
    for (i--;  i >= 0;  i--)
        fprintf(ofile, "%09ld", Xdec[i]);
    fprintf(ofile, "\n");

    free(Xdec);
}



void PrintNumHex(FILE *ofile, u_long *X, int N)
{
    int i;

    /* Print number in Hex in the form for a C data statement */
    fprintf(ofile, "/* Prime number as a C data statement */\n");
    fprintf(ofile, "static u_long xxxx[] =\n   {0x%08lX", X[0]);

    for (i = 1;  i < N;  i++)
    {
        if (i % 6 == 0)
            fprintf(ofile, ",\n    ");
        else
            fprintf(ofile, ", ");

        fprintf(ofile, "0x%08lX", X[i]);
    }

    fprintf(ofile, "};\n");
}



void main()
{
    int i, N, Temp;
    u_long NBits, NDigits, *X;
    char fname[80];
    FILE *ofile;
    static char *Msg[] = {
"This is a sample program that uses the ExtMath library to do arithmetic on",
"very large numbers.\n",
"The program finds, at random, very large prime numbers.  Try it first enter-",
"ing 100 for the number of digits.  It will usually find a prime number in a",
"few seconds.  It can find prime numbers a thousand digits long and more, but",
"that takes much longer.\n\n",
    NULL};

    /* Display introduction */
    clrscr();
    gotoxy(27, 1);
    printf("< Welcome to FINDPRIM >\n\n");

    for (i = 0;  Msg[i] != NULL;  i++)
        printf("  %s\n", Msg[i]);

    /* Prompt for and open an output file */
    ofile = NULL;
    for (;;)
    {
        printf("Write results to a file (Y or N)? ");
        gets(fname);
        strupr(fname);

        if (fname[0] == 'Y' || fname[0] == 'N') break;

        printf("  Must respond Y or N!\n");
    }
    
    if (fname[0] == 'Y')
        for (;;)
        {
            printf("Enter new file name (or press ENTER for none): ");
            gets(fname);
            strupr(fname);

            if (fname[0] == 0) break;

            if ((ofile = fopen(fname, "w")) != NULL) break;

            printf("  File open failed.  Try again.\n");
        }

    /* Main loop */
    for (;;)
    {
        /* Ask for number of digits in the prime number */
        for (;;)
        {
            printf("\nEnter approximate number of decimal digits (0 to exit): ");
            scanf("%lu", &NDigits);
            if (NDigits <= 78894) break;

            printf("  %lu is a little too long.  Try again.\n", NDigits);
        }

        /* Finished if 0 entered */
        if (NDigits <= 0) break;

        /* Determine roughly the number of bits this requires */
        NBits = BitsForDigits(NDigits);
        printf("%ld bits required.\n", NBits);

        /* Determine the number of long integers this uses */
        N = (NBits + 31) / 32;

        /* Allocate the space for the number and set to a random number */
        X = ExtMAlloc(N);
        ExtRandom(X, N);

        /* Eliminate extra high bits */
        Temp = NBits & 31;
        if (Temp > 0)
            X[N - 1] &= ((u_long)1 << Temp) - 1;

        /* Search down from this number to find a prime number */
        PrimeSearch(X, N);

        /* Print the found prime number */
        PrintNum(stdout, X, N);

        /* If file open, print result to file */
        if (ofile != NULL)
        {
            PrintNum(ofile, X, N);
            fprintf(ofile, "\n");
            PrintNumHex(ofile, X, N);
            fprintf(ofile, "\n");
        }

        /* Release memory */
        free (X);
    }

    if (ofile != NULL)
    {
        fclose(ofile);
        printf("File %s was created!\n", fname);
    }
}
