/*
 *: program Example1
 *: purpose: test shell for threaded RedBlack tree. Mainly for profiling.
 *
 *: Usage:  EXAMPLE1 <filename>
 *.	    file <filename> is a text file which contains a list a words,
 *.	    one per line and left-adjusted. The list may contain duplicated
 *.	    words.  Program reads the list, writes the words to the monitor
 *.	    and inserts them into a threaded RedBlack tree.  Program pauses
 *.	    briefly, then writes the list (with duplicates eliminated) to the
 *.	    monitor in order.  Program pauses briefly again, and then items
 *.	    are randomly deleted from the tree until it is empty. The tree
 *.	    is then disposed and the program halts.
 *
 *  Copyright (c) 1993	Bert C. Hughes
 *			200 N. Saratoga
 *			St.Paul, MN 55104
 *			Compuserve 71211,577
 */

#if defined NDEBUG
#error Compile EXAMPLE1.C with NDEBUG undefined.
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <assert.h>
#include "redblack.h"

#if defined __TURBOC__
#include <conio.h>
#else
#define cprintf printf
#endif

/* Borland C++ - turn off "Possibly incorrect assignment" warning */
#pragma warn -pia

void pause(void)
{
    int c;
    cprintf("\r\nPress <ENTER> to continue...");
    while (getchar() != '\n');
    cprintf("\r\n");
}

void *ident(void *x)   /* function to return identifier of data node */
{			 /* is required by "rb_init".  In this very  */
    return x;		 /* simple case, the identifier is simply what */
}			 /* is being inserted into the tree - a string */

void *make_str(const void *s)
{
    int size = 1 + strlen((char *)s);
    char *p = malloc(size);
    return(memcpy(p,s,size));
}

#define INP_BUFFER_SIZE 1024
char inp_buffer[INP_BUFFER_SIZE];

/* find minimum, maximum path length of the tree*/
static unsigned max_path, min_path = 0xffff;

void min_max(RB_nodeptr p, unsigned path_length)
{
     if (++path_length > max_path)
	  max_path = path_length;

     if ((p->Lbit + p->Rbit == 0) && (path_length < min_path))
	  min_path = path_length;

     if (p->Lbit) min_max(p->Lptr, path_length);
     if (p->Rbit) min_max(p->Rptr, path_length);
}


main (int argc, char *argv[])
{
    RB_treeptr tree;
    RB_nodeptr p;
    unsigned int n,i,k;
    char **name;
    char s[128];

  /* use C library function "strcmp" to compare identifiers of nodes */
    int (*cmp)() = strcmp;

  /* use C library function "strcpy" as data copier "readitem"*/
#pragma warn -sus
    void *(*cpy)() = (void *) strcpy;
#pragma warn .sus

    FILE *fp;

   /* initialize random number generator */
    time_t t;
    srand((unsigned) time(&t));

#if defined __TURBOC__
    directvideo = 0;
#endif

    cprintf("FYI: sizeof(RB_NODE) = %d\r\n",sizeof(RB_NODE));
    pause();

    /* open a file which contains a list of words, one per line */

    if (argc < 2)
	strcpy(s,"WORDLIST");
    else
	strcpy(s,argv[1]);

    assert((fp = fopen(s,"r")) != NULL);

    setvbuf(fp,inp_buffer,_IOFBF,INP_BUFFER_SIZE);

    /* initialize the tree */
    assert((tree = rb_init(cmp,ident,make_str,free,cpy,malloc,free)));

    cprintf("First, \"%s\" will be read and words found inserted into the tree.\r\n",s);
    pause();

    /* build tree of words */
    while (fscanf(fp,"%s",s) != EOF)
	if (!(p = rb_insert(tree,s,NO_REPLACE))) {
	    cprintf("Out of memory!\r\n");
	    return EXIT_FAILURE;
	}

   /* check that all identifiers inserted can be found */

    cprintf("Tree is built. Next, word file will be re-read to verify that\r\nall words are in the tree and can be found.\r\n");
    pause();

    fseek(fp,0,SEEK_SET);
    cprintf("Testing 'Find'\r\n");

    while (fscanf(fp,"%s",s) != EOF) {
	cprintf("%-10.8s",s);
	assert(rb_find(tree,s) != NULL);
    }

    cprintf("\r\n\nFind OK.\r\n");
    cprintf("Next, data items inserted into the tree will be listed in ascending order.\r\n");

    pause();

    fclose(fp);

    cprintf("In order:\r\n");
    p = rb_reset(tree);
    n = 0;
    while (p = rb_succ(p)) {
	n++;
	cprintf("%-10.8s",p->dataptr);	/*dangerous to use dataptr directly*/
    }
    cprintf("\r\n\nNodes = %u\r\n",n);

    min_max(((RB_nodeptr)(tree))->Lptr, 0);
    cprintf("Max_path = %u    Min_path = %u\r\n",max_path,min_path);

    k = n;
    p = rb_reset(tree);

    if ((name = malloc(n * sizeof(char *))) == NULL) {
	cprintf("\r\nOut of memory, can't do random deletions test.\r\n");
	cprintf("Testing count in reverse direction...\r\n");
	while (p = rb_pred(p)) n--;
	assert(n==0);
	cprintf("Reverse direction tested & OK.\r\n");
	return EXIT_SUCCESS;
    }

    cprintf("\r\nNext: An array is filled for random deletions while passing through RBtree\r\nin reverse direction.  Words will not be written to output.\r\n");
    pause();

    while (p = rb_pred(p))
	*(name + --k) = p->dataptr;
				      /* Direct use of a field of RB_NODE*/
				      /* All fields are READONLY! Only RB-*/
				      /* library routines alter tree & */
				      /* node field values. */
    assert(k==0);

    cprintf("\r\nReverse direction tested & OK.\r\n");
    cprintf("\r\nDelete test:  words to delete will be randomly selected from the array just\r\nfilled, written to screen, and then deleted from the RBtree.\r\n");
    pause();

    cprintf("\r\nDelete test:\r\n");

    while (n) {
       /* select random identifier to delete */
	i = --n * ((float) rand() / (float) RAND_MAX);

       /* show the world */
	cprintf("%-10.8s",*(name + i));

       /* delete it */
	rb_delete(tree,*(name + i));

       /* replace deleted name with last name in list - list will shrink */
	*(name + i) = *(name + n);
    }

    p = rb_reset(tree);		  /* make sure the tree is empty */
    if (rb_succ(p) != NULL) {
	cprintf("Can't happen: tree still conatains nodes - successor.\r\n");
	return EXIT_FAILURE;
    }

    p = rb_reset(tree);		  /* make sure the tree is empty */
    if (rb_pred(p) != NULL) {
	cprintf("Can't happen: tree still conatains nodes - predecessor.\r\n");
	return EXIT_FAILURE;
    }

    p = (RB_nodeptr) tree;
    if ((p->Rptr != p) || (p->Lptr != p) || p->Lbit || !p->Rbit) {
	cprintf("Head node (tree node by inheritance) is messed up...\r\n");
	return EXIT_FAILURE;
    }

    cprintf("\r\nDelete OK.\r\n");

    rb_destroy(tree);

    cprintf("OK - tree destroyed - bye.");
    return EXIT_SUCCESS;
}
