/*
 *: program Example3
 *: purpose: test shell for threaded AVL-tree. mainly for profiling.
 *
 *: Usage:  EXAMPLE3 <filename>
 *.	    file <filename> is a text file which contains a list a words,
 *.	    one per line and left-adjusted. The list may contain duplicated
 *.	    words.  Program reads the list, writes the words to the monitor
 *.	    and inserts them into a threaded AVL tree.	Program pauses briefly,
 *.	    then writes the list (with duplicates eliminated) to the monitor
 *.	    in order.  Program pauses briefly again, and then items are
 *.	    randomly deleted from the tree until it is empty. The tree is
 *.	    then disposed and the program halts.
 *
 *.	    EXAMPLE3 is just like EXAMPLE1 except for use allocation &
 *.	    deallocation routines other than "malloc" & "free".
 *
 *  Copyright (c) 1991	Bert C. Hughes
 *			200 N. Saratoga
 *			St.Paul, MN 55104
 *			Compuserve 71211,577
 */

#if defined NDEBUG
#error Compile EXAMPLE3.C with NDEBUG undefined.
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <assert.h>
#include "redblack.h"

#if defined __TURBOC__
#include <conio.h>
#else
#define cprintf printf
#endif

/* Borland C++ - turn off "Possibly incorrect assignment" warning */
#pragma warn -pia

void pause(void)
{
    int c;
    cprintf("\r\nPress <ENTER> to continue...");
    while (getchar() != '\n');
    cprintf("\r\n");
}

/*  Here are dead simple memory allocation & deallocation routines,
    whose only purpose is to illustrate use of initialization function
    "rb_init" with memory allocation routines other than the standard
    "malloc" and "free".  The RedBlack library functions will use only
    these routines to allocate/deallocate memory.
*/

#define BUFFER_MAX  30000

void *alloc(size_t size)
{
    static char membuffer[BUFFER_MAX];
    static unsigned next = 0;

    if ((size + next) <= BUFFER_MAX) {
	next += size;
	return (membuffer+next-size);
    }
    else return NULL;
}

#pragma warn -par
void dealloc(void *p)
{
/*dummy routine does nothing - this memory allocator is a one way street*/
}
#pragma warn .par


void *ident(void *x)	/* function to return identifier of data node */
{			/* is required by "rb_init".  In this very */
    return x;		/* simple case, the identifier is simply what */
}			/* is being inserted into the tree - a string */


void *cpy(const void *p)
/* create a copy of data item p - which must be a string in this example. */
{
    int size = 1 + strlen(p);
    char *s = alloc(size);
    if (s) return(memcpy(s,p,size));
    else   return NULL;
}


void *rdstr(void *dest, const void *src)
{
    return((void *)strcpy(dest,src));
}

main (int argc, char *argv[])
{
    RB_treeptr tree;
    RB_nodeptr p;
    unsigned int n,i,k;
    char **name;
    char s[128];

    int (*cmp)() = strcmp;  /* use C library function "strcmp" to compare */
			    /* identifiers of nodes */
    FILE *fp;

   /* initialize random number generator */
    time_t t;
    srand((unsigned) time(&t));

#if defined __TURBOC__
    directvideo = 0;   /* change to 1 if there is "snow" on your monitor */
#endif

    /* open a file which contains a list of words, one per line */

    if (argc < 2)
	strcpy(s,"SHORTLST");
    else
	strcpy(s,argv[1]);

    assert((fp = fopen(s,"r")) != NULL);

    /* initialize the tree */
    assert((tree = rb_init(cmp,ident,cpy,dealloc,rdstr,alloc,dealloc)));

    cprintf("First, \"%s\" will be read and words found inserted into the tree.\r\n",s);
    pause();

    /* build tree of words */
    while (fscanf(fp,"%s",s) != EOF)
	if (rb_insert(tree,s,NO_REPLACE) == NULL) {
	    cprintf("Out of memory!\r\n");
	    return EXIT_FAILURE;
	}

   /* check that all identifiers inserted can be found */

    cprintf("Tree is built. Next, word file will be re-read to verify that\r\nall words are in the tree and can be found.\r\n");
    pause();

    fseek(fp,0,SEEK_SET);
    cprintf("Testing 'Find'\r\n");

    while (fscanf(fp,"%s",s) != EOF) {
	cprintf("%-10.8s",s);
	assert(rb_find(tree,s) != NULL);
    }
    cprintf("\r\n\nFind OK.\r\n");
    cprintf("Next, data items inserted into the tree will be listed in ascending order.\r\n");

    pause();

    fclose(fp);

    cprintf("In order:\r\n");
    p = rb_reset(tree);
    n = 0;
    while ((p = rb_succ(p)) != NULL) {
	cprintf("%-10.8s",p->dataptr);
	n++;
    }
    cprintf("\r\n\nNodes = %u\r\n",n);

    k = n;
    p = rb_reset(tree);

    if ((name = malloc(n * sizeof(char *))) == NULL) {
	cprintf("\r\nOut of memory, can't do random deletions test.\r\n");
	cprintf("Testing count in reverse direction...\r\n");
	while (p = rb_pred(p)) n--;
	assert(n==0);
	cprintf("Reverse direction tested & OK.\r\n");
	return EXIT_FAILURE;
    }

    cprintf("\r\nNext: An array is filled for random deletions while passing through RedBlack tree\r\nin reverse direction.  Words will not be written to output.\r\n");
    pause();

    while ((p = rb_pred(p)) != NULL)
	*(name + --k) = p->dataptr;

    assert(k==0);

    cprintf("\r\nReverse direction tested & OK.\r\n");
    cprintf("\r\nDelete test:  words to delete will be randomly selected from the array just\r\nfilled, written to screen, and then deleted from the RedBlack tree.\r\n");
    pause();

    cprintf("\r\nDelete test:\r\n");

    while (n) {
       /* select random identifier to delete */
	i = --n * ((float) rand() / (float) RAND_MAX);

       /* show the world */
	cprintf("%-10.8s",*(name + i));

       /* delete it */
	rb_delete(tree,*(name + i));

       /* replace deleted name with last name in list - list will shrink */
	*(name + i) = *(name + n);
    }

    p = rb_reset(tree);		  /* make sure the tree is empty */
    if (rb_succ(p) != NULL) {
	cprintf("Can't happen: tree still conatains nodes - successor.\r\n");
	return EXIT_FAILURE;
    }

    p = rb_reset(tree);		  /* make sure the tree is empty */
    if (rb_pred(p) != NULL) {
	cprintf("Can't happen: tree still conatains nodes - predecessor.\r\n");
	return EXIT_FAILURE;
    }

    cprintf("\r\nDelete OK.\r\n");

    rb_destroy(tree);

    cprintf("OK - tree destroyed - bye.");
    return EXIT_SUCCESS;
}
