/*: RB_INS.C	 Threaded RedBlack Tree Library   Version 1.51	8-feb-93
 *
 *: Prototype:	RB_nodeptr rb_insert(RB_treeptr, void *item, int replace)
 *: Purpose:	Insert item into RB_TREE;  "item" may be anything,
 *.		but the user-defined function "key_of()" given when tree was
 *.		initialized by "rb_init()" must be able to read an
 *.		identifier from "item" which can be compared with other
 *.		indentifiers via the user defined function "cmp()".
 *
 *.		If a datanode is found such that identifier(datanode) equals
 *.		identifier(item), the new data from item replaces the data
 *.		existing in datanode if & only if parameter replace == 1.
 *
 *  RELEASED TO THE PUBLIC DOMAIN
 *
 *  author:		    Bert C. Hughes
 *			    200 N.Saratoga
 *			    St.Paul, MN 55104
 *			    Compuserve 71211,577
 */

#include <assert.h>    /* production versions of the library should be */
#include "redblack.h"  /* compiled with NDEBUG defined to remove assertions */
#include "rb_priv.h"

RB_nodeptr rb_insert(RB_treeptr tree, void *item, int replace)
		/*
		Using the user supplied (key_of) & (cmp) functions, *tree
		is searched for a node which matches *item. If a match is
		found, the new item replaces the old if & only if
		replace != 0.  If no match is found the item is inserted
		into *tree.  "rb_insert" returns a pointer to the node
		inserted into or found in *tree. "rb_insert" returns
		NULL if & only if it is unable to allocate memory for
		a new node.
		*/
{
     register RB_nodeptr p,q,f,z;
     register int cmpval = -1;
     RB_nodeptr y;

    /* locate insertion point for item */

     q = &tree->head;  p = Leftchild(q);

     while (p) {
	  q = p;
	  cmpval = (*tree->cmp)((*tree->key_of)(item),\
				(*tree->key_of)(p->dataptr));
	  if (cmpval < 0)
	       p = Leftchild(p);
	  else if (cmpval > 0)
	       p = Rightchild(p);
	  else {
	       if (replace) {
		    void *temp = (*tree->make_item)(item);
		    if (temp) {
			 (*tree->free_item)(p->dataptr);
			 p->dataptr = temp;
		    }
		    else p = NULL;
	       }
	       return p;
	  }
     }/* while (p) */

    /* wasn't found - create new node as child of "q" */

     y = (*tree->alloc)(sizeof(RB_NODE));

     if (y) {
	  y->Lbit  = THREAD;
	  y->Rbit  = THREAD;
	  y->color = RED;
	  if ((y->dataptr = (*tree->make_item)(item)) == NULL) {
	       (*tree->dealloc)(y);
	       return NULL; /* out of memory */
	  }
     }
     else return NULL;	    /* out of memory */

     if (cmpval < 0) {	    /* connect to tree & thread it */
	  y->rchild = 0;
	  y->Lptr = q->Lptr;
	  y->Rptr = q;
	  q->Lbit = LINK;
	  q->Lptr = y;
     }
     else {
	  y->rchild = 1;
	  y->Rptr = q->Rptr;
	  y->Lptr = q;
	  q->Rbit = LINK;
	  q->Rptr = y;
     }

    /* rebalance tree if necessary */

     p = y;

     while (p != tree->head.Lptr  &&  (q=parentOf(p))->color == RED) {
	  f = parentOf(q);
	  assert((!Is_Head(f)) && (f->color == BLACK));
	  if (q->rchild) {
	       z = Leftchild(f);
	       if (colorOf(z) == RED) {
		    q->color = BLACK;
		    z->color = BLACK;
		    f->color = RED;
		    p = f;
		    continue; /* unnecessary - here for human clarity */
	       }
	       else {
		    if (!p->rchild) {
			 rotateRight(q);
			 z = q; q = p; p = z;
		    }
		    q->color = BLACK;
		    f->color = RED;
		    rotateLeft(f);
		    break;
	       }
	  }
	  else {
	       z = Rightchild(f);
	       if (colorOf(z) == RED) {
		    q->color = BLACK;
		    z->color = BLACK;
		    f->color = RED;
		    p = f;
		    continue; /* ditto previous comment */
	       }
	       else {
		    if (p->rchild) {
			 rotateLeft(q);
			 z = q; q = p; p = z;
		    }
		    q->color = BLACK;
		    f->color = RED;
		    rotateRight(f);
		    break;
	       }
	  }
     } /* while rebalancing */

     tree->head.Lptr->color = BLACK;

     return y;
} /*rb_insert*/
