#if !defined RBtree_H
#define RBtree_H
/*: REDBLACK.H	 Threaded RedBlack Tree Library   Version 1.51	8-feb-93
 *
 *: Module:   threaded RedBlack tree library
 *: Purpose:  contains prototypes for all threaded RedBlack tree functions
 *
 *  RELEASED TO THE PUBLIC DOMAIN
 *
 *  Author:		Bert C. Hughes
 *			200 N. Saratoga
 *			St.Paul, MN 55104
 *			Compuserve 71211,577
 */

/* Constants for "replace" parameter of "rb_insert" */
#define REPLACE     1
#define NO_REPLACE  0

/*  Constants are possible return values of "rb_setdata" */
#define TREE_OK		 0  /* No error. */
#define TREE_NOMEM	 1  /* Out of memory error */
#define TREE_ILLEGAL_OP  2  /* Requested operation would disrupt the
			       rb_tree structure; operation cancelled! */

#include <stddef.h>	    /* for definition of NULL, size_t */

typedef struct rbnode *RB_nodeptr;
typedef struct rbtree *RB_treeptr;

/* function prototypes */

RB_treeptr rb_init(int (*compare)(void *key1, void *key2),
		    void *(*key_of)(void *DataObject),
		    void *(*make_item)(const void *DataObject),
		    void (*free_item)(void *DataObject),
		    void *(*copy_item)(void *Destination_DataObject,\
				 const void *Source_DataObject),
		    void *(*alloc)(size_t),
		    void (*dealloc)(void *)
		    );
	    /*
	    Returns pointer to empty tree on success, NULL if insufficient
	    memory.  The function pointers passed to "rb_init" determine
	    how that instance of rb_tree will behave & how it will use
	    dynamic memory.

		parameters-
		  compare:	Compares identifiers, same form as "strcmp".
		  key_of:	Gets pointer to a data object's identifier.
		  make_item:	Creates new data object that is a copy of
				*DataObject.
		  free_item:	Complements make_item. Releases any memory
				allocated to DataObject by "make_item".
		  copy_item:	Copies data object *Source to buffer *Dest
		  alloc:	Memory allocator.
		  dealloc:	Deallocates dynamic memory - complements
				"alloc"
	    */


int rb_setdata(RB_treeptr tree, RB_nodeptr p, void *item);
	    /*
	    Replace data contents of *p with *item.
	    returns:
		0  ................ OK
		TREE_NOMEM ........ out of memory (heap space)
		TREE_ILLEGAL_OP ...
		     (*tree->key_of)(p->dataptr) != (*tree->key_of)(item)

	    Uses "make_item" and "free_item". See rb_init.
	    */

void *rb_getdata(RB_treeptr tree, RB_nodeptr p, void *buffer);
	    /*
	    A safe method of reading the data contained in rbnode.
	    If user/programmer uses "dataptr" for anything other than
	    reading the data "dataptr" points to, the rb_tree will
	    be corrupted.  Returns *buffer;  Data will be copied to
	    buffer using method "copy_item"; see rb_init.
	    */

RB_nodeptr rb_insert(RB_treeptr tree, void *item, int replace);
	    /*
	    Using the user supplied "key_of" & "compare" functions,
	    *tree is searched for a node which matches *item. If a
	    match is found, the new item replaces the old if & only
	    if replace != 0.  If no match is found the item is
	    inserted into *tree.  "rb_insert" returns a pointer to
	    the node inserted or found, or NULL if there is not enough
	    memory to create a new node and copy "item".  Uses functions
	    "key_of" and "compare" for comparisons and to retrieve
	    identifiers from data objects, "make_item" to create a copy
	    of "item", "alloc" to get memory for the new tree node, and
	    "dealloc" if "make_item" fails.
	    */

int rb_delete(RB_treeptr tree, void *key);
	    /*
	    Delete node identified by "key" from *tree.
	    Returns 1 if found and deleted, 0 if not found.
	    Uses "compare", "key_of", "free_item" and "dealloc".
	    See function rb_init.
	    */

void rb_remove(RB_treeptr tree, RB_nodeptr p);
	    /*
	    Removes the node pointed to; Node p must be a pointer
	    return value from rb_find(), rb_succ(), rb_pred(),
	    or rb_insert(). All memory used by node is freed using
	    "free_item" and "dealloc". (See rb_init() above).
	    */

void rb_delete_all(RB_treeptr tree);
	    /*
	    Remove all data nodes from tree, release memory used.
	    */

void rb_destroy(RB_treeptr tree);
	    /*
	    Destroy the tree. Uses functions "free_item" and "dealloc"
	    to restore pool memory used. See function rb_init.
	    */

RB_nodeptr rb_find(RB_treeptr tree, void *key);
	    /*
	    Returns pointer to node which contains data item
	    in *tree whose identifier equals "key". Uses "key_of"
	    to retrieve identifier of data items in the tree,
	    "compare" to compare the identifier retrieved with
	    *key.  Returns NULL if *key is not found.
	    */

/********************************************************************
    Following three functions allow you to treat rb_trees as a
    doubly linked sorted list with a head node.  This is the point
    of threaded trees - it is almost as efficient to move from node
    to node or back with a threaded tree as it is with a linked list.
*********************************************************************/

RB_nodeptr rb_reset(RB_treeptr tree);
	    /*
	    Returns pointer to begin/end of *tree (the head node).
	    A subsequent call to rb_succ will return a pointer
	    to the node containing first (least) item in the tree;
	    just as a call to rb_pred would return the last
	    (greatest).  Pointer returned can only be used a parameter
	    to "rb_succ" or "rb_pred" - the head node contains no
	    user data.
	    */

RB_nodeptr rb_succ(RB_nodeptr p);
	    /*
	    Returns successor of "p", or NULL if "p" has no successor.
	    */

RB_nodeptr rb_pred(RB_nodeptr p);
	    /*
	    Returns predecessor of "p", or NULL if no predecessor.
	    */

/**************      END PUBLIC DEFINITIONS	*******************/

/* Private: for internal use by rb*.c library routines only! */

typedef struct rbnode {
	    void *dataptr;
	    RB_nodeptr Lptr, Rptr;
	    unsigned int rchild : 1;  /* are we left or right child of parent */
	    unsigned int Lbit	: 1;
	    unsigned int Rbit	: 1;
	    unsigned int color	: 1;
	} RB_NODE;

typedef struct rbtree {
	    RB_NODE head; /* struct rbtree "inherits" struct rbnode */
	    int (*cmp)(void *, void *);
	    void *(*key_of)(void *);
	    void *(*make_item)(const void *);
	    void (*free_item)(void *);
	    void *(*copy_item)(void *, const void *);
	    void *(*alloc)(size_t);
	    void (*dealloc)(void *);
	} RB_TREE;

/* end private */

#endif

