/***********************************************************************/
/* QUERY.C -                                                           */
/* This file contains functions related to QUERY,STATUS and EXTRACT    */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1995 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Id: query.c 2.0 1995/01/26 16:31:43 MH Release MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"
#include "query.h"

#ifdef PROTO
static short extract_pending(short,CHARTYPE *);
static short extract_point(short,CHARTYPE *);
static short extract_prefix(short,CHARTYPE *);
#else
static short extract_pending();
static short extract_point();
static short extract_prefix();
#endif

/*man-start*********************************************************************


========================================================================
QUERY, EXTRACT and STATUS
========================================================================

     The following lists the valid variables that can be queried and
     also shows what values are returned. For both QUERY and STATUS,
     the values are concatenated together and displayed as a single
     line. For EXTRACT the REXX variables that are set are defined.
     The capitalised portion of the variable is the minimum abbreviation
     recognised.
     The bracketed text at the end of the description indicates from
     which commands a valid response will be supplied (Q-Query,E-Extract
     M-Modify and S-Status).

ALT
     The number of alterations to the current file since the last SAVE
     or AUTOSAVE.
     (QES)

     alt.0           - 2
     alt.1           - Number of alterations since last SAVE or AUTOSAVE
     alt.2           - Number of alterations since last SAVE

ARBchar
     The status of ARBCHAR and the ARBCHAR characters.
     (QEMS)

     arbchar.0       - 3
     arbchar.1       - ON|OFF
     arbchar.2       - multiple match character
     arbchar.3       - single match character

AUtosave
     The status of AUTOSAVE and/or the frequency setting.
     (QEMS)

     autosave.0      - 1
     autosave.1      - OFF|n

BACKup
     Indicates if a .bak file is kept after editing.
     (QEMS)

     backup.0        - 1
     backup.1        - ON|OFF|TEMP|KEEP

BEEP
     Indicates if the bell is sounded on display of error messages.
     (QEMS)

     beep.0          - 1
     beep.1          - ON|OFF

BLOCK
     Returns information about the marked block, if any.
     (E)

     block.0         - 6 if a marked block exists, or 1 for NONE
     block.1         - type of marked block (LINE|BOX|NONE|WORD|COLUMN)
     block.2         - line number of start of block
     block.3         - column number of start of block
     block.4         - line number of end of block
     block.5         - column number of end of block
     block.6         - file name containing marked block

CASE
     The settings related to the CASE of characters entered, searched
     for, changed and sorted.
     (QEMS)

     case.0          - 4
     case.1          - MIXED|UPPER|LOWER
     case.2          - RESPECT|IGNORE
     case.3          - RESPECT|IGNORE
     case.4          - RESPECT|IGNORE

CLEARScreen
     Indicates if the screen is cleared on exit from THE.
     (QEMS)

     clearscreen.0   - 1
     clearscreen.1   - ON|OFF

CLOCK
     Indicates if the time is displayed on the status line.
     (QEMS)

     clock.0         - 1
     clock.1         - ON|OFF

CMDArrows
     Returns the settings for arrow key behaviour.
     (QEMS)

     cmdarrows.0     - 1
     cmdarrows.1     - RETRIEVE|TAB

CMDline
     The settings for the command line.
     (QEMS)

     If CMDLINE OFF

     cmdline.0       - 1
     cmdline.1       - OFF

     If CMDLINE ON, BOTTOM or TOP

     cmdline.0       - 3
     cmdline.1       - ON|TOP|BOTTOM
     cmdline.2 *     - line number within window of command line
     cmdline.3 *     - contents of command line
               * these values are only returned with EXTRACT

COMPAT
     The settings for the compatibility mode.
     (QEMS)

     compat.0        - 1
     compat.1        - THE|XEDIT|KEDIT

CURline
     The value and position of the current line within the current file.
     (QEMS)

     With EXTract command:

     curline.0       - 3
     curline.1       - curline setting
     curline.2       - line number within window of current line
     curline.3       - contents of current line

     With Query,Modify and Status commands:

     curline.0       - 1
     curline.1       - curline setting

CURSor
     The position of the cursor within the displayed screen and file
     at the time the EXTRACT /CURSOR/ is issued and the position of
     the cursor at the time the macro was issued.
     If the cursor is not in the file area, then line and column
     values will be set to -1.
     (QE)

     cursor.0        - 8
     cursor.1        - line number of cursor in screen (current)
     cursor.2        - column number of cursor in screen (current)
     cursor.3        - line number of cursor in file (current)
     cursor.4        - column number of cursor in file (current)
     cursor.5        - line number of cursor in screen (original)
     cursor.6        - column number of cursor in screen (original)
     cursor.7        - line number of cursor in file (original)
     cursor.8        - column number of cursor in file (original)

DIRFILEID
     The value of the path and filename of the focus line in a DIR.DIR
     file.
     (E)

     dirfileid.0     - 2
     dirfileid.1     - full path of directory
     dirfileid.2     - file name at focus line

DISPlay
     Indicates the range of selection levels current.
     (QEMS)

     display.0       - 2
     display.1       - display low value
     display.2       - display high value

EOF
     Indicates if the current line is on the *** Bottom of file *** line.
     (QES)

     eof.0           - 1
     eof.1           - ON|OFF

EOLout
     Returns the value of the end-of-line character(s).
     (QEMS)

     eolout.0        - 1
     eolout.1        - LF|CRLF

ETMODE
     Indicates if extended display mode is set.
     (QEMS)

     etmode.0        - 1
     etmode.1        - ON|OFF

FNAME
     The file name of the current file.
     (E)

     fname.0         - 1
     fname.1         - File name.

FPATH
     The path name of the current file. This includes a trailing
     directory separator.
     (E)

     fpath.0         - 1
     fpath.1         - File path.

FTYPE
     The extension of the current file. The characters following
     the trailing '.' character.
     (E)

     ftype.0         - 1
     ftype.1         - File extension.

GETENV variable
     The value of the supplied environment variable or ***invalid***
     if the variable does not exist.
     (E)

     getenv.0        - 1
     getenv.1        - value of variable

HEX
     Indicates if HEX is on or off.
     (QEMS)

     hex.0           - 1
     hex.1           - ON|OFF

HEXDISPlay
     Indicates if the current character is displayed on the status line.
     (QEMS)

     hexdisplay.0    - 1
     hexdisplay.1    - ON|OFF

HEXShow
     Returns details of HEXSHOW placement.
     (QEMS)

     hexshow.0       - 2
     hexshow.1       - ON|OFF
     hexshow.2       - line displaying first hexshow line

IDline
     Indicates if the id line is displayed for a file.
     (QEMS)

     idline.0        - 1
     idline.1        - ON|OFF

IMPMACro
     Indicates if implied macro processing is on or off.
     (QEMS)

     impmacro.0      - 1
     impmacro.1      - ON|OFF

IMPOS
     Indicates if implied operating system command processing is on 
     or off.
     (QEMS)

     impos.0         - 1
     impos.1         - ON|OFF

INSERTmode
     Indicates if currently in insert mode or overstrike mode.
     (QEMS)

     insertmode.1    - 1
     insertmode.1    - ON|OFF

LASTmsg
     Return the text of the last error message generated.
     (E)

     lastmsg.0       - 1
     lastmsg.1       - text of last message.

LASTRC
     Return code from last command issued from command line.
     (QES)

     lastrc.0        - 1
     lastrc.1        - Last return code.

LENgth
     Length of the current line.
     (QES)

     length.0        - 1
     length.1        - Length of current line.

LIne
     Line number of current line in current file.
     (QES)

     line.0          - 1
     line.1          - Line number

LINENd
     Indicates if multiple commands allowed on command line
     and the delimiter.
     (QEMS)

     linend.0        - 2
     linend.1        - ON|OFF
     linend.2        - delimiter

MACROExt
     The current setting for a macro's file extension.
     (QEMS)

     macroext.0      - 1
     macroext.1      - Default file extension

MACROPath
     The path that THE looks for by default for macro files.
     (QEMS)

     macropath.0     - 1
     macropath.1     - Path for macro files.

MARgins
     The settings for left and right margins and paragraph indent.
     (QEMS)

     margins.0       - 3
     margins.1       - left column
     margins.2       - right column
     margins.3       - indent value (column or offset from left margin)

MONITOR
     Indicates if the combination of monitor and the curses package
     supports colour. If the curses package supports colour, then
     monitor.3 is set to COLOR and monitor.2 can be COLOR or MONO
     depending on whether the monitor supports colour. If monitor.3
     is MONO then monitor.2 will also be set to MONO.
     (QES)

     monitor.0       - 2
     monitor.1       - COLOR|MONO
     monitor.2       - COLOR|MONO

MSGLine
     Returns details of where the message line is displayed.
     (QEMS)

     msgline.0       - 2
     msgline.1       - ON
     msgline.2       - line position of message line
     msgline.3       - number of message lines available
     msgline.4       - OVERLAY

MSGMode
     Indicates if MSGMODE setting is on or off.
     (QEMS)

     msgmode.0       - 1
     msgmode.1       - ON|OFF

NBFile
     Returns with the number of files currently in the ring.
     (QES)

     nbfile.0        - 1
     nbfile.1        - Number of files in ring

NEWLines
     Indicates if NEWLINES variable is set to LEFT or ALIGNED.
     (QEMS)

     newlines.0      - 1
     newlines.1      - ALIGNED|LEFT

NONDisp
     Returns the character that is displayed for extended characters
     that are not displayed.
     (QEMS)

     nondisp.0       - 1
     nondisp.1       - char

NUMber
     Indicates if line numbers are displayed in the prefix area.
     (QEMS)

     number.0        - 1
     number.1        - ON|OFF

PENDing [BLOCK] [OLDNAME] name|*
     Returns information about pening prefix commands.
     (E)

     pending.0       - 4
     pending.1       - line number in file
     pending.2       - newname - actual name entered in prefix area
     pending.3       - oldname - original name of macro after
                                 synonym resolution
     pending.4       - BLOCK or null

Point [*]
     Returns the name and line number of the focus line, or names and
     line numbers of all lines in a file if '*' is specified.
     (QEMS)

     When 'EXTract /Point' is specified:

     point.0         - 0 or 1       (0 if focus line not named)
     point.1         - line number and name of line (if line is named)

     When 'EXTract /Point *' is specified:
     (E)

     point.0         - number of named lines in the file
     point.1         - line number and name for first named line
     point.n         - line number and name for nth named line

POSition
     Indicates if LINE/COL is displayed on IDLINE
     (QEMS)

     position.0      - 3
     position.1      - ON|OFF
     position.2      - current or focus line
     position.3      - current or focus column

PREfix [Synonym name|*]
     Indicates if prefix is displayed for the view and if displayed
     where is is displayed.
     (QEMS)

     prefix.0        - 1 or 2     (1 if prefix.1 is OFF, 2 otherwise)
     prefix.1        - ON|OFF|NULLS
     prefix.2        - LEFT|RIGHT        (if prefix.1 is ON or NULLS)

     With [Synonym name] option, the name of the macrofile (oldname) 
     is returned that is associated with the synonym, name. If name 
     is not a synonym then name is returned.
     (E)

     prefix.0        - 1
     prefix.1        - oldname

     With [Synonym *] option, all prefix synonyms are returned.
     (E)

     prefix.0        - number of prefix synonyms
     prefix.1        - newname oldname
     prefix.n        - newname oldname

PRINTER
     Returns the value of the printer port or spooler.
     (QEMS)

     printer.0       - 1
     printer.1       - port or spooler name

REPROFile
     Indicates if the profile file is to be re-executed for each
     subsequent file to be edited.
     (QEMS)

     reprofile.0     - 1
     reprofile.1     - ON|OFF

REESERved
     Return with list of the screen rows that have been reserved.
     (QES)

     reserved.0      - 0 if no reserved lines; 1 otherwise
     reserved.1      - list of screen row numbers reserved

REXXOUTput
     Indicates if REXX output is captured to a file or not and the
     line number limit of lines to be displayed.
     (QEMS)

     rexxoutput.0    - 2
     rexxoutput.1    - FILE|DISPLAY
     rexxoutput.2    - line number limit

SCOPE
     Returns the status of [SET] SCOPE.
     (QEMS)

     scope.0         - 1
     scope.1         - ALL|DISPLAY

SCALe
     Returns details of SCALE line.
     (QEMS)

     scale.0         - 2
     scale.1         - ON|OFF
     scale.2         - line displaying scale line

SELect
     Returns the selection level of the focus line and the
     maximum selection level for the file.
     (QES)

     select.0        - 2
     select.1        - selection level of focus line
     select.2        - maximum selection level for file

SHADow
     Returns the status of the display of shadow lines.
     (QEMS)

     shadow.0        - 1
     shadow.1        - ON|OFF

SIze
     Returns the number of lines in the current file.
     (QES)

     size.0          - 1
     size.1          - Lines in current file.

STATUSLine
     The settings for the status line.
     (QEMS)

     statusline.0    - 1
     statusline.1    - TOP|BOTTOM|OFF

STAY
     Indicates if STAY is on or off.
     (QEMS)

     stay.0          - 1
     stay.1          - ON|OFF

TABKey
     Returns settings about behaviour of TAB key.
     tabkey.1 is behaviour while in insert mode
     tabkey.2 is behaviour while not in insert mode
     (QEMS)

     tabkey.0        - 2
     tabkey.1        - TAB|CHARACTER
     tabkey.2        - TAB|CHARACTER

TABLine
     Returns details of TABLine line.
     (QEMS)

     tabline.0       - 2
     tabline.1       - ON|OFF
     tabline.2       - line displaying tab line

TABS
     Returns settings about tab columns.
     (QEMS)

     tabs.0          - 2
     tabs.1          - tab columns

TABSIn
     Indicates if TABSIN processing is on or off and the size of the
     tabs setting.
     (QEMS)

     tabsin.0        - 2
     tabsin.1        - ON|OFF
     tabsin.2        - size of tabs

TABSOut
     Indicates if TABSOUT processing is on or off and the size of the
     tabs setting.
     (QEMS)

     tabsout.0       - 2
     tabsout.1       - ON|OFF
     tabsout.2       - size of tabs

TERMinal
     Identifies the terminal type currently being used.
     (QES)

     terminal.0      - 1
     terminal.1      - DOS|OS2|$TERM value under Unix

TOF
     Indicates if the current line is on the *** Top of file *** line.
     (QES)

     tof.0           - 1
     tof.1           - ON|OFF

Verify
     Returns verify column settings.
     (QEMS)

     verify.0        - 1
     verify.1        - Column pair of verify start and end columns.

VERShift
     Returns vertical shift value.
     (QES)

     vershift.0      - 1
     vershift.1      - VERSHIFT value

VERSION
     Returns information about name of application (THE) and version
     information.
     (QES)

     version.0       - 4
     version.1       - THE
     version.2       - version string eg. 1.5
     version.3       - platform version (DOS,OS2,UNIX)
     version.4       - version status information eg. beta

Width
     Returns maximum line width setting.
     (QES)

     width.0         - 1
     width.1         - Maximum line width value.

WORDWrap
     Indicates if WORDWARP is on or off.
     (QEMS)

     wordwrap.0      - 1
     wordwrap.1      - ON|OFF

Zone
     Returns zone column settings.
     (QEMS)

     zone.0          - 2
     zone.1          - Zone start column
     zone.2          - Zone end column



========================================================================
IMPLIED EXTRACT, BOOLEAN and OTHER FUNCTIONS
========================================================================

IMPLIED EXTRACT

     The above REXX variables set above by the EXTRACT command may also
     be obtained by a REXX macro as an implied EXTRACT. Each variable
     above that may be set by an explicit EXTRACT command may also be
     used as an external function to REXX to obtain the same information.
     eg. The REXX commands:

         'EXTRACT /SIZE/CURLINE/'
         Say size.1 curline.1

     may be substituted with:

         Say size.1() curline.1()

BOOLEAN FUNCTIONS

     THE also provides other information to the REXX interpreter via
     boolean functions. These functions return either '1' or '0'
     depending on the information queried.

  after()
     Returns '1' if the cursor is currently after the last non-blank
     character on the line, or if the line is blank.

  blank()
     Returns '1' if the line the cursor is on is completely blank.

  block()
     Returns '1' if the marked block is within the current view.

  before()
     Returns '1' if the cursor is currently before the first non-blank
     character on the line, or if the line is blank.

  bottomedge()
     Returns '1' if the cursor is on the bottom edge of the filearea
     or prefix area.

  command()
     Returns '1' if the command line is on. This is different to
     KEDIT's definition; "Returns '1' if the cursor is on the 
     command line." To get the equivalent KEDIT functionality
     use incommand();

  current()
     Returns '1' if the cursor is on the current line.

  dir()
     Returns '1' if the current file is the special DIR.DIR file.

  end()
     Returns '1' if the cursor is on the last non-blank character on
     the line.

  eof()
     Returns '1' if the cursor is on the '*** Bottom of file ***' line
     and the cursor is not on the command line.

  first()
     Returns '1' if the cursor is in column 1 of the current window.

  focuseof()
     Returns '1' if the focus line is the '*** Bottom of file ***' line
     whether the cursor is on it or not.

  focustof()
     Returns '1' if the focus line is the '*** Top of file ***' line
     whether the cursor is on it or not.

  inblock()
     Returns '1' if the cursor is in the marked block.

  incommand()
     Returns '1' if the cursor is on the command line.

  initial()
     Returns '1' if the function is called from the profile file.

  inprefix()
     Returns '1' if the cursor is located in the prefix area.

  leftedge()
     Returns '1' if the cursor is on the left edge of the file area.

  modifiable()
     Returns '1' if the cursor is located in an area that can be changed.
     ie. not on '*** Top of File ***' or '*** Bottom of File ***' lines.

  rightedge()
     Returns '1' if the cursor is on the right edge of the file area.

  spacechar()
     Returns '1' if the cursor is on a space character.

  tof()
     Returns '1' if the cursor is on the '*** Top of file ***' line
     and the cursor is not on the command line.

  topedge()
     Returns '1' if the cursor is on the top edge of the file area.

  verone()
     Returns '1' if the column 1 of the file is being displayed in
     column 1.

OTHER FUNCTIONS

   The following functions provide features to simplify THE macros
   written in REXX.

  valid_target(target[,anything])
     Returns 'ERROR' if the supplied target is invalid.
     Returns 'NOTFOUND' if the supplied target is valid, but not found.

     The first argument is the target to be validated. If a second,
     optional argument is supplied, the target to be validated can
     consist of a target followed by any optional characters. This can
     be useful if the arguments to a macro consist of a target followed
     by another argument. If a valid target is supplied, the remainder
     of the string passed to valid_target() is returned following the
     first line affected and the number of lines to the target.

     If a valid target, returns the first line affected by the target
     followed by the number of lines to the target, and optionally the
     remainder of the argument.
     eg. if the focus line is 12 and valid_target() is called as
         result = valid_target(":7") , the return value is "12 -5"
         if the focus line is 12 and valid_target is called as
         result = valid_target(":7 /fred bloggs/",junk), the return
         value is "12 -5 /fred bloggs/"

**man-end**********************************************************************/

struct query_item
 {
  CHARTYPE query;                                /* valid query response ? */
  CHARTYPE *name;                                          /* name of item */
  short min_len;                       /* minimum length of abbreviation */
  short item_number;                           /* unique number for item */
  short number_values;                      /* number of values returned */
 };
typedef struct query_item QUERY_ITEM;

/***********************************************************************/
/* Keep the following items in alphabetic order of name.               */
/***********************************************************************/
 static QUERY_ITEM query_item[] =
  {
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"alt",3,ITEM_ALT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"arbchar",3,ITEM_ARBCHAR,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"autosave",2,ITEM_AUTOSAVE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"backup",4,ITEM_BACKUP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"beep",4,ITEM_BEEP,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"block",5,ITEM_BLOCK,0},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"case",4,ITEM_CASE,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"clearscreen",6,ITEM_CLEARSCREEN,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"clock",5,ITEM_CLOCK,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"cmdarrows",4,ITEM_CMDARROWS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"cmdline",3,ITEM_CMDLINE,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"compat",6,ITEM_COMPAT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"curline",3,ITEM_CURLINE,3},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"cursor",4,ITEM_CURSOR,8},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"dirfileid",9,ITEM_DIRFILEID,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"display",4,ITEM_DISPLAY,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"eof",3,ITEM_EOF,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"eolout",3,ITEM_EOLOUT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"etmode",6,ITEM_ETMODE,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"fname",5,ITEM_FNAME,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"fpath",5,ITEM_FPATH,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"ftype",5,ITEM_FTYPE,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"getenv",6,ITEM_GETENV,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"hex",3,ITEM_HEX,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"hexdisplay",7,ITEM_HEXDISPLAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"hexshow",4,ITEM_HEXSHOW,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"idline",2,ITEM_IDLINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"impmacro",6,ITEM_IMPMACRO,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"impos",5,ITEM_IMPOS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"insertmode",6,ITEM_INSERTMODE,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"lastmsg",4,ITEM_LASTMSG,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"lastrc",6,ITEM_LASTRC,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"length",3,ITEM_LENGTH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"line",2,ITEM_LINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"linend",5,ITEM_LINEND,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"macroext",6,ITEM_MACROEXT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"macropath",6,ITEM_MACROPATH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"margins",3,ITEM_MARGINS,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"monitor",7,ITEM_MONITOR,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"msgline",4,ITEM_MSGLINE,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"msgmode",4,ITEM_MSGMODE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"nbfile",3,ITEM_NBFILE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"newlines",4,ITEM_NEWLINES,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"nondisp",4,ITEM_NONDISP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"number",3,ITEM_NUMBER,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"pending",4,ITEM_PENDING,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"point",1,ITEM_POINT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"position",3,ITEM_POSITION,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"prefix",3,ITEM_PREFIX,0},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"printer",7,ITEM_PRINTER,1},
   {QUERY_READV                                        ,(CHARTYPE *)"readv",5,ITEM_READV,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"reprofile",6,ITEM_REPROFILE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"reserved",5,ITEM_RESERVED,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"rexxoutput",7,ITEM_REXXOUTPUT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"scale",4,ITEM_SCALE,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"scope",5,ITEM_SCOPE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"select",3,ITEM_SELECT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"shadow",4,ITEM_SHADOW,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"size",2,ITEM_SIZE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"statusline",7,ITEM_STATUSLINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"stay",4,ITEM_STAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabkey",4,ITEM_TABKEY,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabline",4,ITEM_TABLINE,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabs",4,ITEM_TABS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabsin",5,ITEM_TABSIN,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabsout",5,ITEM_TABSOUT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"terminal",4,ITEM_TERMINAL,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"tof",3,ITEM_TOF,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"verify",1,ITEM_VERIFY,1},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"vershift",4,ITEM_VERSHIFT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"version",7,ITEM_VERSION,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"width",1,ITEM_WIDTH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"wordwrap",5,ITEM_WORDWRAP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"zone",1,ITEM_ZONE,2},
   {0                                                   ,NULL,0,0,0},
  };
CHARTYPE *block_name[] = {
                          (CHARTYPE *)"",
                          (CHARTYPE *)"LINE",
                          (CHARTYPE *)"BOX",
                          (CHARTYPE *)"STREAM",
                          (CHARTYPE *)"COLUMN",
                          (CHARTYPE *)"WORD"
                         };

VALUE item_values[MAX_VARIABLES_RETURNED];
/***********************************************************************/
#ifdef PROTO
short find_item(CHARTYPE *item_name,CHARTYPE query_type)
#else
short find_item(item_name,query_type)
CHARTYPE *item_name;
CHARTYPE query_type;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short itemno = (-1);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :find_item");
#endif
/*---------------------------------------------------------------------*/
/* Using the supplied abbreviation for an item, find the entry in the  */
/* query_item[] array. It should return with the index pointer (for    */
/* QUERY_EXTRACT) or an index to the first item in the function_item[] */
/* rexx.c for other query types.                                       */
/*---------------------------------------------------------------------*/
 for (i=0;query_item[i].name != NULL;i++)
    {
     if (equal(query_item[i].name,item_name,query_item[i].min_len)
     && (query_item[i].query & query_type))
       {
        if (query_type == QUERY_EXTRACT)
           itemno = i;
        else
           itemno = query_item[i].item_number;
        break;
       }
    }
#ifdef TRACE
 trace_return();
#endif
 return(itemno);
}
/***********************************************************************/
#ifdef PROTO
short show_status(void)
#else
short show_status()
#endif
/***********************************************************************/
{
#define STATUS_COLS 4
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0,k=0;
 short lineno=0,colno=0;
 short number_variables=0;
 short item_width=0,column=0,column_width=0,col[STATUS_COLS];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :show_status");
#endif
/*---------------------------------------------------------------------*/
/* For each item that is displayable, display it...                    */
/*---------------------------------------------------------------------*/
 column = 0;
 column_width = COLS / STATUS_COLS;
 col[0] = 0;
 for (i=1;i<STATUS_COLS;i++)
    col[i] = col[i-1]+column_width+1;
 wclear(stdscr);
 for (i=0;query_item[i].name != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* Get only those settings that are queryable...                       */
/*---------------------------------------------------------------------*/
     if (query_item[i].query & QUERY_STATUS)
       {
        number_variables = get_item_values(query_item[i].item_number,"",QUERY_STATUS,0L,NULL,0L);
        item_width = 0;
/*---------------------------------------------------------------------*/
/* Obtain the total length of the setting values...                    */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
           item_width += item_values[j].len+1;
        item_width--;     /* reduce by 1 for last value's blank at end */
/*---------------------------------------------------------------------*/
/* If the length of the variables is > the screen width, go to next    */
/* line.                                                               */
/*---------------------------------------------------------------------*/
        if (item_width+col[column] > COLS)
          {
           column = colno = 0;
           lineno++;
          }
        else
           colno = col[column];
/*---------------------------------------------------------------------*/
/* Display the variables. For the first value, display in BOLD.        */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
          {
           if (j == 0)
              attrset(A_BOLD);
           mvaddstr(lineno,colno,item_values[j].value);
           attrset(A_NORMAL);
           colno += item_values[j].len+1;
          }
        colno--;
/*---------------------------------------------------------------------*/
/* Now have to determine where to display the next set of variables... */
/* If the just-displayed values go past the last column, or we are     */
/* already in the last column, go to the next line...                  */
/*---------------------------------------------------------------------*/
        if (colno >= col[STATUS_COLS-1] || column == STATUS_COLS-1)
          {
           lineno++;
           column = 0;
          }
        else
          {
/*---------------------------------------------------------------------*/
/* ...else if the the just-displayed values go past the 2nd column...  */
/* If the just-displayed values go past the last column, go on to the  */
/* next line...                                                        */
/*---------------------------------------------------------------------*/
           for (k=0;k<STATUS_COLS;k++)
              if (colno < col[k])
                {
                 column = k;
                 break;
                }
          }
       }
    }
 mvaddstr(LINES-2,0,HIT_ANY_KEY);
 refresh();
 (void)my_getch(stdscr);
 wclear(stdscr);
 refresh();

 restore_THE();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short save_status(CHARTYPE *filename)
#else
short save_status(filename)
CHARTYPE *filename;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE sp_path[MAX_FILE_NAME+1] ;
 extern CHARTYPE sp_fname[MAX_FILE_NAME+1] ;
 extern CHARTYPE *the_version;
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 short number_variables=0,rc=RC_OK;
 FILE *fp=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :save_status");
#endif
/*---------------------------------------------------------------------*/
/* Get the fully qualified filename from the supplied filename.        */
/*---------------------------------------------------------------------*/
 strtrans(filename,OSLASH,ISLASH);
 if ((rc = splitpath(filename)) != RC_OK)
   {
    display_error(10,filename,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* splitpath() has set up sp_path  and sp_fname. Append the filename.  */
/*---------------------------------------------------------------------*/
 strcat(sp_path,sp_fname);
/*---------------------------------------------------------------------*/
/* If the file already exists, display an error.                       */
/*---------------------------------------------------------------------*/
  if (file_exists(sp_path))
   {
    display_error(8,filename,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 fp = fopen(sp_path,"w");
/*---------------------------------------------------------------------*/
/* For each item that is modifiable, get its value...                  */
/*---------------------------------------------------------------------*/
 fprintf(fp,"/* This file generated by STATUS command of THE %s */\n",the_version);
 for (i=0;query_item[i].name != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* Get only those settings that are modifiable...                      */
/*---------------------------------------------------------------------*/
     if (query_item[i].query & QUERY_MODIFY)
       {
        number_variables = get_item_values(query_item[i].item_number,"",QUERY_MODIFY,0L,NULL,0L);
        fputs("'set",fp);
/*---------------------------------------------------------------------*/
/* Write the variables to the file...                                  */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
          {
           fputc(' ',fp);
           fputs(item_values[j].value,fp);
          }
        fprintf(fp,"'\n");
       }
    }
 fflush(fp);
 fclose(fp);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
short set_extract_variables(short itemno)
#else
short set_extract_variables(itemno)
short itemno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[MAX_VARIABLES_RETURNED];
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short rc=RC_OK,number_values=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("rexx.c:    set_extract_variables");
#endif
 number_values = atoi(item_values[0].value);
 for (i=0;i<number_values+1;i++)
  {
   if ((rc = set_rexx_variable(query_item[itemno].name,item_values[i].value,i)) != RC_OK)
      break;
  } 
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
short get_item_values(short itemno,CHARTYPE *itemargs,CHARTYPE query_type,LINETYPE argc,CHARTYPE *arg,LINETYPE arglen)
#else
short get_item_values(itemno,itemargs,query_type,argc,arg,arglen)
short itemno;
CHARTYPE *itemargs;
CHARTYPE query_type;
LINETYPE argc;
CHARTYPE *arg;
LINETYPE arglen;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE term_name[20];
 extern CHARTYPE dir_path[MAX_FILE_NAME+1];
 extern CHARTYPE *temp_cmd;
 extern unsigned short file_start;
 extern CHARTYPE *the_version;
 extern CHARTYPE the_macro_path[MAX_FILE_NAME+1];
 extern CHARTYPE macro_suffix[12];
 extern CHARTYPE CLEARSCREENx;
 extern CHARTYPE CMDARROWSTABCMDx;
 extern bool REPROFILEx;
 extern bool CAPREXXOUTx;
 extern LINETYPE CAPREXXMAXx;
 extern CHARTYPE TABI_ONx;
 extern CHARTYPE TABI_Nx;
 extern bool BEEPx;
 extern bool CLOCKx;
 extern bool HEXDISPLAYx;
 extern CHARTYPE number_of_files;
 extern bool INSERTMODEx;
 extern CHARTYPE tabkey_insert;
 extern CHARTYPE tabkey_overwrite;
 extern short lastrc;
 extern CHARTYPE last_message[160];
 extern bool in_profile;
 extern bool rexx_support;
 extern unsigned short rec_len;
 extern unsigned short cmd_rec_len;
 extern unsigned short pre_rec_len;
 extern bool colour_support;
 extern bool ETMODEx;
 extern bool NONDISPx;
 extern ROWTYPE STATUSLINEx;
 extern LINETYPE prefix_current_line;
 extern bool in_prefix_macro;
 extern CHARTYPE *cmd_rec;
 extern CHARTYPE *rec;
 extern short compatible;
 extern LINETYPE original_screen_line;
 extern LINETYPE original_screen_column;
 extern LINETYPE original_file_line;
 extern LINETYPE original_file_column;

#if defined(UNIX) || defined(OS2)
 extern CHARTYPE *spooler_name;
#endif
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 static CHARTYPE num0[3];
 static CHARTYPE num1[10];
 static CHARTYPE num2[10];
 static CHARTYPE num3[40];
 static CHARTYPE num4[40];
 static CHARTYPE num5[10];
 static CHARTYPE num6[10];
 static CHARTYPE num7[10];
 static CHARTYPE num8[10];
 static CHARTYPE rsrvd[512];
 static LINE *curr;
 static RESERVED *curr_rsrvd;
 static TARGET target={NULL,0L,0L,NULL,0,0,FALSE};
 register int i=0;
 short y=0,x=0;
 short number_variables = query_item[itemno].number_values;
 LINETYPE true_line=0L,num_lines=0L;
 bool bool_flag=FALSE;
 CHARTYPE cursor_char=0;
 CHARTYPE *tmpbuf=NULL;
 CHARTYPE *tarbuf=NULL;     /* only to be used in valid_target section */
 short target_type=0;
 short rc=RC_OK;
 LINETYPE current_screen_line=(-1L);
 LINETYPE current_screen_column=(-1L);
 LINETYPE current_file_line=(-1L);
 LINETYPE current_file_column=(-1L);
 int key=0;
 CHARTYPE *keyname=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   get_item_values");
#endif
/*---------------------------------------------------------------------*/
/* Now that we have a valid item, set up its values...                 */
/*---------------------------------------------------------------------*/
 switch(itemno)
   {
    case ITEM_AFTER_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if ((x+CURRENT_VIEW->verify_col-1) >= rec_len)
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (x >= cmd_rec_len)
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:
                 if (x >= pre_rec_len)
                    bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_ALT:
         sprintf(num1,"%d",CURRENT_FILE->autosave_alt);
         sprintf(num2,"%d",CURRENT_FILE->save_alt);
         item_values[1].value = num1;
         item_values[2].value = num2;
         item_values[1].len = strlen(num1);
         item_values[2].len = strlen(num2);
         break;
    case ITEM_ARBCHAR:
         if (CURRENT_VIEW->arbchar_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         num1[0] = CURRENT_VIEW->arbchar_multiple;
         num1[1] ='\0';
         item_values[2].value = num1;
         item_values[2].len = 1;
         num2[0] = CURRENT_VIEW->arbchar_single;
         num2[1] ='\0';
         item_values[3].value = num2;
         item_values[3].len = 1;
         break;
    case ITEM_AUTOSAVE:
         sprintf(num1,"%d",CURRENT_FILE->autosave);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         if (CURRENT_FILE->autosave == 0)
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_BACKUP:
         switch(CURRENT_FILE->backup)
           {
            case BACKUP_OFF:
                 item_values[1].value = (CHARTYPE *)"OFF";
                 break;
            case BACKUP_TEMP:
                 item_values[1].value = (CHARTYPE *)"TEMP";
                 break;
            case BACKUP_KEEP:
                 item_values[1].value = (CHARTYPE *)"KEEP";
                 break;
           }
         item_values[1].len = strlen(item_values[1].value);
         break;
    case ITEM_BEFORE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if ((x+CURRENT_VIEW->verify_col-1) < memne(rec,' ',rec_len))
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (x < memne(cmd_rec,' ',cmd_rec_len))
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:/* cursor can't go before 1st non-blank */
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_BEEP:
         if (BEEPx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_BLANK_FUNCTION:
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if (rec_len == 0)
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:
                 if (pre_rec_len == 0)
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (cmd_rec_len == 0)
                    bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_BLOCK:
         if (MARK_VIEW == NULL)
           {
            item_values[1].value = (CHARTYPE *)"NONE";
            item_values[1].len = 4;
            number_variables = 1;
           }
          else
           {
            number_variables = 6;
            item_values[1].value = block_name[MARK_VIEW->mark_type];
            item_values[1].len = strlen(item_values[1].value);
            sprintf(num1,"%d",MARK_VIEW->mark_start_line);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
            sprintf(num2,"%d",MARK_VIEW->mark_start_col);
            item_values[3].value = num2;
            item_values[3].len = strlen(num2);
            sprintf(num3,"%d",MARK_VIEW->mark_end_line);
            item_values[4].value = num3;
            item_values[4].len = strlen(num3);
            sprintf(num4,"%d",MARK_VIEW->mark_end_col);
            item_values[5].value = num4;
            item_values[5].len = strlen(num4);
            strcpy(temp_cmd,MARK_FILE->fpath);
            strcat(temp_cmd,MARK_FILE->fname);
            item_values[6].value = (CHARTYPE *)temp_cmd;
            item_values[6].len = strlen(temp_cmd);
           }
         break;
    case ITEM_BLOCK_FUNCTION:
         if (CURRENT_VIEW == MARK_VIEW)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_BOTTOMEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  y == CURRENT_SCREEN.rows[WINDOW_MAIN]-1)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_CASE:
         switch(CURRENT_VIEW->case_enter)
           {
            case CASE_MIXED:
                 item_values[1].value = (CHARTYPE *)"MIXED";
                 break;
            case CASE_UPPER:
                 item_values[1].value = (CHARTYPE *)"UPPER";
                 break;
            case CASE_LOWER:
                 item_values[1].value = (CHARTYPE *)"LOWER";
                 break;
            default:
                 break;
           }
         item_values[1].len = 5;
         switch(CURRENT_VIEW->case_locate)
           {
            case CASE_IGNORE:
                 item_values[2].value = (CHARTYPE *)"IGNORE";
                 item_values[2].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[2].value = (CHARTYPE *)"RESPECT";
                 item_values[2].len = 7;
                 break;
            default:
                 break;
           }
         switch(CURRENT_VIEW->case_change)
           {
            case CASE_IGNORE:
                 item_values[3].value = (CHARTYPE *)"IGNORE";
                 item_values[3].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[3].value = (CHARTYPE *)"RESPECT";
                 item_values[3].len = 7;
                 break;
            default:
                 break;
           }
         switch(CURRENT_VIEW->case_sort)
           {
            case CASE_IGNORE:
                 item_values[4].value = (CHARTYPE *)"IGNORE";
                 item_values[4].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[4].value = (CHARTYPE *)"RESPECT";
                 item_values[4].len = 7;
                 break;
            default:
                 break;
           }
         break;
    case ITEM_CLEARSCREEN:
         if (CLEARSCREENx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_CLOCK:
         if (CLOCKx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_COMMAND_FUNCTION:
         if (CURRENT_WINDOW_COMMAND != NULL)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_CMDARROWS:
         if (CMDARROWSTABCMDx)
           {
            item_values[1].value = (CHARTYPE *)"TAB";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"RETRIEVE";
            item_values[1].len = 8;
           }
         break;
    case ITEM_CMDLINE:
         switch(CURRENT_VIEW->cmd_line)
           {
            case 'B':
                 item_values[1].value = (CHARTYPE *)"BOTTOM";
                 item_values[1].len = 6;
                 break;
            case 'T':
                 item_values[1].value = (CHARTYPE *)"TOP";
                 item_values[1].len = 3;
                 break;
            case 'O':
                 item_values[1].value = (CHARTYPE *)"OFF";
                 item_values[1].len = 3;
                 number_variables = 1;
                 break;
           }
         if (CURRENT_VIEW->cmd_line == 'O')
            break;
         if (query_type == QUERY_EXTRACT
         ||  query_type == QUERY_FUNCTION)
           {
            sprintf(num1,"%d",CURRENT_SCREEN.start_row[WINDOW_COMMAND]+1);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
            item_values[3].value = cmd_rec;
            item_values[3].len = cmd_rec_len;
            number_variables = 3;
           }
         else
            number_variables = 1;
         break;
    case ITEM_COMPAT:
         switch(compatible)
           {
            case COMPAT_THE:
                 item_values[1].value = (CHARTYPE *)"THE";
                 item_values[1].len = 3;
                 break;
            case COMPAT_XEDIT:
                 item_values[1].value = (CHARTYPE *)"XEDIT";
                 item_values[1].len = 5;
                 break;
            case COMPAT_KEDIT:
                 item_values[1].value = (CHARTYPE *)"KEDIT";
                 item_values[1].len = 5;
                 break;
           }
         break;
    case ITEM_CURLINE:
         if (CURRENT_VIEW->current_base == POSITION_MIDDLE)
            sprintf(rsrvd,"M%+d",CURRENT_VIEW->current_off);
         else
            sprintf(rsrvd,"%d",CURRENT_VIEW->current_off);
         item_values[1].value = rsrvd;
         item_values[1].len = strlen(rsrvd);
         if (query_type == QUERY_EXTRACT
         ||  query_type == QUERY_FUNCTION)
           {
            sprintf(num1,"%d",CURRENT_VIEW->current_row+1);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
            curr = lll_find(CURRENT_FILE->first_line,get_true_line());
            item_values[3].value = (CHARTYPE *)curr->line;
            item_values[3].len = curr->length;
           }
         else
            number_variables = 1;
         break;
    case ITEM_CURSOR:
         get_cursor_position(&current_screen_line,&current_screen_column,
                             &current_file_line,&current_file_column);
         sprintf(num1,"%d",current_screen_line);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         sprintf(num2,"%d",current_screen_column);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         sprintf(num3,"%d",current_file_line);
         item_values[3].value = num3;
         item_values[3].len = strlen(num3);
         sprintf(num4,"%d",current_file_column);
         item_values[4].value = num4;
         item_values[4].len = strlen(num4);
         sprintf(num5,"%d",original_screen_line);
         item_values[5].value = num5;
         item_values[5].len = strlen(num5);
         sprintf(num6,"%d",original_screen_column);
         item_values[6].value = num6;
         item_values[6].len = strlen(num6);
         sprintf(num7,"%d",original_file_line);
         item_values[7].value = num7;
         item_values[7].len = strlen(num7);
         sprintf(num8,"%d",original_file_column);
         item_values[8].value = num8;
         item_values[8].len = strlen(num8);
         break;
    case ITEM_CURRENT_FUNCTION:
         if (CURRENT_VIEW->current_window != WINDOW_COMMAND
         &&  CURRENT_VIEW->focus_line == CURRENT_VIEW->current_line)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_DIR_FUNCTION:
         if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_DIRFILEID:
         true_line = (-1L);
         if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
           {
            if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
               if (CURRENT_TOF || CURRENT_BOF)
                  true_line = (-1L);
               else
                  true_line = CURRENT_VIEW->current_line;
            else
               if (FOCUS_TOF || FOCUS_BOF)
                  true_line = (-1L);
               else
                  true_line = CURRENT_VIEW->focus_line;
           }
         if (true_line == (-1L))
           {
            item_values[1].value = (CHARTYPE *)"";
            item_values[1].len = 0;
            item_values[2].value = (CHARTYPE *)"";
            item_values[2].len = 0;
           }
         else
           {
            curr = lll_find(CURRENT_FILE->first_line,true_line);
            item_values[1].value = (CHARTYPE *)dir_path;
            item_values[1].len = strlen(dir_path);
            if (curr->length < file_start)
              {
               item_values[2].value = (CHARTYPE *)"";
               item_values[2].len = 0;
              }
            else
              {
               item_values[2].value = (CHARTYPE *)curr->line+file_start;
               item_values[2].len = strlen(curr->line+file_start);
              }
           }
         break;
    case ITEM_END_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  (x+CURRENT_VIEW->verify_col) == rec_len)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_DISPLAY:
         sprintf(num1,"%d",CURRENT_VIEW->display_low);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         sprintf(num2,"%d",CURRENT_VIEW->display_high);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         break;
    case ITEM_EOF:
         if (CURRENT_BOF)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_EOF_FUNCTION:
         if (FOCUS_BOF && CURRENT_VIEW->current_window != WINDOW_COMMAND)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_EOLOUT:
         if (CURRENT_FILE->eolout == EOLOUT_LF)
           {
            item_values[1].value = (CHARTYPE *)"LF";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"CRLF";
            item_values[1].len = 4;
           }
         break;
    case ITEM_ETMODE:
         if (ETMODEx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_FIRST_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (x == 0)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_FOCUSEOF_FUNCTION:
         if (FOCUS_BOF)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_FOCUSTOF_FUNCTION:
         if (FOCUS_TOF)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_FNAME:
         item_values[1].value = (CHARTYPE *)CURRENT_FILE->fname;
         item_values[1].len = strlen(CURRENT_FILE->fname);
         break;
    case ITEM_FPATH:
         item_values[1].value = (CHARTYPE *)CURRENT_FILE->fpath;
         item_values[1].len = strlen(CURRENT_FILE->fpath);
         break;
    case ITEM_FTYPE:
         x = strzreveq(CURRENT_FILE->fname,'.');
         if (x == (-1))
           {
            item_values[1].value = (CHARTYPE *)"";
            item_values[1].len = 0;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)CURRENT_FILE->fname+x+1;
            item_values[1].len = strlen(CURRENT_FILE->fname+x+1);
           }
         break;
    case ITEM_GETENV:
         if (query_type == QUERY_FUNCTION)
            tmpbuf = (CHARTYPE *)getenv(arg);
         else
            tmpbuf = (CHARTYPE *)getenv(itemargs);
         if (tmpbuf == NULL)
            item_values[1].value = (CHARTYPE *)"***invalid***";
         else
            item_values[1].value = tmpbuf;
         item_values[1].len = strlen(item_values[1].value);
         break;
    case ITEM_HEX:
         if (CURRENT_VIEW->hex)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_HEXDISPLAY:
         if (HEXDISPLAYx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_HEXSHOW:
         if (CURRENT_VIEW->hexshow_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->hexshow_base == POSITION_MIDDLE)
            sprintf(rsrvd,"M%+d",CURRENT_VIEW->hexshow_off);
         else
            sprintf(rsrvd,"%d",CURRENT_VIEW->hexshow_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen(rsrvd);
         break;
    case ITEM_IDLINE:
         if (CURRENT_VIEW->id_line)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_IMPMACRO:
         if (CURRENT_VIEW->imp_macro)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_IMPOS:
         if (CURRENT_VIEW->imp_os)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_INBLOCK_FUNCTION:
         bool_flag = FALSE;
         if (CURRENT_VIEW == MARK_VIEW
         &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
           {
            getyx(CURRENT_WINDOW_MAIN,y,x);
            switch(MARK_VIEW->mark_type)
              {
               case M_LINE:
                    if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                    &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line))
                      bool_flag = TRUE;
                    break;
               case M_BOX:
                    if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                    &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
                    &&  (x + CURRENT_VIEW->verify_start >= MARK_VIEW->mark_start_col)
                    &&  (x + CURRENT_VIEW->verify_start <= MARK_VIEW->mark_end_col))
                      bool_flag = TRUE;
                    break;
               default:
                    break;
              }
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INCOMMAND_FUNCTION:
         if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INITIAL_FUNCTION:
         if (in_profile)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INPREFIX_FUNCTION:
         if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INSERTMODE:
         if (INSERTMODEx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_LASTMSG:
         item_values[1].value = (CHARTYPE *)last_message;
         item_values[1].len = strlen(last_message);
         break;
    case ITEM_LASTRC:
         sprintf(num1,"%d",lastrc);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_LEFTEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  x == 0)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_LENGTH:
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_VIEW->current_line);
         sprintf(num1,"%d",curr->length);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_LINE:
         sprintf(num1,"%d",get_true_line());
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_LINEND:
         if (CURRENT_VIEW->linend_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         num1[0] = CURRENT_VIEW->linend_value;
         num1[1] = '\0';
         item_values[2].value = num1;
         item_values[2].len = 1;
         break;
    case ITEM_MACROEXT:
         if (strlen(macro_suffix) == 0)
            item_values[1].value = (CHARTYPE *)macro_suffix;
         else
            item_values[1].value = (CHARTYPE *)macro_suffix+1;
         item_values[1].len = strlen(macro_suffix);
         break;
    case ITEM_MACROPATH:
         item_values[1].value = (CHARTYPE *)the_macro_path;
         item_values[1].len = strlen(the_macro_path);
         break;
    case ITEM_MARGINS:
         sprintf(num1,"%d",CURRENT_VIEW->margin_left);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         sprintf(num2,"%d",CURRENT_VIEW->margin_right);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         if (CURRENT_VIEW->margin_indent_offset)
            sprintf(num3,"%+d",CURRENT_VIEW->margin_indent);
         else
            sprintf(num3,"%d",CURRENT_VIEW->margin_indent);
         item_values[3].value = num3;
         item_values[3].len = strlen(num3);
         break;
    case ITEM_MODIFIABLE_FUNCTION:
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 getyx(CURRENT_WINDOW,y,x);
                 if (FOCUS_TOF 
                 ||  FOCUS_BOF
                 ||  CURRENT_SCREEN.sl[y].line_type == LINE_SHADOW)
                    bool_flag = FALSE;
                 else
                    bool_flag = TRUE;
                 break;
            default:
                 bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_MONITOR:
#ifdef A_COLOR
         if (colour_support)
           {
            item_values[1].value = (CHARTYPE *)"COLOR";
            item_values[1].len = 5;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"MONO";
            item_values[1].len = 4;
           }
          item_values[2].value = (CHARTYPE *)"COLOR";
          item_values[2].len = 5;
          break;
#else
          item_values[1].value = (CHARTYPE *)"MONO";
          item_values[1].len = 4;
          item_values[2].value = (CHARTYPE *)"MONO";
          item_values[2].len = 4;
         break;
#endif
    case ITEM_MSGLINE:
         item_values[1].value = (CHARTYPE *)"ON";
         item_values[1].len = 2;
         if (CURRENT_VIEW->msgline_base == POSITION_MIDDLE)
            sprintf(rsrvd,"M%+d",CURRENT_VIEW->msgline_off);
         else
            sprintf(rsrvd,"%d",CURRENT_VIEW->msgline_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen(rsrvd);
         sprintf(num1,"%d",CURRENT_VIEW->msgline_rows);
         item_values[3].value = num1;
         item_values[3].len = strlen(num1);
         item_values[4].value = (CHARTYPE *)"OVERLAY";
         item_values[4].len = 7;
         break;
    case ITEM_MSGMODE:
         if (CURRENT_VIEW->msgmode_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_NBFILE:
         sprintf(num1,"%d",number_of_files);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_NEWLINES:
         if (CURRENT_VIEW->newline_aligned)
           {
            item_values[1].value = (CHARTYPE *)"ALIGNED";
            item_values[1].len = 7;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"LEFT";
            item_values[1].len = 4;
           }
         break;
    case ITEM_NONDISP:
         num1[0] = NONDISPx;
         num1[1] ='\0';
         item_values[1].value = num1;
         item_values[1].len = 1;
         break;
    case ITEM_NUMBER:
         if (CURRENT_VIEW->number)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_PENDING:
         number_variables = extract_pending(itemno,itemargs);
         break;
    case ITEM_POINT:
         switch(query_type)
           {
            case QUERY_STATUS:
            case QUERY_QUERY:
            case QUERY_MODIFY:
                 true_line = get_true_line();
                 curr = lll_find(CURRENT_FILE->first_line,true_line);
                 if (curr->name == NULL)  /* line not named */
                   {
                    item_values[1].value = (CHARTYPE *)"";
                    item_values[1].len = 0;
                   }
                 else
                   {
                    item_values[1].value = curr->name;
                    item_values[1].len = strlen(curr->name);
                   }
                 break;
            default:
                 number_variables = extract_point(itemno,itemargs);
                 break;
           }
         break;
    case ITEM_POSITION:
         if (CURRENT_VIEW->position_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         get_current_position(&true_line,&x);
         sprintf(num1,"%ld",true_line);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         sprintf(num2,"%d",x);
         item_values[3].value = num2;
         item_values[3].len = strlen(num2);
         break;
    case ITEM_PREFIX:
         if (rexx_support)
           {
            if (strcmp(itemargs,"") != 0)
              {
               number_variables = extract_prefix(itemno,itemargs);
               break;
              }
           }
         else
           {
            if (strcmp(itemargs,"") != 0)
              {
               display_error(2,(CHARTYPE *)itemargs,FALSE);
               number_variables = EXTRACT_ARG_ERROR;
               break;
              }
           }
         if (CURRENT_VIEW->prefix == PREFIX_OFF)
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
            item_values[2].value = (CHARTYPE *)"";  /* this set to empty deliberately */
            item_values[2].len = 0;
            number_variables = 1;
            break;
           }
         if ((CURRENT_VIEW->prefix&PREFIX_STATUS_MASK) == PREFIX_ON)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"NULLS";
            item_values[1].len = 5;
           }
         if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT)
           {
            item_values[2].value = (CHARTYPE *)"LEFT";
            item_values[2].len = 4;
           }
         else
           {
            item_values[2].value = (CHARTYPE *)"RIGHT";
            item_values[2].len = 5;
           }
         number_variables = 2;
         break;
    case ITEM_PRINTER:
#if defined(UNIX) || defined(OS2)
         item_values[1].value = (CHARTYPE *)spooler_name;
         item_values[1].len = strlen(spooler_name);
#else
         item_values[1].value = (CHARTYPE *)"LPT1";
         item_values[1].len = 4;
#endif
         break;
    case ITEM_REPROFILE:
         if (REPROFILEx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_READV:
         key = my_getch(stdscr);
         keyname = get_key_name(key);
         if (keyname == NULL)
           {
            item_values[1].value = (CHARTYPE *)"";
            item_values[1].len = 0;
           }
         else
           {
            item_values[1].value = keyname;
            item_values[1].len = strlen(keyname);
           }
         if (key < 256)
           {
            sprintf(num1,"%d",key);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
           }
         else
           {
            item_values[2].value = (CHARTYPE *)"";
            item_values[2].len = 0;
           }
         sprintf(num2,"%d",key);
         item_values[3].value = num2;
         item_values[3].len = strlen(num2);
         break;
    case ITEM_RESERVED:
         curr_rsrvd = CURRENT_FILE->first_reserved;
         strcpy(rsrvd,"");
         while(curr_rsrvd != NULL)
           {
            y = strlen(curr_rsrvd->spec) + 1;
            if ((x + y) > 80)
               break;
            strcat(rsrvd,curr_rsrvd->spec);
            strcat(rsrvd," ");
            x += y;
            curr_rsrvd = curr_rsrvd->next;
           }
         if (x == 0)
            number_variables = 0;
          else
           {
            item_values[1].value = rsrvd;
            item_values[1].len = x;
           }
         break;
    case ITEM_REXXOUTPUT:
         if (CAPREXXOUTx)
           {
            item_values[1].value = (CHARTYPE *)"FILE";
            item_values[1].len = 4;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"DISPLAY";
            item_values[1].len = 7;
           }
         sprintf(num1,"%d",CAPREXXMAXx);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
    case ITEM_RIGHTEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  x == getmaxx(CURRENT_WINDOW)-1)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_SCOPE:
         if (CURRENT_VIEW->scope_all)
           {
            item_values[1].value = (CHARTYPE *)"ALL";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"DISPLAY";
            item_values[1].len = 7;
           }
         break;
    case ITEM_SCALE:
         if (CURRENT_VIEW->scale_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->scale_base == POSITION_MIDDLE)
            sprintf(rsrvd,"M%+d",CURRENT_VIEW->scale_off);
         else
            sprintf(rsrvd,"%d",CURRENT_VIEW->scale_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen(rsrvd);
         break;
    case ITEM_SELECT:
         curr = lll_find(CURRENT_FILE->first_line,get_true_line());
         sprintf(num1,"%d",curr->select);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         x = 0;
         curr = lll_find(CURRENT_FILE->first_line,1L);
         while(curr->next != NULL)
           {
            if (curr->select > x)
               x = curr->select;
            curr = curr->next;
           }
         sprintf(num2,"%d",x);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         break;
    case ITEM_SHADOW:
         if (CURRENT_VIEW->shadow)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_SIZE:
         sprintf(num1,"%d",CURRENT_FILE->number_lines);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_SPACECHAR_FUNCTION:
         cursor_char = (CHARTYPE)(winch(CURRENT_WINDOW) & A_CHARTEXT);
         if (cursor_char == ' ')
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_STATUSLINE:
         switch(STATUSLINEx)
           {
            case 'B':
                 item_values[1].value = (CHARTYPE *)"BOTTOM";
                 item_values[1].len = 6;
                 break;
            case 'T':
                 item_values[1].value = (CHARTYPE *)"TOP";
                 item_values[1].len = 3;
                 break;
            case 'O':
                 item_values[1].value = (CHARTYPE *)"OFF";
                 item_values[1].len = 3;
                 break;
            case 'G':
                 item_values[1].value = (CHARTYPE *)"GUI";
                 item_values[1].len = 3;
                 break;
           }
         break;
    case ITEM_STAY:
         if (CURRENT_VIEW->stay)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_TABKEY:
         if (tabkey_insert == 'T')
           {
            item_values[1].value = (CHARTYPE *)"TAB";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"CHARACTER";
            item_values[1].len = 9;
           }
         if (tabkey_overwrite == 'T')
           {
            item_values[2].value = (CHARTYPE *)"TAB";
            item_values[2].len = 3;
           }
          else
           {
            item_values[2].value = (CHARTYPE *)"CHARACTER";
            item_values[2].len = 9;
           }
         break;
    case ITEM_TABLINE:
         if (CURRENT_VIEW->tab_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->tab_base == POSITION_MIDDLE)
            sprintf(rsrvd,"M%+d",CURRENT_VIEW->tab_off);
         else
            sprintf(rsrvd,"%d",CURRENT_VIEW->tab_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen(rsrvd);
         break;
    case ITEM_TABS:
         strcpy(rsrvd,"");
         for (i=0;i<CURRENT_VIEW->numtabs;i++)
           {
            sprintf(num1,"%d ",CURRENT_VIEW->tabs[i]);
            strcat(rsrvd,num1);
           }
         if (query_type == QUERY_STATUS
         ||  query_type == QUERY_QUERY
         ||  query_type == QUERY_MODIFY)
            rsrvd[COLS-7] = '\0';
         item_values[1].value = rsrvd;
         item_values[1].len = strlen(rsrvd);
         break;
    case ITEM_TABSIN:
         if (TABI_ONx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         sprintf(num1,"%d",TABI_Nx);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
    case ITEM_TABSOUT:
         if (CURRENT_FILE->tabsout_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         sprintf(num1,"%d",CURRENT_FILE->tabsout_num);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
    case ITEM_TERMINAL:
         item_values[1].value = term_name;
         item_values[1].len = strlen(term_name);
         break;
    case ITEM_TOF:
         if (CURRENT_TOF)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_TOF_FUNCTION:
         if (FOCUS_TOF && CURRENT_VIEW->current_window != WINDOW_COMMAND)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_TOPEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  y == 0)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_VALID_TARGET_FUNCTION:
         target_type=TARGET_NORMAL|TARGET_BLOCK_CURRENT|TARGET_ALL;
         if (argc < 1 || argc > 2)  /* incorrect no of arguments - error */
           {
            item_values[1].value = (CHARTYPE *)"ERROR";
            item_values[1].len = 5;
            break;
           }
         if (argc == 2)
            target_type = target_type | TARGET_SPARE;
/* allocate sufficient space for the spare string and 2 longs */
         if (tarbuf == NULL)
            tarbuf = (CHARTYPE *)(*the_malloc)(sizeof(CHARTYPE)*(arglen+30));
         else
            tarbuf = (CHARTYPE *)(*the_realloc)(tarbuf,sizeof(CHARTYPE)*(arglen+30));
         if (tarbuf == (CHARTYPE *)NULL)
           {
            item_values[1].value = (CHARTYPE *)"ERROR";
            item_values[1].len = 5;
            free_target(&target);
            break;
           }
         memcpy(tarbuf,arg,arglen);
         *(tarbuf+arglen) = '\0';
         if (in_prefix_macro)
            true_line = prefix_current_line;
         else
            true_line = get_true_line();

         initialise_target(&target);
         rc = validate_target(tarbuf,&target,target_type,true_line,FALSE,FALSE);
         if (rc == RC_TARGET_NOT_FOUND)
           {
            item_values[1].value = (CHARTYPE *)"NOTFOUND";
            item_values[1].len = 8;
            free_target(&target);
            break;
           }
         if (rc != RC_OK)
           {
            item_values[1].value = (CHARTYPE *)"ERROR";
            item_values[1].len = 5;
            free_target(&target);
            break;
           }
         if (argc == 2
         &&  target.spare != (-1))
            sprintf(tarbuf,"%ld %ld %s",target.true_line,target.num_lines,
                                        target.rt[target.spare].string);
         else
             sprintf(tarbuf,"%ld %ld",target.true_line,target.num_lines);
         item_values[1].value = tarbuf;
         item_values[1].len = strlen(tarbuf);
         free_target(&target);
         break;
    case ITEM_VERIFY:
         sprintf(num3,"%d %d",CURRENT_VIEW->verify_start,CURRENT_VIEW->verify_end);
         item_values[1].value = num3;
         item_values[1].len = strlen(num3);
         break;
    case ITEM_VERSHIFT:
         sprintf(rsrvd,"%ld",(LINETYPE)CURRENT_VIEW->verify_col - (LINETYPE)CURRENT_VIEW->verify_start);
         item_values[1].value = rsrvd;
         item_values[1].len = strlen(rsrvd);
         break;
    case ITEM_VERONE_FUNCTION:
         if (CURRENT_VIEW->verify_col == 1)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_VERSION:
         item_values[1].value = (CHARTYPE *)"THE";
         item_values[1].len = 3;
         item_values[2].value = (CHARTYPE *)the_version;
         item_values[2].len = strlen(the_version);
         item_values[3].value = (CHARTYPE *)"???";
#if defined(DOS)
         item_values[3].value = (CHARTYPE *)"DOS";
#endif
#if defined(OS2)
         item_values[3].value = (CHARTYPE *)"OS2";
#endif
#if defined(UNIX)
         item_values[3].value = (CHARTYPE *)"UNIX";
#endif
#if defined(MSWIN)
         item_values[3].value = (CHARTYPE *)"MS-WINDOWS";
#endif
         item_values[3].len = strlen(item_values[3].value);
         item_values[4].value = (CHARTYPE *)"26-Jan-95";
         item_values[4].len = strlen(item_values[4].value);
         break;
    case ITEM_WIDTH:
         sprintf(num1,"%d",max_line_length);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_WORDWRAP:
         if (CURRENT_VIEW->wordwrap)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_ZONE:
         sprintf(num1,"%d",CURRENT_VIEW->zone_start);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         sprintf(num2,"%d",CURRENT_VIEW->zone_end);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* If an "error" condition exists, do not set any values.              */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
    switch(query_type)
      {
       case QUERY_EXTRACT:
       case QUERY_FUNCTION:
       case QUERY_READV:
            sprintf(num0,"%d",number_variables);
            item_values[0].value = num0;
            item_values[0].len = strlen(num0);
            break;
       case QUERY_STATUS:
       case QUERY_QUERY:
       case QUERY_MODIFY:
            item_values[0].value = query_item[itemno].name;
            item_values[0].len = strlen(query_item[itemno].name);
            break;
       default:
            break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef PROTO
static short extract_point(short itemno,CHARTYPE *params)
#else
static short extract_point(itemno,params)
short itemno;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short number_variables = query_item[itemno].number_values;
 static CHARTYPE num4[15];
 LINE *curr=NULL;
 LINETYPE true_line=0L;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_point");
#endif
 if (strcmp(params,"") == 0) /* get name for focus line only */
   {
    true_line = get_true_line();
    curr = lll_find(CURRENT_FILE->first_line,true_line);
    if (curr->name == NULL)  /* line not named */
       number_variables = 0;
    else
      {
       sprintf(num4,"%d %s",true_line,curr->name);
       item_values[1].value = num4;
       item_values[1].len = strlen(num4);
       number_variables = 1;
      }
   }
 else
   {
    if (strcmp(params,"*") != 0)     /* argument must be "*" */
      {
       display_error(1,(CHARTYPE *)params,FALSE);
       number_variables = EXTRACT_ARG_ERROR;
      }
    else
      {
       curr = CURRENT_FILE->first_line;
       for(true_line=0,i=0;curr != NULL;true_line++)
         {
          if (curr->name != NULL)  /* line is named */
            {
             sprintf(num4,"%d %s",true_line,curr->name);
             rc = set_rexx_variable(query_item[itemno].name,num4,++i);
             if (rc == RC_SYSTEM_ERROR)
               {
                display_error(54,(CHARTYPE *)"",FALSE);
#ifdef TRACE
                trace_return();
#endif
                return(EXTRACT_ARG_ERROR);
               }
            }
          curr = curr->next;
         }
       sprintf(num4,"%d",i);
       rc = set_rexx_variable(query_item[itemno].name,num4,0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(CHARTYPE *)"",FALSE);
          number_variables = EXTRACT_ARG_ERROR;
         }
       else
          number_variables = EXTRACT_VARIABLES_SET;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef PROTO
static short extract_pending(short itemno,CHARTYPE *params)
#else
static short extract_pending(itemno,params)
short itemno;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
#define PEN_PARAMS  4
 CHARTYPE *word[PEN_PARAMS+1];
 unsigned short num_params=0;
 short number_variables = query_item[itemno].number_values;
 bool find_block=FALSE;
 bool find_oldname=FALSE;
 bool valid_args = FALSE;
 CHARTYPE *name=NULL;
 static CHARTYPE num1[7];
 PPC *curr_ppc=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_pending");
#endif
 num_params = param_split(params,word,PEN_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* No parameters, error.                                               */
/*---------------------------------------------------------------------*/
    case 0:
           display_error(3,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 1  parameter, only name.                                            */
/*---------------------------------------------------------------------*/
    case 1:
           if (equal((CHARTYPE *)"block",word[0],5)
           ||  equal((CHARTYPE *)"oldname",word[0],7))
             {
              display_error(3,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           name = word[0];
           break;
/*---------------------------------------------------------------------*/
/* 2  parameters, BLOCK and name or OLDNAME and name.                  */
/*---------------------------------------------------------------------*/
    case 2:
           if (equal((CHARTYPE *)"block",word[0],5)
           &&  equal((CHARTYPE *)"oldname",word[1],7))
             {
              display_error(3,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           if (equal((CHARTYPE *)"block",word[1],5)
           &&  equal((CHARTYPE *)"oldname",word[0],7))
             {
              display_error(3,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           if (equal((CHARTYPE *)"block",word[0],5))
             {
              name = word[1];
              find_block = TRUE;
              break;
             }
           if (equal((CHARTYPE *)"oldname",word[0],7))
             {
              name = word[1];
              find_oldname = TRUE;
              break;
             }
           display_error(1,word[0],FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 3  parameters, BLOCK and OLDNAME and name.                          */
/*---------------------------------------------------------------------*/
    case 3:
           if (equal((CHARTYPE *)"block",word[0],5)
           &&  equal((CHARTYPE *)"oldname",word[1],7))
              valid_args = TRUE;
           if (equal((CHARTYPE *)"block",word[1],5)
           &&  equal((CHARTYPE *)"oldname",word[0],7))
              valid_args = TRUE;
           if (!valid_args)
             {
              display_error(1,word[0],FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           find_block = find_oldname = TRUE;
           name = word[2];
           break;
/*---------------------------------------------------------------------*/
/* Too many parameters, error.                                         */
/*---------------------------------------------------------------------*/
    default:
           display_error(2,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
   }
/*---------------------------------------------------------------------*/
/* If the validation of parameters is successful...                    */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
/*---------------------------------------------------------------------*/
/* No pending prefix commands, return 0.                               */
/*---------------------------------------------------------------------*/
    if (CURRENT_FILE->first_ppc == NULL)
       number_variables = 0;
    else
/*---------------------------------------------------------------------*/
/* Have to search each pending prefix command...                       */
/*---------------------------------------------------------------------*/
      {
/*---------------------------------------------------------------------*/
/* If we are to look for OLDNAME, find a synonym for it if one exists..*/
/*---------------------------------------------------------------------*/
       if (find_oldname)
          name = find_prefix_oldname(name);
/*---------------------------------------------------------------------*/
/* For each pending prefix command...                                  */
/*---------------------------------------------------------------------*/
       curr_ppc = CURRENT_FILE->first_ppc;
       while(1)
         {
          if (curr_ppc == NULL)
             break;
/*---------------------------------------------------------------------*/
/* If we want to match on any name...                                  */
/*---------------------------------------------------------------------*/
          if (strcmp(name,"*") == 0)
            {
/*---------------------------------------------------------------------*/
/* Are we matching on any BLOCK command...                             */
/*---------------------------------------------------------------------*/
             if (find_block)
               {
                if (curr_ppc->ppc_block_command)
/*---------------------------------------------------------------------*/
/* We have found the first BLOCK command with any name.                */
/*---------------------------------------------------------------------*/
                   break;
                else
                  {
/*---------------------------------------------------------------------*/
/* Go back and look for another...                                     */
/*---------------------------------------------------------------------*/
                   curr_ppc = curr_ppc->next;
                   continue;
                  }
               }
             else
/*---------------------------------------------------------------------*/
/* We have found the first command with any name.                      */
/*---------------------------------------------------------------------*/
                break;
            }
/*---------------------------------------------------------------------*/
/* We want to find a specific command...                               */
/*---------------------------------------------------------------------*/
          if (strcmp(curr_ppc->ppc_command,name) == 0)
            {
/*---------------------------------------------------------------------*/
/* Are we looking for a specific BLOCK command...                      */
/*---------------------------------------------------------------------*/
             if (find_block)
               {
                if (curr_ppc->ppc_block_command)
/*---------------------------------------------------------------------*/
/* We have found the first specific BLOCK command.                     */
/*---------------------------------------------------------------------*/
                   break;
               }
             else
/*---------------------------------------------------------------------*/
/* We have found the first specific command.                           */
/*---------------------------------------------------------------------*/
                break;
            }
          curr_ppc = curr_ppc->next;
         }
/*---------------------------------------------------------------------*/
/* Did we find a matching pending prefix command ?                     */
/*---------------------------------------------------------------------*/
       if (curr_ppc == NULL)
          number_variables = 0;
       else
         {
/*---------------------------------------------------------------------*/
/* Yes we did. Set all of the REXX variables to the correct values...  */
/*---------------------------------------------------------------------*/
          sprintf(num1,"%d",curr_ppc->ppc_line_number);
          item_values[1].value = num1;
          item_values[1].len = strlen(num1);
          item_values[2].value = curr_ppc->ppc_command;
          item_values[2].len = strlen(item_values[2].value);
          item_values[3].value = find_prefix_synonym(curr_ppc->ppc_command);
          item_values[3].len = strlen(item_values[2].value);
          if (curr_ppc->ppc_block_command)
             item_values[4].value = (CHARTYPE *)"BLOCK";
          else
             item_values[4].value = (CHARTYPE *)"";
          item_values[4].len = strlen(item_values[4].value);
         }
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef PROTO
static short extract_prefix(short itemno,CHARTYPE *params)
#else
static short extract_prefix(itemno,params)
short itemno;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LINE *first_prefix_synonym;
/*--------------------------- local data ------------------------------*/
 register short i=0;
#define PRE_PARAMS  3
 CHARTYPE *word[PRE_PARAMS+1];
 unsigned short num_params=0;
 short number_variables = query_item[itemno].number_values;
 static CHARTYPE num4[15];
 LINE *curr=NULL;
 short rc=RC_OK;
 CHARTYPE *tmpbuf=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_prefix");
#endif
 num_params = param_split(params,word,PRE_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* None or 1 parameters, error.                                        */
/*---------------------------------------------------------------------*/
    case 0:
    case 1:
           display_error(3,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 2  parameters, Synonym and name.                                    */
/*---------------------------------------------------------------------*/
    case 2:
           if (!equal((CHARTYPE *)"synonym",word[0],1))
             {
              display_error(13,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           break;
/*---------------------------------------------------------------------*/
/* Too many parameters, error.                                         */
/*---------------------------------------------------------------------*/
    default:
           display_error(2,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
   }
/*---------------------------------------------------------------------*/
/* If the validation of parameters is successful...                    */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
    if (strcmp(word[1],"*") == 0)
/*---------------------------------------------------------------------*/
/* Get oldname for all synonyms...                                     */
/*---------------------------------------------------------------------*/
      {
       curr = first_prefix_synonym;
       i = 0;
       while(curr != NULL)
         {
          tmpbuf = (CHARTYPE *)(*the_malloc)(sizeof(CHARTYPE)*(strlen(curr->name)+strlen(curr->line)+1));
          if (tmpbuf == (CHARTYPE *)NULL)
            {
             display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
          strcpy(tmpbuf,curr->name);
          strcat(tmpbuf," ");
          strcat(tmpbuf,curr->line);
          rc = set_rexx_variable(query_item[itemno].name,tmpbuf,++i);
          (*the_free)(tmpbuf);
          if (rc == RC_SYSTEM_ERROR)
            {
             display_error(54,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
          curr = curr->next;
         }
       sprintf(num4,"%d",i);
       rc = set_rexx_variable(query_item[itemno].name,num4,0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(CHARTYPE *)"",FALSE);
          number_variables = EXTRACT_ARG_ERROR;
         }
       else
          number_variables = EXTRACT_VARIABLES_SET;
      }
    else
/*---------------------------------------------------------------------*/
/* Get oldname for named synonym...                                    */
/*---------------------------------------------------------------------*/
      {
       item_values[1].value = find_prefix_synonym(word[1]);
       item_values[1].len = strlen(item_values[1].value);
       number_variables = 1;
      }
 }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
