/*
 Timer functions for Djgpp;
 File:timerdj.h;
 By Dhonn Lushine; ludhonn@aol.com; http://members.aol.com/ludhonn
 Includes: interrupt handling, delay functions and update functions

 I have added a getmovementspeed() function. It updates moves with the timer.
 For example if you have a game and for some reason it slows down, this 
 function will bring it characters/positions to correct place.
 I have an example later in the timerdj.c.

 Also note install_timer(my_function,TIMER_DEF); that
 the my_function() should not have any interupts (cause of re-entrancy
 and it should thats function must be fast.

 Warnings: I am not responsible for any damages.
*/

#include <stdio.h>
#include <dos.h>
#include <dpmi.h>			/* Djgpp stuff */
#include <go32.h>
#include <sys\farptr.h>

#define TIMER_DEF	0xffff		/* Its for Default 18.2 hz timer rate */
#define TIMER_(x)	(1193180/(x))   /* For intializing any rate greater, use numbers>=18 */ 

#define getimer0x46c()  (_farpeekl(_dos_ds,0x46c)) /* Gets the 18.2hz system timer */
long timer1ch;                          /* Counter for the new timer interrupt */ 

_go32_dpmi_seginfo newtimer,oldtimer;	/* seginfo for the timers */

void newint1c(void);
void newint1c_end(void);		/* helps to lock newint1c code */
void (*newint1cfunc)(void);		/* function in our new timer interrupt */
void install_timer(void (*function)(void),int rate);	/* install routine */
void uninstall_timer(void);		/* uninstall routine */
void setimerate(int rate);		/* setting rate */
void delayer(int delay);		/* 18.2hz delayer, the dumb delayer is for example, for(i=0;i<100000;i++){} , which is dumb. */
int getmovementspeed(void);		/* gets an update if your delayed, for game movements and timing */

void install_timer(void (*function)(void),int rate)
{
 timer1ch=0;				/* set our incremental var */
 newint1cfunc=function;			/* link you function to the timer */
 newtimer.pm_offset=(int)newint1c;	/* Getting location of newint1c() */
 newtimer.pm_selector=_go32_my_cs();
 _go32_dpmi_lock_data(newint1cfunc,sizeof(newint1cfunc));
 _go32_dpmi_lock_data(&timer1ch,sizeof(timer1ch));	/* lock the variable, paging will affect it */
 _go32_dpmi_lock_code(newint1c,newint1c_end-newint1c);	/* lock the data of the interrupt handler */
 _go32_dpmi_get_protected_mode_interrupt_vector(0x1c,&oldtimer); /* Save the old int, we need that */
 _go32_dpmi_allocate_iret_wrapper(&newtimer); /* Make a function into an interrupt */
 _go32_dpmi_set_protected_mode_interrupt_vector(0x1c,&newtimer); /* Set new interrupt code */
 setimerate(rate);			/* set timer rate */
 getmovementspeed();			/* update movementspeed */
}

void uninstall_timer(void)
{
 setimerate(TIMER_DEF);	/* set default timer */
 _go32_dpmi_set_protected_mode_interrupt_vector(0x1c,&oldtimer);  /* Install the old interrupt code */
 _go32_dpmi_free_iret_wrapper(&newtimer);
}

void setimerate(int rate)
{
 outportb(0x43,0x3c);			/* using control and setting bits for a lowbyte then high byte a read|write */
 outportb(0x40,rate&0x00ff);		/* using timer #1; that 40h */
 outportb(0x40,(rate&0xff00)>>8);	/* hibyte */
}

void newint1c(void)			/* interrupt code, never call this by function, hardware calls it */
{
 asm("cli;pusha");			/* save all regs and cli ints */
 if(newint1cfunc)			/* call this function if there is a function */
  newint1cfunc();
 timer1ch++;				/* inc timer1ch */
 outportb(0x20,0x20);			/* make an eoi */
 asm("sti;popa");                       /* set interrupts and pop regs */
}
void newint1c_end(void){}		/* this is not a function, it holds the total bytes of the newint1c */
					/* is used for locking */

void delayer(int delay)			/* delays with delay * 1/18.2hz */
{
 long t2;
 t2=getimer0x46c()+delay;               /* current timer+delay */
 while(t2>getimer0x46c()){}             /* wait for the current time to reach the timer+delay */
}

int getmovementspeed(void)		/* this function causes program variables, for example: players/charaters/backgrounds, to be up to time with the timer rate */
{
 static int time1;			/* create a static time val so it has the last value from the last call */
 int movementspeed;
 movementspeed=timer1ch-time1;		/* movementspeed is now equal to the new time compared to the last time */
 if(movementspeed)			/* if the movementspeed was fast dont update the old time with the new time */
  time1=timer1ch;			/* if movementspeed was called too fast and the old time was updated every call with the new time then the movementspeed return will always be zero and thats bad */
 return movementspeed;			/* returns 0 if up to speed with timer and positive if its slower */
}

/* Now it's time to show you a real world example. */
/* see timerdj.c */

/* these are how all this functions are called

Examples:

 newint1c();			// never call this function because the hardware with call it for you

 install_timer(NULL,TIMER_DEF);	// returns nothing; sets no funtion and uses default timer
 install_timer(myfunction,TIMER_(50));	// returns nothing; calls myfunction() every 50hz/second

 uninstall_timer();		// returns noting; just uninstalls timer			

 setimerate(TIMER_DEF);		// sets timer to default 18.2
 setimerate(TIMER_(30));	// sets timer to 30hz a second

 getimer0x46c();		// gets a 18.2hz incremental number

 delayer(x);			// delays x/18.2hz a second
 delayer(10);			// delays 10/18.2hz a second

 TIMER_(x)			// macro return (1193180/(x))
 TIMER_(100)			// returns 1193180/100; its used for the install_timer function

 movementspeed=getmovementspeed();	// return 0 if on time with rate or number of timer cycles off

*/
