/* VDIFF.C	- Visual (or vertical) difference
 *
 * Produce a side-by-side comparison of two text files in a manner
 * analogous to the DIFF/PARALLEL command in VMS.
 *
 * Algorithmically rather simplistic and it can easily become confused when
 * there are a number of identical lines near a changed area. However it
 * quickly recovers and mostly does the job pretty well.
 *
 * Added match window to help avoid bad resynching.
 *
 *
 * Written by Andrew Davison, August 1993.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <fcntl.h>

#ifndef __MSDOS__
#	include <sys/stat.h>
#	include <dirent.h>
#else
#	include <sys\stat.h>
#	ifdef __TURBOC__
#		if __TURBOC__ >= 0x300
#			include <dirent.h>
#		else
#			define NODIR
#		endif
#	else
#		include <dirent.h>
#	endif
#endif

#ifdef __STDC__
#define PROTOTYPES
#endif
#ifdef __cplusplus		/* TURBO C++ before v3 */
#define PROTOTYPES__
#endif
#ifdef __CPLUSPLUS__
#define PROTOTYPES
#endif

static int screen_width = 80;
static int tab_stop = 8;
static int to_a_file = 0;
static int line_nbr = 0;
static int jump_to_change = 0;
static int next_file = 0;
static int ansi_tty = 1;
static int search_window = 20;
static int match_window = 2;
static char save1[256], save2[256];
static char *mem1, *mem2, *src1, *src2;

#define WIDTH ((screen_width / 2) - 1)
#define TAB_STOP tab_stop
#define PAGE_SIZE 24
#define SEARCH_WINDOW search_window
#define MATCH_WINDOW match_window

#ifdef __MSDOS__
#define SEPARATOR '\\'
#else
#define SEPARATOR '/'
#endif

#ifdef __MSDOS__
#define DIM (ansi_tty ? "\033[1m" : "")
#define BAR (ansi_tty ? "|" : "|")
#else
#define DIM (ansi_tty ? "\033[4m" : "")
#define BAR (ansi_tty ? " " : "|")
#endif

#define REVERSE (ansi_tty ? "\033[7m" : "")
#define BOLD (ansi_tty ? "\033[1m" : "")
#define NORMAL (ansi_tty ? "\033[0m" : "")

static enum {READ_AND_PRINT_1, READ_AND_PRINT_2, READ_1, READ_2, READ_12, COMPARE_12} state = READ_12;

#ifdef PROTOTYPES
static char *getline(char *buf, char *mem)
#else
static char *getline(buf, mem)
char *buf;
char *mem;
#endif
{
while (*mem && (*mem != '\r') && (*mem != '\n'))
  *buf++ = *mem++;

*buf = '\0';

if (*mem == '\r')
  mem++;				/* Get rid of CR */

if (*mem == '\n')
  mem++;				/* Get rid of NL */

return (*mem == '\0' ? NULL : mem);
}

#ifdef PROTOTYPES
static int find_match(char *buf, char *source, char *target)
#else
static int find_match(buf, source, target)
char *buf;
char *source;
char *target;
#endif
{
char buf_source[256], buf_target[256];
char *pos_target = target;
char *pos_source = source;
int cnt = 0, matches = 0, save_cnt = 0;
char *save_target;
char *save_buf = buf;

/*
 * Look for required number of matches to resync. Try until search
 * window has been exhausted or end of file.
 */

while (matches < MATCH_WINDOW)
  {
  if ((cnt > SEARCH_WINDOW) || !(target = getline(buf_target, target)))
    {
    break;
    }

  cnt++;

  /*
   * Check for match of source and target buffers...
   */
  
  if (!strcmp(buf, buf_target))
    {
    /*
     * On an initial match save the target position in case
     * we have to resume the search.
     */

    if (!matches++)
      {
      save_cnt = cnt;
      save_target = target;
      }

    /*
     * If the search runs out of source for matching then give up.
     */

    if (!(source = getline(buf_source, source)))
      {
      break;
      }

    buf = buf_source;
    cnt--;				/* don't penalise the search */
    continue;
    }

  /*
   * If source matching has failed then resume original search
   * from where it left off.
   */

  if (matches)
    {
    target = save_target;			/* resume target */
    cnt = save_cnt;

    source = pos_source;			/* original source */
    buf = save_buf;

    matches = 0;
    }
  }

target = pos_target;
source = pos_source;

return (matches ? save_cnt : 0);
}

#ifdef PROTOTYPES
static void output(char *vid, char *buf1, char *buf2)
#else
static output(vid, buf1, buf2)
char *vid, *buf1, *buf2;
#endif
{
char tmp[256];
char *dst = tmp;
char *src;
int cnt;

if (line_nbr++ == 0)
  if (!to_a_file)
    output(DIM, save1, save2);

/* 
 * Basically, the story here is to de-TAB into an output buffer
 * and then print just the requisite width.
 */

dst += sprintf(dst, "%s", vid);

src = buf1;
cnt = 0;

while (*src && (cnt < WIDTH))
  {
  if (*src == '\t')
    {
    int extras = TAB_STOP - (cnt % TAB_STOP);

    while ((extras-- > 0) && (cnt < WIDTH))
      {
      *dst++ = ' ';
      cnt++;
      }

    src++;
    }
  else if (!isprint(*src))
    {
    src++;
    }
  else
    {
    *dst++ = *src++;
    cnt++;
    }
  }

while (cnt < WIDTH)
  {
  *dst++ = ' ';
  cnt++;
  }

dst += sprintf(dst, "%s%s%s", DIM, BAR, vid);

src = buf2;
cnt = 0;

while (*src && (cnt < WIDTH))
  {
  if (*src == '\t')
    {
    int extras = TAB_STOP - (cnt % TAB_STOP);

    while ((extras-- > 0) && (cnt < WIDTH))
      {
      *dst++ = ' ';
      cnt++;
      }

    src++;
    }
  else if (!isprint(*src))
    {
    src++;
    }
  else
    {
    *dst++ = *src++;
    cnt++;
    }
  }

while (cnt < WIDTH)
  {
  *dst++ = ' ';
  cnt++;
  }

sprintf(dst, "%s", NORMAL);

if (jump_to_change)
  {
  line_nbr = 0;
  return;
  }

printf("%s\n", tmp);

if (line_nbr >= PAGE_SIZE)
  {
  line_nbr = 0;

  if (!to_a_file)
    {
    printf("\r%sVDIFF <q>uit, <j>ump, <n>ext, or RETURN to continue...%s", BOLD, NORMAL);
    fflush(stdout);
    switch (getchar())
      {
      case 'q': exit(0);
      case 'j': jump_to_change = 1; break;
      case 'n': next_file = 1; break;
      }
    }
  }
}

#ifdef PROTOTYPES
static int vdiff(char *filename1, char *filename2)
#else
static int vdiff(filename1, filename2)
char *filename1, *filename2;
#endif
{
int file1, file2;
char buf1[256], buf2[256];
int left, gap1, gap2, done;
struct stat statbuf;

if ((file1 = open(filename1, O_RDONLY)) < 0)
  return -1;

if ((file2 = open(filename2, O_RDONLY)) < 0)
  {
  close(file1);
  return 0;
  }

stat(filename1, &statbuf);
mem1 = malloc(statbuf.st_size+1);
read(file1, mem1, statbuf.st_size);
mem1[statbuf.st_size] = '\0';
src1 = mem1;
close(file1);

stat(filename2, &statbuf);
mem2 = malloc(statbuf.st_size+1);
read(file2, mem2, statbuf.st_size);
mem2[statbuf.st_size] = '\0';
src2 = mem2;
close(file2);

line_nbr = 0;
jump_to_change = 0;
next_file = 0;
done = 0;

strcpy(save1, filename1);
strcpy(save2, filename2);

state = READ_12;

while (!next_file && !done)
  {
  switch (state)
    {
    /*
     * Read and print a (possibly predetermined) number of lines.
     */

    case READ_AND_PRINT_1:

      if (!(src1 = getline(buf1, src1)))
        {
        done = 1;
        break;
        }

      output(REVERSE, buf1, "");

      if (--gap1 == 0)
         state = READ_1;

      break;

    /*
     * Read and print a (possibly predetermined) number of lines.
     */

    case READ_AND_PRINT_2:

      if (!(src2 = getline(buf2, src2)))
        {
        done = 1;
        break;
        }

      output(REVERSE, "", buf2);

      if (--gap2 == 0)
        state = READ_2;

      break;

    /*
     * Read the next line ready for comparison.
     */

    case READ_1:

      if (!(src1 = getline(buf1, src1)))
        {
	output(REVERSE, "", buf2);
	state = READ_AND_PRINT_2;
	gap2 = -1;
	break;
	}

      state = COMPARE_12;
      break;

    /*
     * Read the next line ready for comparison.
     */

    case READ_2:

      if (!(src2 = getline(buf2, src2)))
        {
	output(REVERSE, buf1, "");
	state = READ_AND_PRINT_1;
	gap1 = -1;
	break;
	}

      state = COMPARE_12;
      break;

    /*
     * Read lines from each file ready for comparison.
     */

    case READ_12:

      if (!(src1 = getline(buf1, src1)))
	{
        state = READ_AND_PRINT_2;
        gap1 = -1;
        break;
        }

      if (!(src2 = getline(buf2, src2)))
        {
	output(REVERSE, buf1, "");
	state = READ_AND_PRINT_1;
	gap2 = -1;
	break;
	}

      state = COMPARE_12;
      break;

    /*
     * Compare lines from each file.
     */

    case COMPARE_12:

      /*
       * If the lines are the same then print them both
       * and just continue on.
       */

      if (!strcmp(buf1, buf2))
        {
	output(NORMAL, buf1, buf2);
	state = READ_12;
	break;
	}

      /*
       * They aren't the same, so try to find matches for each
       * line in the opposing file. This records the 'gap' that
       * needs printing.
       */

      gap1 = find_match(buf1, src1, src2);
      gap2 = find_match(buf2, src2, src1);

      /*
       * Print the smallest gap first.
       */

      if (gap1 <= gap2)
        {
	output(REVERSE, buf1, "");
	state = (--gap1 > 0 ? READ_AND_PRINT_1 : READ_1);
	break;
	}
      else
        {
	output(REVERSE, "", buf2);
	state = (--gap2 > 0 ? READ_AND_PRINT_2 : READ_2);
	break;
	}
    }
  }

free(mem1);
free(mem2);

/*
 * Fill out the screen to the bottom line.
 */

left = PAGE_SIZE - line_nbr;

while (left-- > 0)
  output(NORMAL, "", "");

return 1;
}

#ifdef NODIR
#pragma argsused
#endif

#ifdef PROTOTYPES
static void process(char *filename1, char *filename2, int fileidx)
#else
static process(filename1, filename2, fileidx)
char *filename1;
char *filename2;
int fileidx;
#endif
{
#ifdef NODIR

switch (vdiff(filename1, filename2))
  {
  case -1: printf("Sorry, could not open '%s'\n", filename1); break;
  case 0:  printf("Sorry, could not open '%s'\n", filename2); break;
  }

#else

char tmp1[256], tmp2[256];
DIR *dirp1, *dirp2;

if ((dirp1 = opendir(filename1)) == NULL)	/* is a file */
  {
  if ((dirp2 = opendir(filename2)) == NULL)	/* file1 file2 */
    {
    switch (vdiff(filename1, filename2))
      {
      case -1: printf("Sorry, could not open '%s'\n", filename1); break;
      case 0:  printf("Sorry, could not open '%s'\n", filename2); break;
      }
    }
  else						/* file1 dir2 */
    {
    char *ptr;

    closedir(dirp2);
    ptr = strrchr(filename1, SEPARATOR);
    sprintf(tmp2, "%s%c%s", filename2, SEPARATOR, (ptr ? ++ptr : filename1));	

    switch (vdiff(filename1, tmp2))
      {
      case -1: printf("Sorry, could not open '%s'\n", filename1); break;
      case 0:  if (!fileidx) printf("Sorry, could not open '%s'\n", tmp2); break;
      }
    }
  }
else						/* is a directory */
  {
  if ((dirp2 = opendir(filename2)) == NULL)	/* dir1 file2 */
    {
    printf("Sorry, '%s' must be a directory\n", filename2);
    }
  else						/* dir1 dir2 */
    {
    struct dirent *dp;

    while ((dp = readdir(dirp1)) != NULL)
      {
      char *ptr = strrchr(dp->d_name, '.');

      if (!ptr)
        continue;

      if (!strcmp(ptr, ".c"))
        ;
      else if (!strcmp(ptr, ".h"))
        ;
      else if (!strcmp(ptr, ".cpp"))
        ;
      else
        continue;

      sprintf(tmp1, "%s%c%s", filename1, SEPARATOR, dp->d_name);	
      sprintf(tmp2, "%s%c%s", filename2, SEPARATOR, dp->d_name);	

      switch (vdiff(tmp1, tmp2))
        {
        case -1: printf("Sorry, could not open '%s'\n", tmp1); break;
        case 0:  if (!fileidx) printf("Sorry, could not open '%s'\n", tmp2); break;
        }
      }

    closedir(dirp2);
    }

  closedir(dirp1);
  }

#endif
}

#ifdef PROTOTYPES
main(int ac, char *av[])
#else
main(ac, av)
int ac;
char *av[];
#endif
{
char *filename1, *filename2;
int i, args = 0, fileidx = 0, filecnt = 0;

for (i = 1; i < ac; i++)
  {
  if (sscanf(av[i], "-w:%d", &screen_width))
    ;
  else if (sscanf(av[i], "-t:%d", &tab_stop))
    ;
  else if (sscanf(av[i], "-search:%d", &search_window))
    ;
  else if (sscanf(av[i], "-match:%d", &match_window))
    ;
  else if (!strcmp(av[i], "-f"))
    {
    to_a_file = 1;
    ansi_tty = 0;
    }
  else if (!strcmp(av[i], "-noansi"))
    {
    ansi_tty = 0;
    }
  else if (av[i+1])
    {
    if (filecnt++ == 0)
      {
      fileidx = i;
      args = 1;
      }
    }
  else
    {
    filename2 = av[i];
    args++;
    }
  }

if (args != 2)
    {
    printf("Usage: vdiff [options] file1 file2\n");
    printf("\n");
    printf("       vdiff fred.c ../backups/fred.c\n");
#ifndef NODIR
    printf("       vdiff fred.c ../backups\n");
    printf("       vdiff fred.h fred.c ../backups\n");
#ifndef __MSDOS__
    printf("       vdiff *.h *.c ../backups\n");
#endif
    printf("       vdiff . ../backups\n");
#endif
    printf("\n");
#ifndef NODIR
    printf(" Note, when file1 is a list or directory then file2 must be a directory.\n");
#endif
    printf("\n");
    printf(" Options...\n\n");
    printf("       -f            output is to a file (continuous)\n");
    printf("       -noansi       output is to a non-ANSI display\n");
    printf("       -w:132        132 column output (default is %d)\n", WIDTH);
    printf("       -t:4          TAB every 4 chars (default is %d)\n", TAB_STOP);
    printf("       -search:100   search window is 100 lines (default is %d)\n", SEARCH_WINDOW);
    printf("       -match:3      match window is 3 (default is %d)\n", MATCH_WINDOW);
    printf("\n");
    exit(1);
    }

do
  {
  filename1 = av[fileidx++];
  process(filename1, filename2, fileidx);
  }
  while	(--filecnt > 0);


return 0;
}
