/* xlapi.c - xlisp api routines */
/*      Copyright (c) 1995, by David Michael Betz
        All Rights Reserved
        Permission is granted for unrestricted non-commercial use       */

#include "xlisp.h"

/* external variables */
extern LVAL s_eval,s_load,s_unbound,eof_object;
extern LVAL *xlcatch;

/* xlCallFunction - call a function */
int xlCallFunction(LVAL *pvalue,LVAL fun,int argc,...)
{
    LVAL *save_catch;
    va_list ap;
    int valc;
    
    /* initialize the catch list */
    save_catch = xlcatch;
    xlcatch = NULL;
    
    /* execute the function call */
    va_start(ap,argc);
    valc = InvokeInterpreter(pvalue,fun,v_nil,argc,ap);
    va_end(ap);
    
    /* restore the old catch list */
    xlcatch = save_catch;
    
    /* return the status */
    return valc >= 0 ? xlsSuccess : xlsError;
}

/* xlCallFunctionByName - call a function by name */
int xlCallFunctionByName(LVAL *pvalue,char *fname,int argc,...)
{
    LVAL *save_catch,sym,fun;
    ERRORTARGET target;
    va_list ap;
    int valc;
    
    /* initialize the catch list */
    save_catch = xlcatch;
    xlcatch = NULL;
    
    /* setup the error handler */
    pushtarget(&target);
    if (setjmp(target.target) != 0) {
    	xlcatch = save_catch;
        poptarget();
        return xlsError;
    }

    /* get the function */
    sym = xlenter(fname);
    if ((fun = getvalue(sym)) == s_unbound)
        xlerror("unbound symbol",sym);

    /* only needed error handler for dealing with unbound symbols */
    poptarget();
    
    /* execute the function call */
    va_start(ap,argc);
    valc = InvokeInterpreter(pvalue,fun,v_nil,argc,ap);
    va_end(ap);
    
    /* restore the old catch list */
    xlcatch = save_catch;
    
    /* return the status */
    return valc >= 0 ? xlsSuccess : xlsError;
}

/* xlSendMessage - send a message to an object */
EXPORT int xlSendMessage(LVAL *pvalue,LVAL obj,LVAL selector,int argc,...)
{
    LVAL *save_catch;
    va_list ap;
    int valc;
    
    /* initialize the catch list */
    save_catch = xlcatch;
    xlcatch = NULL;

    /* count the selector as an argument */
    ++argc;
    
    /* execute the function call */
    va_start(ap,argc);
    valc = InvokeInterpreter(pvalue,obj,selector,-argc,ap);
    va_end(ap);
    
    /* restore the old catch list */
    xlcatch = save_catch;
    
    /* return the status */
    return valc >= 0 ? xlsSuccess : xlsError;
}

/* xlSendMessageByName - send a message to an object */
EXPORT int xlSendMessageByName(LVAL *pvalue,LVAL obj,char *sname,int argc,...)
{
    LVAL *save_catch,selector;
    ERRORTARGET target;
    va_list ap;
    int valc;
    
     /* initialize the catch list */
    save_catch = xlcatch;
    xlcatch = NULL;
    
   /* setup the error handler */
    pushtarget(&target);
    if (setjmp(target.target) != 0) {
    	xlcatch = save_catch;
        poptarget();
        return xlsError;
    }

    /* get the selector symbol (and count it as an argument) */
    selector = xlenter(sname);
    ++argc;

    /* only needed error handler for dealing with entering selector */
    poptarget();
    
    /* execute the function call */
    va_start(ap,argc);
    valc = InvokeInterpreter(pvalue,obj,selector,-argc,ap);
    va_end(ap);
    
    /* restore the old catch list */
    xlcatch = save_catch;
    
    /* return the status */
    return valc >= 0 ? xlsSuccess : xlsError;
}

/* xlEvaluateString - evaluate an expression from a string */
EXPORT int xlEvaluateString(char *str,FIXTYPE len,LVAL *pval)
{
    LVAL val;
    int sts;
    if ((sts = xlReadFromString(str,len,&val)) != xlsSuccess)
        return sts;
    return xlEvaluate(val,pval);    
}

/* xlReadFromString - read an expression from a string */
EXPORT int xlReadFromString(char *str,FIXTYPE len,LVAL *pval)
{
    ERRORTARGET target;
    int sts;

    /* trap errors */
    pushtarget(&target);
    if (setjmp(target.target)) {
        poptarget();
        return xlsError;
    }
 
    /* create the string stream */
    cpush(cvustream(str,len));
    
    /* read from the stream */
    if (!(sts = xlread(pop(),pval)))
        *pval = eof_object;
    poptarget();
    
    /* return status */
    return sts ? xlsSuccess : xlsEndOfFile;
}

/* xlLoadFile - load an XLISP source file */
EXPORT int xlLoadFile(char *fname)
{
    ERRORTARGET target;
    LVAL val;
    int valc;

    /* trap errors */
    pushtarget(&target);
    if (setjmp(target.target)) {
        poptarget();
        return xlsError;
    }
 
    /* convert the filename */
    cpush(cvcstring(fname));
    poptarget();
    
    /* call the LOAD function */
    valc = xlCallFunction(&val,getvalue(s_load),1,pop());

    /* return status */
    return valc >= 0 && val == v_true ? xlsSuccess : xlsError;
}

/* xlEvaluate - evaluate an expression */
EXPORT int xlEvaluate(LVAL expr,LVAL *pval)
{
    int valc = xlCallFunction(pval,getvalue(s_eval),1,expr);
    return valc >= 0 ? xlsSuccess : xlsError;
}

/* xlPrintToString - print an expression to a string */
EXPORT char *xlPrintToString(LVAL expr,char *buf,FIXTYPE len)
{
    ERRORTARGET target;
    char *p = buf;
    LVAL stream;
    int ch;

    /* trap errors */
    pushtarget(&target);
    if (setjmp(target.target)) {
        poptarget();
        return NULL;
    }

    /* create a stream */
    cpush(expr);
    stream = newustream();
    
    /* print the expression to the stream */
    xlprin1(expr,stream);
    drop(1);
    
    /* copy the stream characters to the buffer */
    while ((ch = xlgetc(stream)) != EOF) {
        if (--len <= 1)
            return NULL;
        *p++ = ch;
    }
    poptarget();
    *p = '\0';
    
    /* return successfully */
    return buf;
}

/* xlProtect - protect a pointer */
EXPORT int xlProtect(LVAL *p)
{
    return protectptr(p) ? xlsSuccess : xlsError;
}
