/* xlcom.c - the xlisp bytecode compiler */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"
#include "xlbcode.h"

/* size of code buffer */
#define CMAX	40000

/* continuation types */
#define C_RETURN	-1
#define C_NEXT		-2

/* macro to check for a lambda list keyword */
#define lambdakey(x)	((x) == lk_optional \
		      || (x) == lk_rest \
		      || (x) == lk_key \
		      || (x) == lk_allow_other_keys \
		      || (x) == lk_aux \
		      || slambdakey(x))

/* macro to check for a scheme lambda list keyword */
#define slambdakey(x)	((x) == slk_optional \
		      || (x) == slk_rest)

/* external variables */
extern LVAL lk_optional,lk_rest,lk_key,lk_allow_other_keys,lk_aux;
extern LVAL slk_optional,slk_rest;

/* local variables */
static LVAL info;		/* compiler info */

/* code buffer */
static unsigned char *cbuff = NULL;	/* base of code buffer */
static int cbase;			/* base for current function */
static int cptr;			/* code buffer pointer */

/* forward declarations */
static void do_expr(LVAL expr,int cont);
static int in_ntab(LVAL expr,int cont);
static int in_ftab(LVAL expr,int cont);
static void do_define(LVAL form,int cont);
static void define1(LVAL list,LVAL body,int cont);
static void do_setq(LVAL form,int cont);
static void do_setvar(LVAL form,int cont);
static void do_quote(LVAL form,int cont);
static void do_lambda(LVAL form,int cont);
static void do_namedlambda(LVAL form,int cont);
static void cd_fundefinition(LVAL fun,LVAL fargs,LVAL body);
static void parse_lambda_expr(LVAL fargs,LVAL body,int mflag);
static int count_arguments(LVAL fargs,int *prargc,int *poargc,LVAL *prestarg,int *pkargc);
static void add_extra_arguments(LVAL fargs);
static void parse_optional_arguments(LVAL key,LVAL *pfargs,int base);
static void parse_optional_argument(LVAL form,LVAL *parg,LVAL *pdef,LVAL *psvar);
static void parse_key_arguments(LVAL *pfargs,int base);
static void parse_key_argument(LVAL form,LVAL *parg,LVAL *pkey,LVAL *pdef,LVAL *psvar);
static void parse_aux_arguments(LVAL *pfargs);
static void parse_aux_argument(LVAL form,LVAL *parg,LVAL *pdef);
static void add_argument_name(LVAL name);
static void patch_argument_name(LVAL name);
static int get_argument_offset(LVAL name);
static void do_delay(LVAL form,int cont);
static void do_let(LVAL form,int cont);
static void do_named_let(LVAL form,int cont);
static LVAL extract_let_variables(LVAL bindings,int *pcnt);
static void do_unnamed_let(LVAL form,int cont);
static void do_letrec(LVAL form,int cont);
static void do_letstar(LVAL form,int cont);
static void letstar1(LVAL blist,LVAL body,int cont);
static int push_dummy_values(LVAL blist);
static int push_init_expressions(LVAL blist);
static void generate_let_setup_code(LVAL blist);
static void parse_let_variables(LVAL blist,int *pcnt,int *pextra);
static void set_bound_variables(LVAL blist);
static void do_mvbind(LVAL form,int cont);
static void do_mvcall(LVAL form,int cont);
static LVAL make_code_object(LVAL fun);
static void do_cond(LVAL form,int cont);
static void do_and(LVAL form,int cont);
static void do_or(LVAL form,int cont);
static void do_if(LVAL form,int cont);
static void do_begin(LVAL form,int cont);
static void do_while(LVAL form,int cont);
static void do_catch(LVAL form,int cont);
static void do_unwindprotect(LVAL form,int cont);
static void do_call(LVAL form,int cont);
static int push_args(LVAL form);
static void do_nary(int op,int n,LVAL form,int cont);
static void push_nargs(LVAL form,int n);
static void do_literal(LVAL lit,int cont);
static void do_identifier(LVAL sym,int cont);
static void do_continuation(int cont);
static void add_frame(void);
static void remove_frame(void);
static int add_level(void);
static void remove_level(int oldcbase);
static int findvariable(int opcode,LVAL sym,int *plev,int *poff);
static int findcvariable(int opcode,LVAL frame,LVAL sym,int *poff);
static int findliteral(LVAL lit);
static void cd_variable(int op,LVAL sym);
static void cd_evariable(int op,int lev,int off);
static void cd_literal(LVAL lit);
static int nextcaddr(void);
static int putcbyte(int b);
static int putcword(int w);
static void fixup(int chn);

/* integrable function table */
typedef struct { char *nt_name; int nt_code,nt_args; } NTDEF;
static NTDEF ntab[] = {
{	"ATOM",			OP_ATOM,	1	},
{	"EQ?",			OP_EQ,		2	},
{	"NULL?",		OP_NULL,	1	},
{	"NOT",			OP_NULL,	1	},
{	"CONS",			OP_CONS,	2	},
{	"CAR",			OP_CAR,		1	},
{	"CDR",			OP_CDR,		1	},
{	"SET-CAR!",		OP_SETCAR,	2	},
{	"SET-CDR!",		OP_SETCDR,	2	},
{	"+",			OP_ADD,		-2	},
{	"-",			OP_SUB,		-2	},
{	"*",			OP_MUL,		-2	},
{	"QUOTIENT",		OP_QUO,		-2	},
{	"<",			OP_LSS,		-2	},
{	"=",			OP_EQL,		-2	},
{	">",			OP_GTR,		-2	},
{0,0,0}
};

/* special form table */
typedef struct { char *ft_name; void (*ft_fcn)(LVAL,int); } FTDEF;
static FTDEF ftab[] = {
{	"QUOTE",		do_quote		},
{	"LAMBDA",		do_lambda		},
{	"NAMED-LAMBDA",		do_namedlambda		},
{	"DELAY",		do_delay		},
{	"LET",			do_let			},
{	"LET*",			do_letstar		},
{	"LETREC",		do_letrec		},
{	"MULTIPLE-VALUE-BIND",	do_mvbind		},
{	"MULTIPLE-VALUE-CALL",	do_mvcall		},
{	"DEFINE",		do_define		},
{	"SET!",			do_setq			},
{	"IF",			do_if			},
{	"COND",			do_cond			},
{	"BEGIN",		do_begin		},
{	"SEQUENCE",		do_begin		},
{	"AND",			do_and			},
{	"OR",			do_or			},
{	"WHILE",		do_while		},
{	"CATCH",		do_catch		},
{	"UNWIND-PROTECT",	do_unwindprotect	},
{0,0}
};

/* xlcompile - compile an expression */
LVAL xlcompile(LVAL expr,LVAL ctenv)
{
    /* allocate the code buffer on the first call */
    if (cbuff == NULL) {
        if ((cbuff = osalloc(CMAX)) == NULL)
            xlfatal("insufficient memory");
    }

    /* initialize the compile time environment */
    info = cons(v_nil,v_nil); cpush(info);
    rplaca(info,newframe(ENV,ctenv,FIRSTENV));
    rplacd(info,cons(v_nil,v_nil));

    /* setup the base of the code for this function */
    cbase = cptr = 0;

    /* setup the entry code */
    putcbyte(OP_ARGSEQ);
    putcbyte(0);

    /* compile the expression */
    do_expr(expr,C_RETURN);

    /* build the code object */
    settop(make_code_object(v_nil));
    return pop();
}

/* xlmethod - compile a method */
LVAL xlmethod(LVAL fun,LVAL fargs,LVAL body,LVAL ctenv)
{
    /* initialize the compile time environment */
    info = cons(v_nil,v_nil); cpush(info);
    rplaca(info,newframe(MENV,ctenv,FIRSTENV));
    rplacd(info,cons(v_nil,v_nil));

    /* setup the base of the code for this function */
    cbase = cptr = 0;

    /* add 'self' to the argument list */
    cpush(cons(xlenter("SELF"),fargs));

    /* compile the lambda list and the function body */
    parse_lambda_expr(top(),body,TRUE);
    
    /* build the code object */
    settop(make_code_object(fun));
    return pop();
}

/* do_expr - compile an expression */
static void do_expr(LVAL expr,int cont)
{
    LVAL fun;
    if (consp(expr)) {
	fun = car(expr);
 	if (!symbolp(fun) || (!in_ntab(expr,cont) && !in_ftab(expr,cont)))
	    do_call(expr,cont);
    }
    else if (symbolp(expr))
	do_identifier(expr,cont);
    else
	do_literal(expr,cont);
}

/* in_ntab - check for a function in ntab */
static int in_ntab(LVAL expr,int cont)
{
    NTDEF *nptr;
    char *pname = getstring(getpname(car(expr)));
    for (nptr = ntab; nptr->nt_name; ++nptr)
	if (strcmp(pname,nptr->nt_name) == 0) {
	    do_nary(nptr->nt_code,nptr->nt_args,expr,cont);
	    return TRUE;
	}
    return FALSE;
}

/* in_ftab - check for a function in ftab */
static int in_ftab(LVAL expr,int cont)
{
    FTDEF *fptr;
    char *pname = getstring(getpname(car(expr)));
    for (fptr = ftab; fptr->ft_name; ++fptr)
	if (strcmp(pname,fptr->ft_name) == 0) {
	    (*fptr->ft_fcn)(cdr(expr),cont);
	    return TRUE;
	}
    return FALSE;
}

/* do_define - handle the (DEFINE ... ) expression */
static void do_define(LVAL form,int cont)
{
    if (atom(form))
	xlerror("expecting symbol or function template",form);
    define1(car(form),cdr(form),cont);
}

/* define1 - helper routine for do_define */
static void define1(LVAL list,LVAL body,int cont)
{
    int opcode,off;

    /* check for procedure definition */
    if (consp(list)) {
	cpush(car(list));
	if (!symbolp(top()))
	    xlerror("expecting function name",top());
	else if (atom(cdr(list)))
	    xlerror("expecting argument list",cdr(list));
	cd_fundefinition(car(list),cdr(list),body);
    }
    else {
	cpush(list);
	do_begin(body,C_NEXT);
    }
	
    /* define the variable value */
    if ((opcode = findcvariable(OP_ESET,car(info),top(),&off)) != 0)
	cd_evariable(opcode,0,off);
    else
	cd_variable(OP_GSET,top());
    do_literal(top(),cont);
    drop(1);
}

/* do_setq - compile the (SET! ... ) expression */
static void do_setq(LVAL form,int cont)
{
    check(1);
    while (consp(form) && consp(cdr(form))) {
	push(cdr(cdr(form)));
	if (atom(form))
	    xlerror("expecting symbol",form);
	else if (symbolp(car(form)))
	    do_setvar(form,top() ? C_NEXT : cont);
	else
	    xlerror("expecting symbol",form);
	form = pop();
    }
    if (form != v_nil)
        xlfmterror("bad syntax in set! ~S",form);
}

/* do_setvar - compile the (SET! var value) expression */
static void do_setvar(LVAL form,int cont)
{
    int opcode,lev,off;

    /* get the variable name */
    cpush(car(form));

    /* compile the value expression */
    form = cdr(form);
    if (atom(form))
	xlerror("expecting value expression",form);
    do_expr(car(form),C_NEXT);

    /* set the variable value */
    if ((opcode = findvariable(OP_ESET,top(),&lev,&off)) != 0)
	cd_evariable(opcode,lev,off);
    else
	cd_variable(OP_GSET,top());
    do_continuation(cont);
    drop(1);
}

/* do_quote - compile the (QUOTE ... ) expression */
static void do_quote(LVAL form,int cont)
{
    if (atom(form))
	xlerror("expecting quoted expression",form);
    do_literal(car(form),cont);
}

/* do_lambda - compile the (LAMBDA ... ) expression */
static void do_lambda(LVAL form,int cont)
{
    if (atom(form))
	xlerror("expecting argument list",form);
    cd_fundefinition(v_nil,car(form),cdr(form));
    do_continuation(cont);
}

/* do_namedlambda - compile the (NAMED-LAMBDA ... ) expression */
static void do_namedlambda(LVAL form,int cont)
{
    if (atom(form) || !symbolp(car(form)))
	xlerror("expecting function name",form);
    else if (atom(cdr(form)))
	xlerror("expecting argument list",form);
    cd_fundefinition(car(form),car(cdr(form)),cdr(cdr(form)));
    do_continuation(cont);
}

/* cd_fundefinition - compile the function */
static void cd_fundefinition(LVAL fun,LVAL fargs,LVAL body)
{
    int oldcbase;

    /* establish a new environment frame */
    oldcbase = add_level();

    /* compile the lambda list and the function body */
    parse_lambda_expr(fargs,body,FALSE);

    /* build the code object */
    cpush(make_code_object(fun));
    
    /* restore the previous environment */
    remove_level(oldcbase);

    /* compile code to create a closure */
    do_literal(pop(),C_NEXT);
    putcbyte(OP_CLOSE);
}

/* parse_lambda_expr - parse a lambda expression */
static void parse_lambda_expr(LVAL fargs,LVAL body,int mflag)
{
    int rargc,oargc,kargc,extra;
    LVAL arg,key,restarg;
    
    /* count the arguments */
    extra = count_arguments(fargs,&rargc,&oargc,&restarg,&kargc);
    
    /* output the entry code */
    if (kargc == 0 && restarg == v_nil) {
	if (oargc == 0) {
	    putcbyte(OP_ARGSEQ);
	    putcbyte(rargc);
	}
	else {
	    putcbyte(OP_ARGSBT);
	    putcbyte(rargc);
	    putcbyte(rargc + oargc);
	}
    }
    else {
	putcbyte(OP_ARGSGE);
	putcbyte(rargc);
    }
    
    /* handle each required argument */
    while (consp(fargs)
    &&     (arg = car(fargs)) != v_nil
    &&     !lambdakey(arg)) {
	add_argument_name(arg);
	fargs = cdr(fargs);
    }

    /* insert the code to mark this frame as a method frame */
    if (mflag)
	putcbyte(OP_METHOD);

    /* handle extra variables */
    if (extra > FIRSTENV) {

	/* add an extra frame for optional, key, rest and aux variables */
	add_frame();
	add_extra_arguments(fargs);
	putcbyte(OP_FRAME);
	putcbyte(0);
	putcbyte(extra);
	putcbyte(findliteral(getenvnames(car(info))));

	/* check for &optional or #!optional arguments */
	if (consp(fargs) && (car(fargs) == lk_optional || car(fargs) == slk_optional)) {
	    key = car(fargs);
	    fargs = cdr(fargs);
	    parse_optional_arguments(key,&fargs,FIRSTENV + rargc);
	}

	/* check for the &rest argument */
	if (consp(fargs) && (car(fargs) == lk_rest || car(fargs) == slk_rest))
	    fargs = cdr(cdr(fargs));
	if (restarg)
    	    patch_argument_name(restarg);

	/* check for &key arguments */
	if (consp(fargs) && car(fargs) == lk_key) {
	    fargs = cdr(fargs);
	    parse_key_arguments(&fargs,FIRSTENV + rargc + oargc);
	    if (consp(fargs) && car(fargs) == lk_allow_other_keys)
		fargs = cdr(fargs);
	}

	/* check for &aux arguments */
	if (consp(fargs) && car(fargs) == lk_aux) {
	    fargs = cdr(fargs);
	    parse_aux_arguments(&fargs);
	}

	/* output instruction to build the '&rest' argument list */
	if (restarg) {
	    putcbyte(OP_REST);
	    putcbyte(FIRSTENV + rargc + oargc);
	    putcbyte(get_argument_offset(restarg));
	}

    }
    
    /* compile the function body */
    do_begin(body,C_RETURN);

    /* remove the extra variable frame */
    if (extra > FIRSTENV)
	remove_frame();
}

/* count_arguments - count the arguments */
static int count_arguments(LVAL fargs,int *prargc,int *poargc,LVAL *prestarg,int *pkargc)
{
    int extra=FIRSTENV,rargc=0,oargc=0,kargc=0;
    LVAL arg,restarg=NULL,key,def,svar;
    
    /* skip each required argument */
    while (consp(fargs)
    &&     (arg = car(fargs)) != v_nil
    &&     !lambdakey(arg)) {
	if (!symbolp(arg))
	    xlerror("variable must be a symbol",arg);
	fargs = cdr(fargs);
	++rargc;
    }

    /* check for '&optional and #!optional arguments */
    if (consp(fargs) && (car(fargs) == lk_optional || car(fargs) == slk_optional)) {
	key = car(fargs);
	fargs = cdr(fargs);
	while (consp(fargs)
	&&     (arg = car(fargs)) != v_nil
	&&     !lambdakey(arg)) {
	    if (key == lk_optional)
		parse_optional_argument(arg,&arg,&def,&svar);
	    else {
		if (!symbolp(arg))
		    xlerror("#!optional argument must be a symbol",arg);
		svar = v_nil;
	    }
	    if (svar)
		++extra;
	    fargs = cdr(fargs);
	    ++extra;
	    ++oargc;
	}
    }

    /* check for the &rest or #!rest argument */
    if (consp(fargs) && (car(fargs) == lk_rest || car(fargs) == slk_rest)) {
	fargs = cdr(fargs);
	if (consp(fargs)
	&&  (arg = car(fargs)) != v_nil
	&&  !lambdakey(arg)) {
	    if (!symbolp(arg))
		xlerror("&rest variable must be a symbol",arg);
	    fargs = cdr(fargs);
	    restarg = arg;
	}
	else
	    xlerror("expecting the &rest variable",fargs);
    }

    /* check for &key arguments */
    if (consp(fargs) && car(fargs) == lk_key) {
	fargs = cdr(fargs);
	while (consp(fargs)
	&&     (arg = car(fargs)) != v_nil
	&&     !lambdakey(arg)) {
	    parse_key_argument(arg,&arg,&key,&def,&svar);
	    if (svar)
		++extra;
	    fargs = cdr(fargs);
	    ++extra;
	    ++kargc;
	}
	if (consp(fargs) && car(fargs) == lk_allow_other_keys)
	    fargs = cdr(fargs);
    }

    /* check for &aux arguments */
    if (consp(fargs) && car(fargs) == lk_aux) {
	fargs = cdr(fargs);
	while (consp(fargs)
	&&     (arg = car(fargs)) != v_nil
	&&     !lambdakey(arg)) {
	    parse_aux_argument(arg,&arg,&def);
	    fargs = cdr(fargs);
	    ++extra;
	}
    }

    /* check for the a dotted tail */
    if (restarg == v_nil && symbolp(fargs)) {
	restarg = fargs;
	fargs = v_nil;
    }

    /* add the &rest argument */
    if (restarg)
	++extra;
    
    /* check for the end of the argument list */
    if (fargs != v_nil)
	xlerror("bad argument list tail",fargs);

    /* return the argument counts */
    *prargc = rargc;
    *poargc = oargc;
    *prestarg = restarg;
    *pkargc = kargc;
    return extra;
}

/* add_extra_arguments - add extra (optional, key, rest, aux) arguments */
static void add_extra_arguments(LVAL fargs)
{
    LVAL arg,restarg=NULL,key,def,svar;
    
    /* skip each required argument */
    while (consp(fargs)
    &&     (arg = car(fargs)) != v_nil
    &&     !lambdakey(arg)) {
	if (!symbolp(arg))
	    xlerror("variable must be a symbol",arg);
	fargs = cdr(fargs);
    }

    /* check for '&optional and #!optional arguments */
    if (consp(fargs) && (car(fargs) == lk_optional || car(fargs) == slk_optional)) {
	key = car(fargs);
	fargs = cdr(fargs);
	while (consp(fargs)
	&&     (arg = car(fargs)) != v_nil
	&&     !lambdakey(arg)) {
	    if (key == lk_optional)
		parse_optional_argument(arg,&arg,&def,&svar);
	    else {
		if (!symbolp(arg))
		    xlerror("#!optional argument must be a symbol",arg);
		svar = v_nil;
	    }
	    add_argument_name(v_nil);	/* arg */
	    if (svar)
		add_argument_name(v_nil);	/* svar */
	    fargs = cdr(fargs);
	}
    }

    /* check for the &rest or #!rest argument */
    if (consp(fargs) && (car(fargs) == lk_rest || car(fargs) == slk_rest)) {
	fargs = cdr(fargs);
	if (consp(fargs)
	&&  (arg = car(fargs)) != v_nil
	&&  !lambdakey(arg)) {
	    if (!symbolp(arg))
		xlerror("&rest variable must be a symbol",arg);
	    fargs = cdr(fargs);
	    restarg = arg;
	}
	else
	    xlerror("expecting the &rest variable",fargs);
    }

    /* check for &key arguments */
    if (consp(fargs) && car(fargs) == lk_key) {
	fargs = cdr(fargs);
	while (consp(fargs)
	&&     (arg = car(fargs)) != v_nil
	&&     !lambdakey(arg)) {
	    parse_key_argument(arg,&arg,&key,&def,&svar);
	    add_argument_name(v_nil);	/* arg */
	    if (svar)
		add_argument_name(v_nil);	/* svar */
	    fargs = cdr(fargs);
	}
	if (consp(fargs) && car(fargs) == lk_allow_other_keys)
	    fargs = cdr(fargs);
    }

    /* check for &aux arguments */
    if (consp(fargs) && car(fargs) == lk_aux) {
	fargs = cdr(fargs);
	while (consp(fargs)
	&&     (arg = car(fargs)) != v_nil
	&&     !lambdakey(arg)) {
	    parse_aux_argument(arg,&arg,&def);
	    add_argument_name(v_nil);	/* arg */
	    fargs = cdr(fargs);
	}
    }

    /* check for the a dotted tail */
    if (restarg == v_nil && symbolp(fargs)) {
	restarg = fargs;
	fargs = v_nil;
    }

    /* add the &rest argument */
    if (restarg)
	add_argument_name(v_nil);		/* rest */
    
    /* check for the end of the argument list */
    if (fargs != v_nil)
	xlerror("bad argument list tail",fargs);
}

/* parse_optional_arguments - parse the &optional arguments */
static void parse_optional_arguments(LVAL key,LVAL *pfargs,int base)
{
    extern LVAL default_object;
    int patch,patch2,chain,off,oargc=0;
    LVAL fargs,arg,def,svar;

    /* generate the conditional branches */
    fargs = *pfargs;
    while (consp(fargs)
    &&     (arg = car(fargs)) != v_nil
    &&     !lambdakey(arg)) {
	
	/* parse the argument form */
	if (key == lk_optional)
	    parse_optional_argument(arg,&arg,&def,&svar);
	else {
	    def = default_object;
	    svar = v_nil;
	}

	/* output code to check for the optional argument */
	patch = putcbyte(OP_OPTARG);
	putcbyte(base + oargc);
	putcbyte(0);

	/* set the supplied-p variable if present */
	if (svar)
	    cd_evariable(OP_ESET,0,0);

	/* compile the default value expression */
	if (def) {
	    putcbyte(OP_BRT);
	    chain = putcword(0);
	    if (def == default_object)
		do_literal(def,C_NEXT);
	    else
		do_expr(def,C_NEXT);
	    patch2 = nextcaddr();
	    cd_evariable(OP_ESET,0,0);
	    fixup(chain);
	}

	/* add the argument name to the name list */
	patch_argument_name(arg);
	off = get_argument_offset(arg);
	cbuff[cbase+patch+2] = off;

	/* check for a supplied-p variable */
	if (svar) {
	    patch_argument_name(svar);
	    cbuff[cbase+patch+5] = get_argument_offset(svar);
	}

	/* patch the setting of the default value */
	if (def)
	    cbuff[cbase+patch2+2] = off;

	/* move the formal argument list pointer ahead */
	fargs = cdr(fargs);
	++oargc;
    }

    /* update the reference parameters */
    *pfargs = fargs;
}

/* parse_optional_argument - parse a single &optional argument */
static void parse_optional_argument(LVAL form,LVAL *parg,LVAL *pdef,LVAL *psvar)
{
    *pdef = *psvar = v_nil;
    if (consp(form)) {
	if ((*pdef = cdr(form)) != v_nil)
	    if (consp(*pdef)) {
		if ((*psvar = cdr(*pdef)) != v_nil)
		    if (consp(*psvar)) {
			*psvar = car(*psvar);
			if (!symbolp(*psvar))
			    xlfmterror("supplied-p variable must be a symbol");
		    }
		    else
			xlfmterror("expecting supplied-p variable");
		*pdef = car(*pdef);
	    }
	    else
		xlfmterror("expecting init expression");
	*parg = car(form);
    }
    else
	*parg = form;
    if (!symbolp(*parg))
	xlerror("&optional variable must be a symbol",*parg);
}

/* parse_key_arguments - parse the &key arguments */
static void parse_key_arguments(LVAL *pfargs,int base)
{
    LVAL fargs,arg,key,def,svar;
    int patch,patch2,chain,off;

    /* generate the conditional branches */
    fargs = *pfargs;
    while (consp(fargs)
    &&     (arg = car(fargs)) != v_nil
    &&     !lambdakey(arg)) {
	
	/* parse the argument form */
	parse_key_argument(arg,&arg,&key,&def,&svar);
	
	/* check for the &key argument */
	patch = putcbyte(OP_KEYARG);
	putcbyte(findliteral(key));
	putcbyte(base);
	putcbyte(0);

	/* set the supplied-p variable if present */
	if (svar)
	    cd_evariable(OP_ESET,0,0);

	/* compile the default value expression */
	if (def) {
	    putcbyte(OP_BRT);
	    chain = putcword(0);
	    do_expr(def,C_NEXT);
	    patch2 = nextcaddr();
	    cd_evariable(OP_ESET,0,0);
	    fixup(chain);
	}

	/* add the argument name to the name list */
	patch_argument_name(arg);
	off = get_argument_offset(arg);
	cbuff[cbase+patch+3] = off;

	/* set the supplied-p variable */
	if (svar) {
	    patch_argument_name(svar);
	    cbuff[cbase+patch+6] = get_argument_offset(svar);
	}

	/* patch the setting of the default value */
	if (def)
	    cbuff[cbase+patch2+2] = off;

	/* move the formal argument list pointer ahead */
	fargs = cdr(fargs);
    }

    /* update the reference parameters */
    *pfargs = fargs;
}

/* parse_key_argument - parse a single &key argument */
static void parse_key_argument(LVAL form,LVAL *parg,LVAL *pkey,LVAL *pdef,LVAL *psvar)
{
    extern LVAL keywordpackage;
    LVAL key;
    *pkey = *pdef = *psvar = v_nil;
    if (consp(form)) {
	if ((*pdef = cdr(form)) != v_nil)
	    if (consp(*pdef)) {
		if ((*psvar = cdr(*pdef)) != v_nil) {
		    if (consp(*psvar)) {
			*psvar = car(*psvar);
			if (!symbolp(*psvar))
			    xlerror("supplied-p variable must be a symbol",*psvar);
		    }
		    else
			xlfmterror("expecting supplied-p variable");
		}
		*pdef = car(*pdef);
	    }
	    else
		xlfmterror("expecting init expression");
	if ((*parg = car(form)) != v_nil) {
	    if (consp(*parg)) {
		*pkey = car(*parg);
		if (!symbolp(*pkey))
		    xlerror("&key keyword must be a symbol",*pkey);
		if ((*parg = cdr(*parg)) != v_nil)
		    *parg = car(*parg);
		else
		    xlfmterror("expecting keyword variable");
	    }
	}
	else
	    xlfmterror("expecting keyword variable");
    }
    else
	*parg = form;
    if (!symbolp(*parg))
	xlerror("&key variable must be a symbol",*parg);
    if (*pkey == v_nil) {
	*pkey = intern(getpname(*parg),keywordpackage,&key);
	export(*pkey,keywordpackage);
    }
}

/* parse_aux_arguments - parse the &aux arguments */
static void parse_aux_arguments(LVAL *pfargs)
{
    LVAL fargs,arg,def;
    fargs = *pfargs;
    while (consp(fargs)
    &&     (arg = car(fargs)) != v_nil
    &&     !lambdakey(arg)) {
	
	/* parse the argument form */
	parse_aux_argument(arg,&arg,&def);

	/* compile the initialization expression */
	if (def)
	    do_expr(def,C_NEXT);

	/* add the argument name */
	patch_argument_name(arg);
	
	/* store the initialization value */
	if (def)
	    cd_evariable(OP_ESET,0,get_argument_offset(arg));
	
	/* move the formal argument list pointer ahead */
	fargs = cdr(fargs);
    }

    /* update the reference parameters */
    *pfargs = fargs;
}

/* parse_aux_argument - parse a single &aux argument */
static void parse_aux_argument(LVAL form,LVAL *parg,LVAL *pdef)
{
    *pdef = v_nil;
    if (consp(form)) {
	if ((*pdef = cdr(form)) != v_nil)
	    if (consp(*pdef))
		*pdef = car(*pdef);
	    else
		xlfmterror("expecting init expression");
	*parg = car(form);
    }
    else
	*parg = form;
    if (!symbolp(*parg))
	xlerror("&aux variable must be a symbol",*parg);
}

/* add_argument_name - add an argument name to the argument list */
static void add_argument_name(LVAL name)
{
    int level = FIRSTENV;
    LVAL last;
    if ((last = getenvnames(car(info))) != v_nil) {
	for (;;) {
	    if (name && name == car(last))
		xlerror("duplicate argument name",name);
	    if (cdr(last) == v_nil)
	        break;
	    last = cdr(last);
            ++level;
	}
	rplacd(last,cons(name,v_nil));
    }
    else {
        if (level < 255)
	    setenvnames(car(info),cons(name,v_nil));
        else
            xlfmterror("too many environment variables");
    }
}

/* patch_argument_name - patch a nil argument name in the argument list */
static void patch_argument_name(LVAL name)
{
    int found=FALSE;
    LVAL next;
    if ((next = getenvnames(car(info))) != v_nil) {
	while (next) {
	    if (name == car(next))
		xlerror("duplicate argument name",name);
	    if (!found && car(next) == v_nil) {
	        rplaca(next,name);
		found = TRUE;
	    }
	    next = cdr(next);
	}
    }
    if (!found)
	xlerror("trouble patching argument",name);
}

/* get_argument_offset - get the offset to an argument in the environment frame */
static int get_argument_offset(LVAL arg)
{
    int off;
    findcvariable(OP_EREF,car(info),arg,&off);
    return off;
}

/* do_delay - compile the (DELAY ... ) expression */
static void do_delay(LVAL form,int cont)
{
    int oldcbase;

    /* check argument list */
    if (atom(form))
	xlerror("expecting delay expression",form);

    /* establish a new environment frame */
    oldcbase = add_level();

    /* setup the entry code */
    putcbyte(OP_ARGSEQ);
    putcbyte(0);

    /* compile the expression */
    do_expr(car(form),C_RETURN);

    /* build the code object */
    cpush(make_code_object(v_nil));
    
    /* restore the previous environment */
    remove_level(oldcbase);

    /* compile code to create a closure */
    do_literal(pop(),C_NEXT);
    putcbyte(OP_DELAY);
    do_continuation(cont);
}

/* do_let - compile the (LET ... ) expression */
static void do_let(LVAL form,int cont)
{
    /* handle named let */
    if (consp(form) && symbolp(car(form)))
	do_named_let(form,cont);
    
    /* handle unnamed let */
    else
        do_unnamed_let(form,cont);
}

/* do_named_let - compile the (LET name (...) ... ) expression */
static void do_named_let(LVAL form,int cont)
{
    int oldcbase,rargc,opcode,lev,off;

    /* push the procedure */
    putcbyte(OP_NIL);
    putcbyte(OP_PUSH);
    
    /* establish a new environment frame */
    add_frame();
    add_argument_name(car(form));

    /* push a new environment frame */
    putcbyte(OP_FRAME);
    putcbyte(1);
    putcbyte(FIRSTENV);
    putcbyte(findliteral(getenvnames(car(info))));

    /* make sure there is a binding list */
    if (atom(cdr(form)) || !listp(car(cdr(form))))
	xlerror("expecting binding list",form);
    
    /* push the initialization expressions */
    push_init_expressions(car(cdr(form)));
    
    /* establish a new environment frame */
    oldcbase = add_level();
    
    /* build a function */
    cpush(extract_let_variables(car(cdr(form)),&rargc));
    parse_lambda_expr(top(),cdr(cdr(form)),FALSE);
    
    /* build the code object */
    settop(make_code_object(car(form)));
    
    /* restore the previous environment */
    remove_level(oldcbase);

    /* compile code to create a closure */
    do_literal(pop(),C_NEXT);
    putcbyte(OP_CLOSE);

    /* store the procedure */
    opcode = findvariable(OP_ESET,car(form),&lev,&off);
    cd_evariable(opcode,lev,off);
    
    /* apply the function */
    putcbyte(cont == C_RETURN ? OP_TCALL : OP_CALL);
    putcbyte(rargc);
    if (cont == C_NEXT) {
	putcbyte(OP_UNFRAME);
	do_continuation(cont);
    }

    /* restore the previous environment */
    remove_frame();
}

/* extract_let_variables - extract a list of variable names from a let binding list */
static LVAL extract_let_variables(LVAL bindings,int *pcnt)
{
    LVAL this,last;
    check(2);
    push(bindings);
    push(v_nil);
    for (*pcnt = 0; consp(bindings); bindings = cdr(bindings), ++(*pcnt)) {
	LVAL def = car(bindings),sym;
	if (symbolp(def))
	    sym = def;
	else if (consp(def) && symbolp(car(def)))
	    sym = car(def);
	else
	    xlerror("invalid binding",def);
	this = cons(sym,v_nil);
	if (top() == v_nil)
	    settop(this);
	else
	    rplacd(last,this);
	last = this;
    }
    this = pop();
    drop(1);
    return this;
}

/* do_unnamed_let - compile the (LET (...) ... ) expression */
static void do_unnamed_let(LVAL form,int cont)
{
    /* make sure there is a binding list */
    if (atom(form) || !listp(car(form)))
	xlerror("expecting binding list",form);

    /* push the initialization expressions */
    push_init_expressions(car(form));

    /* establish a new environment frame */
    add_frame();

    /* compile the binding list */
    generate_let_setup_code(car(form));

    /* compile the body of the let */
    do_begin(cdr(form),cont);
    if (cont == C_NEXT) {
	putcbyte(OP_UNFRAME);
	do_continuation(cont);
    }

    /* restore the previous environment */
    remove_frame();
}

/* do_letrec - compile the (LETREC ... ) expression */
static void do_letrec(LVAL form,int cont)
{
    /* make sure there is a binding list */
    if (atom(form) || !listp(car(form)))
	xlerror("expecting binding list",form);

    /* push the initialization expressions */
    push_dummy_values(car(form));

    /* establish a new environment frame */
    add_frame();

    /* compile the binding list */
    generate_let_setup_code(car(form));

    /* compile instructions to set the bound variables */
    set_bound_variables(car(form));
    
    /* compile the body of the letrec */
    do_begin(cdr(form),cont);
    if (cont == C_NEXT) {
	putcbyte(OP_UNFRAME);
	do_continuation(cont);
    }

    /* restore the previous environment */
    remove_frame();
}

/* do_letstar - compile the (LET* ... ) expression */
static void do_letstar(LVAL form,int cont)
{
    /* make sure there is a binding list */
    if (atom(form) || !listp(car(form)))
	xlerror("expecting binding list",form);

    /* build the nested lambda expressions */
    if (consp(car(form)))
	letstar1(car(form),cdr(form),cont);
    
    /* handle the case where there are no bindings */
    else
	do_begin(cdr(form),cont);
}

/* letstar1 - helper routine for let* */
static void letstar1(LVAL blist,LVAL body,int cont)
{
    /* push the next initialization expressions */
    cpush(cons(car(blist),v_nil));
    push_init_expressions(top());

    /* establish a new environment frame */
    add_frame();

    /* handle the case where there are more bindings */
    if (consp(cdr(blist))) {
	generate_let_setup_code(top());
	letstar1(cdr(blist),body,cont);
    }
    
    /* handle the last binding */
    else {
	generate_let_setup_code(top());
	do_begin(body,cont);
    }

    /* remove the frame if more code follows */
    if (cont == C_NEXT) {
	putcbyte(OP_UNFRAME);
	do_continuation(cont);
    }
    drop(1);
    	
    /* restore the previous environment */
    remove_frame();
    do_continuation(cont);
}

/* push_dummy_values - push dummy values for a 'letrec' expression */
static int push_dummy_values(LVAL blist)
{
    int n=0;
    if (consp(blist)) {
	putcbyte(OP_NIL);
	for (; consp(blist); blist = cdr(blist), ++n)
	    putcbyte(OP_PUSH);
    }
    return n;
}

/* push_init_expressions - push init expressions for a 'let' expression */
static int push_init_expressions(LVAL blist)
{
    int argc;
    if (consp(blist)) {
	argc = push_init_expressions(cdr(blist));
	if (consp(car(blist)) && consp(cdr(car(blist))))
	    do_expr(car(cdr(car(blist))),C_NEXT);
	else
	    putcbyte(OP_NIL);
	putcbyte(OP_PUSH);
	return argc + 1;
    }
    return 0;
}

/* generate_let_setup_code - generate the setup code for an in-line let */
static void generate_let_setup_code(LVAL blist)
{
    int rargc,extra;
    parse_let_variables(blist,&rargc,&extra);    
    putcbyte(OP_FRAME);
    putcbyte(rargc);
    putcbyte(extra);
    putcbyte(findliteral(getenvnames(car(info))));
}

/* parse_let_variables - parse the binding list */
static void parse_let_variables(LVAL blist,int *prargc,int *pextra)
{
    int rargc;
    LVAL arg;
    
    /* initialize the argument name list and slot number */
    rargc = 0;
    
    /* handle each required argument */
    while (consp(blist) && (arg = car(blist)) != v_nil) {

	/* make sure the argument is a symbol */
	if (symbolp(arg))
	    ;
	else if (consp(arg) && symbolp(car(arg)))
	    arg = car(arg);
	else
	    xlerror("invalid binding",arg);

	/* add the argument name to the name list */
	add_argument_name(arg);
	
	/* move the formal argument list pointer ahead */
	blist = cdr(blist);
	++rargc;
    }

    /* return the binding count and extra slot count */
    *prargc = rargc;
    *pextra = FIRSTENV;
}

/* set_bound_variables - set bound variables in a 'letrec' expression */
static void set_bound_variables(LVAL blist)
{
    int opcode,lev,off;
    for (; consp(blist); blist = cdr(blist)) {
	if (consp(car(blist)) && consp(cdr(car(blist)))) {
	    do_expr(car(cdr(car(blist))),C_NEXT);
	    if ((opcode = findvariable(OP_ESET,car(car(blist)),&lev,&off)) != 0)
		cd_evariable(opcode,lev,off);
	    else
		xlerror("compiler error -- can't find",car(car(blist)));
	}
    }
}

/* do_mvbind - compile the (MULTIPLE-VALUE-BIND (...) ... ) expression */
static void do_mvbind(LVAL form,int cont)
{
    LVAL blist,arg,p;
    int size,n;
    
    /* get the binding list */
    if (atom(form) || !listp(car(form)))
	xlerror("expecting binding list",form);
    blist = car(form);

    /* compile the multiple value expression */
    if (!consp(cdr(form)))
	xlerror("expecting expression",cdr(form));
    do_expr(car(cdr(form)),C_NEXT);

    /* establish a new environment frame */
    add_frame();

    /* handle each symbol to bind */
    for (p = blist; consp(p) && (arg = car(p)) != v_nil; p = cdr(p)) {
	if (!symbolp(arg))
	    xlerror("invalid binding",arg);
	add_argument_name(arg);
    }

    /* establish a new environment frame */
    add_frame();
    size = FIRSTENV;

    /* handle each symbol to bind */
    for (p = blist; consp(p) && (arg = car(p)) != v_nil; p = cdr(p)) {
	if (!symbolp(arg))
	    xlerror("invalid binding",arg);
	add_argument_name(arg);
	++size;
    }

    /* compile the binding list */
    putcbyte(OP_MVFRAME);
    putcbyte(size);
    putcbyte(findliteral(getenvnames(getnextframe(car(info)))));
    putcbyte(findliteral(getenvnames(car(info))));

    /* move the arguments into the correct stack frame slots */
    for (p = blist, n = FIRSTENV; consp(p); p = cdr(p), ++n) {
    	putcbyte(OP_OPTARG);
	putcbyte(n);
	putcbyte(get_argument_offset(car(p)));
    }

    /* compile the body of the multiple-value-bind */
    do_begin(cdr(cdr(form)),cont);
    if (cont == C_NEXT) {
	putcbyte(OP_UNFRAME);
	putcbyte(OP_UNFRAME);
	do_continuation(cont);
    }

    /* restore the previous environment */
    remove_frame();
    remove_frame();
}

/* make_code_object - build a code object */
static LVAL make_code_object(LVAL fun)
{
    unsigned char *src,*dst,*end;
    LVAL code,p;
    int i;

    /* create a code object */
    code = newcode(FIRSTLIT + length(car(cdr(info)))); cpush(code);
    setbcode(code,xlnewstring(cptr - cbase));
    setcname(code,fun);			       	/* function name */
    setvnames(code,getenvnames(car(info)));	/* lambda list variables */

    /* copy the literals into the code object */
    for (i = FIRSTLIT, p = car(cdr(info)); consp(p); p = cdr(p), ++i)
	setelement(code,i,car(p));

    /* copy the byte codes */
    for (src = &cbuff[cbase], end = &cbuff[cptr], dst = getcodestr(code); src < end; )
	*dst++ = *src++;

    /* return the new code object */
    return pop();
}

/* do_cond - compile the (COND ... ) expression */
static void do_cond(LVAL form,int cont)
{
    int nxt,end;
    if (consp(form)) {
	for (end = 0; consp(form); form = cdr(form)) {
	    if (atom(car(form)))
		xlerror("expecting a cond clause",form);
	    do_expr(car(car(form)),C_NEXT);
	    putcbyte(OP_BRF);
	    nxt = putcword(0);
	    if (cdr(car(form)))
		do_begin(cdr(car(form)),cont);
	    else
		do_continuation(cont);
	    if (cont == C_NEXT) {
		putcbyte(OP_BR);
		end = putcword(end);
	    }
	    fixup(nxt);
	}
	fixup(end);
    }
    else
	putcbyte(OP_NIL);
    do_continuation(cont);
}

/* do_and - compile the (AND ... ) expression */
static void do_and(LVAL form,int cont)
{
    int end;
    if (consp(form)) {
	for (end = 0; consp(form); form = cdr(form)) {
	    if (cdr(form)) {
		do_expr(car(form),C_NEXT);
		putcbyte(OP_BRF);
		end = putcword(end);
	    }
	    else
		do_expr(car(form),cont);
	}
	fixup(end);
    }
    else
	putcbyte(OP_T);
    do_continuation(cont);
}

/* do_or - compile the (OR ... ) expression */
static void do_or(LVAL form,int cont)
{
    int end;
    if (consp(form)) {
	for (end = 0; consp(form); form = cdr(form)) {
	    if (cdr(form)) {
		do_expr(car(form),C_NEXT);
		putcbyte(OP_BRT);
		end = putcword(end);
	    }
	    else
		do_expr(car(form),cont);
	}
	fixup(end);
    }
    else
	putcbyte(OP_NIL);
    do_continuation(cont);
}

/* do_if - compile the (IF ... ) expression */
static void do_if(LVAL form,int cont)
{
    int nxt,end;

    /* compile the test expression */
    if (atom(form))
	xlerror("expecting test expression",form);
    do_expr(car(form),C_NEXT);

    /* skip around the 'then' clause if the expression is false */
    putcbyte(OP_BRF);
    nxt = putcword(0);

    /* skip to the 'then' clause */
    form = cdr(form);
    if (atom(form))
	xlerror("expecting then clause",form);

    /* compile the 'then' and 'else' clauses */
    if (consp(cdr(form))) {
	if (cont == C_NEXT) {
	    do_expr(car(form),C_NEXT);
	    putcbyte(OP_BR);
	    end = putcword(0);
	}
	else {
	    do_expr(car(form),cont);
	    end = -1;
	}
	fixup(nxt);
	do_expr(car(cdr(form)),cont);
	nxt = end;
    }

    /* compile just a 'then' clause */
    else
	do_expr(car(form),cont);

    /* handle the end of the statement */
    if (nxt >= 0) {
	fixup(nxt);
	do_continuation(cont);
    }
}

/* do_begin - compile the (BEGIN ... ) expression */
static void do_begin(LVAL form,int cont)
{
    if (consp(form))
	for (; consp(form); form = cdr(form))
	    if (consp(cdr(form)))
		do_expr(car(form),C_NEXT);
	    else
		do_expr(car(form),cont);
    else {
	putcbyte(OP_NIL);
	do_continuation(cont);
    }
}

/* do_while - compile the (WHILE ... ) expression */
static void do_while(LVAL form,int cont)
{
    int loop,nxt;

    /* make sure there is a test expression */
    if (atom(form))
	xlerror("expecting test expression",form);

    /* skip around the 'body' to the test expression */
    putcbyte(OP_BR);
    nxt = putcword(0);

    /* compile the loop body */
    loop = cptr - cbase;
    do_begin(cdr(form),C_NEXT);

    /* label for the first iteration */
    fixup(nxt);

    /* compile the test expression */
    nxt = cptr - cbase;
    do_expr(car(form),C_NEXT);

    /* skip around the 'body' if the expression is false */
    putcbyte(OP_BRT);
    putcword(loop);

    /* compile the continuation */
    do_continuation(cont);
}

/* do_catch - compile the (CATCH ... ) expression */
static void do_catch(LVAL form,int cont)
{
    int nxt;
    
    /* make sure there is a tag expression */
    if (atom(form))
	xlerror("expecting tag expression",form);

    /* compile the catch tag expression */
    do_expr(car(form),C_NEXT);
    
    /* output a catch instruction to push a catch frame */
    putcbyte(OP_CATCH);
    nxt = putcword(0);

    /* compile the catch body */
    do_begin(cdr(form),C_NEXT);
    
    /* pop the catch frame */
    putcbyte(OP_UNCATCH);
    fixup(nxt);
    
    /* compile the continuation */
    do_continuation(cont);
}

/* do_unwindprotect - compile the (UNWIND-PROTECT ... ) expression */
static void do_unwindprotect(LVAL form,int cont)
{
    /* make sure there is a protected expression */
    if (atom(form))
	xlerror("expecting protected expression",form);

    /* check for cleanup forms */
    if (cdr(form) == v_nil)
        do_expr(car(form),cont);
    else {
        cd_fundefinition(v_nil,v_nil,cdr(form));
	putcbyte(OP_PROTECT);
	do_expr(car(form),C_NEXT);
	putcbyte(OP_MVPUSH);
	putcbyte(OP_UNPROTECT);
	putcbyte(OP_CALL);
	putcbyte(0);
	putcbyte(OP_MVPOP);
	do_continuation(cont);
    }
}

/* do_call - compile a function call */
static void do_call(LVAL form,int cont)
{
    int n;
    
    /* compile each argument expression */
    n = push_args(cdr(form));

    /* compile the function itself */
    do_expr(car(form),C_NEXT);

    /* apply the function */
    putcbyte(cont == C_RETURN ? OP_TCALL : OP_CALL);
    putcbyte(n);
}

/* do_mvcall - compile a multiple value function call */
static void do_mvcall(LVAL form,int cont)
{
    /* check the syntax */
    if (!consp(cdr(form)))
	xlerror("expecting multiple value expression",cdr(form));
    else if (!null(cdr(cdr(form))))
	xlerror("only one multiple value expression allowed",cdr(form));

    /* compile each argument expression */
    do_expr(car(cdr(form)),C_NEXT);
    putcbyte(OP_MVPUSH);

    /* compile the function itself */
    do_expr(car(form),C_NEXT);

    /* apply the function */
    putcbyte(cont == C_RETURN ? OP_MVTCALL : OP_MVCALL);
}

/* push_args - compile the arguments for a function call */
static int push_args(LVAL form)
{
    int argc;
    if (consp(form)) {
	argc = push_args(cdr(form));
	do_expr(car(form),C_NEXT);
	putcbyte(OP_PUSH);
	return argc + 1;
    }
    return 0;
}

/* do_nary - compile nary operator expressions */
static void do_nary(int op,int n,LVAL form,int cont)
{
    if (n < 0 && (n = (-n)) != length(cdr(form)))
	do_call(form,cont);
    else {
	push_nargs(form,n);
	putcbyte(op);
	do_continuation(cont);
    }
}

/* push_nargs - compile the arguments for an inline function call */
static void push_nargs(LVAL form,int n)
{
    int cnt=0;
    if (consp(cdr(form))) {
	cnt = push_args(cdr(cdr(form))) + 1;
	do_expr(car(cdr(form)),C_NEXT);
    }
    if (cnt > n)
	xlerror("too many arguments",form);
    else if (cnt < n)
	xlerror("too few arguments",form);
}

/* do_literal - compile a literal */
static void do_literal(LVAL lit,int cont)
{
    cd_literal(lit);
    do_continuation(cont);
}

/* do_identifier - compile an identifier */
static void do_identifier(LVAL sym,int cont)
{
    int opcode,lev,off;
    if (sym == v_true)
	putcbyte(OP_T);
    else if ((opcode = findvariable(OP_EREF,sym,&lev,&off)) != 0)
	cd_evariable(opcode,lev,off);
    else
	cd_variable(OP_GREF,sym);
    do_continuation(cont);
}

/* do_continuation - compile a continuation */
static void do_continuation(int cont)
{
    switch (cont) {
    case C_RETURN:
	putcbyte(OP_RETURN);
	break;
    case C_NEXT:
	break;
    }
}

/* add_frame - add a new environment frame */
static void add_frame(void)
{
    rplaca(info,newframe(ENV,car(info),FIRSTENV));
}

/* remove_frame - remove an environment frame */
static void remove_frame(void)
{
    rplaca(info,getnextframe(car(info)));
}

/* add_level - add a nesting level */
static int add_level(void)
{
    int oldcbase;
    
    /* add a new environment frame */
    add_frame();
    
    /* add a new literal list */
    rplacd(info,cons(v_nil,cdr(info)));
    
    /* setup the base of the code for this function */
    oldcbase = cbase;
    cbase = cptr;

    /* return the old code base */
    return oldcbase;
}

/* remove_level - remove a nesting level */
static void remove_level(int oldcbase)
{
    /* restore the previous environment */
    remove_frame();
    
    /* remove the previous literal list */
    rplacd(info,cdr(cdr(info)));

    /* restore the base and code pointer */
    cptr = cbase;
    cbase = oldcbase;
}

/* findvariable - find an environment variable */
static int findvariable(int opcode,LVAL sym,int *plev,int *poff)
{
    LVAL frame = car(info);
    int newop;
    for (*plev = 0; frame != v_nil; frame = getnextframe(frame), ++(*plev))
	if ((newop = findcvariable(opcode,frame,sym,poff)) != 0)
	    return newop;
    return 0;
}

/* findcvariable - find an environment variable in the current frame */
static int findcvariable(int opcode,LVAL frame,LVAL sym,int *poff)
{
    LVAL names = getenvnames(frame);
    for (*poff = FIRSTENV; names != v_nil; ++(*poff), names = cdr(names))
	if (sym == car(names))
	    return opcode;
    if (menvp(frame)) {
	names = getivar(getenvelement(getnextframe(frame),FIRSTENV),IV_IVARS);
	for (*poff = FIRSTIVAR; names != v_nil; ++(*poff), names = cdr(names))
	    if (sym == car(names))
		return opcode + 1;
    }
    return 0;
}

/* findliteral - find a literal in the literal frame */
static int findliteral(LVAL lit)
{
    int o = FIRSTLIT;
    LVAL t,p;

    /* first check to see if the literal already exists */
    if ((t = car(cdr(info))) != v_nil) {
	for (p = v_nil; consp(t); p = t, t = cdr(t), ++o)
	    if (eq(lit,car(t)))
		return o;
    }

    /* make sure there aren't too many literals */
    if (o > 255)
        xlfmterror("too many literals");

    /* add the new literal */
    if (car(cdr(info)) == v_nil)
	rplaca(cdr(info),cons(lit,v_nil));
    else
	rplacd(p,cons(lit,v_nil));
    return o;
}

/* cd_variable - compile a variable reference */
static void cd_variable(int op,LVAL sym)
{
    putcbyte(op);
    putcbyte(findliteral(sym));
}

/* cd_evariable - compile an environment variable reference */
static void cd_evariable(int op,int lev,int off)
{
    putcbyte(op);
    putcbyte(lev);
    putcbyte(off);
}

/* cd_literal - compile a literal reference */
static void cd_literal(LVAL lit)
{
    if (lit == v_nil)
	putcbyte(OP_NIL);
    else if (lit == v_true)
	putcbyte(OP_T);
    else {
	putcbyte(OP_LIT);
	putcbyte(findliteral(lit));
    }
}

/* nextcaddr - get the next code address */
static int nextcaddr(void)
{
    return cptr - cbase;
}

/* putcbyte - put a code byte into data space */
static int putcbyte(int b)
{
    int adr;
    if (cptr >= CMAX)
	xlfmtabort("insufficient code space");
    adr = (cptr - cbase);
    cbuff[cptr++] = b;
    return adr;
}

/* putcword - put a code word into data space */
static int putcword(int w)
{
    int adr;
    adr = putcbyte(w >> 8);
    putcbyte(w);
    return adr;
}

/* fixup - fixup a reference chain */
static void fixup(int chn)
{
    int val,hval,nxt;

    /* store the value into each location in the chain */
    val = cptr - cbase; hval = val >> 8;
    for (; chn; chn = nxt) {
	nxt = (cbuff[cbase+chn] << 8) | (cbuff[cbase+chn+1]);
	cbuff[cbase+chn] = hval;
	cbuff[cbase+chn+1] = val;
    }
}

/* length - find the length of a list */
FIXTYPE length(LVAL list)
{
    FIXTYPE len;
    for (len = 0; consp(list); list = cdr(list))
	++len;
    return len;
}
