/* xldbg.c - debugging routines */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"
#include "xlbcode.h"

/* instruction output formats */
#define FMT_NONE	0
#define FMT_BYTE	1
#define FMT_2BYTE	2
#define FMT_LOFF	3
#define FMT_WORD	4
#define FMT_EOFF	5
#define FMT_IOFF	6
#define FMT_OPTARG	7
#define FMT_KEYARG	8
#define FMT_FRAME	9
#define FMT_MVFRAME	10
#define FMT_FOFF	11
#define FMT_ARGS	12
#define FMT_ARGS2	13

typedef struct { int ot_code; char *ot_name; int ot_fmt; } OTDEF;
OTDEF otab[] = {
{	OP_BRT,		"BRT",		FMT_WORD	},
{	OP_BRF,		"BRF",		FMT_WORD	},
{	OP_BR,		"BR",		FMT_WORD	},
{	OP_LIT,		"LIT",		FMT_LOFF	},
{	OP_GREF,	"GREF",		FMT_LOFF	},
{	OP_GSET,	"GSET",		FMT_LOFF	},
{	OP_EREF,	"EREF",		FMT_EOFF	},
{	OP_IREF,	"IREF",		FMT_IOFF	},
{	OP_ESET,	"ESET",		FMT_EOFF	},
{	OP_ISET,	"ISET",		FMT_IOFF	},
{	OP_CALL,	"CALL",		FMT_BYTE	},
{	OP_MVCALL,	"MVCALL",	FMT_NONE	},
{	OP_TCALL,	"TCALL",	FMT_BYTE	},
{	OP_MVTCALL,	"MVTCALL",	FMT_NONE	},
{	OP_RETURN,	"RETURN",	FMT_NONE	},
{	OP_FRAME,	"FRAME",	FMT_FRAME	},
{	OP_MVFRAME,	"MVFRAME",	FMT_MVFRAME	},
{	OP_UNFRAME,	"UNFRAME",	FMT_NONE	},
{	OP_MVPUSH,	"MVPUSH",	FMT_NONE	},
{	OP_MVPOP,	"MVPOP",	FMT_NONE	},
{	OP_T,		"T",		FMT_NONE	},
{	OP_NIL,		"NIL",		FMT_NONE	},
{	OP_PUSH,	"PUSH",		FMT_NONE	},
{	OP_CLOSE,	"CLOSE",	FMT_NONE	},
{	OP_DELAY,	"DELAY",	FMT_NONE	},

{	OP_ARGSEQ,	"ARGSEQ",	FMT_ARGS	},
{	OP_ARGSGE,	"ARGSGE",	FMT_ARGS	},
{	OP_ARGSBT,	"ARGSBT",	FMT_ARGS2	},
{	OP_OPTARG,	"OPTARG",	FMT_OPTARG	},
{	OP_KEYARG,	"KEYARG",	FMT_KEYARG	},
{	OP_REST,	"REST",		FMT_2BYTE	},

{	OP_METHOD,	"METHOD",	FMT_NONE	},

{	OP_CATCH,	"CATCH",	FMT_WORD	},
{	OP_UNCATCH,	"UNCATCH",	FMT_NONE	},
{	OP_PROTECT,	"PROTECT",	FMT_NONE	},
{	OP_UNPROTECT,	"UNPROTECT",	FMT_NONE	},

/* integrable functions */
{	OP_ATOM,	"ATOM",		FMT_NONE	},
{	OP_EQ,		"EQ?",		FMT_NONE	},
{	OP_NULL,	"NULL?",	FMT_NONE	},
{	OP_NULL,	"NOT",		FMT_NONE	},
{	OP_CONS,	"CONS",		FMT_NONE	},
{	OP_CAR,		"CAR",		FMT_NONE	},
{	OP_CDR,		"CDR",		FMT_NONE	},
{	OP_SETCAR,	"SET-CAR!",	FMT_NONE	},
{	OP_SETCDR,	"SET-CDR!",	FMT_NONE	},
{	OP_ADD,		"+",		FMT_NONE	},
{	OP_SUB,		"-",		FMT_NONE	},
{	OP_MUL,		"*",		FMT_NONE	},
{	OP_QUO,		"QUOTIENT",	FMT_NONE	},
{	OP_LSS,		"<",		FMT_NONE	},
{	OP_EQL,		"=",		FMT_NONE	},
{	OP_GTR,		">",		FMT_NONE	},

{0,0,0}
};

/* forward declarations */
static LVAL findenvname(LVAL env,int lev,int off);
static LVAL findivarname(LVAL env,int lev,int off);

/* decode_procedure - decode the instructions in a code object */
void decode_procedure(LVAL fptr,LVAL code,LVAL env)
{
    unsigned char *cp;
    FIXTYPE len,lc;
    int n;
    cpush(env);
    settop(newframe(ENV,top(),FIRSTENV));
    setenvnames(top(),getvnames(code));
    len = getslength(getbcode(code));
    for (lc = 0; lc < len; lc += n) {
	n = decode_instruction(fptr,code,lc,top());
	cp = getcodestr(code) + lc;
	switch (*cp) {
	case OP_FRAME:
	    settop(newframe(ENV,top(),FIRSTENV));
	    setenvnames(top(),getelement(code,cp[3]));
	    break;
	case OP_MVFRAME:
	    settop(newframe(ENV,top(),FIRSTENV));
	    setenvnames(top(),getelement(code,cp[2]));
	    settop(newframe(ENV,top(),FIRSTENV));
	    setenvnames(top(),getelement(code,cp[3]));
	    break;
	case OP_UNFRAME:
	    settop(getnextframe(top()));
	    break;
	}
    }
    drop(1);
}

/* decode_instruction - decode a single bytecode instruction */
int decode_instruction(LVAL fptr,LVAL code,FIXTYPE lc,LVAL env)
{
    unsigned char *cp;
    char buf[100];
    OTDEF *op;
    LVAL tmp;
    int n=1;

    /* start on a new line */
    xlterpri(fptr);

    /* get a pointer to the bytecodes for this instruction */
    cp = getcodestr(code) + lc;

    /* show the address and opcode */
    if ((tmp = getcname(code)) == v_nil) {
	sprintf(buf,AFMT,code); xlputstr(fptr,buf);
    	sprintf(buf,":%04x %02x ",lc,*cp);
    }
    else
	sprintf(buf,"%s:%04x %02x ",getstring(getpname(tmp)),lc,*cp);
    xlputstr(fptr,buf);

    /* display the operands */
    for (op = otab; op->ot_name; ++op)
	if (*cp == op->ot_code) {
	    switch (op->ot_fmt) {
	    case FMT_NONE:
		sprintf(buf,"         %s",
			op->ot_name);
		xlputstr(fptr,buf);
		break;
	    case FMT_BYTE:
		sprintf(buf,"%02x       %s %02x",
			cp[1],
			op->ot_name,cp[1]);
		xlputstr(fptr,buf);
		n += 1;
		break;
	    case FMT_2BYTE:
		sprintf(buf,"%02x %02x    %s %02x %02x",
			cp[1],cp[2],
			op->ot_name,cp[1],cp[2]);
		xlputstr(fptr,buf);
		n += 2;
		break;
	    case FMT_LOFF:
		sprintf(buf,"%02x       %s %02x ; ",
			cp[1],
			op->ot_name,cp[1]);
		xlputstr(fptr,buf);
		xlprin1(getelement(code,cp[1]),fptr);
		n += 1;
		break;
	    case FMT_WORD:
		sprintf(buf,"%02x %02x    %s %02x%02x",
			cp[1],cp[2],
			op->ot_name,cp[1],cp[2]);
		xlputstr(fptr,buf);
		n += 2;
		break;
	    case FMT_EOFF:
		sprintf(buf,"%02x %02x    %s %02x %02x ; ",
			cp[1],cp[2],
			op->ot_name,cp[1],cp[2]);
		xlputstr(fptr,buf);
		xlprin1(findenvname(env,cp[1],cp[2]),fptr);
		n += 2;
		break;
	    case FMT_FOFF:
		sprintf(buf,"%02x       %s %02x ; ",
			cp[1],
			op->ot_name,cp[1]);
		xlputstr(fptr,buf);
		xlprin1(findenvname(env,0,cp[1]),fptr);
		n += 1;
		break;
	    case FMT_IOFF:
		sprintf(buf,"%02x %02x    %s %02x %02x ; ",
			cp[1],cp[2],
			op->ot_name,cp[1],cp[2]);
		xlputstr(fptr,buf);
		xlprin1(findivarname(env,cp[1],cp[2]),fptr);
		n += 2;
		break;
	    case FMT_OPTARG:
		sprintf(buf,"%02x %02x    %s %02x %02x",
			cp[1],cp[2],
			op->ot_name,cp[1],cp[2]);
		xlputstr(fptr,buf);
		n += 2;
		break;
	    case FMT_KEYARG:
		sprintf(buf,"%02x %02x %02x %s %02x %02x %02x ; ",
			cp[1],cp[2],cp[3],
			op->ot_name,cp[1],cp[2],cp[3]);
		xlputstr(fptr,buf);
		xlprin1(getelement(code,cp[1]),fptr);
		n += 3;
		break;
	    case FMT_ARGS:
		sprintf(buf,"%02x       %s %02x ; ",
			cp[1],
			op->ot_name,cp[1]);
		xlputstr(fptr,buf);
		xlprin1(getvnames(code),fptr);
		n += 1;
		break;
	    case FMT_ARGS2:
		sprintf(buf,"%02x %02x    %s %02x %02x ; ",
			cp[1],cp[2],
			op->ot_name,cp[1],cp[2]);
		xlputstr(fptr,buf);
		xlprin1(getvnames(code),fptr);
		n += 2;
		break;
	    case FMT_FRAME:
		sprintf(buf,"%02x %02x %02x %s %02x %02x %02x ; ",
			cp[1],cp[2],cp[3],
			op->ot_name,cp[1],cp[2],cp[3]);
		xlputstr(fptr,buf);
		xlprin1(getelement(code,cp[3]),fptr);
		n += 3;
		break;
	    case FMT_MVFRAME:
		sprintf(buf,"%02x %02x %02x %s %02x %02x %02x ; ",
			cp[1],cp[2],cp[3],
			op->ot_name,cp[1],cp[2],cp[3]);
		xlputstr(fptr,buf);
		xlprin1(getelement(code,cp[2]),fptr);
		xlputstr(fptr," ");
		xlprin1(getelement(code,cp[3]),fptr);
		n += 3;
		break;
	    }
	    return n;
	}
    
    /* unknown opcode */
    sprintf(buf,"      <UNKNOWN>");
    xlputstr(fptr,buf);
    return n;
}

/* findenvname - find the name of an environment variable */
static LVAL findenvname(LVAL env,int lev,int off)
{
    while (env != v_nil && --lev >= 0)
	    env = getnextframe(env);
	if (env == v_nil)
	    return v_nil;
	env = getenvnames(env);
    while (env != v_nil && --off >= FIRSTENV)
	env = cdr(env);
    return env == v_nil ? v_nil : car(env);
}

/* findivarname - find the name of an instance variable */
static LVAL findivarname(LVAL env,int lev,int off)
{
    while (env != v_nil && --lev >= 0)
	    env = getnextframe(env);
	if (env == v_nil)
	    return v_nil;
	env = getivar(getenvelement(getnextframe(env),FIRSTENV),IV_IVARS);
    while (env != v_nil && --off >= FIRSTIVAR)
	env = cdr(env);
    return env == v_nil ? v_nil : car(env);
}
