/* xldmem.c - xlisp dynamic memory management routines */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

#undef DEBUG_GC

/* virtual machine registers */
LVAL xlfun;	/* current function */
LVAL xlenv;	/* current environment */
LVAL xlval;	/* value of most recent instruction */
LVAL *xlsp;	/* value stack pointer */
LVAL *xlcsp;	/* control stack pointer */

/* important values */
LVAL v_true;
LVAL v_false;
#ifndef v_nil
LVAL v_nil;
#endif

/* stack limits */
LVAL *xlstkbase;	/* base of the stack space - 1 */
LVAL *xlstktop;		/* top of the stack + 1 */

/* variables shared with xlimage.c */
FIXTYPE total;		/* total number of bytes of memory in use */
FIXTYPE gccalls;	/* number of calls to the garbage collector */

/* node space */
PPBLOCK *ppointers;	/* protected pointers */
FIXTYPE nssize = NSSIZE;/* number of nodes per segment */
NSEGMENT *nsegments;	/* list of node segments */
NSEGMENT *nslast;	/* last node segment */
int nscount;		/* number of node segments */
FIXTYPE nnodes;		/* total number of nodes */
FIXTYPE nfree;		/* number of nodes in free list */
LVAL fnodes;		/* list of free nodes */

/* vector (and string) space */
FIXTYPE vssize = VSSIZE;/* number of LVALS per vector segment */
VSEGMENT *vsegments;	/* list of vector segments */
VSEGMENT *vscurrent;	/* current vector segment */
int vscount;		/* number of vector segments */
LVAL *vfree;		/* next free location in vector space */
LVAL *vtop;		/* top of vector space */

/* external variables */
extern LVAL packages;		/* list of packages */
extern LVAL s_unbound;		/* unbound indicator */
extern LVAL default_object;	/* default object */
extern LVAL eof_object;		/* eof object */

/* forward declarations */
static LVAL allocnode(int);
static void findmemory(void);
static LVAL allocvector(int,FIXTYPE);
static int findvmemory(FIXTYPE);
static void markvector(LVAL);
static void markcontinuation(LVAL);
static void compact(void);
static void compact_vector(struct vsegment *);
static void sweep(void);
static void sweep_segment(struct nsegment *);
static void freeforeignptr(LVAL fptr);

/* cons - construct a new cons node */
LVAL cons(LVAL x,LVAL y)
{
    LVAL nnode;

    /* get a free node */
    if ((nnode = fnodes) == v_nil) {
	check(2);
	push(x);
	push(y);
	findmemory();
	if ((nnode = fnodes) == v_nil)
	    xlfmtabort("insufficient node space");
	drop(2);
    }

    /* unlink the node from the free list */
    fnodes = cdr(nnode);
    --nfree;

    /* initialize the new node */
    nnode->n_type = CONS;
    rplaca(nnode,x);
    rplacd(nnode,y);

    /* return the new node */
    return nnode;
}

/* newframe - create a new environment frame */
LVAL newframe(int type,LVAL parent,FIXTYPE size)
{
    LVAL env;
    env = newvector(size);
    setnextframe(env,parent);
    env->n_type = type;
    return env;
}

/* cvstring - convert a string to a string node */
LVAL cvstring(char *str,FIXTYPE len)
{
    LVAL val = xlnewstring(len);
    memcpy(getstring(val),str,(size_t)len);
    return val;
}

/* cvcstring - convert a c string to a string node */
LVAL cvcstring(char *str)
{
    size_t len = strlen(str);
    LVAL val = xlnewstring((FIXTYPE)len);
    memcpy(getstring(val),str,len);
    return val;
}

/* copystring - copy a string */
LVAL copystring(LVAL str)
{
    return cvstring(getstring(str),getslength(str));
}

/* cvfstream - convert a file pointer to a fstream */
LVAL cvfstream(FILE *fp,short flags)
{
    LVAL val = newstream(FSTREAM,flags);
    setsdata(val,fp);
    return val;
}

/* cvustream - convert a character array to a ustream */
LVAL cvustream(char *buf,FIXTYPE len)
{
    /* create an unnamed stream */
    cpush(newustream());
    
    /* copy the characters into the stream */
    while (--len >= 0)
        xlputc(top(),*buf++);

    /* return the new stream */
    return pop();
}

/* cvostream - convert an object to an ostream */
LVAL cvostream(LVAL obj,short flags)
{
    LVAL val = newstream(OSTREAM,flags);
    setsdata(val,obj);
    return val;
}

/* cvsymbol - convert a string to a symbol */
LVAL cvsymbol(LVAL pname)
{
    LVAL val;
    cpush(pname);
    val = allocvector(SYMBOL,SYMSIZE);
    setvalue(val,s_unbound);
    setpname(val,pop());
    setplist(val,v_nil);
    return val;
}

/* cvfixnum - convert an integer to a fixnum node */
LVAL cvfixnum(FIXTYPE n)
{
    LVAL val;
    if (sfixnump(n))
	return cvsfixnum(n);
    val = allocnode(FIXNUM);
    val->n_int = n;
    return val;
}

/* cvflonum - convert a floating point number to a flonum node */
LVAL cvflonum(FLOTYPE n)
{
    LVAL val;
    val = allocnode(FLONUM);
    val->n_flonum = n;
    return val;
}

/* cvchar - convert an integer to a character node */
LVAL cvchar(int ch)
{
    LVAL val;
    val = allocnode(CHARACTER);
    val->n_chcode = ch;
    return val;
}

/* cvclosure - convert code and an environment to a closure */
LVAL cvclosure(LVAL code,LVAL env)
{
    LVAL val;
    val = cons(code,env);
    val->n_type = CLOSURE;
    return val;
}

/* cvpromise - convert a procedure to a promise */
LVAL cvpromise(LVAL code,LVAL env)
{
    LVAL val;
    val = cons(cvclosure(code,env),v_nil);
    val->n_type = PROMISE;
    return val;
}

/* cvsubr - convert a function to a subr/xsubr */
LVAL cvsubr(int type,FUNDEF *def)
{
    LVAL val;
    val = allocnode(type);
    val->n_subr = def->fd_subr;
    val->n_fundef = def;
    return val;
}

/* newvector - allocate and initialize a new vector */
LVAL newvector(FIXTYPE size)
{
    if (size < 0) xlerror("vector length negative",cvfixnum(size));
    return allocvector(VECTOR,size);
}

/* newtable - allocate and initialize a new table */
LVAL newtable(FIXTYPE size)
{
    if (size < 0) xlerror("vector length negative",cvfixnum(size));
    return allocvector(TABLE,size);
}

/* xlnewstring - allocate and initialize a new string */
LVAL xlnewstring(FIXTYPE size)
{
    LVAL val;
    if (size < 0) xlerror("string length negative",cvfixnum(size));
    val = allocvector(STRING,btow_size(size + 1));
    getstring(val)[size] = '\0'; /* in case we need to use it as a c string */
    val->n_vsize = size;
    return val;
}

/* newpackage - create a new package */
LVAL newpackage(char *name)
{
    LVAL pack;
    if (findpackage(name) != v_nil)
	xlerror("duplicate package name",cvcstring(name));
    pack = allocvector(PACKAGE,PAKSIZE);
    cpush(pack);
    setnames(pack,cons(cvcstring(name),v_nil));
    setextern(pack,newvector(HSIZE));
    setintern(pack,newvector(HSIZE));
    setuses(pack,v_nil);
    setusedby(pack,v_nil);
    setnextpackage(pack,packages);
    packages = pack;
    return pop();
}

/* newcode - create a new code object */
LVAL newcode(FIXTYPE nlits)
{
    return allocvector(CODE,nlits);
}

/* newcontinuation - create a new continuation object */
LVAL newcontinuation(FIXTYPE size)
{
    return allocvector(CONTINUATION,size);
}

/* newstream - allocate and initialize a new stream */
LVAL newstream(int type,short flags)
{
    LVAL val = allocnode(type);
    setpflags(val,flags);
    setsavech(val,'\0');
    return val;
}

/* newustream - create a new unnamed stream */
LVAL newustream(void)
{
    LVAL val;
    cpush(allocvector(VECTOR,USTRSIZE));
    val = newstream(USTREAM,PF_INPUT | PF_OUTPUT | PF_BOL);
    setsdata(val,pop());
    setstrhead(val,v_nil);
    setstrtail(val,v_nil);
    setstriptr(val,cvsfixnum(0));
    setstroptr(val,cvsfixnum(0));
    return val;
}

/* cvforeignptr - convert a c pointer to a foreign pointer */
LVAL cvforeignptr(LVAL type,void *p)
{
    LVAL val;
    cpush(type);
    val = allocnode(FOREIGNPTR);
    setfptype(val,pop());
    setfptr(val,p);
    return val;
}

/* allocnode - allocate a new node */
static LVAL allocnode(int type)
{
    LVAL nnode;

    /* get a free node */
    if ((nnode = fnodes) == v_nil) {
	findmemory();
	if ((nnode = fnodes) == v_nil)
	    xlfmtabort("insufficient node space");
    }

    /* unlink the node from the free list */
    fnodes = cdr(nnode);
    --nfree;

    /* initialize the new node */
    nnode->n_type = type;
    rplacd(nnode,v_nil);

    /* return the new node */
    return nnode;
}

/* findmemory - garbage collect, then add more node space if necessary */
static void findmemory(void)
{
    /* first try garbage collecting */
    gc();

    /* expand memory only if less than one segment is free */
    if (nfree < nssize)
	nexpand(nssize);
    
    /* expand vector space if less than one segment is free */
    if (VSFREE(vfree,vtop) < vssize / 2)
        vexpand(vssize);
}

/* nexpand - expand node space */
int nexpand(FIXTYPE size)
{
    NSEGMENT *newseg;
    LVAL p;

    /* allocate the new segment */
    if ((newseg = newnsegment(size)) != NULL) {

	/* add each new node to the free list */
	for (p = &newseg->ns_data[0]; size > 0; ++p, --size) {
	    p->n_type = FREE;
	    p->n_flags = 0;
	    rplacd(p,fnodes);
	    fnodes = p;
	}
    }
    return newseg != NULL;
}

/* allocvector - allocate and initialize a new vector node */
static LVAL allocvector(int type,FIXTYPE size)
{
    register LVAL val,*p;

    /* get a free node */
    if ((val = fnodes) == v_nil) {
	findmemory();
	if ((val = fnodes) == v_nil)
	    xlfmtabort("insufficient node space");
    }

    /* unlink the node from the free list */
    fnodes = cdr(fnodes);
    --nfree;

    /* initialize the vector node */
    val->n_type = type;
    val->n_vsize = size;
    val->n_vdata = NULL;
    cpush(val);

    /* add space for the backpointer and length */
    size += 2;
    
    /* make sure there's enough space */
    if (!VCOMPARE(vfree,size,vtop)
    &&  !checkvmemory(size)
    &&  !findvmemory(size))
	xlfmtabort("insufficient vector space");

    /* allocate the next available block */
    p = vfree;
    vfree += size;
    
    /* store the backpointer and length */
    *p++ = top();
    *p++ = (LVAL)size;
    val->n_vdata = p;

    /* set all the elements to NIL */
    for (size -= 2; --size >= 0; )
	*p++ = v_nil;

    /* return the new vector */
    return pop();
}

/* checkvmemory - check for vector memory (used by 'xlimage.c') */
int checkvmemory(FIXTYPE size)
{
    VSEGMENT *vseg;
    for (vseg = vsegments; vseg != NULL; vseg = vseg->vs_next)
	if (vseg != vscurrent && VCOMPARE(vseg->vs_free,size,vseg->vs_top)) {
	    if (vscurrent != NULL)
		vscurrent->vs_free = vfree;
	    vfree = vseg->vs_free;
	    vtop = vseg->vs_top;
	    vscurrent = vseg;
	    return TRUE;
	}	
    return FALSE;
}
    
/* findvmemory - find vector memory */
static int findvmemory(FIXTYPE size)
{
    /* try garbage collecting */
    gc();

    /* check to see if we found enough memory */
    if (VCOMPARE(vfree,size,vtop) || checkvmemory(size))
	return TRUE;

    /* expand vector space */
    return makevmemory(size);
}

/* makevmemory - make vector memory (used by 'xlimage.c') */
int makevmemory(FIXTYPE size)
{
    return vexpand(size < vssize ? vssize : size);
}

/* vexpand - expand vector space */
int vexpand(FIXTYPE size)
{
    VSEGMENT *vseg;

    /* allocate the new segment */
    if ((vseg = newvsegment(size)) != NULL) {
	if (vscurrent != NULL)
	    vscurrent->vs_free = vfree;
	vfree = vseg->vs_free;
	vtop = vseg->vs_top;
	vscurrent = vseg;
    }
    return vseg != NULL;
}

/* newnsegment - create a new node segment */
NSEGMENT *newnsegment(FIXTYPE n)
{
    NSEGMENT *newseg;

    /* allocate the new segment */
    if ((newseg = (NSEGMENT *)osalloc(nsegsize(n))) == NULL)
	return NULL;

    /* initialize the new segment */
    newseg->ns_size = n;
    newseg->ns_next = NULL;
    if (nsegments == NULL)
	nsegments = newseg;
    else
	nslast->ns_next = newseg;
    nslast = newseg;

    /* update the statistics */
    total += nsegsize(n);
    nnodes += n;
    nfree += n;
    ++nscount;

    /* return the new segment */
    return newseg;
}
 
/* newvsegment - create a new vector segment */
VSEGMENT *newvsegment(FIXTYPE n)
{
    VSEGMENT *newseg;

    /* allocate the new segment */
    if ((newseg = (VSEGMENT *)osalloc(vsegsize(n))) == NULL)
	return NULL;

    /* initialize the new segment */
    newseg->vs_free = newseg->vs_data;
    newseg->vs_top = newseg->vs_free + n;
    newseg->vs_next = vsegments;
    vsegments = newseg;

    /* update the statistics */
    total += vsegsize(n);
    ++vscount;

    /* return the new segment */
    return newseg;
}
 
/* gc - garbage collect */
void gc(void)
{
    register LVAL *p;
    PPBLOCK *ppb;
    
#ifdef DEBUG_GC
/*if ((gccalls % 10) == 0) */
{ char buf[20];
  sprintf(buf,"\n[GC %ld",gccalls);
  ostputs(buf);
}
#endif

    /* reset the mark flags on the control stack */
    for (p = xlcsp; p > xlstkbase; )
 	p = cdunmark(p);
 
    /* mark the current package and environment */
    mark(packages);
    mark(xlfun);
    mark(xlenv);
    mark(xlval);
    mark(default_object);
    mark(eof_object);
    mark(v_true);
    mark(v_false);

    /* mark the value stack */
    for (p = xlsp; p < xlstktop; )
	mark(*p++);

    /* mark the control stack */
    for (p = xlcsp; p > xlstkbase; )
	p = cdmark(p);

    /* mark protected pointers */
    for (ppb = ppointers; ppb != NULL; ppb = ppb->next) {
        LVAL **pp = ppb->pointers;
        int count = ppb->count;
        for (; --count >= 0; ++pp)
            if (*pp) mark(**pp);
    }
    
    /* compact vector space */
    gc_protect(compact);

    /* sweep memory collecting all unmarked nodes */
    sweep();

#ifdef DEBUG_GC
/*if ((gccalls % 10) == 0)*/
  ostputs(" - done]");
#endif

    /* count the gc call */
    ++gccalls;
}

/* mark - mark all accessible nodes */
void mark(LVAL ptr)
{
    register LVAL this,prev,tmp;

    /* check for a non-pointer */
    if (!ispointer(ptr))
	return;

    /* initialize */
    prev = v_nil;
    this = ptr;

    /* mark this node */
    for (;;) {

	/* descend as far as we can */
	while (!(this->n_flags & MARK))

	    /* mark this node and trace its children */
	    switch (this->n_type) {
	    case CONS:		/* mark cons-like nodes */
	    case CLOSURE:
	    case PROMISE:
		this->n_flags |= MARK;
		tmp = car(this);
		if (ispointer(tmp)) {
		    this->n_flags |= LEFT;
		    rplaca(this,prev);
		    prev = this;
		    this = tmp;
		}
		else {
		    tmp = cdr(this);
		    if (ispointer(tmp)) {
			rplacd(this,prev);
			prev = this;
			this = tmp;
		    }
		}
		break;
	    case SYMBOL:	/* mark vector-like nodes */
	    case VECTOR:
	    case CODE:
	    case PACKAGE:
	    case ENV:
	    case SENV:
	    case MSENV:
	    case MENV:
	    case SMENV:
	    case OBJECT:
            case TABLE:
		this->n_flags |= MARK;
		markvector(this);
		break;
	    case USTREAM:
	    case OSTREAM:
		this->n_flags |= MARK;
	    	mark((LVAL)getsdata(this));
	    	break;
	    case CONTINUATION:
		this->n_flags |= MARK;
		markcontinuation(this);
		break;
	    case FIXNUM:	/* mark objects that don't contain pointers */
	    case FLONUM:
	    case STRING:
	    case FSTREAM:
	    case SUBR:
	    case XSUBR:
	    case CHARACTER:
	    case FOREIGNPTR:
		this->n_flags |= MARK;
		break;
	    default:		/* bad object type */
		xlfatal("%lx: bad object type %d",this,this->n_type);
		break;
	    }

	/* backup to a point where we can continue descending */
	for (;;)

	    /* make sure there is a previous node */
	    if (prev != v_nil) {
		if (prev->n_flags & LEFT) {	/* came from left side */
		    prev->n_flags &= ~LEFT;
		    tmp = car(prev);
		    rplaca(prev,this);
		    this = cdr(prev);
		    if (ispointer(this)) {
			rplacd(prev,tmp);			
			break;
		    }
		}
		else {				/* came from right side */
		    tmp = cdr(prev);
		    rplacd(prev,this);
		}
		this = prev;			/* step back up the branch */
		prev = tmp;
	    }

	    /* no previous node, must be done */
	    else
		return;
    }
}

/* markvector - mark a vector-like node */
static void markvector(LVAL vect)
{
    register FIXTYPE n;
    register LVAL *p;
    if ((p = vect->n_vdata) != NULL)
	for (n = getsize(vect); --n >= 0; )
	    mark(*p++);
}

/* markcontinuation - mark a continuation node */
static void markcontinuation(LVAL cont)
{
    register LVAL *p,*sp;
    FIXTYPE vsize,csize;

    /* make sure there is data */
    if (cont->n_vdata != NULL) {

	/* get the stack sizes */
	vsize = getfixnum(getelement(cont,0));
	csize = getsize(cont) - vsize - 1;

	/* mark the value stack */
	for (sp = &cont->n_vdata[1]; --vsize >= 0; )
	    mark(*sp++);
	    
	/* mark the control stack */
	for (p = &sp[csize]; p > sp; )
	    p = cdmark(p);
    }
}

/* compact - compact vector space */
static void compact(void)
{
    VSEGMENT *vseg;

    /* store the current segment information */
    if (vscurrent != NULL)
	vscurrent->vs_free = vfree;

    /* compact each vector segment */
    for (vseg = vsegments; vseg != NULL; vseg = vseg->vs_next)
	compact_vector(vseg);

    /* make the first vector segment current */
    if ((vscurrent = vsegments) != NULL) {
	vfree = vscurrent->vs_free;
	vtop = vscurrent->vs_top;
    }
}

/* compact_vector - compact a vector segment */
static void compact_vector(VSEGMENT *vseg)
{
    register LVAL *vdata,*vnext,*vfree,vector;
    register FIXTYPE vsize;

    vdata = vnext = vseg->vs_data;
    vfree = vseg->vs_free;
    while (vdata < vfree) {
	vector = *vdata;
	vsize = (FIXTYPE)vdata[1];
	if (vector->n_flags & MARK) {
	    if (vdata == vnext) {
		vdata += vsize;
		vnext += vsize;
	    }
	    else {
		vector->n_vdata = vnext + 2;
		while (--vsize >= 0)
		    *vnext++ = *vdata++;
	    }
	}
	else
	    vdata += vsize;
    }
    vseg->vs_free = vnext;
}

/* sweep - sweep all unmarked nodes and add them to the free list */
static void sweep(void)
{
    NSEGMENT *nseg;

    /* empty the free list */
    fnodes = v_nil;
    nfree = 0L;

    /* sweep each node segment */
    for (nseg = nsegments; nseg != NULL; nseg = nseg->ns_next)
	sweep_segment(nseg);
}

/* sweep_segment - sweep a node segment */
static void sweep_segment(NSEGMENT *nseg)
{
    register FIXTYPE n;
    register LVAL p;

    /* add all unmarked nodes */
    for (p = &nseg->ns_data[0], n = nseg->ns_size; --n >= 0L; ++p)
	if (!(p->n_flags & MARK)) {
	    switch (p->n_type) {
	    case FSTREAM:
		if (getfile(p))
		    osclose(getfile(p));
		break;
	    case FOREIGNPTR:
	    	freeforeignptr(p);
	        break;
	    }
	    rplaca(p,(LVAL)(OFFTYPE)p->n_type); /* for debugging */
	    p->n_type = FREE;
	    rplacd(p,fnodes);
	    fnodes = p;
	    ++nfree;
	}
	else
	    p->n_flags &= ~MARK;
}

/* freeforeignptr - free a foreign pointer */
static void freeforeignptr(LVAL fptr)
{
    extern LVAL s_freeptr;
    void *ptr = getfptr(fptr);
    if (ptr != NULL) {
	LVAL val = getfptype(fptr);
     	if (symbolp(val)) {
            val = xlgetprop(val,s_freeptr);
            if (fixp(val))
                (*(void (*)(void *))getfixnum(val))(ptr);
	}
    }
}

/* protectptr - protect a pointer */
int protectptr(LVAL *p)
{
    PPBLOCK *ppb = ppointers;
    if (ppb == NULL || ppb->count >= PPBSIZE) {
	if ((ppb = (PPBLOCK *)osalloc(sizeof(PPBLOCK))) == NULL)
	    return FALSE;
	ppb->next = ppointers;
	ppointers = ppb;
	ppb->count = 0;
    }
    ppb->pointers[ppb->count++] = p;
    *p = v_nil; /* initialize the pointer */
    return TRUE;
}

/* xlminit - initialize the dynamic memory module */
void xlminit(FIXTYPE ssize)
{
    FIXTYPE n;

    /* initialize some important variables */
    ppointers = NULL;
#ifndef v_nil
    v_nil = (LVAL)0;
#endif
    v_true = v_false = v_nil;

    /* initialize structures that are marked by the collector */
    packages = s_unbound = default_object = eof_object = v_nil;
    xlfun = xlenv = xlval = v_nil;

    /* initialize our internal variables */
    gccalls = 0;
    total = 0L;

    /* initialize node space */
    nsegments = nslast = NULL;
    nscount = 0;
    nnodes = nfree = 0L;
    fnodes = v_nil;

    /* initialize vector space */
    vsegments = vscurrent = NULL;
    vscount = 0;
    vfree = vtop = NULL;
    
    /* allocate the value and control stack space */
    n = ssize * sizeof(LVAL);
    if ((xlstkbase = (LVAL *)osalloc(n)) == NULL)
	xlfatal("insufficient memory");
    total += n;

    /* initialize the stacks */
    xlstktop = xlstkbase + ssize;
    xlresetstack();
}

/* xlresetstack - reset the stacks */
void xlresetstack(void)
{
    xlsp = xlstktop;
    xlcsp = xlstkbase;
}

