/* xlfasl.c - fast load file handler */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* prototypes */
static int faslwritecode(LVAL fptr,LVAL code);
static LVAL faslreadcode(LVAL fptr);
static LVAL faslreadexpr(LVAL fptr);
static LVAL faslread(LVAL fptr);
static int gethexbyte(LVAL fptr);
static int faslskip(LVAL fptr);
static int faslgetc(LVAL fptr);
static void fasleof(void);

/* xfaslwriteprocedure - write a procedure to a fasl file */
LVAL xfaslwriteprocedure(void)
{
    LVAL fun,fptr;

    /* parse the argument list */
    fun = xlgaclosure();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* write out the code portion of the closure */
    return faslwritecode(fptr,getcode(fun)) ? v_true : v_false;
}

/* faslwritecode - write a code object to a fasl file */
static int faslwritecode(LVAL fptr,LVAL code)
{
    LVAL bytecodes = getbcode(code);
    long nbytes = getslength(bytecodes);
    long nlits = getsize(code);
    char buf[100];
    long i;
    
    /* write the function name */
    xlputstr(fptr,"C ");
    xlprin1(getcname(code),fptr);
    xlputstr(fptr," ");

    /* write the required argument names */
    xlprin1(getvnames(code),fptr);
    xlterpri(fptr);

    /* write the code object size and number of bytecodes */
    sprintf(buf,"%ld %ld\n",nlits - FIRSTLIT,nbytes);
    xlputstr(fptr,buf);

    /* write the bytecodes */
    for (i = 0; i < nbytes; ) {
	sprintf(buf,"%02x ",getstring(bytecodes)[i]);
	xlputstr(fptr,buf);
	if ((++i % 16) == 0)
	    xlterpri(fptr);
    }
    if ((i % 16) != 0)
	xlterpri(fptr);

    /* write the literals */
    for (i = FIRSTLIT; i < nlits; ++i) {
	LVAL lit = getelement(code,i);
	if (codep(lit))
	    faslwritecode(fptr,lit);
	else {
	    xlputstr(fptr,"D ");
	    xlprin1(lit,fptr);
            xlterpri(fptr);
	}
    }

    /* return successfully */
    return TRUE;
}

/* xfaslreadprocedure - read a procedure from a fasl file */
LVAL xfaslreadprocedure(void)
{
    extern LVAL eof_object;
    LVAL fptr;
    int ch;
    
    /* parse the argument list */
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();

    /* check for eof */
    if ((ch = faslskip(fptr)) == EOF)
	return eof_object;

    /* make sure the next expression is a code object */
    else if (ch != 'C')
	xlfmterror("bad fasl file ~A",cvchar(ch));
    xlungetc(fptr,'C');

    /* write out the code portion of the closure */
    return cvclosure(faslreadexpr(fptr),v_nil);
}

/* faslreadcode - read a code object from a fasl file */
static LVAL faslreadcode(LVAL fptr)
{
    long nlits,nbytes,i;
    LVAL val;
    
    /* read the function name and argument names */
    check(2);
    push(faslread(fptr));
    push(faslread(fptr));

    /* get the code object size and number of bytecodes */
    val = faslread(fptr); nlits = getfixnum(val) + FIRSTLIT;
    val = faslread(fptr); nbytes = getfixnum(val);
    
    /* allocate the code object */
    val = newcode(nlits);
    setvnames(val,pop());
    setcname(val,pop());
    push(val);

    /* allocate the bytecode array */
    val = xlnewstring(nbytes);
    setbcode(top(),val);
    
    /* read the bytecodes */
    for (i = 0; i < nbytes; ++i)
        getstring(val)[i] = gethexbyte(fptr);

    /* read the literals */
    for (i = FIRSTLIT; i < nlits; ++i)
	setelement(top(),i,faslreadexpr(fptr));

    /* return the code object */
    return pop();
}

/* faslreadexpr - read the next expression from a fasl file */
static LVAL faslreadexpr(LVAL fptr)
{
    int ch;

    /* dispatch on the expression type */
    switch (ch = faslskip(fptr)) {
    case 'D':
	return faslread(fptr);
    case 'C':
	return faslreadcode(fptr);
	break;
    default:
	xlfmterror("unknown fasl expression type ~A",cvchar(ch));
        break;
    }
}

/* faslread - read an expression from a fasl file */
LVAL faslread(LVAL fptr)
{
    LVAL val;
    if (!xlread(fptr,&val))
	fasleof();
    return val;
}

/* gethexbyte - get a hex byte from a fasl file */
static int gethexbyte(LVAL fptr)
{
    int ch1,ch2;

    /* read the first hex digit */
    if ((ch1 = faslskip(fptr)) == EOF)
        fasleof();
    ch1 = toupper(ch1);
    if ((ch1 -= '0') > 9)
	ch1 += '0' - 'A' + 10;

    /* read the second hex digit */
    ch2 = faslgetc(fptr);
    ch2 = toupper(ch2);
    if ((ch2 -= '0') > 9)
	ch2 += '0' - 'A' + 10;

    /* return the byte */
    return (ch1 << 4) | ch2;
}
    
/* faslskip - read the next non-space character from a fasl file */
static int faslskip(LVAL fptr)
{
    int ch;
    while ((ch = xlgetc(fptr)) != EOF && isspace(ch))
        ;
    return ch;
}

/* faslgetc - read the next character from a fasl file */
static int faslgetc(LVAL fptr)
{
    int ch;
    if ((ch = xlgetc(fptr)) == EOF)
	fasleof();
    return ch;
}

/* fasleof - unexpected eof in fasl file */
static void fasleof(void)
{
    xlfmterror("unexpected eof in fasl file");
}
