/* xlfun1.c - xlisp built-in functions - part 1 */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* gensym variables */
static char gsprefix[STRMAX+1] = { 'G',0 };	/* gensym prefix string */
static FIXTYPE gsnumber = 1;			/* gensym number */

/* external variables */
extern LVAL xlenv,xlval,default_object;
extern LVAL s_unbound,s_package,s_eql,k_uses,k_test,k_testnot,k_key;
extern LVAL packages,xlisppackage,schemepackage,systempackage;

/* forward declarations */
static LVAL cxr(char *adstr);
static LVAL member(int (*fcn)(LVAL,LVAL));
static LVAL assoc(int (*fcn)(LVAL,LVAL));
static LVAL nth(int carflag);
static LVAL copytree(LVAL tree);
static LVAL getpackagenamearg(void);
static LVAL nametostring(LVAL arg);
static LVAL getpackagearg(void);
static LVAL nametopackage(LVAL arg);
static LVAL vref(LVAL vector);
static LVAL vset(LVAL vector);
static LVAL makearray1(int argc,LVAL *argv);
static int stringequal(LVAL v1,LVAL v2);
static int vectorequal(LVAL v1,LVAL v2);

/* xcons - built-in function 'cons' */
LVAL xcons(void)
{
    LVAL carval,cdrval;
    carval = xlgetarg();
    cdrval = xlgetarg();
    xllastarg();
    return cons(carval,cdrval);
}

/* xacons - built-in function 'acons' */
LVAL xacons(void)
{
    LVAL key,datum;
    key = xlgetarg();
    datum = xlgetarg();
    xlval = xlgalist();
    xllastarg();
    return cons(cons(key,datum),xlval);
}

/* xcar - built-in function 'car' */
LVAL xcar(void)
{
    LVAL list;
    list = xlgalist();
    xllastarg();
    return list ? car(list) : v_nil;
}

/* xicar - built-in function '%car' */
LVAL xicar(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return car(arg);
}

/* xcdr - built-in function 'cdr' */
LVAL xcdr(void)
{
    LVAL arg;
    arg = xlgalist();
    xllastarg();
    return arg ? cdr(arg) : v_nil;
}

/* xicdr - built-in function '%cdr' */
LVAL xicdr(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return cdr(arg);
}

/* cxxr functions */
LVAL xcaar(void) { return cxr("aa"); }
LVAL xcadr(void) { return cxr("da"); }
LVAL xcdar(void) { return cxr("ad"); }
LVAL xcddr(void) { return cxr("dd"); }

/* cxxxr functions */
LVAL xcaaar(void) { return cxr("aaa"); }
LVAL xcaadr(void) { return cxr("daa"); }
LVAL xcadar(void) { return cxr("ada"); }
LVAL xcaddr(void) { return cxr("dda"); }
LVAL xcdaar(void) { return cxr("aad"); }
LVAL xcdadr(void) { return cxr("dad"); }
LVAL xcddar(void) { return cxr("add"); }
LVAL xcdddr(void) { return cxr("ddd"); }

/* cxxxxr functions */
LVAL xcaaaar(void) { return cxr("aaaa"); }
LVAL xcaaadr(void) { return cxr("daaa"); }
LVAL xcaadar(void) { return cxr("adaa"); }
LVAL xcaaddr(void) { return cxr("ddaa"); }
LVAL xcadaar(void) { return cxr("aada"); }
LVAL xcadadr(void) { return cxr("dada"); }
LVAL xcaddar(void) { return cxr("adda"); }
LVAL xcadddr(void) { return cxr("ddda"); }
LVAL xcdaaar(void) { return cxr("aaad"); }
LVAL xcdaadr(void) { return cxr("daad"); }
LVAL xcdadar(void) { return cxr("adad"); }
LVAL xcdaddr(void) { return cxr("ddad"); }
LVAL xcddaar(void) { return cxr("aadd"); }
LVAL xcddadr(void) { return cxr("dadd"); }
LVAL xcdddar(void) { return cxr("addd"); }
LVAL xcddddr(void) { return cxr("dddd"); }

/* cxr - common car/cdr routine */
static LVAL cxr(char *adstr)
{
    LVAL list;

    /* get the list */
    list = xlgalist();
    xllastarg();

    /* perform the car/cdr operations */
    while (*adstr && consp(list))
	list = (*adstr++ == 'a' ? car(list) : cdr(list));

    /* make sure the operation succeeded */
    if (*adstr && list)
	xlbadtype(list);

    /* return the result */
    return list;
}

/* xsetcar - built-in function 'set-car!' */
LVAL xsetcar(void)
{
    LVAL arg,newcar;

    /* get the cons and the new car */
    arg = xlgacons();
    newcar = xlgetarg();
    xllastarg();

    /* replace the car */
    rplaca(arg,newcar);
    return arg;
}

/* xisetcar - built-in function '%set-car!' */
LVAL xisetcar(void)
{
    LVAL arg,newcar;

    /* get the cons and the new car */
    arg = xlgetarg();
    newcar = xlgetarg();
    xllastarg();

    /* replace the car */
    rplaca(arg,newcar);
    return arg;
}

/* xsetcdr - built-in function 'set-cdr!' */
LVAL xsetcdr(void)
{
    LVAL arg,newcdr;

    /* get the cons and the new cdr */
    arg = xlgacons();
    newcdr = xlgetarg();
    xllastarg();

    /* replace the cdr */
    rplacd(arg,newcdr);
    return arg;
}

/* xisetcdr - built-in function '%set-cdr!' */
LVAL xisetcdr(void)
{
    LVAL arg,newcdr;

    /* get the cons and the new cdr */
    arg = xlgetarg();
    newcdr = xlgetarg();
    xllastarg();

    /* replace the cdr */
    rplacd(arg,newcdr);
    return arg;
}

/* xnappend - destructively append lists */
LVAL xnappend(void)
{
    LVAL next,last;

    /* initialize */
    xlval = v_nil;
    
    /* concatenate each argument */
    if (moreargs()) {
	while (xlargc > 1) {

	    /* ignore everything except lists */
	    if ((next = nextarg()) != v_nil && consp(next)) {

		/* concatenate this list to the result list */
		if (xlval) rplacd(last,next);
		else xlval = next;

		/* find the end of the list */
		while (consp(cdr(next)))
		    next = cdr(next);
		last = next;
	    }
	}

	/* handle the last argument */
	if (xlval) rplacd(last,nextarg());
	else xlval = nextarg();
    }

    /* return the list */
    return xlval;
}

/* xlist - built-in function 'list' */
LVAL xlist(void)
{
    LVAL last,next;

    /* initialize the list */
    xlval = v_nil;

    /* add each argument to the list */
    if (moreargs()) {
        xlval = last = cons(nextarg(),v_nil);
        while (moreargs()) {
	    next = cons(nextarg(),v_nil);
	    rplacd(last,next);
	    last = next;
	}
    }

    /* return the list */
    return xlval;
}

/* xliststar - built-in function 'list*' */
LVAL xliststar(void)
{
    LVAL last,next;

    /* initialize the list */
    xlval = v_nil;

    /* add each argument to the list */
    if (moreargs()) {
        for (;;) {
	    next = nextarg();
	    if (moreargs()) {
		next = cons(next,v_nil);
		if (xlval) rplacd(last,next);
		else xlval = next;
		last = next;
	    }
	    else {
		if (xlval) rplacd(last,next);
		else xlval = next;
		break;
	    }
	}
    }

    /* return the list */
    return xlval;
}

/* xpairlis - built-in function 'pairlis' */
LVAL xpairlis(void)
{
    LVAL keys,data;
    keys = xlgalist();
    data = xlgalist();
    xlval = moreargs() ? xlgalist() : v_nil;
    check(2);
    push(keys);
    push(data);
    while (consp(keys) && consp(data)) {
	xlval = cons(cons(car(keys),car(data)),xlval);
	keys = cdr(keys);
	data = cdr(data);
    }
    drop(2);
    return xlval;
}

/* xcopylist - built-in function 'copy-list' */
LVAL xcopylist(void)
{
    LVAL last;
    xlval = xlgalist();
    xllastarg();
    cpush(v_nil);
    if (xlval) {
	last = cons(car(xlval),v_nil); settop(last);
	for (xlval = cdr(xlval); consp(xlval); xlval = cdr(xlval)) {
	    rplacd(last,cons(car(xlval),v_nil));
	    last = cdr(last);
	}
    }
    return pop();
}

/* copytree - copytree helper function */
static LVAL copytree(LVAL tree)
{
    if (consp(tree)) {
	cpush(copytree(car(tree)));
	tree = copytree(cdr(tree));
	tree = cons(pop(),tree);
    }
    return tree;
}

/* xcopytree - built-in function 'copy-tree' */
LVAL xcopytree(void)
{
    xlval = xlgalist();
    xllastarg();
    return copytree(xlval);
}

/* xcopyalist - built-in function 'copy-alist' */
LVAL xcopyalist(void)
{
    LVAL last,entry;
    xlval = xlgalist();
    xllastarg();
    cpush(v_nil);
    if (xlval) {
	entry = car(xlval);
	if (consp(entry)) entry = cons(car(entry),cdr(entry));
	last = cons(entry,v_nil); settop(last);
	for (xlval = cdr(xlval); consp(xlval); xlval = cdr(xlval)) {
	    entry = car(xlval);
	    if (consp(entry)) entry = cons(car(entry),cdr(entry));
	    rplacd(last,cons(entry,v_nil));
	    last = cdr(last);
	}
    }
    return pop();
}

/* xappend - built-in function 'append' */
LVAL xappend(void)
{
    LVAL next,this,last;

    /* append each argument */
    for (xlval = last = v_nil; xlargc > 1; )

	/* append each element of this list to the result list */
	for (next = xlgalist(); consp(next); next = pop()) {
	    cpush(cdr(next));
	    this = cons(car(next),v_nil);
	    if (last) rplacd(last,this);
	    else xlval = this;
	    last = this;
	}

    /* tack on the last argument */
    if (moreargs()) {
	if (last) rplacd(last,xlgetarg());
	else xlval = xlgetarg();
    }

    /* return the list */
    return xlval;
}

/* xreverse - built-in function 'reverse' */
LVAL xreverse(void)
{
    LVAL val;
    
    /* get the list to reverse */
    xlval = xlgalist();
    xllastarg();

    /* append each element of this list to the result list */
    for (val = v_nil; consp(xlval); xlval = cdr(xlval))
	val = cons(car(xlval),val);

    /* return the list */
    return val;
}

/* xlength - built-in function 'length' */
LVAL xlength(void)
{
    FIXTYPE n;
    LVAL arg;

    /* get the argument */
    arg = xlgalist();
    xllastarg();

    /* find the length */
    for (n = 0; consp(arg); ++n)
	arg = cdr(arg);

    /* return the length */
    return cvfixnum(n);
}

/* xxmember - built-in function 'member' */
LVAL xxmember(void)
{
    return member(equal);
}

/* xxmemv - built-in function 'memv' */
LVAL xxmemv(void)
{
    return member(eqv);
}

/* xxmemq - built-in function 'memq' */
LVAL xxmemq(void)
{
    return member(eq);
}

/* member - common routine for member/memv/memq */
static LVAL member(int (*fcn)(LVAL,LVAL))
{
    LVAL x,list,val;

    /* get the expression to look for and the list */
    x = xlgetarg();
    list = xlgalist();
    xllastarg();

    /* look for the expression */
    for (val = v_nil; consp(list); list = cdr(list))
	if ((*fcn)(x,car(list))) {
	    val = list;
	    break;
	}

    /* return the result */
    return val;
}

/* xxassoc - built-in function 'assoc' */
LVAL xxassoc(void)
{
    return assoc(equal);
}

/* xxassv - built-in function 'assv' */
LVAL xxassv(void)
{
    return assoc(eqv);
}

/* xxassq - built-in function 'assq' */
LVAL xxassq(void)
{
    return assoc(eq);
}

/* assoc - common routine for assoc/assv/assq */
static LVAL assoc(int (*fcn)(LVAL,LVAL))
{
    LVAL x,alist,pair,val;

    /* get the expression to look for and the association list */
    x = xlgetarg();
    alist = xlgalist();
    xllastarg();

    /* look for the expression */
    for (val = v_nil; consp(alist); alist = cdr(alist))
	if ((pair = car(alist)) != v_nil && consp(pair))
	    if ((*fcn)(x,car(pair))) {
		val = pair;
		break;
	    }

    /* return the result */
    return val;
}

/* xlast - return the last cons of a list */
LVAL xlast(void)
{
    LVAL list;

    /* get the list */
    list = xlgalist();
    xllastarg();

    /* find the last cons */
    if (consp(list))
	while (consp(cdr(list)))
	    list = cdr(list);

    /* return the last element */
    return list;
}

/* xlistref - built-in function 'list-ref' */
LVAL xlistref(void)
{
    LVAL list,num;
    FIXTYPE n;

    /* get the list and n */
    list = xlgalist();
    num = xlgafixnum();
    xllastarg();

    /* make sure the number isn't negative */
    if ((n = getfixnum(num)) < 0)
	xlfmterror("bad argument");

    /* find the nth element */
    while (consp(list) && --n >= 0)
	list = cdr(list);

    /* return the list beginning at the nth element */
    return consp(list) ? car(list) : v_nil;
}

/* xlisttail - return the nth cdr of a list */
LVAL xlisttail(void)
{
    LVAL list,num;
    FIXTYPE n;

    /* get n and the list */
    list = xlgalist();
    num = xlgafixnum();
    xllastarg();

    /* make sure the number isn't negative */
    if ((n = getfixnum(num)) < 0)
	xlfmterror("bad argument");

    /* find the nth element */
    while (consp(list) && --n >= 0)
	list = cdr(list);

    /* return the list beginning at the nth element */
    return list;
}

/* xmkpackage - make a package */
LVAL xmkpackage(void)
{
    LVAL uses;
    xlval = getpackagenamearg();
    xlval = newpackage(getstring(xlval));
    if (xlgetkeyarg(k_uses,v_nil,&uses)) {
	cpush(uses);
	for (; consp(uses); uses = cdr(uses))
	    usepackage(nametopackage(car(uses)),xlval);
	drop(1);
    }
    else {
	usepackage(xlisppackage,xlval);
	usepackage(schemepackage,xlval);
	usepackage(systempackage,xlval);
    }
    xlpopargs();
    return xlval;
}

/* xinpackage - switch to a package */
LVAL xinpackage(void)
{
    xlval = getpackagearg();
    xllastarg();
    setvalue(s_package,xlval);
    return xlval;
}

/* xfindpackage - find a package by name */
LVAL xfindpackage(void)
{
    xlval = getpackagenamearg();
    xllastarg();
    return findpackage(getstring(xlval));
}

/* xlistallpackages - return a list of all packages */
LVAL xlistallpackages(void)
{
    LVAL pack;
    xllastarg();
    for (xlval = v_nil, pack = packages; packagep(pack); pack = getnextpackage(pack))
	xlval = cons(pack,xlval);
    return xlval;
}

/* xpackagename - get the name of a package */
LVAL xpackagename(void)
{
    xlval = getpackagearg();
    xllastarg();
    return car(getnames(xlval));
}

/* xpkgnicknames - get the nicknames of a package */
LVAL xpkgnicknames(void)
{
    xlval = getpackagearg();
    xllastarg();
    return cdr(getnames(xlval));
}

/* xusepackage - use a package */
LVAL xusepackage(void)
{
    LVAL dst;
    xlval = xlgetarg();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    if (listp(xlval))
	for (; consp(xlval); xlval = cdr(xlval))
	    usepackage(nametopackage(car(xlval)),dst);
    else
	usepackage(nametopackage(xlval),dst);
    return v_true;
}

/* xunusepackage - unuse a package */
LVAL xunusepackage(void)
{
    LVAL dst;
    xlval = xlgetarg();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    if (listp(xlval))
	for (; consp(xlval); xlval = cdr(xlval))
	    unusepackage(nametopackage(car(xlval)),dst);
    else
	unusepackage(nametopackage(xlval),dst);
    return v_true;
}

/* xpkguselist - get a package use list */
LVAL xpkguselist(void)
{
    xlval = getpackagearg();
    xllastarg();
    return getuses(xlval);
}

/* xpkgusedbylist - get a package used-by list */
LVAL xpkgusedbylist(void)
{
    xlval = getpackagearg();
    xllastarg();
    return getusedby(xlval);
}

/* xexport - export symbols from a package */
LVAL xexport(void)
{
    LVAL dst;
    xlval = xlgetarg();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    if (listp(xlval))
	for (; consp(xlval); xlval = cdr(xlval))
	    if (symbolp(car(xlval)))
		export(car(xlval),dst);
	    else
		xlerror("expecting a symbol",car(xlval));
    else if (symbolp(xlval))
	export(xlval,dst);
    else
	xlerror("expecting a symbol or list of symbols",xlval);
    return v_true;
}

/* xunexport - unexport symbols from a package */
LVAL xunexport(void)
{
    LVAL dst;
    xlval = xlgetarg();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    if (listp(xlval))
	for (; consp(xlval); xlval = cdr(xlval))
	    if (symbolp(car(xlval)))
		unexport(car(xlval),dst);
	    else
		xlerror("expecting a symbol",car(xlval));
    else if (symbolp(xlval))
	unexport(xlval,dst);
    else
	xlerror("expecting a symbol or list of symbols",xlval);
    return v_true;
}

/* ximport - import symbols into a package */
LVAL ximport(void)
{
    LVAL dst;
    xlval = xlgetarg();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    if (listp(xlval))
	for (; consp(xlval); xlval = cdr(xlval))
	    if (symbolp(car(xlval)))
		import(car(xlval),dst);
	    else
		xlerror("expecting a symbol",car(xlval));
    else if (symbolp(xlval))
	import(xlval,dst);
    else
	xlerror("expecting a symbol or list of symbols",xlval);
    return v_true;
}

/* xmksymbol - make an uninterned symbol */
LVAL xmksymbol(void)
{
    xlval = xlgastring();
    xllastarg();
    return cvsymbol(copystring(xlval));
}

/* xintern - intern a symbol in a package */
void xintern(void)
{
    LVAL dst,key;
    xlval = xlgastring();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    xlval = intern(xlval,dst,&key);
    xlargc = 2;
    cpush(key);
    drop(1);
    cdrestore();
}

/* xfindsymbol - find a symbol in a package */
void xfindsymbol(void)
{
    LVAL dst,key;
    xlval = xlgastring();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    xlval = findsymbol(getstring(xlval),dst,&key);
    xlargc = 2;
    cpush(key);
    drop(1);
    cdrestore();
}

/* xunintern - unintern a symbol from a package */
LVAL xunintern(void)
{
    LVAL dst;
    xlval = xlgasymbol();
    dst = (moreargs() ? getpackagearg() : getvalue(s_package));
    xllastarg();
    unintern(xlval,dst);
    return v_true;
}

/* getpackagenamearg - get a package name argument */
static LVAL getpackagenamearg(void)
{
    return nametostring(xlgetarg());
}

/* nametostring - convert package name to a string */
static LVAL nametostring(LVAL arg)
{
    if (symbolp(arg))
	return getpname(arg);
    else if (stringp(arg))
	return arg;
    xlerror("expecting a package name",arg);
}

/* getpackagearg - get a package argument */
static LVAL getpackagearg(void)
{
    return nametopackage(xlgetarg());
}

/* nametopackage - convert a name to a package */
static LVAL nametopackage(LVAL arg)
{
    LVAL pack;
    if (packagep(arg))
	return arg;
    else if (symbolp(arg))
	arg = getpname(arg);
    else if (!stringp(arg))
	xlerror("expecting a package name",arg);
    if ((pack = findpackage(getstring(arg))) == v_nil)
	xlerror("no package",arg);
    return pack;
}

/* xboundp - is this a value bound to this symbol? */
LVAL xboundp(void)
{
    LVAL sym,env,tmp;
    int off;
    
    /* parse the arguments */
    sym = xlgasymbol();
    env = moreargs() ? xlgetenv() : v_nil;
    xllastarg();

    /* check the global environment */
    if (env == v_nil || (tmp = findvar(env,sym,&off)) == v_nil)
        return boundp(sym) ? v_true : v_false;

    /* bound as an instance variable or local variable */
    else
        return v_true;
}

/* xsymname - get the print name of a symbol */
LVAL xsymname(void)
{
    LVAL sym;
    sym = xlgasymbol();
    xllastarg();
    return getpname(sym);
}

/* xsymvalue - get the value of a symbol */
LVAL xsymvalue(void)
{
    LVAL sym,env,tmp;
    int off;
    sym = xlgasymbol();
    env = moreargs() ? xlgetenv() : v_nil;
    xllastarg();

    /* return a global value */
    if (env == v_nil || (tmp = findvar(env,sym,&off)) == v_nil)
    	return getvalue(sym);

    /* return an instance variable */
    else if objectp(tmp)
        return getivar(tmp,off);
    
    /* return a local variable */
    else
        return getenvelement(tmp,off);
}

/* xsetsymvalue - set the value of a symbol */
LVAL xsetsymvalue(void)
{
    LVAL sym,val,env,tmp;
    int off;

    /* get the symbol */
    sym = xlgasymbol();
    val = xlgetarg();
    env = moreargs() ? xlgetenv() : v_nil;
    xllastarg();

    /* set the global value */
    if (env == v_nil || (tmp = findvar(env,sym,&off)) == v_nil)
    	setvalue(sym,val);

    /* set an instance variable */
    else if (objectp(tmp))
        setivar(tmp,off,val);

    /* set a local variable */
    else 
    	setenvelement(tmp,off,val);

    /* return the value */
    return val;
}

/* xsympackage - get the home package of a symbol */
LVAL xsympackage(void)
{
    LVAL sym;
    sym = xlgasymbol();
    xllastarg();
    return getpackage(sym);
}

/* xsymplist - get the property list of a symbol */
LVAL xsymplist(void)
{
    LVAL sym;

    /* get the symbol */
    sym = xlgasymbol();
    xllastarg();

    /* return the property list */
    return getplist(sym);
}

/* xsetsymplist - set the property list of a symbol */
LVAL xsetsymplist(void)
{
    LVAL sym,val;

    /* get the symbol */
    sym = xlgasymbol();
    val = xlgetarg();
    xllastarg();

    /* set the property list */
    setplist(sym,val);
    return val;
}

/* xget - get the value of a property */
LVAL xget(void)
{
    LVAL sym,prp;

    /* get the symbol and property */
    sym = xlgasymbol();
    prp = xlgasymbol();
    xllastarg();

    /* retrieve the property value */
    return xlgetprop(sym,prp);
}

/* xput - set the value of a property */
LVAL xput(void)
{
    LVAL sym,val,prp;

    /* get the symbol and property */
    sym = xlgasymbol();
    prp = xlgasymbol();
    val = xlgetarg();
    xllastarg();

    /* set the property value */
    xlputprop(sym,val,prp);

    /* return the value */
    return val;
}

/* xremprop - remove a property value from a property list */
LVAL xremprop(void)
{
    LVAL sym,prp;

    /* get the symbol and property */
    sym = xlgasymbol();
    prp = xlgasymbol();
    xllastarg();

    /* remove the property */
    xlremprop(sym,prp);

    /* return nil */
    return v_nil;
}

/* xtheenvironment - built-in function 'the-environment' */
LVAL xtheenvironment(void)
{
    LVAL unstack_environment();
    xllastarg();
    xlenv = unstack_environment(xlenv);
    return xlenv;
}

/* xprocenvironment - built-in function 'procedure-environment' */
LVAL xprocenvironment(void)
{
    LVAL arg;
    arg = xlgaclosure();
    xllastarg();
    return getenvironment(arg);
}

/* xenvp - built-in function 'environment?' */
LVAL xenvp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return envp(arg) ? v_true : v_false;
}

/* xenvbindings - built-in function 'environment-bindings' */
LVAL xenvbindings(void)
{
    LVAL names,this,last;
    FIXTYPE len,i;

    /* get the environment */
    xlval = xlgetenv();
    xllastarg();

    /* initialize */
    names = getenvnames(xlval);
    len = getenvsize(xlval);
    cpush(v_nil);

    /* build a list of dotted pairs */
    for (last = v_nil, i = FIRSTENV; i < len; ++i, names = cdr(names)) {
	this = cons(cons(car(names),getenvelement(xlval,i)),v_nil);
	if (last == v_nil) settop(this);
	else rplacd(last,this);
	last = this;
    }
    return pop();
}

/* xenvparent - built-in function 'environment-parent' */
LVAL xenvparent(void)
{
    LVAL env = xlgetenv();
    xllastarg();
    return env == v_nil ? v_nil : getnextframe(env);
}

/* xobjectp - built-in function 'object?' */
LVAL xobjectp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return objectp(arg) ? v_true : v_false;
}

/* xvector - built-in function 'vector' */
LVAL xvector(void)
{
    LVAL vect,*p;
    vect = newvector(xlargc);
    for (p = &vect->n_vdata[0]; moreargs(); )
	*p++ = xlgetarg();
    return vect;
}

/* xmakevector - built-in function 'make-vector' */
LVAL xmakevector(void)
{
    LVAL arg,val,*p;
    FIXTYPE len;
    
    /* get the vector size */
    arg = xlgafixnum();
    len = getfixnum(arg);

    /* check for an initialization value */
    if (moreargs()) {
	arg = xlgetarg();	/* get the initializer */
	xllastarg();		/* make sure that's the last argument */
	cpush(arg);		/* save the initializer */
	val = newvector(len);	/* create the vector */
	p = &val->n_vdata[0];	/* initialize the vector */
	for (arg = pop(); --len >= 0; )
	    *p++ = arg;
    }

    /* no initialization value */
    else
	val = newvector(len);	/* defaults to initializing to NIL */
    
    /* return the new vector */
    return val;
}

/* xvlength - built-in function 'vector-length' */
LVAL xvlength(void)
{
    LVAL arg;
    arg = xlgavector();
    xllastarg();
    return cvfixnum((FIXTYPE)getsize(arg));
}

/* xivlength - built-in function '%vector-length' */
LVAL xivlength(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return cvfixnum((FIXTYPE)getsize(arg));
}

/* xvref - built-in function 'vector-ref' */
LVAL xvref(void)
{
    return vref(xlgavector());
}

/* xivref - built-in function '%vector-ref' */
LVAL xivref(void)
{
    return vref(xlgetarg());
}

/* vref - common code for xvref and xivref */
static LVAL vref(LVAL vector)
{
    LVAL index;
    FIXTYPE i;

    /* get the index */
    index = xlgafixnum();
    xllastarg();

    /* range check the index */
    if ((i = getfixnum(index)) < 0 || i >= getsize(vector))
	xlerror("index out of range",index);

    /* return the vector element */
    return getelement(vector,i);
}

/* xvset - built-in function 'vector-set!' */
LVAL xvset(void)
{
    return vset(xlgavector());
}

/* xivset - built-in function '%vector-set!' */
LVAL xivset(void)
{
    return vset(xlgetarg());
}

/* vset - common code for xvset and xivset */
static LVAL vset(LVAL vector)
{
    LVAL index,val;
    FIXTYPE i;

    /* get the index and the new value */
    index = xlgafixnum();
    val = xlgetarg();
    xllastarg();

    /* range check the index */
    if ((i = getfixnum(index)) < 0 || i >= getsize(vector))
	xlerror("index out of range",index);

    /* set the vector element and return the value */
    setelement(vector,i,val);
    return val;
}

/* xibase - built-in function '%vector-base' */
LVAL xivbase(void)
{
    LVAL vector;
    vector = xlgetarg();
    xllastarg();
    return cvfixnum((FIXTYPE)vector->n_vdata);
}

/* xvectlist - built-in function 'vector->list' */
LVAL xvectlist(void)
{
    FIXTYPE size;
    LVAL vect;

    /* get the vector */
    vect = xlgavector();
    xllastarg();
    
    /* make a list from the vector */
    cpush(vect);
    size = getsize(vect);
    for (xlval = v_nil; --size >= 0; )
	xlval = cons(getelement(vect,size),xlval);
    drop(1);
    return xlval;
}

/* xlistvect - built-in function 'list->vector' */
LVAL xlistvect(void)
{
    LVAL vect,*p;
    FIXTYPE size;

    /* get the list */
    xlval = xlgalist();
    xllastarg();

    /* make a vector from the list */
    size = length(xlval);
    vect = newvector(size);
    for (p = &vect->n_vdata[0]; --size >= 0; xlval = cdr(xlval))
	*p++ = car(xlval);
    return vect;
}

/* xmakearray - built-in function 'make-array' */
LVAL xmakearray(void)
{
    LVAL val;
    val = makearray1(xlargc,xlsp);
    xlpopargs();
    return val;
}

/* makearray1 - helper function for xmakearray */
static LVAL makearray1(int argc,LVAL *argv)
{
    FIXTYPE size,i;
    LVAL arg;

    /* check for the end of the list of dimensions */
    if (--argc < 0)
	return v_nil;

    /* get this dimension */
    arg = *argv++;
    if (!fixp(arg))
	xlbadtype(arg);
    size = getfixnum(arg);

    /* make the new array */
    cpush(newvector(size));

    /* fill the array and return it */
    for (i = 0; i < size; ++i)
	setelement(top(),i,makearray1(argc,argv));
    return pop();
}

/* xaref - built-in function 'array-ref' */
LVAL xaref(void)
{
    LVAL array,index;
    FIXTYPE i;

    /* get the array */
    array = xlgavector();

    /* get each array index */
    while (xlargc > 1) {
	index = xlgafixnum(); i = getfixnum(index);
	if (i < 0 || i > getsize(array))
	    xlerror("index out of range",index);
	array = getelement(array,i);
	if (!vectorp(array))
	    xlbadtype(array);
    }
    cpush(array); ++xlargc;
    return xvref();
}

/* xaset - built-in function 'array-set!' */
LVAL xaset(void)
{
    LVAL array,index;
    FIXTYPE i;

    /* get the array */
    array = xlgavector();

    /* get each array index */
    while (xlargc > 2) {
	index = xlgafixnum(); i = getfixnum(index);
	if (i < 0 || i > getsize(array))
	    xlerror("index out of range",index);
	array = getelement(array,i);
	if (!vectorp(array))
	    xlbadtype(array);
    }
    cpush(array); ++xlargc;
    return xvset();
}

/* xmaketable - built-in function 'make-table' */
LVAL xmaketable(void)
{
    FIXTYPE len = HSIZE;
    
    /* get the vector size */
    if (moreargs()) {
        xlval = xlgafixnum();
        len = getfixnum(xlval);
    }
    xllastarg();

    /* make the table */
    return newtable(len);
}

/* xtablep - built-in function 'table?' */
LVAL xtablep(void)
{
    xlval = xlgetarg();
    xllastarg();
    return tablep(xlval) ? v_true : v_false;
}

/* xtableref - built-in function 'table-ref' */
LVAL xtableref(void)
{
    LVAL key;

    /* parse the arguments */
    xlval = xlgatable();
    key = xlgetarg();
    xllastarg();

    /* find the entry */
    xlval = findentryintable(xlval,key);
    return xlval == v_nil ? v_nil : cdr(xlval);
}

/* xtableset - built-in function 'table-set!' */
LVAL xtableset(void)
{
    LVAL key,val;

    /* parse the arguments */
    xlval = xlgatable();
    key = xlgetarg();
    val = xlgetarg();
    xllastarg();

    /* add the entry */
    addentrytotable(xlval,key,val);
    return xlval;
}

/* xtableremove - built-in function 'table-remove!' */
LVAL xtableremove(void)
{
    LVAL key;

    /* parse the arguments */
    xlval = xlgatable();
    key = xlgetarg();
    xllastarg();

    /* remove the entry */
    xlval = removeentryfromtable(xlval,key);
    return xlval == v_nil ? v_nil : cdr(xlval);
}

/* xemptytable - built-in function 'empty-table!' */
LVAL xemptytable(void)
{
    FIXTYPE size,i;

    /* parse the arguments */
    xlval = xlgatable();
    xllastarg();

    /* empty the table */
    size = getsize(xlval);
    for (i = 0; i < size; ++i)
        setelement(xlval,i,v_nil);
    return xlval;
}

/* xmapovertableentries - built-in function 'map-over-table-entries' */
LVAL xmapovertableentries(void)
{
    LVAL table,fun,list,last,val;
    FIXTYPE size,i;
    
    /* parse the arguments */
    table = xlgatable();
    fun = xlgetarg();
    xllastarg();

    /* save the table and function */
    check(3);
    push(table);
    push(fun);

    /* initialize */
    size = getsize(table);
    xlval = v_nil;
    last = v_nil;

    /* map over the table entries */
    for (i = 0; i < size; ++i) {
        for (list = getelement(table,i); consp(list); list = cdr(list)) {
            LVAL entry = car(list);
            push(list);
            CallFunction(&val,fun,2,car(entry),cdr(entry));
            if (last == v_nil) {
                xlval = cons(val,v_nil);
                last = xlval;
            }
            else {
                rplacd(last,cons(val,v_nil));
                last = cdr(last);
            }
            list = pop();
        }
    }
    drop(2);

    /* return the list of values */
    return xlval;
}

/* xiaddrof - built-in function '%address-of' */
LVAL xiaddrof(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return cvfixnum((FIXTYPE)arg);
}

/* xifmtaddr - built-in function '%format-address' */
LVAL xifmtaddr(void)
{
    char buf[20];
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    sprintf(buf,AFMT,arg);
    return cvcstring(buf);
}

/* xnull - built-in function 'null?' */
LVAL xnull(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return null(arg) ? v_true : v_false;
}

/* xatom - built-in function 'atom?' */
LVAL xatom(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return atom(arg) ? v_true : v_false;
}

/* xlistp - built-in function 'list?' */
LVAL xlistp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return listp(arg) ? v_true : v_false;
}

/* xendp - built-in function 'endp' */
LVAL xendp(void)
{
    LVAL arg;
    arg = xlgalist();
    xllastarg();
    return null(arg) ? v_true : v_false;
}

/* xnumberp - built-in function 'number?' */
LVAL xnumberp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return numberp(arg) ? v_true : v_false;
}

/* xbooleanp - built-in function 'boolean?' */
LVAL xbooleanp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return arg == v_true || arg == v_false ? v_true : v_false;
}

/* xpairp - built-in function 'pair?' */
LVAL xpairp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return consp(arg) ? v_true : v_false;
}

/* xsymbolp - built-in function 'symbol?' */
LVAL xsymbolp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return symbolp(arg) ? v_true : v_false;
}

/* xintegerp - built-in function 'integer?' */
LVAL xintegerp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return fixp(arg) ? v_true : v_false;
}

/* xrealp - built-in function 'real?' */
LVAL xrealp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return floatp(arg) || fixp(arg) ? v_true : v_false;
}

/* xcharp - built-in function 'char?' */
LVAL xcharp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return charp(arg) ? v_true : v_false;
}

/* xstringp - built-in function 'string?' */
LVAL xstringp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return stringp(arg) ? v_true : v_false;
}

/* xvectorp - built-in function 'vector?' */
LVAL xvectorp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return vectorp(arg) ? v_true : v_false;
}

#define isprocedure(x) \
(closurep(x) || continuationp(x) || subrp(x) || xsubrp(x))

/* xprocedurep - built-in function 'procedure?' */
LVAL xprocedurep(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return isprocedure(arg) ? v_true : v_false;
}

/* xdefaultobjectp - built-in function 'default-object?' */
LVAL xdefaultobjectp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return arg == default_object ? v_true : v_false;
}

/* xeq - built-in function 'eq?' */
LVAL xeq(void)
{
    LVAL arg1,arg2;
    arg1 = xlgetarg();
    arg2 = xlgetarg();
    xllastarg();
    return eq(arg1,arg2) ? v_true : v_false;
}

/* eq - internal 'eq?' function */
int eq(LVAL arg1,LVAL arg2)
{
    return arg1 == arg2;
}

/* xeqv - built-in function 'eqv?' */
LVAL xeqv(void)
{
    LVAL arg1,arg2;
    arg1 = xlgetarg();
    arg2 = xlgetarg();
    xllastarg();
    return eqv(arg1,arg2) ? v_true : v_false;
}

/* eqv - internal 'eqv?' function */
int eqv(LVAL arg1,LVAL arg2)
{
    /* try the eq test first */
    if (arg1 == arg2)
	return TRUE;

    /* compare fixnums, flonums and characters */
    if (!null(arg1) && !null(arg2)) {
	switch (ntype(arg1)) {
	case FIXNUM:
	    switch (ntype(arg2)) {
	    case FIXNUM:
	        return getfixnum(arg1) == getfixnum(arg2);
	    case FLONUM:
	        return (double)getfixnum(arg1) == getflonum(arg2);
	    default:
		return FALSE;
	    }
	case FLONUM:
	    switch (ntype(arg2)) {
	    case FIXNUM:
	        return getflonum(arg1) == (double)getfixnum(arg2);
	    case FLONUM:
	        return getflonum(arg1) == getflonum(arg2);
	    default:
		return FALSE;
	    }
	case CHARACTER:
	    return charp(arg2) && getchcode(arg1) == getchcode(arg2);
	case STRING:
	    return stringp(arg2) && stringequal(arg1,arg2);
	}
    }
    return FALSE;
}

/* xequal - built-in function 'equal?' */
LVAL xequal(void)
{
    LVAL arg1,arg2;
    arg1 = xlgetarg();
    arg2 = xlgetarg();
    xllastarg();
    return equal(arg1,arg2) ? v_true : v_false;
}

/* equal - internal 'equal?' function */
int equal(LVAL arg1,LVAL arg2)
{
    /* try the eq test first */
    if (arg1 == arg2)
	return TRUE;

    /* compare fixnums, flonums, characters, strings, vectors and conses */
    if (!null(arg1) && !null(arg2)) {
	switch (ntype(arg1)) {
	case FIXNUM:
	    switch (ntype(arg2)) {
	    case FIXNUM:
	        return getfixnum(arg1) == getfixnum(arg2);
	    case FLONUM:
	        return (double)getfixnum(arg1) == getflonum(arg2);
	    default:
		return FALSE;
	    }
	case FLONUM:
	    switch (ntype(arg2)) {
	    case FIXNUM:
	        return getflonum(arg1) == (double)getfixnum(arg2);
	    case FLONUM:
	        return getflonum(arg1) == getflonum(arg2);
	    default:
		return FALSE;
	    }
	case CHARACTER:
	    return charp(arg2) && getchcode(arg1) == getchcode(arg2);
	case STRING:
	    return stringp(arg2) && stringequal(arg1,arg2);
	case VECTOR:
	    return vectorp(arg2) && vectorequal(arg1,arg2);
	case CONS:
	    return consp(arg2) && equal(car(arg1),car(arg2))
	    		       && equal(cdr(arg1),cdr(arg2));
	}
    }
    return FALSE;
}

/* stringequal - compare two strings */
static int stringequal(LVAL s1,LVAL s2)
{
    char *p1 = getstring(s1);
    char *p2 = getstring(s2);
    FIXTYPE len;

    /* compare the vector lengths */
    if ((len = getslength(s1)) != getslength(s2))
	return FALSE;

    /* compare the vector elements */
    while (--len >= 0)
	if (*p1++ != *p2++)
	    return FALSE;
    return TRUE;
}

/* vectorequal - compare two vectors */
static int vectorequal(LVAL v1,LVAL v2)
{
    FIXTYPE len,i;

    /* compare the vector lengths */
    if ((len = getsize(v1)) != getsize(v2))
	return FALSE;

    /* compare the vector elements */
    for (i = 0; i < len; ++i)
	if (!equal(getelement(v1,i),getelement(v2,i)))
	    return FALSE;
    return TRUE;
}

/* xidentity - built-in function 'identity' */
LVAL xidentity(void)
{
    xlval = xlgetarg();
    xllastarg();
    return xlval;
}

/* xgensym - generate a symbol */
LVAL xgensym(void)
{
    char sym[STRMAX+11]; /* enough space for prefix and number */
    LVAL x;

    /* get the prefix or number */
    if (moreargs()) {
	if ((x = xlgetarg()) == v_nil)
	    xlerror("bad argument type",x);
	else
	    switch (ntype(x)) {
	    case SYMBOL:
		x = getpname(x);
	    case STRING:
		strncpy(gsprefix,getstring(x),STRMAX);
		gsprefix[STRMAX] = '\0';
		break;
	    case FIXNUM:
		gsnumber = getfixnum(x);
		break;
	    default:
		xlerror("bad argument type",x);
	    }
    }
    xllastarg();

    /* create the pname of the new symbol */
    sprintf(sym,"%s%d",gsprefix,gsnumber++);

    /* make a symbol with this print name */
    return cvsymbol(cvcstring(sym));
}

/* xlgetkeyarg - get a keyword argument */
int xlgetkeyarg(LVAL key,LVAL def,LVAL *pval)
{
    LVAL *p;
    int n;
    for (n = xlargc, p = xlsp; n >= 2; n -= 2, p += 2)
	if (*p == key) {
	    *pval = p[1];
	    return TRUE;
	}
    *pval = def;
    return FALSE;
}

/* xlgkfixnum - get a fixnum keyword argument */
int xlgkfixnum(LVAL key,FIXTYPE def,FIXTYPE *pval)
{
    LVAL arg;
    if (xlgetkeyarg(key,v_nil,&arg)) {
	if (!fixp(arg))
	    xlbadtype(arg);
        *pval = getfixnum(arg);
	return TRUE;
    }
    *pval = def;
    return FALSE;
}

/* xlgettest - get the :test or :test-not keyword argument */
void xlgettest(LVAL def,LVAL *pfcn,LVAL *ptresult)
{
    if (xlgetkeyarg(k_test,def,pfcn))		/* :test */
	*ptresult = v_true;
    else if (xlgetkeyarg(k_testnot,def,pfcn))	/* :test-not */
	*ptresult = v_false;
    else
	*ptresult = v_true;
}

/* xlgetport - get a port */
LVAL xlgetport(void)
{
    LVAL arg;

    /* get a file, unnamed or object stream or nil */
    if ((arg = xlgetarg()) != v_nil) {
	if (fstreamp(arg)) {
	    if (getfile(arg) == NULL && (getpflags(arg) & PF_TERMINAL) == 0)
		xlerror("port not open",arg);
	}
	else if (!ustreamp(arg) && !ostreamp(arg))
	    xlbadtype(arg);
    }
    return arg;
}

/* xlgetiport - get an input port */
LVAL xlgetiport(void)
{
    LVAL arg = xlgetport();
    if (portp(arg) && (getpflags(arg) & PF_INPUT) == 0)
	xlerror("expecting input port",arg);
    return arg;
}

/* xlgetoport - get an output port */
LVAL xlgetoport(void)
{
    LVAL arg = xlgetport();
    if (portp(arg) && (getpflags(arg) & PF_OUTPUT) == 0)
	xlerror("expecting output port",arg);
    return arg;
}

/* xlgetenv - get an environment */
LVAL xlgetenv(void)
{
    LVAL val = xlgetarg();
    if (closurep(val))
	val = getenvironment(val);
    else if (!envp(val))
	xlbadtype(val);
    return val;
}
