/* xlfun2.c - xlisp built-in functions - part 2 */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* local definitions */
#define fix(n)	cvfixnum((FIXTYPE)(n))
#define TLEFT	1
#define TRIGHT	2

/* external variables */
extern LVAL eof_object;
extern LVAL xlfun,xlenv,xlval;
extern LVAL s_stdin,s_stdout,s_stderr,s_error,k_start,k_end,k_1start,k_1end,k_2start,k_2end;
extern int prbreadth,prdepth;
extern FILE *tfp;

/* forward declarations */
static LVAL setit(int *pvar);
static LVAL openfile(short flags,char *mode);
static void format(LVAL stream);
static void getbounds(LVAL str,LVAL skey,LVAL ekey,FIXTYPE *pstart,FIXTYPE *pend);
static char *radixnumtostr(FIXTYPE n,int radix,char *buf,int len);
static int inbag(int ch,LVAL bag);
static LVAL strcompare(int fcn,int icase);
static LVAL strsearch(int icase);
static LVAL chrcompare(int fcn,int icase);
static LVAL changecase(int fcn,int destructive);
static LVAL trim(int fcn);
static LVAL room(void);

/* xsymstr - built-in function 'symbol->string' */
LVAL xsymstr(void)
{
    xlval = xlgasymbol();
    xllastarg();
    return getpname(xlval);
}

/* xstrsym - built-in function 'string->symbol' */
LVAL xstrsym(void)
{
    extern LVAL s_package;
    LVAL key;
    xlval = xlgastring();
    xllastarg();
    return intern(xlval,getvalue(s_package),&key);
}

/* xreadline - read a line from a file */
LVAL xreadline(void)
{
    LVAL fptr;
    int ch;

    /* get file pointer */
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();

    /* get character and check for eof */
    if ((ch = xlgetc(fptr)) == EOF)
        xlval = v_nil;
    else {
	char buf[STRMAX],*p = buf;
	FIXTYPE len = 0;
	xlval = v_nil;
	cpush(fptr);
        while (ch != EOF && ch != '\n') {
	    if (++len > STRMAX) {
	        if (xlval == v_nil) {
	            FIXTYPE cnt = STRMAX;
	            xlval = newustream();
	            for (p = buf; --cnt >= 0; )
	                xlputc(xlval,*p++);    
	        }
	        xlputc(xlval,ch);
	    }
	    else
	        *p++ = ch;
	    ch = xlgetc(fptr);
	}
	xlval = xlval == v_nil ? cvstring(buf,len) : getstroutput(xlval);
	drop(1);
    }

    /* return the string */
    return xlval;
}

/* xrdchar - built-in function 'read-char' */
LVAL xrdchar(void)
{
    LVAL fptr;
    int ch;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    return (ch = xlgetc(fptr)) == EOF ? eof_object : cvchar(ch);
}

/* xunreadchar - built-in function 'unread-char' */
LVAL xunreadchar(void)
{
    LVAL fptr,ch;
    ch = xlgachar();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlungetc(fptr,(int)getchcode(ch));
    return v_true;
}

/* xpkchar - peek at a character from a file */
LVAL xpkchar(void)
{
    LVAL fptr;
    int ch;

    /* get file pointer */
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();

    /* return the character */
    return (ch = xlpeek(fptr)) == EOF ? eof_object : cvchar(ch);
}

/* xcharready - built-in function 'char-read?' */
LVAL xcharready(void)
{
    LVAL fptr;
    
    /* parse the argument list */
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    
    /* return true if there is a character to read */
    return xliready(fptr) ? v_true : v_false;
}

/* xclearinput - built-in function 'clear-input' */
LVAL xclearinput(void)
{
    LVAL fptr;
    
    /* parse the argument list */
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    
    /* flush buffered input */
    xliflush(fptr);
    return v_nil;
}

/* xrdbyte - built-in function 'read-byte' */
LVAL xrdbyte(void)
{
    LVAL fptr;
    int ch;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    return (ch = xlgetc(fptr)) == EOF ? eof_object : cvfixnum((FIXTYPE)ch);
}

/* xrdshort - built-in function 'read-short' */
LVAL xrdshort(void)
{
    unsigned char *p;
    short int val=0;
    LVAL fptr;
    int ch,n;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    for (n = sizeof(short int), p = (unsigned char *)&val; --n >= 0; ) {
        if ((ch = xlgetc(fptr)) == EOF)
	    return eof_object;
        *p++ = ch;
    }
    return cvfixnum((FIXTYPE)val);
}

/* xrdshorthf - built-in function 'read-short-high-first' */
LVAL xrdshorthf(void)
{
    short int val;
    LVAL fptr;
    int ch;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val = (ch & 0xff) << 8;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= ch & 0xff;
    return cvfixnum((FIXTYPE)val);
}

/* xrdshortlf - built-in function 'read-short-low-first' */
LVAL xrdshortlf(void)
{
    short int val;
    LVAL fptr;
    int ch;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val = ch & 0xff;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= (ch & 0xff) << 8;
    return cvfixnum((FIXTYPE)val);
}

/* xrdlong - built-in function 'read-long' */
LVAL xrdlong(void)
{
    unsigned char *p;
    long int val=0;
    LVAL fptr;
    int ch,n;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    for (n = sizeof(long int), p = (unsigned char *)&val; --n >= 0; ) {
        if ((ch = xlgetc(fptr)) == EOF)
	    return eof_object;
        *p++ = ch;
    }
    return cvfixnum((FIXTYPE)val);
}

/* xrdlonghf - built-in function 'read-long-high-first' */
LVAL xrdlonghf(void)
{
    long int val;
    LVAL fptr;
    int ch;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val = (ch & 0xff) << 24;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= (ch & 0xff) << 16;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= (ch & 0xff) << 8;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= ch & 0xff;
    return cvfixnum((FIXTYPE)val);
}

/* xrdlonglf - built-in function 'read-long-low-first' */
LVAL xrdlonglf(void)
{
    long int val;
    LVAL fptr;
    int ch;
    fptr = moreargs() ? xlgetiport() : curinput();
    xllastarg();
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val = ch & 0xff;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= (ch & 0xff) << 8;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= (ch & 0xff) << 16;
    if ((ch = xlgetc(fptr)) == EOF)
	return eof_object;
    val |= (ch & 0xff) << 24;
    return cvfixnum((FIXTYPE)val);
}

/* xeofobjectp - built-in function 'eof-object?' */
LVAL xeofobjectp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return arg == eof_object ? v_true : v_false;
}

/* xwrite - built-in function 'write' */
LVAL xwrite(void)
{
    LVAL fptr,val;

    /* get expression to print and file pointer */
    val = xlgetarg();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* print the value */
    xlprin1(val,fptr);
    return v_true;
}

/* xprint - built-in function 'print' */
LVAL xprint(void)
{
    LVAL fptr,val;

    /* get expression to print and file pointer */
    val = xlgetarg();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* print the value one a new line */
    xlterpri(fptr);
    xlprin1(val,fptr);
    xlputc(fptr,' ');
    return v_true;
}

/* xwrchar - built-in function 'write-char' */
LVAL xwrchar(void)
{
    LVAL fptr,ch;
    ch = xlgachar();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlputc(fptr,(int)getchcode(ch));
    return v_true;
}

/* xwrbyte - built-in function 'write-byte' */
LVAL xwrbyte(void)
{
    LVAL fptr,ch;
    ch = xlgafixnum();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlputc(fptr,(int)getfixnum(ch));
    return v_true;
}

/* xwrshort - built-in function 'write-short' */
LVAL xwrshort(void)
{
    unsigned char *p;
    short int val;
    LVAL fptr,v;
    int n;
    v = xlgafixnum(); val = (short int)getfixnum(v);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    for (n = sizeof(short int), p = (unsigned char *)&val; --n >= 0; )
        xlputc(fptr,*p++);
    return v_true;
}

/* xwrshorthf - built-in function 'write-short-high-first' */
LVAL xwrshorthf(void)
{
    short int val;
    LVAL fptr,v;
    v = xlgafixnum(); val = (short int)getfixnum(v);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlputc(fptr,(val >> 8) & 0xff);
    xlputc(fptr,val & 0xff);
    return v_true;
}

/* xwrshortlf - built-in function 'write-short-low-first' */
LVAL xwrshortlf(void)
{
    short int val;
    LVAL fptr,v;
    v = xlgafixnum(); val = (short int)getfixnum(v);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlputc(fptr,val & 0xff);
    xlputc(fptr,(val >> 8) & 0xff);
    return v_true;
}

/* xwrlong - built-in function 'write-long' */
LVAL xwrlong(void)
{
    unsigned char *p;
    long int val;
    LVAL fptr,v;
    int n;
    v = xlgafixnum(); val = (long int)getfixnum(v);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    for (n = sizeof(long int), p = (unsigned char *)&val; --n >= 0; )
        xlputc(fptr,*p++);
    return v_true;
}

/* xwrlonghf - built-in function 'write-long-high-first' */
LVAL xwrlonghf(void)
{
    short int val;
    LVAL fptr,v;
    v = xlgafixnum(); val = (short int)getfixnum(v);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlputc(fptr,(val >> 24) & 0xff);
    xlputc(fptr,(val >> 16) & 0xff);
    xlputc(fptr,(val >> 8) & 0xff);
    xlputc(fptr,val & 0xff);
    return v_true;
}

/* xwrlonglf - built-in function 'write-long-low-first' */
LVAL xwrlonglf(void)
{
    short int val;
    LVAL fptr,v;
    v = xlgafixnum(); val = (short int)getfixnum(v);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();
    xlputc(fptr,val & 0xff);
    xlputc(fptr,(val >> 8) & 0xff);
    xlputc(fptr,(val >> 16) & 0xff);
    xlputc(fptr,(val >> 24) & 0xff);
    return v_true;
}

/* xdisplay - built-in function 'display' */
LVAL xdisplay(void)
{
    LVAL fptr,val;

    /* get expression to print and file pointer */
    val = xlgetarg();
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* print the value */
    xlprinc(val,fptr);
    return v_true;
}

/* xnewline - terminate the current print line */
LVAL xnewline(void)
{
    LVAL fptr;

    /* get file pointer */
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* terminate the print line and return nil */
    xlterpri(fptr);
    return v_true;
}

/* xfreshline - start a fresh print line */
LVAL xfreshline(void)
{
    LVAL fptr;

    /* get file pointer */
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* terminate the print line and return nil */
    xlfreshline(fptr);
    return v_true;
}

/* xwritesize - built-in function 'write-size' */
LVAL xwritesize(void)
{
    extern FIXTYPE xlfsize;
    LVAL val;

    /* get expression to compute the size of */
    val = xlgetarg();
    xllastarg();

    /* compute the size */
    xlfsize = 0;
    xlprin1(val,v_nil);
    return cvfixnum(xlfsize);
}

/* xdisplaysize - built-in function 'display-size' */
LVAL xdisplaysize(void)
{
    extern FIXTYPE xlfsize;
    LVAL val;

    /* get expression to compute the size of */
    val = xlgetarg();
    xllastarg();

    /* compute the size */
    xlfsize = 0;
    xlprinc(val,v_nil);
    return cvfixnum(xlfsize);
}

/* xprbreadth - set the maximum number of elements to be printed */
LVAL xprbreadth(void)
{
    return setit(&prbreadth);
}

/* xprdepth - set the maximum depth of nested lists to be printed */
LVAL xprdepth(void)
{
    return setit(&prdepth);
}

/* setit - common routine for prbreadth/prdepth */
static LVAL setit(int *pvar)
{
    LVAL arg;

    /* get the optional argument */
    if (moreargs()) {
	arg = xlgetarg();
	xllastarg();
	*pvar = (fixp(arg) ? (int)getfixnum(arg) : -1);
    }

    /* return the value of the variable */
    return *pvar >= 0 ? cvfixnum((FIXTYPE)*pvar) : v_nil;
}

/* xfilemodtime - built-in function 'file-modification-time' */
LVAL xfilemodtime(void)
{
    FIXTYPE mtime;

    /* parse the arguments */
    xlval = xlgastring();
    xllastarg();

    /* get the file modification time */
    return osfmodtime(getstring(xlval),&mtime) ? cvfixnum(mtime) : v_nil;
}

/* xopeni - built-in function 'open-input-file' */
LVAL xopeni(void)
{
    return openfile(PF_INPUT,"r");
}

/* xopeno - built-in function 'open-output-file' */
LVAL xopeno(void)
{
    return openfile(PF_OUTPUT | PF_BOL,"w");
}

/* xopena - built-in function 'open-append-file' */
LVAL xopena(void)
{
    return openfile(PF_OUTPUT | PF_BOL,"a");
}

/* xopenu - built-in function 'open-update-file' */
LVAL xopenu(void)
{
    return openfile(PF_INPUT | PF_OUTPUT | PF_BOL,"r+");
}

/* openfile - open an ascii or binary file */
static LVAL openfile(short flags,char *mode)
{
    LVAL file,modekey;
    char *name;
    FILE *fp;

    /* get the file name and direction */
    name = getstring(xlgastring());
    modekey = moreargs() ? xlgasymbol() : v_nil;
    xllastarg();

    /* check for binary mode */
    if (modekey != v_nil) {
	if (modekey == xlenter("BINARY"))
	    flags |= PF_BINARY;
	else if (modekey != xlenter("TEXT"))
	    xlerror("unrecognized open mode",modekey);
    }

    /* try to open the file */
    file = cvfstream(NULL,flags);
    fp = ((flags & PF_BINARY) == 0 ? osaopen(name,mode) : osbopen(name,mode));
    if (fp == NULL)
	return v_nil;
    setsdata(file,fp);
    return file;
}

/* xclose - built-in function 'close-port' */
LVAL xclose(void)
{
    LVAL fptr;
    fptr = xlgaport();
    xllastarg();
    if (fstreamp(fptr) && getfile(fptr) != NULL) {
	osclose(getfile(fptr));
        setsdata(fptr,NULL);
    }
    return v_nil;
}

/* xclosei - built-in function 'close-input-port' */
LVAL xclosei(void)
{
    LVAL fptr;
    fptr = xlgaiport();
    xllastarg();
    if (fstreamp(fptr) && getfile(fptr) != NULL) {
	osclose(getfile(fptr));
        setsdata(fptr,NULL);
    }
    return v_nil;
}

/* xcloseo - built-in function 'close-output-port' */
LVAL xcloseo(void)
{
    LVAL fptr;
    fptr = xlgaoport();
    xllastarg();
    if (fstreamp(fptr) && getfile(fptr) != NULL) {
	osclose(getfile(fptr));
        setsdata(fptr,NULL);
    }
    return v_nil;
}

/* xgetfposition - built-in function 'get-file-position' */
LVAL xgetfposition(void)
{
    LVAL fptr;
    fptr = xlgafstream();
    xllastarg();
    return cvfixnum(ostell(getfile(fptr)));
}

/* xsetfposition - built-in function 'set-file-position!' */
LVAL xsetfposition(void)
{
    LVAL fptr,val;
    long position;
    int whence;
    fptr = xlgafstream();
    val = xlgafixnum(); position = getfixnum(val);
    val = xlgafixnum(); whence = (int)getfixnum(val);
    xllastarg();
    return osseek(getfile(fptr),position,whence) == 0 ? v_true : v_false;
}

/* xcurinput - built-in function 'current-input-port' */
LVAL xcurinput(void)
{
    xllastarg();
    return curinput();
}

/* curinput - get the current input port */
LVAL curinput(void)
{
    return getvalue(s_stdin);
}

/* xcuroutput - built-in function 'current-output-port' */
LVAL xcuroutput(void)
{
    xllastarg();
    return curoutput();
}

/* curoutput - get the current output port */
LVAL curoutput(void)
{
    return getvalue(s_stdout);
}

/* xcurerror - built-in function 'current-error-port' */
LVAL xcurerror(void)
{
    xllastarg();
    return curerror();
}

/* curerror - get the current error port */
LVAL curerror(void)
{
    return getvalue(s_stderr);
}

/* xportp - built-in function 'port?' */
LVAL xportp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return portp(arg) ? v_true : v_false;
}

/* xinputportp - built-in function 'input-port?' */
LVAL xinputportp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return iportp(arg) ? v_true : v_false;
}

/* xoutputportp - built-in function 'output-port?' */
LVAL xoutputportp(void)
{
    LVAL arg;
    arg = xlgetarg();
    xllastarg();
    return oportp(arg) ? v_true : v_false;
}

/* xmkstrinput - make a string input stream */
LVAL xmkstrinput(void)
{
    FIXTYPE start,end,len;
    LVAL val;
    
    /* get the string and length */
    xlval = xlgastring();
    len = getslength(xlval);

    /* get the starting offset */
    if (moreargs()) {
	val = xlgafixnum();
	start = getfixnum(val);
    }
    else start = 0;

    /* get the ending offset */
    if (moreargs()) {
	val = xlgafixnum();
	end = getfixnum(val);
    }
    else end = len;
    xllastarg();

    /* check the bounds */
    if (start < 0 || start > len)
	xlerror("string index out of bounds",cvfixnum((FIXTYPE)start));
    if (end < 0 || end > len)
	xlerror("string index out of bounds",cvfixnum((FIXTYPE)end));

    /* create the stream */
    cpush(newustream());
    
    /* copy the characters into the stream */
    for (; start < end; ++start) {
	char *p = getstring(xlval) + start;
        xlputc(top(),*p);
    }

    /* return the new stream */
    return pop();
}

/* xmkstroutput - make a string output stream */
LVAL xmkstroutput(void)
{
    xllastarg();
    return newustream();
}

/* xgetstroutput - get output stream string */
LVAL xgetstroutput(void)
{
    LVAL stream;
    stream = xlgaustream();
    xllastarg();
    return getstroutput(stream);
}

/* xmkobjstream - built-in function 'make-object-stream' */
LVAL xmkobjstream(void)
{
    /* parse the argument list */
    xlval = xlgaobject();
    xllastarg();

    /* create the object stream */
    return cvostream(xlval,PF_INPUT | PF_OUTPUT | PF_BOL);
}

/* xformat - formatted output function */
LVAL xformat(void)
{
    LVAL stream,val;
    
    /* get the stream */
    stream = xlgetarg();
    if (stream == v_nil)
	val = stream = newustream();
    else {
	if (stream == v_true)
	    stream = curoutput();
	else if (!oportp(stream))
	    xlbadtype(stream);
	val = v_nil;
    }
    
    /* do the formatted output */
    format(stream);

    /* return the value */
    return val == v_nil ? v_nil : getstroutput(val);
}

/* format - finish 'format' and 'error' handling */
static void format(LVAL stream)
{
    int atseen,argseen,arg,argc,ch,i;
    LVAL fmtstring,*argp;
    FIXTYPE len;

    /* get the format string */
    fmtstring = xlgastring();
    len = getslength(fmtstring);
    i = 0;

    /* setup the argument pointer and count */
    argp = xlsp;
    argc = xlargc;
    
    /* protect the stream and format string */
    check(2);
    push(stream);
    push(fmtstring);

    /* process the format string */
    for (;;) {
        char *fmt = getstring(fmtstring);
        if (i >= len)
            break;
	else if ((ch = fmt[i++]) == '~' && i < len) {
            
            /* check for the '@' modifier */
            if (atseen = (fmt[i] == '@'))
                ++i;
            
            /* check for a numeric argument */
            if (argseen = isdigit(fmt[i]))
                for (arg = 0; isdigit(fmt[i]); ++i)
                    arg = arg * 10 + fmt[i] - '0';
                
            /* dispatch on the formatting directive */
	    switch (ch = fmt[i++]) {
	    case 'a': case 'A':
		if (--argc < 0)
		    xltoofew();
		xlprinc(*argp++,stream);
		break;
	    case 's': case 'S':
		if (--argc < 0)
		    xltoofew();
		xlprin1(*argp++,stream);
		break;
	    case '%':
		xlterpri(stream);
		break;
	    case '~':
		xlputc(stream,'~');
		break;
	    case '\n':
		while ((ch = fmt[i]) != '\0' && isspace(ch) && ch != '\n')
		    ++i;
		break;
	    default:
		xlerror("unknown format directive",cvchar(ch));
	    }
	}
	else
	    xlputc(stream,ch);
    }
    
    /* get rid of the format arguments */
    drop(xlargc + 2);
}

/* xtranson - built-in function 'transcript-on' */
LVAL xtranson(void)
{
    char *name;

    /* get the file name and direction */
    name = getstring(xlgastring());
    xllastarg();

    /* close any currently open transcript file */
    if (tfp) { osclose(tfp); tfp = NULL; }

    /* try to open the file */
    return (tfp = osaopen(name,"w")) == NULL ? v_false : v_true;
}

/* xtransoff - built-in function 'transcript-off' */
LVAL xtransoff(void)
{
    /* make sure there aren't any arguments */
    xllastarg();

    /* make sure the transcript is open */
    if (tfp == NULL)
	return v_false;

    /* close the transcript and return successfully */
    osclose(tfp); tfp = NULL;
    return v_true;
}

/* xmakestring - built-in function 'make-string' */
LVAL xmakestring(void)
{
    FIXTYPE size;
    int fill;
    char *p;
    
    /* parse the argument list */
    xlval = xlgafixnum(); size = getfixnum(xlval);
    fill = moreargs() ? getchcode(xlgachar()) : -1;
    xllastarg();
    
    /* make the string */
    xlval = xlnewstring(size);
    
    /* fill it if necessary */
    if (fill != -1)
        for (p = getstring(xlval); --size >= 0; )
            *p++ = fill;
    
    /* return the new string */
    return xlval;
}

/* xstrlen - built-in function 'string-length' */
LVAL xstrlen(void)
{
    LVAL str;
    str = xlgastring();
    xllastarg();
    return cvfixnum((FIXTYPE)(getslength(str)));
}

/* xstrnullp - built-in function 'string-null?' */
LVAL xstrnullp(void)
{
    LVAL str;
    str = xlgastring();
    xllastarg();
    return getslength(str) == 0 ? v_true : v_false;
}

/* xstrappend - built-in function 'string-append' */
LVAL xstrappend(void)
{
    LVAL *savesp,arg;
    int saveargc;
    FIXTYPE len;
    char *str;

    /* save the argument list */
    saveargc = xlargc;
    savesp = xlsp;

    /* find the length of the new string */
    for (len = 0; moreargs(); ) {
	arg = xlgastring();
	len += getslength(arg);
    }

    /* restore the argument list */
    xlargc = saveargc;
    xlsp = savesp;
    
    /* create the result string */
    xlval = xlnewstring(len);
    str = getstring(xlval);

    /* combine the strings */
    while (moreargs()) {
	arg = nextarg();
	len = getslength(arg);
	memcpy(str,getstring(arg),(size_t)len);
	str += len;
    }

    /* return the new string */
    return xlval;
}

/* xstrref - built-in function 'string-ref' */
LVAL xstrref(void)
{
    LVAL str,num;
    FIXTYPE n;

    /* get the string and the index */
    str = xlgastring();
    num = xlgafixnum();
    xllastarg();

    /* range check the index */
    if ((n = getfixnum(num)) < 0 || n >= getslength(str))
	xlerror("index out of range",num);

    /* return the character */
    return cvchar(((unsigned char *)getstring(str))[n]);
}

/* xstrset - built-in function 'string-set!' */
LVAL xstrset(void)
{
    LVAL str,num,ch;
    FIXTYPE n;

    /* get the string and the index */
    str = xlgastring();
    num = xlgafixnum();
    ch = xlgachar();
    xllastarg();

    /* range check the index */
    if ((n = getfixnum(num)) < 0 || n >= getslength(str))
	xlerror("index out of range",num);

    /* return the character */
    getstring(str)[n] = getchcode(ch);
    return ch;
}

/* xsubstring - built-in function 'substring' */
LVAL xsubstring(void)
{
    FIXTYPE start,end,len;
    LVAL src,dst;
    char *srcp;

    /* get string and starting and ending positions */
    src = xlgastring();

    /* get the starting position */
    dst = xlgafixnum(); start = getfixnum(dst);
    if (start < 0 || start > getslength(src))
	xlerror("index out of range",dst);

    /* get the ending position */
    if (moreargs()) {
	dst = xlgafixnum(); end = getfixnum(dst);
	if (end < 0 || end > getslength(src))
	    xlerror("index out of range",dst);
    }
    else
	end = getslength(src);
    xllastarg();

    /* setup the source pointer */
    srcp = getstring(src) + start;
    len = end - start;

    /* make a destination string and setup the pointer */
    dst = xlnewstring(len);

    /* copy the source to the destination */
    memcpy(getstring(dst),srcp,(size_t)len);

    /* return the substring */
    return dst;
}

/* xstrlist - built-in function 'string->list' */
LVAL xstrlist(void)
{
    FIXTYPE size;
    LVAL str;

    /* get the vector */
    str = xlgastring();
    xllastarg();
    
    /* make a list from the vector */
    cpush(str);
    size = getslength(str);
    for (xlval = v_nil; --size >= 0; )
	xlval = cons(cvchar(((unsigned char *)getstring(str))[size]),xlval);
    drop(1);
    return xlval;
}

/* xliststring - built-in function 'list->string' */
LVAL xliststring(void)
{
    FIXTYPE size;
    LVAL str;
    char *p;

    /* get the list */
    xlval = xlgalist();
    xllastarg();

    /* make a vector from the list */
    size = length(xlval);
    str = xlnewstring(size);
    for (p = getstring(str); --size >= 0; xlval = cdr(xlval))
	if (charp(car(xlval)))
	    *p++ = getchcode(car(xlval));
	else
	    xlbadtype(car(xlval));
    return str;
}

/* case conversion functions */
LVAL xupcase(void)   { return changecase('U',FALSE); }
LVAL xdowncase(void) { return changecase('D',FALSE); }

/* destructive case conversion functions */
LVAL xnupcase(void)   { return changecase('U',TRUE); }
LVAL xndowncase(void) { return changecase('D',TRUE); }

/* changecase - change case */
static LVAL changecase(int fcn,int destructive)
{
    FIXTYPE start,end,len,i;
    char *srcp,*dstp;
    LVAL src,dst;
    int ch;

    /* get the arguments */
    src = xlgastring();
    getbounds(src,k_start,k_end,&start,&end);
    xlpopargs();

    /* make a destination string */
    len = getslength(src);
    dst = (destructive ? src : xlnewstring(len));

    /* setup the string pointers */
    srcp = getstring(src);
    dstp = getstring(dst);

    /* copy the source to the destination */
    for (i = 0; i < len; ++i) {
	ch = *srcp++;
	if (i >= start && i < end)
	    switch (fcn) {
	    case 'U':	if (islower(ch)) ch = toupper(ch); break;
	    case 'D':	if (isupper(ch)) ch = tolower(ch); break;
	    }
	*dstp++ = ch;
    }

    /* return the new string */
    return dst;
}

/* trim functions */
LVAL xtrim(void)      { return trim(TLEFT|TRIGHT); }
LVAL xlefttrim(void)  { return trim(TLEFT); }
LVAL xrighttrim(void) { return trim(TRIGHT); }

/* trim - trim character from a string */
static LVAL trim(int fcn)
{
    char *leftp,*rightp,*dstp;
    LVAL bag,src,dst;

    /* get the bag and the string */
    bag = xlgastring();
    src = xlgastring();
    xllastarg();

    /* setup the string pointers */
    leftp = getstring(src);
    rightp = leftp + getslength(src) - 1;

    /* trim leading characters */
    if (fcn & TLEFT)
	while (leftp <= rightp && inbag(*leftp,bag))
	    ++leftp;

    /* trim character from the right */
    if (fcn & TRIGHT)
	while (rightp >= leftp && inbag(*rightp,bag))
	    --rightp;

    /* make a destination string and setup the pointer */
    dst = xlnewstring(rightp - leftp + 1);
    dstp = getstring(dst);

    /* copy the source to the destination */
    while (leftp <= rightp)
	*dstp++ = *leftp++;

    /* return the new string */
    return dst;
}

/* getbounds - get the start and end bounds of a string */
static void getbounds(LVAL str,LVAL skey,LVAL ekey,FIXTYPE *pstart,FIXTYPE *pend)
{
    FIXTYPE len;

    /* get the length of the string */
    len = getslength(str);

    /* get the starting and ending indicies */
    xlgkfixnum(skey,0,pstart);
    xlgkfixnum(ekey,len,pend);

    /* check the starting and ending indicies */
    if (*pstart < 0 || *pstart > len)
	xlerror("string index out of bounds",cvfixnum((FIXTYPE)*pstart));
    if (*pend < 0 || *pend > len)
        xlerror("string index out of bounds",cvfixnum((FIXTYPE)*pend));

    /* make sure the start is less than or equal to the end */
    if (*pstart > *pend)
	xlerror("starting index error",cvfixnum((FIXTYPE)*pstart));
}

/* inbag - test if a character is in a bag */
static int inbag(int ch,LVAL bag)
{
    char *p;
    for (p = getstring(bag); *p != '\0'; ++p)
	if (*p == ch)
	    return TRUE;
    return FALSE;
}

/* string comparision functions */
LVAL xstrlss(void) { return strcompare('<',FALSE); } /* string< */
LVAL xstrleq(void) { return strcompare('L',FALSE); } /* string<= */
LVAL xstreql(void) { return strcompare('=',FALSE); } /* string= */
LVAL xstrneq(void) { return strcompare('#',FALSE); } /* string/= */
LVAL xstrgeq(void) { return strcompare('G',FALSE); } /* string>= */
LVAL xstrgtr(void) { return strcompare('>',FALSE); } /* string> */

/* string comparison functions (not case sensitive) */
LVAL xstrilss(void) { return strcompare('<',TRUE); } /* string-lessp */
LVAL xstrileq(void) { return strcompare('L',TRUE); } /* string-not-greaterp */
LVAL xstrieql(void) { return strcompare('=',TRUE); } /* string-equal */
LVAL xstrineq(void) { return strcompare('#',TRUE); } /* string-not-equal */
LVAL xstrigeq(void) { return strcompare('G',TRUE); } /* string-not-lessp */
LVAL xstrigtr(void) { return strcompare('>',TRUE); } /* string-greaterp */

/* strcompare - compare strings */
static LVAL strcompare(int fcn,int icase)
{
    FIXTYPE start1,end1,start2,end2;
    LVAL str1,str2;
    char *p1,*p2;
    int ch1,ch2;

    /* get the strings */
    str1 = xlgastring();
    str2 = xlgastring();

    /* get the substring specifiers */
    getbounds(str1,k_1start,k_1end,&start1,&end1);
    getbounds(str2,k_2start,k_2end,&start2,&end2);
    xlpopargs();

    /* setup the string pointers */
    p1 = &getstring(str1)[start1];
    p2 = &getstring(str2)[start2];

    /* compare the strings */
    for (; start1 < end1 && start2 < end2; ++start1,++start2) {
	ch1 = *p1++;
	ch2 = *p2++;
	if (icase) {
	    if (isupper(ch1)) ch1 = tolower(ch1);
	    if (isupper(ch2)) ch2 = tolower(ch2);
	}
	if (ch1 != ch2)
	    switch (fcn) {
	    case '<':	return ch1 < ch2 ? cvfixnum(start1) : v_false;
	    case 'L':	return ch1 <= ch2 ? cvfixnum(start1) : v_false;
	    case '=':	return v_false;
	    case '#':	return cvfixnum(start1);
	    case 'G':	return ch1 >= ch2 ? cvfixnum(start1) : v_false;
	    case '>':	return ch1 > ch2 ? cvfixnum(start1) : v_false;
	    }
    }

    /* check the termination condition */
    switch (fcn) {
    case '<':	return start1 >= end1 && start2 < end2 ? cvfixnum(start1) : v_false;
    case 'L':	return start1 >= end1 ? cvfixnum(start1) : v_false;
    case '=':	return start1 >= end1 && start2 >= end2 ? v_true : v_false;
    case '#':	return start1 >= end1 && start2 >= end2 ? v_false : cvfixnum(start1);
    case 'G':	return start2 >= end2 ? cvfixnum(start1) : v_false;
    case '>':	return start2 >= end2 && start1 < end1 ? cvfixnum(start1) : v_false;
    }
}

/* xstrsearch - built-in function 'string-search' */
LVAL xstrsearch(void)
{
    return strsearch(FALSE);
}

/* xstrisearch - built-in function 'string-search-ci' */
LVAL xstrisearch(void)
{
    return strsearch(TRUE);
}

/* strsearch - string search */
static LVAL strsearch(int icase)
{
    FIXTYPE start1,end1,start2,end2,last2,i;
    LVAL str1,str2;
    char *p1,*p2;
    int ch1,ch2;

    /* get the strings */
    str1 = xlgastring();
    str2 = xlgastring();

    /* get the substring specifiers */
    getbounds(str1,k_1start,k_1end,&start1,&end1);
    getbounds(str2,k_2start,k_2end,&start2,&end2);
    xlpopargs();

    /* compare the strings */
    for (last2 = end2 - end1 + start1; start2 <= last2; ++start2) {
	p1 = &getstring(str1)[start1];
	p2 = &getstring(str2)[start2];
        for (i = start1; i < end1; ++i) {
	    ch1 = *p1++;
	    ch2 = *p2++;
	    if (icase) {
	        if (isupper(ch1)) ch1 = tolower(ch1);
	        if (isupper(ch2)) ch2 = tolower(ch2);
	    }
	    if (ch1 != ch2)
	        break;
	}
	if (i >= end1)
	    return cvfixnum(start2);
    }

    /* check the termination condition */
    return v_false;
}

/* xchupcase - built-in function 'char-upcase' */
LVAL xchupcase(void)
{
    LVAL arg;
    int ch;
    arg = xlgachar(); ch = getchcode(arg);
    xllastarg();
    return islower(ch) ? cvchar(toupper(ch)) : arg;
}

/* xchdowncase - built-in function 'char-downcase' */
LVAL xchdowncase(void)
{
    LVAL arg;
    int ch;
    arg = xlgachar(); ch = getchcode(arg);
    xllastarg();
    return isupper(ch) ? cvchar(tolower(ch)) : arg;
}

/* xdigitchar - built-in function 'digit->char' */
LVAL xdigitchar(void)
{
    LVAL arg;
    int n;
    arg = xlgafixnum(); n = (int)getfixnum(arg);
    xllastarg();
    if (n >= 0 && n <= 9)
        return cvchar(n + '0');
    else if (n >= 10 && n <= 15)
        return cvchar(n + 'A' - 10);
    return v_nil;
}

/* xstring - return a string consisting of a single character */
LVAL xstring(void)
{
    char *buf;
    LVAL arg;

    /* return a zero length string if called with no arguments (for Scheme compatibility) */
    if (!moreargs())
        return xlnewstring(0);
    
    /* get the first argument */
    arg = xlgetarg();

    /* make sure its not NIL */
    if (null(arg))
	xlbadtype(arg);

    /* check the argument type */
    switch (ntype(arg)) {
    case STRING:
        xllastarg();
	return arg;
    case SYMBOL:
        xllastarg();
	return getpname(arg);
    case CHARACTER:
        xlval = xlnewstring(xlargc + 1);
	buf = getstring(xlval);
	*buf++ = getchcode(arg);
	while (moreargs())
	    *buf++ = getchcode(xlgachar());
	return xlval;
    default:
	xlbadtype(arg);
    }
}

/* xnumstr - built-in function 'number->string' */
LVAL xnumstr(void)
{
    char buf[256],*str;
    int radix = 10;
    LVAL n;
    
    /* parse argument list */
    n = xlgetarg();
    if (moreargs()) {
        xlval = xlgafixnum();
        radix = (int)getfixnum(xlval);
    }
    xllastarg();
    
    /* convert to a string */
    if (fixp(n))
        str = radixnumtostr(getfixnum(n),radix,buf,sizeof(buf));
    else if (floatp(n))
        { sprintf(buf,FFMT,getflonum(n)); str = buf; }
    else
        xlbadtype(n);
    return cvcstring(str);
}

/* radixnumtostr - convert a number to a string in a given radix */
static char *radixnumtostr(FIXTYPE n,int radix,char *buf,int len)
{
    char *p = &buf[len];
    unsigned long un;
    int sign;
    
    /* determine the sign */
    if (n < 0) {
        un = (unsigned long)-n;
        sign = -1;
    }
    else {
        un = (unsigned long)n;
        sign = 1;
    }
    
    /* insert the terminating nul */
    *--p = '\0';
    
    /* convert the number */
    do {
        *--p = "0123456789abcdefghijklmnopqrstuvwxyz"[un % radix];
        un /= radix;
    } while (un > 0);
    
    /* insert the sign */
    if (sign < 0)
        *--p = '-';
        
    /* return the string */
    return p;
}

/* xstrnum - built-in function 'string->number' */
LVAL xstrnum(void)
{
    int radix = 10;
    LVAL str;
    
    /* parse argument list */
    str = xlgastring();
    if (moreargs()) {
        xlval = xlgafixnum();
        radix = (int)getfixnum(xlval);
    }
    xllastarg();
    
    /* convert to a string */
    if (radix == 10)
        return isnumber(getstring(str),&str) ? str : v_nil;
    else
        return isradixnumber(getstring(str),radix,&str) ? str : v_nil;
}

/* xchar - extract a character from a string */
LVAL xchar(void)
{
    LVAL str,num;
    FIXTYPE n;

    /* get the string and the index */
    str = xlgastring();
    num = xlgafixnum();
    xllastarg();

    /* range check the index */
    if ((n = getfixnum(num)) < 0 || n >= getslength(str))
	xlerror("index out of range",num);

    /* return the character */
    return cvchar(getstring(str)[n]);
}

/* xcharint - built-in function 'char->integer' */
LVAL xcharint(void)
{
    LVAL arg;
    arg = xlgachar();
    xllastarg();
    return cvfixnum((FIXTYPE)getchcode(arg));
}

/* xintchar - built-in function 'integer->char' */
LVAL xintchar(void)
{
    LVAL arg;
    arg = xlgafixnum();
    xllastarg();
    return cvchar((int)getfixnum(arg));
}

/* character comparision functions */
LVAL xchrlss(void) { return chrcompare('<',FALSE); } /* char< */
LVAL xchrleq(void) { return chrcompare('L',FALSE); } /* char<= */
LVAL xchreql(void) { return chrcompare('=',FALSE); } /* char= */
LVAL xchrneq(void) { return chrcompare('#',FALSE); } /* char/= */
LVAL xchrgeq(void) { return chrcompare('G',FALSE); } /* char>= */
LVAL xchrgtr(void) { return chrcompare('>',FALSE); } /* char> */

/* character comparision functions (case insensitive) */
LVAL xchrilss(void) { return chrcompare('<',TRUE); } /* char-lessp */
LVAL xchrileq(void) { return chrcompare('L',TRUE); } /* char-not-greaterp */
LVAL xchrieql(void) { return chrcompare('=',TRUE); } /* char-equalp */
LVAL xchrineq(void) { return chrcompare('#',TRUE); } /* char-not-equalp */
LVAL xchrigeq(void) { return chrcompare('G',TRUE); } /* char-not-lessp */
LVAL xchrigtr(void) { return chrcompare('>',TRUE); } /* char-greaterp */

/* chrcompare - compare characters */
static LVAL chrcompare(int fcn,int icase)
{
    int ch1,ch2,icmp;
    LVAL arg;
    
    /* get the characters */
    arg = xlgachar(); ch1 = getchcode(arg);

    /* convert to lowercase if case insensitive */
    if (icase && isupper(ch1))
	ch1 = tolower(ch1);

    /* handle each remaining argument */
    for (icmp = TRUE; icmp && moreargs(); ch1 = ch2) {

	/* get the next argument */
	arg = xlgachar(); ch2 = getchcode(arg);

	/* convert to lowercase if case insensitive */
	if (icase && isupper(ch2))
	    ch2 = tolower(ch2);

	/* compare the characters */
	switch (fcn) {
	case '<':	icmp = (ch1 < ch2); break;
	case 'L':	icmp = (ch1 <= ch2); break;
	case '=':	icmp = (ch1 == ch2); break;
	case '#':	icmp = (ch1 != ch2); break;
	case 'G':	icmp = (ch1 >= ch2); break;
	case '>':	icmp = (ch1 > ch2); break;
	}
    }

    /* return the result */
    return icmp ? v_true : v_false;
}

/* xuppercasep - built-in function 'upper-case-p' */
LVAL xuppercasep(void)
{
    int ch;
    ch = getchcode(xlgachar());
    xllastarg();
    return isupper(ch) ? v_true : v_false;
}

/* xlowercasep - built-in function 'lower-case-p' */
LVAL xlowercasep(void)
{
    int ch;
    ch = getchcode(xlgachar());
    xllastarg();
    return islower(ch) ? v_true : v_false;
}

/* xbothcasep - built-in function 'both-case-p' */
LVAL xbothcasep(void)
{
    int ch;
    ch = getchcode(xlgachar());
    xllastarg();
    return isupper(ch) || islower(ch) ? v_true : v_false;
}

/* xdigitp - built-in function 'digit-char-p' */
LVAL xdigitp(void)
{
    int ch;
    ch = getchcode(xlgachar());
    xllastarg();
    return isdigit(ch) ? cvfixnum((FIXTYPE)(ch - '0')) : v_false;
}

/* xalphanumericp - built-in function 'alphanumericp' */
LVAL xalphanumericp(void)
{
    int ch;
    ch = getchcode(xlgachar());
    xllastarg();
    return isupper(ch) || islower(ch) || isdigit(ch) ? v_true : v_false;
}

/* xwhitespacep - built-in function 'char-whitespace?' */
LVAL xwhitespacep(void)
{
    int ch;
    ch = getchcode(xlgachar());
    xllastarg();
    return isspace(ch) ? v_true : v_false;
}

/* xcompile - built-in function 'compile' */
LVAL xcompile(void)
{
    LVAL env;

    /* get the expression to compile and the environment */
    xlval = xlgetarg();
    env = moreargs() ? xlgaenv() : v_nil;
    xllastarg();
    
    /* build the closure */
    cpush(env);
    xlval = xlcompile(xlval,top());
    xlval = cvclosure(xlval,top());
    drop(1);
    return xlval;
}

/* xdecompile - built-in function 'decompile' */
LVAL xdecompile(void)
{
    LVAL code,env,fptr;

    /* get the closure (or code and env) and file pointer */
    code = xlgetarg();
    if (closurep(code)) {
	env = getenvironment(code);
	code = getcode(code);
    }
    else if (codep(code))
	env = xlgaenv();
    else
	xlerror("expecting a closure or code and environment",code);
    fptr = moreargs() ? xlgetoport() : curoutput();
    xllastarg();

    /* decompile (disassemble) the procedure */
    decode_procedure(fptr,code,env);
    return v_nil;
}

/* xsave - save the memory image */
LVAL xsave(void)
{
    char *name;

    /* get the file name, verbose flag and print flag */
    name = getstring(xlgastring());
    xllastarg();

    /* save the memory image */
    return xlisave(name) ? v_true : v_false;
}

/* xrestore - restore a saved memory image */
LVAL xrestore(void)
{
    char *name;

    /* get the file name, verbose flag and print flag */
    name = getstring(xlgastring());
    xllastarg();

    /* restore the saved memory image */
    if (!xlirestore(name))
	return v_false;

    /* return directly to the top level */
    stdputstr("[ returning to the top level ]\n");
    xltoplevel();
    return v_nil; /* never reached */
}

/* xgc - function to force garbage collection */
LVAL xgc(void)
{
    extern FIXTYPE nssize,vssize;
    FIXTYPE arg1,arg2;
    LVAL arg;
    
    /* check the argument list and call the garbage collector */
    if (moreargs()) {
	arg = xlgafixnum(); arg1 = getfixnum(arg);
	arg = xlgafixnum(); arg2 = getfixnum(arg);
	xllastarg();
	while (--arg1 >= 0) nexpand(nssize);
	while (--arg2 >= 0) vexpand(vssize);
    }
    else
	gc();
    return room();
}

/* xroom - return the amount of memory currently available */
LVAL xroom(void)
{
    xllastarg();
    return room();
}

/* room - create a list containing memory allocation statistics */
static LVAL room(void)
{
    extern FIXTYPE nnodes,nfree,gccalls,total;
    extern int nscount,vscount;
    xlval = cons(cvfixnum(total),v_nil);
    xlval = cons(cvfixnum((FIXTYPE)vscount),xlval);
    xlval = cons(cvfixnum((FIXTYPE)nscount),xlval);
    xlval = cons(cvfixnum(nfree),xlval);
    xlval = cons(cvfixnum(nnodes),xlval);
    xlval = cons(cvfixnum(gccalls),xlval);
    return xlval;
}

/* xerror - built-in function 'error' */
LVAL xerror(void)
{
    /* display the error message */
    errputstr("\nerror: ");
    format(curerror());

    /* show the error context */
    xlshowerr(find_top_procedure());

    /* call the handler */
    callerrorhandler();
    return v_nil; /* never reached */
}

/* xgetarg - return a command line argument */
LVAL xgetarg(void)
{
    extern char **clargv;
    extern int clargc;
    LVAL arg;
    int n;
    arg = xlgafixnum(); n = (int)getfixnum(arg);
    xllastarg();
    return n >= 0 && n < clargc ? cvcstring(clargv[n]) : v_nil;
}

/* xshowstack - built-in function 'show-stack' */
void xshowstack(void)
{
    int levels = 20;
    if (moreargs()) {
	xlval = xlgafixnum();
        levels = (int)getfixnum(xlval);
    }
    xllastarg();
    show_call_stack(levels);
    mvreturn(0);
}

/* xshowcontrolstack - built-in function 'show-control-stack' */
void xshowcontrolstack(void)
{
    int levels = 20;
    if (moreargs()) {
	xlval = xlgafixnum();
        levels = (int)getfixnum(xlval);
    }
    xllastarg();
    show_control_stack(levels);
    mvreturn(0);
}

/* xshowvaluestack - built-in function 'show-value-stack' */
void xshowvaluestack(void)
{
    int levels = 20;
    if (moreargs()) {
	xlval = xlgafixnum();
        levels = (int)getfixnum(xlval);
    }
    xllastarg();
    show_value_stack(levels);
    mvreturn(0);
}

/* xgettime - get the current time */
LVAL xgettime(void)
{
    xllastarg();
    return (cvfixnum((FIXTYPE)ostime()));
}

/* xexit - exit to the operating system */
LVAL xexit(void)
{
    xllastarg();
    xlwrapup();
    return v_nil; /* never reached */
}

/* crecord field types */
#define CRTYPE_CHAR	1
#define CRTYPE_UCHAR	2
#define CRTYPE_SHORT	3
#define CRTYPE_USHORT	4
#define CRTYPE_INT	5
#define CRTYPE_UINT	6
#define CRTYPE_LONG	7
#define CRTYPE_ULONG	8
#define CRTYPE_PTR	9

/* xallocatecmemory - built-in function 'allocate-cmemory' */
LVAL xalloccmemory(void)
{
    LVAL type;
    FIXTYPE size;
    char *ptr;
    
    /* parse the argument list */
    type = xlgasymbol();
    xlval = xlgafixnum(); size = getfixnum(xlval);
    xllastarg();
    
    /* allocate the memory and create the foreign pointer */
    ptr = osalloc(size);
    return ptr ? cvforeignptr(type,ptr) : v_nil;
}

/* xfreecmemory - built-in function 'free-cmemory' */
LVAL xfreecmemory(void)
{
    void *ptr;
    
    /* parse the argument list */
    xlval = xlgaforeignptr(); ptr = getfptr(xlval);
    xllastarg();
    
    /* free the pointer */
    if (ptr) osfree(ptr);
    setfptr(xlval,0);
    return v_nil;
}

/* xforeignptrp - built-in function 'foreign-pointer?' */
LVAL xforeignptrp(void)
{
    LVAL ptr,type;
    
    /* get the pointer */
    ptr = xlgetarg();
    type = moreargs() ? xlgetarg() : v_nil;
    xllastarg();
    
    /* return its type */
    if (!foreignptrp(ptr))
        return v_false;
    else if (type == v_nil)
        return v_true;
    else
        return getfptype(ptr) == type ? v_true : v_false;
}

/* xforeignptrtype - built-in function 'foreign-pointer-type' */
LVAL xforeignptrtype(void)
{
    LVAL ptr;
    
    /* get the pointer */
    ptr = xlgaforeignptr();
    xllastarg();
    
    /* return its type */
    return getfptype(ptr);
}

/* xforeignptreqp - built-in function 'foreign-pointer-eq?' */
LVAL xforeignptreqp(void)
{
    LVAL ptr1,ptr2;
    
    /* get the pointer */
    ptr1 = xlgaforeignptr();
    ptr2 = xlgaforeignptr();
    xllastarg();
    
    /* check the pointers */
    return getfptr(ptr1) == getfptr(ptr2) ? v_true : v_false;
}

/* xsetforeignptrtype - built-in function 'set-foreign-pointer-type!' */
LVAL xsetforeignptrtype(void)
{
    LVAL ptr,type;
    
    /* get the pointer and new type */
    ptr = xlgaforeignptr();
    type = xlgetarg();
    xllastarg();
    
    /* set the type and return the pointer */
    setfptype(ptr,type);
    return ptr;
}

/* xforeignptrtypep - built-in function 'foreign-pointer-type?' */
LVAL xforeignptrtypep(void)
{
    LVAL ptr,type;
    
    /* get the pointer and type */
    ptr = xlgaforeignptr();
    type = xlgetarg();
    xllastarg();
    
    /* check the type */
    return getfptype(ptr) == type ? v_true : v_false;
}

/* xgetcrecfield - built-in function 'get-crecord-field' */
LVAL xgetcrecfield(void)
{
    LVAL record;
    long offset;
    void *ptr;
    int type;
    
    /* parse argument list */
    record = xlgaforeignptr();
    xlval = xlgafixnum(); offset = getfixnum(xlval);
    xlval = xlgafixnum(); type = (int)getfixnum(xlval);
    xllastarg();
    
    /* get the field pointer */
    if ((ptr = getfptr(record)) == 0)
    	xlerror("pointer is null",record);
    ptr = (void *)((char *)ptr + offset);
    
    /* dispatch on field type */
    switch (type) {
    case CRTYPE_CHAR:
    	{   char ival = *(char *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_UCHAR:
    	{   unsigned char ival = *(unsigned char *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_SHORT:
    	{   short ival = *(short *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_USHORT:
    	{   unsigned short ival = *(unsigned short *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_INT:
    	{   int ival = *(int *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_UINT:
    	{   unsigned int ival = *(unsigned int *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_LONG:
    	{   long ival = *(long *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_ULONG:
    	{   unsigned long ival = *(unsigned long *)ptr;
    	    xlval = cvfixnum((FIXTYPE)ival);
    	    break;
    	}
    case CRTYPE_PTR:
    	{   void *ival = *(void **)ptr;
    	    xlval = cvforeignptr(v_nil,ival);
    	    break;
    	}
    default:
	xlerror("bad type code",xlval);
    	break;
    }
    return xlval;
}

/* xgetcrecfieldaddr - built-in function 'get-crecord-field-address' */
LVAL xgetcrecfieldaddr(void)
{
    LVAL record,type;
    long offset;
    void *ptr;
    
    /* parse argument list */
    record = xlgaforeignptr();
    xlval = xlgafixnum(); offset = getfixnum(xlval);
    type = xlgasymbol();
    xllastarg();
    
    /* get the field pointer */
    if ((ptr = getfptr(record)) == 0)
    	xlerror("pointer is null",record);
    ptr = (void *)((char *)ptr + offset);
    
    /* make a pointer to the field */
    return cvforeignptr(type,ptr);
}

/* xsetcrecfield - built-in function 'set-crecord-field!' */
LVAL xsetcrecfield(void)
{
    LVAL record,value;
    long offset;
    void *ptr;
    int type;
    
    /* parse argument list */
    record = xlgaforeignptr();
    xlval = xlgafixnum(); offset = getfixnum(xlval);
    xlval = xlgafixnum(); type = (int)getfixnum(xlval);
    value = xlgetarg();
    xllastarg();
    
    /* get the field pointer */
    if ((ptr = getfptr(record)) == 0)
    	xlerror("pointer is null",record);
    ptr = (void *)((char *)ptr + offset);
    
    /* dispatch on field type */
    switch (type) {
    case CRTYPE_CHAR:
    case CRTYPE_UCHAR:
    	{   if (!fixp(value))
    		xlerror("expecting a fixnum",value);
    	    *(char *)ptr = (char)getfixnum(value);
    	    break;
    	}
    case CRTYPE_SHORT:
    case CRTYPE_USHORT:
    	{   if (!fixp(value))
    		xlerror("expecting a fixnum",value);
    	    *(short *)ptr = (short)getfixnum(value);
    	    break;
    	}
    case CRTYPE_INT:
    case CRTYPE_UINT:
    	{   if (!fixp(value))
    		xlerror("expecting a fixnum",value);
    	    *(int *)ptr = (int)getfixnum(value);
    	    break;
    	}
    case CRTYPE_LONG:
    case CRTYPE_ULONG:
   	{   if (!fixp(value))
    		xlerror("expecting a fixnum",value);
    	    *(long *)ptr = (long)getfixnum(value);
    	    break;
    	}
    case CRTYPE_PTR:
    	{   if (!foreignptrp(value))
    		xlerror("expecting a foreign pointer",value);
    	    *(void **)ptr = getfptr(value);
    	    break;
    	}
    default:
	xlerror("bad type code",xlval);
    	break;
    }
    return value;
} 

/* xgetcrecstring - built-in function 'get-crecord-string' */
LVAL xgetcrecstring(void)
{
    FIXTYPE offset,length;
    char *src,*dst;
    LVAL val;
    
    /* parse argument list */
    xlval = xlgaforeignptr();
    val = xlgafixnum(); offset = getfixnum(val);
    val = xlgafixnum(); length = getfixnum(val);
    xllastarg();
    
    /* get the field pointer */
    if ((src = (char *)getfptr(xlval)) == 0)
    	xlerror("pointer is null",xlval);
    src += offset;
    
    /* make a string */
    val = xlnewstring(length);
    
    /* copy the value */
    for (dst = getstring(val); --length >= 0; )
        *dst++ = *src++;
    
    /* return the string */
    return val;
}
    
/* xsetcrecstring - built-in function 'set-crecord-string!' */
LVAL xsetcrecstring(void)
{
    FIXTYPE offset,length,cnt;
    LVAL record,str;
    char *src,*dst;
    
    /* parse argument list */
    record = xlgaforeignptr();
    xlval = xlgafixnum(); offset = getfixnum(xlval);
    xlval = xlgafixnum(); length = getfixnum(xlval);
    str = xlgastring();
    xllastarg();
    
    /* get the field pointer */
    if ((dst = (char *)getfptr(record)) == 0)
    	xlerror("pointer is null",record);
    dst += offset;
    
    /* compute the number of bytes to copy */
    if ((cnt = getslength(str)) > length)
        cnt = length;
    
    /* copy the value */
    for (src = getstring(str); --cnt >= 0; --length)
        *dst++ = *src++;

    /* fill the rest of the destination with nulls */
    while (--length >= 0)
        *dst++ = '\0';
        
    /* return the string */
    return str;
}
    
/* xgetcrectypesize - built-in function 'get-crecord-type-size' */
LVAL xgetcrectypesize(void)
{
    size_t size;
    int type;
    
    /* parse argument list */
    xlval = xlgafixnum(); type = (int)getfixnum(xlval);
    xllastarg();
    
    /* dispatch on field type */
    switch (type) {
    case CRTYPE_CHAR:
    case CRTYPE_UCHAR:
    	size = sizeof(char);
    	break;
    case CRTYPE_SHORT:
    case CRTYPE_USHORT:
    	size = sizeof(short);
    	break;
    case CRTYPE_INT:
    case CRTYPE_UINT:
    	size = sizeof(int);
    	break;
    case CRTYPE_LONG:
    case CRTYPE_ULONG:
    	size = sizeof(long);
    	break;
    case CRTYPE_PTR:
    	size = sizeof(void *);
    	break;
    default:
	xlerror("bad type code",xlval);
    	break;
    }
    return cvfixnum(size);
}

/* xnullpointerp - built-in function 'null-pointer?' */
LVAL xnullpointerp(void)
{
    LVAL record = xlgaforeignptr();
    xllastarg();
    return getfptr(record) == 0 ? v_true : v_false;
}
