/* xlfun3.c - xlisp built-in functions - part 3 */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* external variables */
extern LVAL xlfun,xlenv,xlval;
extern LVAL s_package;

/* forward declarations */
static char *showstring(char *str,int bch);
static void withfile_continuation(void);
static void withfile_unwind(void);
static void do_withfile(short flags,char *mode);
static void do_load(LVAL print);
static FILE *load_open(LVAL name,char *mode,LVAL pathsym,char *rpath);
static void load_continuation(void);
static void load_unwind(void);
static void do_loadloop(LVAL print,LVAL oldpack);
static void force_continuation(void);
static void pushccontinuation(CCONTINUATION *);

/* xapply - built-in function 'apply' */
void xapply(void)
{
    LVAL args,*p;

    /* get the function and argument list */
    xlval = xlgetarg();
    args = xlgalist();
    xllastarg();

    /* get the argument count and make space on the stack */
    xlargc = (int)length(args);
    check(xlargc);

    /* copy the arguments onto the stack */
    for (xlsp -= xlargc, p = xlsp; consp(args); args = cdr(args))
	*p++ = car(args);

    /* apply the function to the arguments */
    xlnext = xlapply;
}

/* xvalues - return multiple values */
void xvalues(void)
{
    xlval = xlargc > 0 ? top() : v_nil;
    drop(xlargc);
    cdrestore();
}

/* xvalueslist - return a list as multiple values */
void xvalueslist(void)
{
    LVAL p;
    xlval = p = xlgalist();
    xllastarg();
    xlargc = (int)length(xlval);
    check(xlargc);
    for (xlsp -= xlargc; consp(p); p = cdr(p))
	*xlsp++ = car(p);
    xlval = xlval ? car(xlval) : v_nil;
    cdrestore();
}

/* xcallcc - built-in function 'call-with-current-continuation' */
void xcallcc(void)
{
    LVAL make_continuation();

    /* get the function to call */
    xlval = xlgetarg();
    xllastarg();

    /* create a continuation object */
    cpush(make_continuation());
    xlargc = 1;

    /* apply the function */
    xlnext = xlapply;
}

/* ccode_restore - restore a C continuation */
static void ccode_restore(void)
{
    CCONTINUATION *cc;
    cc = (CCONTINUATION *)Cpop();
    (*cc->cc_cont)();
}

/* ccode_mark - mark a C continuation */
static LVAL *ccode_mark(LVAL *p)
{
    CCONTINUATION *cc;
    int cnt;
    cc = (CCONTINUATION *)*--p;
    for (cnt = cc->cc_cnt; --cnt >= 0; )
	mark(*--p);
    return p;
}

/* ccode_skipit - skip a C continuation */
static LVAL *ccode_skipit(LVAL *p)
{
    CCONTINUATION *cc;
    cc = (CCONTINUATION *)*--p;
    return p - cc->cc_cnt;
}

/* ccode_unwind - unwind past a C continuation (just skip over it) */
static void ccode_unwind(void)
{
    CCONTINUATION *cc;
    cc = (CCONTINUATION *)Cpop();
    if (cc->cc_unwind)
	(*cc->cc_unwind)();
    else
	Cdrop(cc->cc_cnt);
}

/* ccode_print - print a C continuation */
static LVAL *ccode_print(LVAL *p)
{
    CCONTINUATION *cc;
    char *str;
    int cnt;
    cc = (CCONTINUATION *)*--p;
    errputstr("\n ");
    str = showstring(cc->cc_names,':');
    for (cnt = cc->cc_cnt; --cnt >= 0; ) {
	errputstr("\n  ");
	str = showstring(str,',');
	errputstr(": ");
	errprint(*--p);
    }
    return p;
}

/* showstring - show the next string up to a given delimiter */
static char *showstring(char *str,int bch)
{
    char buf[20],*p;
    int ch;
    for (p = buf; *str != '\0' && (ch = *str++) != bch; )
	*p++ = ch;
    *p = '\0';
    errputstr(buf);
    return str;
}

/* C continuation dispatch table */
static CDISPATCH cd_ccode = {
    ccode_restore,
    ccode_mark,
    ccode_skipit,
    ccode_unwind,
    ccode_skipit,
    ccode_print
};

/* pushccontinuation - push a C continuation on the control stack */
static void pushccontinuation(CCONTINUATION *cc)
{
    Ccheck(2);
    Cpush((LVAL)cc);
    Cpush((LVAL)&cd_ccode);
}

/* xcallwi - built-in function 'call-with-input-file' */
void xcallwi(void)
{
    do_withfile(PF_INPUT,"r");
}

/* xcallwo - built-in function 'call-with-output-file' */
void xcallwo(void)
{
    do_withfile(PF_OUTPUT | PF_BOL,"w");
}

/* withfile_continuation - withfile continuation */
static void withfile_continuation(void)
{
    LVAL file;
    file = Cpop();
    osclose(getfile(file));
    setsdata(file,NULL);
    svreturn();
}

/* withfile_unwind - withfile unwind handler */
static void withfile_unwind(void)
{
    xlval = Cpop();
    osclose(getfile(xlval));
}

/* withfile continuation structure */
CCONTINUATION withfile_cc = { withfile_continuation,withfile_unwind,1,"Withfile:file" };

/* do_withfile - handle the 'call-with-xxx-file' functions */
static void do_withfile(short flags,char *mode)
{
    extern LVAL cs_withfile1;
    LVAL name,file;
    FILE *fp;

    /* get the function to call */
    name = xlgastring();
    xlval = xlgetarg();
    xllastarg();

    /* create a file object */
    file = cvfstream(NULL,flags);
    if ((fp = osaopen(getstring(name),mode)) == NULL)
	xlerror("can't open file",name);
    setsdata(file,fp);

    /* save a continuation */
    Ccheck(1);
    Cpush(file);
    pushccontinuation(&withfile_cc);

    /* setup the argument list */
    cpush(file);
    xlargc = 1;

    /* apply the function */
    xlnext = xlapply;
}

/* xload - built-in function 'load' */
void xload(void)
{
    do_load(v_false);
}

/* xloadnoisily - built-in function 'load-noisily' */
void xloadnoisily(void)
{
    do_load(v_true);
}

/* do_load - open the file and setup the load loop */
static void do_load(LVAL print)
{
    FILE *fp;

    /* get the function to call */
    xlval = xlgastring();
    xllastarg();

    /* create a file object */
    cpush(cvfstream(NULL,PF_INPUT));
    if ((fp = load_open(xlval,"r",xlenter("*LOAD-PATH*"),NULL)) == NULL) {
        drop(1);
	xlval = v_nil;
	svreturn();
	return;
    }
    xlval = pop();
    setsdata(xlval,fp);

    /* do the first read */
    do_loadloop(print,getvalue(s_package));
}

/* xlLoadOpen - open a file for loading */
FILE *xlLoadOpen(char *name,char *mode,char *pathsym,char *rpath)
{
    FILE *fp;
    cpush(cvcstring(name));
    fp = load_open(top(),mode,xlenter(pathsym),rpath);
    drop(1);
    return fp;
}

/* load_open - open a file for loading */
static FILE *load_open(LVAL name,char *mode,LVAL pathsym,char *rpath)
{
    LVAL dir,fullpath,path;
    FIXTYPE dirlen,namelen;
    char *pathstr;
    FILE *fp;

    /* first try just opening the file with no additional path information */
    if ((fp = osaopen(getstring(name),mode)) != NULL) {
        if (rpath)
            strcpy(rpath,getstring(name));
	return fp;
    }

    /* initialize */
    path = getvalue(pathsym);
    namelen = getslength(name);
    check(2);

    /* try each directory in the load path */
    for (; consp(path); path = cdr(path)) {

	/* try the next directory */
	dir = car(path);
	if (stringp(dir)) {

	    /* build a full path to the file */
	    push(path); push(name);
	    dirlen = getslength(dir);
	    fullpath = xlnewstring(dirlen + namelen);
	    pathstr = getstring(fullpath);
	    memcpy(pathstr,getstring(dir),(size_t)dirlen);
	    memcpy(pathstr + dirlen,getstring(name),(size_t)namelen);
	    drop(2);
	    
	    /* attempt to open the file */
	    if ((fp = osaopen(getstring(fullpath),mode)) != NULL) {
                if (rpath)
                    strcpy(rpath,getstring(fullpath));
		return fp;
            }
	}
    }
    return NULL;
}

/* load_continuation - load continuation */
static void load_continuation(void)
{
    LVAL print;
    check(2);
    push(Cpop());
    xlenv = Cpop();
    push(Cpop());
    if ((print = Cpop()) != v_false) {
	xlterpri(curoutput());
	xlprin1(xlval,curoutput());
    }
    xlval = pop();
    do_loadloop(print,pop());
}

/* load_unwind - load unwind handler */
static void load_unwind(void)
{
    setvalue(s_package,Cpop());
    Cdrop(1);
    xlval = Cpop();
    osclose(getfile(xlval));
    Cdrop(1);
}

/* load continuation structure */
CCONTINUATION load_cc = { load_continuation,load_unwind,4,"Load:package,env,file,print" };

/* do_loadloop - read the next expression and setup to evaluate it */
static void do_loadloop(LVAL print,LVAL oldpack)
{
    extern LVAL cs_load1,s_eval;
    LVAL expr;
    
    /* try to read the next expression from the file */
    check(3);
    push(oldpack);
    push(print);
    if (xlread(xlval,&expr)) {

	/* save a continuation */
	push(expr);
	xlenv = unstack_environment(xlenv);
	expr = pop();
	Ccheck(4);
	Cpush(pop());
	Cpush(xlval);
	Cpush(xlenv);
	Cpush(pop());
	pushccontinuation(&load_cc);

	/* setup the argument list */
	xlval = getvalue(s_eval);
	cpush(expr);
	xlargc = 1;

	/* apply the function */
	xlnext = xlapply;
    }
    else {
	drop(1);
	setvalue(s_package,pop());
	osclose(getfile(xlval));
	setsdata(xlval,NULL);
	xlval = v_true;
	svreturn();
    }
}

/* force_continuation - force continuation */
static void force_continuation(void)
{
    LVAL promise;
    promise = Cpop();
    if (getpproc(promise)) {
        setpvalue(promise,xlval);
        setpproc(promise,v_nil);
    }
    else
        xlval = getpvalue(promise);
    svreturn();
}

/* force continuation structure */
CCONTINUATION force_cc = { force_continuation,NULL,1,"Force:promise" };

/* xforce - built-in function 'force' */
void xforce(void)
{
    extern LVAL cs_force1;

    /* get the promise */
    xlval = xlgetarg();
    xllastarg();

    /* check for a promise */
    if (promisep(xlval)) {

	/* force the promise the first time */
	if ((xlfun = getpproc(xlval)) != v_nil) {
	    Ccheck(1);
	    Cpush(xlval);
	    pushccontinuation(&force_cc);
	    xlval = xlfun;
	    xlargc = 0;
	    xlnext = xlapply;
	}

	/* return the saved value if the promise has already been forced */
	else {
	    xlval = getpvalue(xlval);
	    svreturn();
	}
	
    }
    
    /* otherwise, just return the argument */
    else
	svreturn();
}
