/* xlimage.c - xlisp memory image save/restore functions */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* node space */
extern FIXTYPE nssize;		/* node segment size */
extern NSEGMENT *nsegments;	/* list of node segments */

/* vector (and string) space */
extern VSEGMENT *vsegments;	/* list of vector segments */
extern LVAL *vfree;		/* next free location in vector space */
extern LVAL *vtop;		/* top of vector space */

/* global variables */
extern LVAL schemepackage,xlisppackage,keywordpackage,systempackage;
extern LVAL packages,eof_object,default_object;

/* local variables */
static OFFTYPE off,foff;
static FILE *fp;

/* local prototypes */
static LVAL requirepackage(char *name);
static void freeimage(void);
static void setoffset(void);
static void writenode(LVAL node);
static void writeptr(OFFTYPE off);
static void readnode(int type,LVAL node);
static OFFTYPE readptr(void);
static LVAL cviptr(OFFTYPE o);
static OFFTYPE cvoptr(LVAL p);
static LVAL *getvspace(LVAL node,FIXTYPE size);

/* xlisave - save the memory image */
int xlisave(char *fname)
{
    NSEGMENT *nseg;
    FIXTYPE size,n;
    LVAL p,*vp;
    char *cp;

    /* open the output file */
    if ((fp = osbopen(fname,"w")) == NULL)
	return FALSE;

    /* first call the garbage collector to clean up memory */
    gc();

    /* write out the stack size */
    writeptr((OFFTYPE)(xlstktop-xlstkbase-1));

    /* write out the package list and various constants */
    writeptr(cvoptr(packages));
    writeptr(cvoptr(eof_object));
    writeptr(cvoptr(default_object));

    /* setup the initial file offsets */
    off = foff = (OFFTYPE)2;

    /* write out all nodes that are still in use */
    for (nseg = nsegments; nseg != NULL; nseg = nseg->ns_next) {
	p = &nseg->ns_data[0];
	n = nseg->ns_size;
	for (; --n >= 0; ++p, off += sizeof(NODE))
	    switch (ntype(p)) {
	    case FREE:
		break;
	    case CONS:
	    case CLOSURE:
	    case PROMISE:
		setoffset();
		putc(p->n_type,fp);
		writeptr(cvoptr(car(p)));
		writeptr(cvoptr(cdr(p)));
		foff += sizeof(NODE);
		break;
	    case SYMBOL:
	    case VECTOR:
	    case CODE:
	    case CONTINUATION:
	    case PACKAGE:
	    case ENV:
	    case MENV:
	    case OBJECT:
            case TABLE:
		setoffset();
		putc(p->n_type,fp);
		size = getsize(p);
		writeptr((OFFTYPE)size);
		for (vp = p->n_vdata; --size >= 0; )
		    writeptr(cvoptr(*vp++));
		foff += sizeof(NODE);
		break;
	    case STRING:
		setoffset();
		putc(p->n_type,fp);
		size = getslength(p);
		writeptr((OFFTYPE)size);
		for (cp = getstring(p); --size >= 0; )
		    putc(*cp++,fp);
		foff += sizeof(NODE);
		break;
	    case SUBR:
	    case XSUBR:
		setoffset();
		putc(p->n_type,fp);
		cp = getfundef(p)->fd_name;
		do {
		    putc(*cp,fp);
		} while (*cp++ != '\0');
		foff += sizeof(NODE);
		break;
	    case FOREIGNPTR:
		setoffset();
		putc(p->n_type,fp);
		writeptr(cvoptr(getfptype(p)));
		foff += sizeof(NODE);
		break;
	    case USTREAM:
	    case OSTREAM:
		setoffset();
		putc(p->n_type,fp);
		writeptr((OFFTYPE)getpflags(p));
		writeptr((OFFTYPE)getsavech(p));
		writeptr(cvoptr((LVAL)getsdata(p)));
	    	break;
	    default:
		setoffset();
		writenode(p);
		foff += sizeof(NODE);
		break;
	    }
    }

    /* write the terminator */
    putc(FREE,fp);
    writeptr((OFFTYPE)0);

    /* close the output file */
    osclose(fp);

    /* return successfully */
    return TRUE;
}

/* xlirestore - restore a saved memory image */
int xlirestore(char *fname)
{
    FIXTYPE ssize,size;
    LVAL p,*vp;
    char *cp;
    int type;

    /* open the file */
    if ((fp = osbopen(fname,"r")) == NULL)
	return FALSE;

    /* free the old memory image */
    freeimage();

    /* read the stack size */
    ssize = readptr();

    /* allocate memory for the workspace */
    xlminit(ssize);

    /* read the package list and various constants */
    packages = cviptr(readptr());
    eof_object = cviptr(readptr());
    default_object = cviptr(readptr());
    
    /* read each node */
    for (off = (OFFTYPE)2; (type = getc(fp)) >= 0; )
	switch (type) {
	case FREE:
	    if ((off = readptr()) == (OFFTYPE)0)
		goto done;
	    break;
	case CONS:
	case CLOSURE:
	case PROMISE:
	    p = cviptr(off);
	    p->n_type = type;
	    rplaca(p,cviptr(readptr()));
	    rplacd(p,cviptr(readptr()));
	    off += sizeof(NODE);
	    break;
	case SYMBOL:
	case VECTOR:
	case CODE:
	case CONTINUATION:
	case PACKAGE:
	case ENV:
	case MENV:
	case OBJECT:
        case TABLE:
	    p = cviptr(off);
	    p->n_type = type;
	    p->n_vsize = size = readptr();
	    p->n_vdata = getvspace(p,size);
	    for (vp = p->n_vdata; --size >= 0; )
		*vp++ = cviptr(readptr());
	    off += sizeof(NODE);
	    break;
	case STRING:
	    p = cviptr(off);
	    p->n_type = type;
	    p->n_vsize = size = readptr();
	    p->n_vdata = getvspace(p,btow_size(size));
	    for (cp = getstring(p); --size >= 0; )
		*cp++ = getc(fp);
	    off += sizeof(NODE);
	    break;
	case FSTREAM:
	    p = cviptr(off);
	    readnode(type,p);
	    setsdata(p,NULL);
	    off += sizeof(NODE);
	    break;
	case SUBR:
	    {	char name[100],*cp;
	    	FUNDEF *def;
	    	int type;
	    	p = cviptr(off);
	    	p->n_type = type;
	    	for (cp = name; (*cp++ = getc(fp)) != '\0'; )
	   	    ;
	   	if ((def = osfindsubr(name,&type)) == NULL
	   	&&  (def = xlfindsubr(name,&type)) == NULL)
	   	    xlfatal("no definition for subr: %s",name);
	    	p->n_type = type;
	    	p->n_subr = def->fd_subr;
	    	p->n_fundef = def;
	    	off += sizeof(NODE);
	    }
	    break;
	case XSUBR:
	    {	char name[100],*cp;
	    	FUNDEF *def;
	    	int type;
	    	p = cviptr(off);
	    	p->n_type = type;
	    	for (cp = name; (*cp++ = getc(fp)) != '\0'; )
	   	    ;
	   	if ((def = osfindsubr(name,&type)) == NULL
	   	&&  (def = xlfindsubr(name,&type)) == NULL)
	   	    xlfatal("no definition for xsubr: %s",name);
	   	p->n_type = type;
	    	p->n_subr = def->fd_subr;
	    	p->n_fundef = def;
	    	off += sizeof(NODE);
	    }
	    break;
	case FOREIGNPTR:
	    p = cviptr(off);
	    p->n_type = type;
	    setfptype(p,cviptr(readptr()));
	    setfptr(p,NULL);
	    off += sizeof(NODE);
	    break;
	case USTREAM:
	case OSTREAM:
	    p = cviptr(off);
	    p->n_type = type;
	    setpflags(p,(short)readptr());
	    setsavech(p,(short)readptr());
	    setsdata(p,cviptr(readptr()));
	    break;
	default:
	    readnode(type,cviptr(off));
	    off += sizeof(NODE);
	    break;
	}
done:

    /* close the input file */
    osclose(fp);

    /* collect to initialize the free space */
    gc();

    /* lookup the packages */
    xlisppackage   = requirepackage("XLISP");
    schemepackage  = requirepackage("SCHEME");
    keywordpackage = requirepackage("KEYWORD");
    systempackage  = requirepackage("SYSTEM");

    /* lookup all of the symbols the interpreter uses */
    xlsymbols();

    /* return successfully */
    return TRUE;
}

/* requirepackage - require that a package exist */
static LVAL requirepackage(char *name)
{
    LVAL pack = findpackage(name);
    if (pack == v_nil)
        xlfatal("missing package: %s",name);
    return pack;
}

/* freeimage - free the current memory image */
static void freeimage(void)
{
    NSEGMENT *nextnseg;
    VSEGMENT *nextvseg;
    FIXTYPE n;
    FILE *fp;
    LVAL p;

    /* close all open ports and free each node segment */
    for (; nsegments != NULL; nsegments = nextnseg) {
	nextnseg = nsegments->ns_next;
	p = &nsegments->ns_data[0];
	n = nsegments->ns_size;
	for (; --n >= 0; ++p)
	    switch (ntype(p)) {
	    case FSTREAM:
		if ((fp = getfile(p)) != NULL
		&&  (getpflags(p) & PF_TERMINAL) == 0)
		    osclose(fp);
		break;
	    case FOREIGNPTR:
	        break;
	    }
	osfree(nsegments);
    }

    /* free each vector segment */
    for (; vsegments != NULL; vsegments = nextvseg) {
	nextvseg = vsegments->vs_next;
	osfree(vsegments);
    }
    
    /* free the stack */
    if (xlstkbase)
	osfree(xlstkbase);
}

/* setoffset - output a positioning command if nodes have been skipped */
static void setoffset(void)
{
    if (off != foff) {
	putc(FREE,fp);
	writeptr(off);
	foff = off;
    }
}

/* writenode - write a node to a file */
static void writenode(LVAL node)
{
    char *p = (char *)&node->n_info;
    int n = sizeof(union ninfo);
    putc(node->n_type,fp);
    while (--n >= 0)
	putc(*p++,fp);
}

/* writeptr - write a pointer to a file */
static void writeptr(OFFTYPE off)
{
    char *p = (char *)&off;
    int n = sizeof(OFFTYPE);
    while (--n >= 0)
	putc(*p++,fp);
}

/* readnode - read a node */
static void readnode(int type,LVAL node)
{
    char *p = (char *)&node->n_info;
    int n = sizeof(union ninfo);
    node->n_type = type;
    while (--n >= 0)
	*p++ = getc(fp);
}

/* readptr - read a pointer */
static OFFTYPE readptr(void)
{
    OFFTYPE off;
    char *p = (char *)&off;
    int n = sizeof(OFFTYPE);
    while (--n >= 0)
	*p++ = getc(fp);
    return off;
}

/* cviptr - convert a pointer on input */
static LVAL cviptr(OFFTYPE o)
{
    OFFTYPE off = (OFFTYPE)2;
    OFFTYPE nextoff;
    NSEGMENT *nseg;

    /* check for nil and small fixnums */
    if (o == (OFFTYPE)0 || (o & 1) == 1)
	return (LVAL)o;

    /* compute a pointer for this offset */
    for (nseg = nsegments; nseg != NULL; nseg = nseg->ns_next) {
	nextoff = off + (OFFTYPE)(nseg->ns_size * sizeof(NODE));
	if (o >= off && o < nextoff)
	    return (LVAL)((OFFTYPE)&nseg->ns_data[0] + o - off);
	off = nextoff;
    }

    /* create new segments if necessary */
    for (;;) {

	/* create the next segment */
	if ((nseg = newnsegment(nssize)) == NULL)
	    xlfatal("insufficient memory - segment");

	/* check to see if the offset is in this segment */
	nextoff = off + (OFFTYPE)(nseg->ns_size * sizeof(NODE));
	if (o >= off && o < nextoff)
	    return (LVAL)((OFFTYPE)&nseg->ns_data[0] + o - off);
	off = nextoff;
    }
}

/* cvoptr - convert a pointer on output */
static OFFTYPE cvoptr(LVAL p)
{
    OFFTYPE off = (OFFTYPE)2;
    NSEGMENT *nseg;

    /* check for nil and small fixnums */
    if (!ispointer(p))
	return (OFFTYPE)p;

    /* compute an offset for this pointer */
    for (nseg = nsegments; nseg != NULL; nseg = nseg->ns_next) {
	if (INSEGMENT(p,nseg))
	    return off + ((OFFTYPE)p - (OFFTYPE)&nseg->ns_data[0]);
	off += (OFFTYPE)(nseg->ns_size * sizeof(NODE));
    }

    /* pointer not within any segment */
    xlerror("bad pointer found during image save",p);
    return (OFFTYPE)0; /* never reached */
}

/* getvspace - allocate vector space */
static LVAL *getvspace(LVAL node,FIXTYPE size)
{
    LVAL *p;
    ++size; /* space for the back pointer */
    if (!VCOMPARE(vfree,size,vtop)
    &&  !checkvmemory(size)
    &&  !makevmemory(size))
	xlfatal("insufficient vector space");
    p = vfree;
    vfree += size;
    *p++ = node;
    return p;
}

/* fpdummy_type - dummy foreign pointer type function */
static int fpdummy_type(void *p)
{
    return 0;
}

/* fpdummy_free - dummy foreign pointer free function */
static void fpdummy_free(void *p)
{
}

/* fpdummy_print - dummy foreign pointer print function */
static void fpdummy_print(LVAL val,char *buf)
{
    strcpy(buf,"#<Null Pointer #");
    sprintf(&buf[strlen(buf)],AFMT,val);
    strcat(buf,">");
}

