/* xlint.c - xlisp bytecode interpreter */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"
#include "xlbcode.h"

/* macro to call a SUBR */
#define callsubr(x,c)	(xlargc = (c), (x)())

/* globals */
ERRORTARGET *xlerrtarget=NULL;	/* error target */
LVAL *xlcatch = NULL;		/* catch frame pointer */
int trace=FALSE;		/* trace enable */
int xlargc;			/* number of arguments remaining */
void (*xlnext)(void);		/* next function to call (xlapply or NULL) */
int hacking = FALSE;		/* control debug hacking */

/* external variables */
extern LVAL s_package,s_stdin,s_stdout,s_unbound;
extern LVAL s_unassigned,default_object,s_error;

/* error target (and bytecode dispatch target) */
#define BCD_START	0	/* must be zero */
#define BCD_RETURN	1
#define BCD_NEXT	2
#define BCD_ABORT	3

/* local variables */
static unsigned char *base,*pc;
static LVAL *xltarget;		/* current throw target */
static int sample=SRATE;

/* local prototypes */
static void show_call(LVAL code,LVAL frame);
static void throwtotag(LVAL tag);
static LVAL *findmatchingcatch(LVAL tag);
static void throwtoreturn(void);
static void throwtotarget(LVAL *target);
static void call(void),tcall(void);
static void restore_continuation(void);
static void stkframe(int,int,LVAL);
static void badfuntype(LVAL arg);
static void badargtype(LVAL arg);

/* xtraceon - built-in function 'trace-on' */
LVAL xtraceon(void)
{
    xllastarg()
    trace = TRUE;
    return v_nil;
}

/* xtraceoff - built-in function 'trace-off' */
LVAL xtraceoff(void)
{
    xllastarg()
    trace = FALSE;
    return v_nil;
}

/* code_restore - restore a code continuation */
static void code_restore(void)
{
    xlfun = Cpop();
    base = getcodestr(xlfun);
    pc = base + (FIXTYPE)Cpop();
    xlenv = Cpop();
    if (msenvp(xlenv))
	xlenv = getforwardingaddr(xlenv);
}

/* code_mark - mark a code continuation */
static LVAL *code_mark(LVAL *p)
{
    mark(*--p);	/* xlfun */
    p -= 1;	/* pc */
    mark(*--p);	/* xlenv */
    return p;
}

/* code_unmark - unmark a code continuation */
static LVAL *code_unmark(LVAL *p)
{
    return p - 3;
}

/* code_unwind - unwind a code continuation */
static void code_unwind(void)
{
    Cdrop(3);
}

/* code_unstack - unstack a code continuation */
static LVAL *code_unstack(LVAL *p)
{
    p -= 3;
    *p = unstack_environment(*p);
    return p;
}

/* code_print - print a code continuation */
static LVAL *code_print(LVAL *p)
{
    FIXTYPE val;
    errputstr("\n Code ");
    errprint(*--p);
    errputstr("\n  pc: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  env: ");
    errprint(*--p);
    return p;
}

/* code continuation dispatch table */
static CDISPATCH cd_code = {
    code_restore,
    code_mark,
    code_unmark,
    code_unwind,
    code_unstack,
    code_print
};

/* frame_restore - pop a stack environment frame */
static void frame_restore(void)
{
    xlenv = getnextframe(xlenv);
    Cdrop(NODEWORDS);
    xlsp = ((LVAL)xlcsp)->n_vdata + ((LVAL)xlcsp)->n_vsize;
}

/* frame_mark - mark a stack environment frame */
static LVAL *frame_mark(LVAL *p)
{
    p -= NODEWORDS;	/* environment frame */
    return p;
}

/* frame_unmark - unmark a stack environment frame */
static LVAL *frame_unmark(LVAL *p)
{
    p -= NODEWORDS;
    ((LVAL)p)->n_flags = 0;
    return p;
}

/* frame_unwind - unwind a stack environment frame */
static void frame_unwind(void)
{
    Cdrop(NODEWORDS);
}

/* frame_unstack - unstack a stack environment frame */
static LVAL *frame_unstack(LVAL *p)
{
    p -= NODEWORDS;
    return p;
}

/* frame_print - print a stack environment frame */
static LVAL *frame_print(LVAL *p)
{
    LVAL frame,np;
    int i;
    errputstr("\n Frame ");
    p -= NODEWORDS;
    frame = (LVAL)p;
    errprint(frame);
    if (msenvp(frame))
	frame = getforwardingaddr(frame);
    np = getenvnames(frame);
    for (i = FIRSTENV; np != v_nil; ++i, np = cdr(np)) {
	errputstr("\n  ");
	errprint(car(np));
	errputstr(" = ");
	errprint(getenvelement(frame,i));
    }
    return p;
}

/* stack environment frame dispatch table */
static CDISPATCH cd_frame = {
    frame_restore,
    frame_mark,
    frame_unmark,
    frame_unwind,
    frame_unstack,
    frame_print
};

/* find_top_procedure - find the top procedure on the call stack */
LVAL find_top_procedure(void)
{
    LVAL *p;
    for (p = xlcsp; p > xlstkbase; p = cdunmark(p))
	if ((CDISPATCH *)p[-1] == &cd_code)
	    return p[-2];
    return v_nil;
}

/* show_call - show a single call on the call stack */
static void show_call(LVAL code,LVAL frame)
{
    LVAL name = getcname(code);
    
    /* start the function */
    errputstr("\n  (");
    
    /* print the function name */
    if (name == v_nil)
	errprint(code);
    else
	errprint(name);
    
    /* print the function arguments */
    if (frame != v_nil) {
	FIXTYPE i,max;
	for (i = FIRSTENV, max = getsize(frame); i < max; ++i) {
	    errputstr(" ");
	    errprint(getelement(frame,i));
	}
    }
    
    /* end the function */
    errputstr(")");
}

/* show_call_stack - display the call stack */
void show_call_stack(int cmax)
{
    LVAL lastcode = v_nil,lastframe = v_nil,*p;
    for (p = xlcsp; cmax > 0 && p > xlstkbase; p = cdunmark(p)) {
	CDISPATCH *d = (CDISPATCH *)p[-1];
	if (d == &cd_code) {
	    if (lastcode != v_nil)
	        show_call(lastcode,lastframe);
	    lastcode = p[-2];
	    lastframe = v_nil;
	    --cmax;
	}
	else if (d == &cd_frame) {
	    lastframe = (LVAL)&p[-((int)NODEWORDS + 1)];
	    if (msenvp(lastframe))
		lastframe = getforwardingaddr(lastframe);
	}
    }
    if (lastcode != v_nil)
        show_call(lastcode,lastframe);
}

/* unwind_restore - restore an unwind frame after invoking an unwind-protect cleanup */
static void unwind_restore(void)
{
    LVAL *target,*p;
    int cnt;
    
    /* restore the throw target */
    target = (LVAL *)Cpop();
    xlargc = (int)(FIXTYPE)Cpop();
    
    /* move the results back onto the stack */
    if (xlargc > 0) {
        check(xlargc);
        for (xlsp -= xlargc, p = xlsp, cnt = xlargc; --cnt >= 0; )
            *p++ = Cpop();
    }

    /* continue throwing to the target */
    throwtotarget(target);
}

/* unwind_mark - mark an unwind frame */
static LVAL *unwind_mark(LVAL *p)
{
    int argc;
    p -= 1;			/* xltarget */
    argc = (int)(FIXTYPE)*--p;	/* xlargc */
    if (argc > 0)
        while (--argc >= 0)	/* results */
            mark(*--p);
    return p;
}

/* unwind_unmark - unmark an unwind frame */
static LVAL *unwind_unmark(LVAL *p)
{
    int argc;
    p -= 1;			/* xltarget */
    argc = (int)(FIXTYPE)*--p;	/* xlargc */
    return argc > 0 ? p - argc : p;
}

/* unwind_unwind - unwind an unwind frame */
static void unwind_unwind(void)
{
    int argc;
    Cdrop(1);			/* xltarget */
    argc = (int)(FIXTYPE)Cpop();/* xlargc */
    if (argc > 0)
        Cdrop(argc);
}

/* unwind_unstack - unstack an unwind frame */
static LVAL *unwind_unstack(LVAL *p)
{
    int argc;
    p -= 1;			/* xltarget */
    argc = (int)(FIXTYPE)*--p;	/* xlargc */
    return argc > 0 ? p - argc : p;
}

/* unwind_print - print a unwind frame */
static LVAL *unwind_print(LVAL *p)
{
    FIXTYPE val;
    errputstr("\n Unwind");
    errputstr("\n  target: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  argc: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  results:");
    if (val > 0)
        while (--val >= 0) {
	    errputstr("\n   ");
	    errprint(*--p);
        }
    return p;
}

/* unwind frame dispatch table */
static CDISPATCH cd_unwind = {
    unwind_restore,
    unwind_mark,
    unwind_unmark,
    unwind_unwind,
    unwind_unstack,
    unwind_print
};

/* protect_restore - restore a protect frame after a successful throw */
static void protect_restore(void)
{
    Cdrop(3);
}

/* protect_mark - mark a protect frame */
static LVAL *protect_mark(LVAL *p)
{
    mark(*--p);		/* cleanup */
    p -= 1;  		/* xlsp */
    mark(*--p); 	/* xlenv */
    return p;
}

/* protect_unmark - unmark a protect frame */
static LVAL *protect_unmark(LVAL *p)
{
    return p - 3;
}

/* protect_unwind - unwind a protect frame */
static void protect_unwind(void)
{
    LVAL *p = xlsp + xlargc;
    int cnt = xlargc;
    xlval = Cpop();
    xlsp = (LVAL *)Cpop();
    xlenv = Cpop();
    if (msenvp(xlenv))
	xlenv = getforwardingaddr(xlenv);
    if (xlargc > 0) {
        Ccheck(xlargc + 3);
        while (--cnt >= 0)
            Cpush(*--p);
    }
    else
        Ccheck(3);
    Cpush((LVAL)xlargc);
    Cpush((LVAL)xltarget);
    Cpush((LVAL)&cd_unwind);
    xlargc = 0;
    xlnext = xlapply;
    longjmp(xlerrtarget->target,BCD_NEXT);
}

/* protect_unstack - unstack a protect frame */
static LVAL *protect_unstack(LVAL *p)
{
    return p - 3;
    *p = unstack_environment(*p);
}

/* protect_print - print a protect frame */
static LVAL *protect_print(LVAL *p)
{
    FIXTYPE val;
    errputstr("\n Unwind-protect");
    errputstr("\n  cleanup: ");
    errprint(*--p);
    errputstr("\n  sp: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  env: ");
    errprint(*--p);
    return p;
}

/* protect frame dispatch table */
static CDISPATCH cd_protect = {
    protect_restore,
    protect_mark,
    protect_unmark,
    protect_unwind,
    protect_unstack,
    protect_print
};

/* return_restore - pop a return frame */
static void return_restore(void)
{
    longjmp(xlerrtarget->target,BCD_RETURN);
}

/* return_mark - mark a return frame */
static LVAL *return_mark(LVAL *p)
{
    return p;
}

/* return_unmark - unmark a return frame */
static LVAL *return_unmark(LVAL *p)
{
    return p;
}

/* return_unwind - unwind a return frame */
static void return_unwind(void)
{
}

/* return_unstack - unstack a return frame */
static LVAL *return_unstack(LVAL *p)
{
    return p;
}

/* return_print - print a return frame */
static LVAL *return_print(LVAL *p)
{
    errputstr("\n Return");
    return p;
}

/* return frame dispatch table */
static CDISPATCH cd_return = {
    return_restore,
    return_mark,
    return_unmark,
    return_unwind,
    return_unstack,
    return_print
};

/* catch_restore - restore a catch frame after a successful throw */
static void catch_restore(void)
{
    Cdrop(1);
    xlcatch = (LVAL *)Cpop();
    xlsp = (LVAL *)Cpop();
    xlfun = Cpop();
    base = getcodestr(xlfun);
    pc = base + (FIXTYPE)Cpop();
    xlenv = Cpop();
    if (msenvp(xlenv))
	xlenv = getforwardingaddr(xlenv);
}

/* catch_mark - mark a catch frame */
static LVAL *catch_mark(LVAL *p)
{
    mark(*--p);		/* tag */
    p -= 2;     	/* xlcatch, xlsp */
    mark(*--p);		/* xlfun */
    p -= 1;		/* pc */
    mark(*--p);		/* xlenv */
    return p;
}

/* catch_unmark - unmark a catch frame */
static LVAL *catch_unmark(LVAL *p)
{
    return p - 6;
}

/* catch_unwind - unwind a catch frame */
static void catch_unwind(void)
{
    Cdrop(1);
    xlcatch = (LVAL *)Cpop();
    Cdrop(4);
}

/* catch_unstack - unstack a catch frame */
static LVAL *catch_unstack(LVAL *p)
{
    return p - 6;
    *p = unstack_environment(*p);
}

/* catch_print - print a catch frame */
static LVAL *catch_print(LVAL *p)
{
    FIXTYPE val;
    errputstr("\n Catch");
    errputstr("\n  tag: ");
    errprint(*--p);
    errputstr("\n  catch: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  sp: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  fun: ");
    errprint(*--p);
    errputstr("\n  pc: ");
    val = (FIXTYPE)*--p;
    errprint(cvfixnum(val));
    errputstr("\n  env: ");
    errprint(*--p);
    return p;
}

/* catch frame dispatch table */
static CDISPATCH cd_catch = {
    catch_restore,
    catch_mark,
    catch_unmark,
    catch_unwind,
    catch_unstack,
    catch_print
};

/* opNOP - handler for opcode NOP */
static void opNOP(void)
{
}

/* opBRT - handler for opcode BRT */
static void opBRT(void)
{
    register unsigned int i;
    if (xlval != v_false) {
	i = *pc++ << 8;
	pc = base + (i | *pc);
    }
    else
	pc += 2;
}

/* opBRF - handler for opcode BRF */
static void opBRF(void)
{
    register unsigned int i;
    if (xlval == v_false) {
	i = *pc++ << 8;
	pc = base + (i | *pc);
    }
    else
	pc += 2;
}

/* opBR - handler for opcode BR */
static void opBR(void)
{
    register unsigned int i;
    i = *pc++ << 8;
    pc = base + (i | *pc);
}

/* opLIT - handler for opcode LIT */
static void opLIT(void)
{
    xlval = getelement(xlfun,*pc++);
    xlargc = 1;
}

/* opGREF - handler for opcode GREF */
static void opGREF(void)
{
    extern LVAL s_package;
    register LVAL tmp;
    LVAL key;
    tmp = getelement(xlfun,*pc++);
    if ((xlval = getvalue(tmp)) == s_unbound) {
        xlval = findsymbol("*UNBOUND-HANDLER*",getvalue(s_package),&key);
        if ((xlval = getvalue(xlval)) != v_nil) {
	    FIXTYPE offset;
	    pc -= 2; /* backup the pc */
	    offset = pc - base;
	    oscheck();
	    Ccheck(4);
	    Cpush(xlenv);
	    Cpush((LVAL)offset);
	    Cpush(xlfun);
	    Cpush((LVAL)&cd_code);
	    tmp = make_continuation();
	    check(2);
	    push(tmp);
	    push(getelement(xlfun,pc[1]));
	    xlargc = 2;
	    xlnext = xlapply;
        }
        else
	    xlerror("unbound variable",tmp);
    }
    else
	xlargc = 1;
}

/* opGSET - handler for opcode GSET */
static void opGSET(void)
{
    setvalue(getelement(xlfun,*pc++),xlval);
    xlargc = 1;
}

/* opEREF - handler for opcode EREF */
static void opEREF(void)
{
    register LVAL tmp;
    register int i;
    i = *pc++;
    tmp = xlenv;
    while (--i >= 0) tmp = getnextframe(tmp);
    xlval = getenvelement(tmp,*pc++);
    xlargc = 1;
}

/* opIREF - handler for opcode IREF */
static void opIREF(void)
{
    register LVAL tmp;
    register int i;
    i = *pc++;
    tmp = xlenv;
    while (--i >= 0) tmp = getnextframe(tmp);
    xlval = getivar(getenvelement(tmp,FIRSTENV),*pc++);
    xlargc = 1;
}

/* opESET - handler for opcode ESET */
static void opESET(void)
{
    register LVAL tmp;
    register int i;
    i = *pc++;
    tmp = xlenv;
    while (--i >= 0) tmp = getnextframe(tmp);
    setenvelement(tmp,*pc++,xlval);
    xlargc = 1;
}

/* opISET - handler for opcode ISET */
static void opISET(void)
{
    register LVAL tmp;
    register int i;
    i = *pc++;
    tmp = xlenv;
    while (--i >= 0) tmp = getnextframe(tmp);
    setivar(getenvelement(tmp,FIRSTENV),*pc++,xlval);
    xlargc = 1;
}

/* opCALL - handler for opcode CALL */
static void opCALL(void)
{
    xlargc = *pc++;	/* get argument count */
    call();
}

/* opMVCALL - handler for opcode MVCALL */
static void opMVCALL(void)
{
    LVAL argc = pop();
    xlargc = (int)getsfixnum(argc);
    call();
}

/* call - common code for CALL and MVCALL */
static void call(void)
{
    register FIXTYPE offset = pc - base;
    Ccheck(4);
    Cpush(xlenv);
    Cpush((LVAL)offset);
    Cpush(xlfun);
    Cpush((LVAL)&cd_code);
    xlapply();
}

/* opTCALL - handler for opcode TCALL */
static void opTCALL(void)
{
    xlargc = *pc++;	/* get argument count */
    tcall();
}

/* opMVTCALL - handler for opcode MVTCALL */
static void opMVTCALL(void)
{
    register LVAL argc = pop();
    xlargc = (int)getsfixnum(argc);
    tcall();
}

/* tcall - common code for TCALL and MVTCALL */
static void tcall(void)
{
    register CDISPATCH *cd;
    register LVAL *src;
    register int cnt;
    src = xlsp + xlargc;
    while ((cd = (CDISPATCH *)Cpop()) == &cd_frame)
	(*cd->cd_restore)();
    Cpush((LVAL)cd);
    for (cnt = xlargc; --cnt >= 0; )
	push(*--src);
    xlapply();
}

/* opRETURN - handler for opcode RETURN */
static void opRETURN(void)
{
    LVAL *src,*dst;
    CDISPATCH *cd;
    int cnt;
    if (xlargc > 1) {
	src = xlsp;
	do {
	    cd = (CDISPATCH *)Cpop();
	    (*cd->cd_restore)();
	} while (cd == &cd_frame);
	for (dst = xlsp, cnt = xlargc; --cnt >= 0; )
	    *--dst = *--src;
    }
    else {
	do {
	    cd = (CDISPATCH *)Cpop();
	    (*cd->cd_restore)();
	} while (cd == &cd_frame);
    }
}

/* opARGSEQ - handler for opcode ARGSEQ */
static void opARGSEQ(void)
{
    register int n;
    n = *pc++;

    /* check the argument count */
    if (xlargc < n)
	xltoofew();
    else if (xlargc > n)
	xltoomany();

    /* setup the environment stack frame */
    stkframe(xlargc,FIRSTENV,getvnames(xlfun));
}

/* opARGSGE - handler for opcode ARGSGE */
static void opARGSGE(void)
{
    register int min;
    min = *pc++;

    /* check the argument count */
    if (xlargc < min)
	xltoofew();

    /* setup the environment stack frame */
    stkframe(xlargc,FIRSTENV,getvnames(xlfun));
}

/* opARGSBT - handler for opcode ARGSBT */
static void opARGSBT(void)
{
    register int min,max;
    min = *pc++;
    max = *pc++;

    /* check the argument count */
    if (xlargc < min)
	xltoofew();
    else if (xlargc > max)
	xltoomany();

    /* setup the environment stack frame */
    stkframe(xlargc,FIRSTENV,getvnames(xlfun));
}

/* opMETHOD - handler for opcode METHOD */
static void opMETHOD(void)
{
    setframetype(xlenv,SMENV);
}

/* opOPTARG - handler for opcode OPTARG */
static void opOPTARG(void)
{
    register LVAL pframe = getnextframe(xlenv);
    register int n,arg;
    n = *pc++;
    arg = *pc++;
    if (getenvsize(pframe) > n) {
	setenvelement(xlenv,arg,getenvelement(pframe,n));
	xlval = v_true;
    }
    else
	xlval = v_nil;
}

/* opKEYARG - handler for the opcode KEYARG */
static void opKEYARG(void)
{
    register LVAL pframe = getnextframe(xlenv);
    register int n,arg;
    register LVAL key;
    key = getelement(xlfun,*pc++);
    n = *pc++;
    arg = *pc++;
    for (; n < getenvsize(pframe); n += 2) {
	if (getenvelement(pframe,n) == key) {
	    if (++n >= getenvsize(pframe))
		xlerror("no value following keyword",key);
	    setenvelement(xlenv,arg,getenvelement(pframe,n));
	    xlval = v_true;
	    return;
	}
    }
    xlval = v_nil;
}

/* opREST - handler for opcode REST */
static void opREST(void)
{
    register LVAL pframe = getnextframe(xlenv);
    register FIXTYPE n,arg,i;
    n = *pc++;	/* get the slot number */
    arg = *pc++;
    for (xlval = v_nil, i = getenvsize(pframe); --i >= n; )
        xlval = cons(getenvelement(pframe,i),xlval);
    setenvelement(xlenv,arg,xlval);
}

/* opFRAME - handler for the opcode FRAME */
static void opFRAME(void)
{
    register int rargc,extra,lit;
    rargc = *pc++;
    extra = *pc++;
    lit = *pc++;

    /* create the environment frame */
    stkframe(rargc,extra,getelement(xlfun,lit));
}

/* opMVFRAME - handler for the opcode MVFRAME */
static void opMVFRAME(void)
{
    register int size,lit,lit2;
    size = *pc++;
    lit = *pc++;
    lit2 = *pc++;
    
    /* push the multiple values onto the stack */
    check(xlargc);
    xlsp -= xlargc;
    settop(xlval);

    /* create the environment frame */
    stkframe(xlargc,FIRSTENV,getelement(xlfun,lit));
    stkframe(0,size,getelement(xlfun,lit2));
}

/* opMVPUSH - handler for the opcode MVPUSH */
static void opMVPUSH(void)
{
    if (xlargc > 0) {
        check(xlargc);
        xlsp -= xlargc;
        settop(xlval);
    }
    cpush(cvsfixnum((FIXTYPE)xlargc));
}

/* opMVPOP - handler for opcode MVPOP */
static void opMVPOP(void)
{
    xlval = pop();
    xlargc = (int)getfixnum(xlval);
    if (xlargc <= 0)
        xlval = v_nil;
    else {
        xlval = top();
        xlsp += xlargc;
    }
}

/* opUNFRAME - handler for the opcode UNFRAME */
static void opUNFRAME(void)
{
    LVAL *src,*dst;
    int cnt;
    if (xlargc > 1) {
	src = xlsp;
	cdrestore();
	for (dst = xlsp, cnt = xlargc; --cnt >= 0; )
	    *--dst = *--src;
    }
    else
	cdrestore();
}

/* opT - handler for opcode T */
static void opT(void)
{
    xlval = v_true;
    xlargc = 1;
}

/* opNIL - handler for opcode NIL */
static void opNIL(void)
{
    xlval = v_nil;
    xlargc = 1;
}

/* opPUSH - handler for opcode PUSH */
static void opPUSH(void)
{
    cpush(xlval);
}

/* opCLOSE - handler for opcode CLOSE */
static void opCLOSE(void)
{
    if (!codep(xlval)) badargtype(xlval);
    xlenv = unstack_environment(xlenv);
    xlval = cvclosure(xlval,xlenv);
    xlargc = 1;
}

/* opDELAY - handler for opcode DELAY */
static void opDELAY(void)
{
    if (!codep(xlval)) badargtype(xlval);
    xlenv = unstack_environment(xlenv);
    xlval = cvpromise(xlval,xlenv);
    xlargc = 1;
}

/* opATOM - handler for opcode ATOM */
static void opATOM(void)
{
    xlval = (atom(xlval) ? v_true : v_false);
    xlargc = 1;
}

/* opEQ - handler for opcode EQ */
static void opEQ(void)
{
    xlval = (pop() == xlval ? v_true : v_false);
    xlargc = 1;
}

/* opNULL - handler for opcode NULL */
static void opNULL(void)
{
    xlval = (xlval ? v_false : v_true);
    xlargc = 1;
}

/* opCONS - handler for opcode CONS */
static void opCONS(void)
{
    xlval = cons(xlval,pop());
}

/* opCAR - handler for opcode CAR */
static void opCAR(void)
{
    if (!listp(xlval)) badargtype(xlval);
    xlval = (xlval ? car(xlval) : v_nil);
    xlargc = 1;
}

/* opCDR - handler for opcode CDR */
static void opCDR(void)
{
    if (!listp(xlval)) badargtype(xlval);
    xlval = (xlval ? cdr(xlval) : v_nil);
    xlargc = 1;
}

/* opSETCAR - handler for opcode SETCAR */
static void opSETCAR(void)
{
    register LVAL tmp;
    tmp = pop();
    if (!consp(xlval)) badargtype(xlval);
    rplaca(xlval,tmp);
    xlargc = 1;
}

/* opSETCDR - handler for opcode SETCDR */
static void opSETCDR(void)
{
    register LVAL tmp;
    tmp = pop();
    if (!consp(xlval)) badargtype(xlval);
    rplacd(xlval,tmp);
    xlargc = 1;
}

/* opADD - handler for opcode ADD */
static void opADD(void)
{
    register FIXTYPE val;
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval)) {
        val = getfixnum(xlval) + getfixnum(tmp);
        xlval = fastcvfixnum(val);
    }
    else {
        push(tmp); cpush(xlval);
        xlval = callsubr(xadd,2);
    }
    xlargc = 1;
}

/* opSUB - handler for opcode SUB */
static void opSUB(void)
{
    register FIXTYPE val;
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval)) {
        val = getfixnum(xlval) - getfixnum(tmp);
        xlval = fastcvfixnum(val);
    }
    else {
        push(tmp); cpush(xlval);
        xlval = callsubr(xsub,2);
    }
    xlargc = 1;
}

/* opMUL - handler for opcode MUL */
static void opMUL(void)
{
    register FIXTYPE val;
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval)) {
        val = getfixnum(xlval) * getfixnum(tmp);
        xlval = fastcvfixnum(val);
    }
    else {
        push(tmp); cpush(xlval);
        xlval = callsubr(xmul,2);
    }
    xlargc = 1;
}

/* opQUO - handler for opcode QUO */
static void opQUO(void)
{
    register FIXTYPE val;
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval)) {
        if ((val = getfixnum(tmp)) == (FIXTYPE)0)
	    xlfmterror("division by zero");
        val = getfixnum(xlval) / val;
        xlval = fastcvfixnum(val);
    }
    else if (fixp(tmp))
        badargtype(xlval);
    else
        badargtype(tmp);
    xlargc = 1;
}

/* opLSS - handler for opcode LSS */
static void opLSS(void)
{
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval))
        xlval = (getfixnum(xlval) < getfixnum(tmp) ? v_true : v_false);
    else {
        push(tmp); cpush(xlval);
        xlval = callsubr(xlss,2);
    }
    xlargc = 1;
}

/* opEQL - handler for opcode EQL */
static void opEQL(void)
{
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval))
        xlval = (getfixnum(xlval) == getfixnum(tmp) ? v_true : v_false);
    else {
        push(tmp); cpush(xlval);
        xlval = callsubr(xeql,2);
    }
    xlargc = 1;
}

/* opGTR - handler for opcode GTR */
static void opGTR(void)
{
    register LVAL tmp;
    tmp = pop();
    if (fixp(tmp) && fixp(xlval))
        xlval = (getfixnum(xlval) > getfixnum(tmp) ? v_true : v_false);
    else {
        push(tmp); cpush(xlval);
        xlval = callsubr(xgtr,2);
    }
    xlargc = 1;
}

/* opCATCH - handler for opcode CATCH */
static void opCATCH(void)
{
    register unsigned int offset;
    
    /* get the target offset */
    offset = *pc++ << 8;
    offset |= *pc++;

    /* push a catch frame */
    Ccheck(7);
    Cpush(xlenv);
    Cpush((LVAL)offset);
    Cpush(xlfun);
    Cpush((LVAL)xlsp);
    Cpush((LVAL)xlcatch);
    Cpush(xlval);
    Cpush((LVAL)&cd_catch);
    xlcatch = xlcsp;
}

/* opUNCATCH - handler for opcode UNCATCH */
static void opUNCATCH(void)
{
    Cdrop(2);
    xlcatch = (LVAL *)Cpop();
    Cdrop(4);
}

/* opPROTECT - handler for opcode PROTECT */
static void opPROTECT(void)
{
    Ccheck(4);
    Cpush(xlenv);
    Cpush((LVAL)xlsp);
    Cpush(xlval);
    Cpush((LVAL)&cd_protect);
}

/* opUNPROTECT - handler for opcode UNPROTECT */
static void opUNPROTECT(void)
{
    Cdrop(1);
    xlval = Cpop();
    Cdrop(2);
}

/* opBAD - handler for all bad opcodes */
static void opBAD(void)
{
    xlerror("bad opcode",cvsfixnum((FIXTYPE)*--pc));
}

/* opcode dispatch table */
static void (*optab[256])(void) = {
    opNOP, 	/* 00 - nop */
    opBRT, 	/* 01 - branch on true */
    opBRF, 	/* 02 - branch on false */
    opBR,  	/* 03 - branch unconditionally */
    opLIT, 	/* 04 - load literal */
    opGREF,	/* 05 - global symbol value */
    opGSET,	/* 06 - set global symbol value */
    opEREF,	/* 07 - environment variable value */
    opIREF,	/* 08 - instance variable value */
    opESET,	/* 09 - set environment variable value */
    opISET,	/* 0A - set instance variable value */
    opCALL,	/* 0B - call a function */
    opTCALL,	/* 0C - tail recursive call */
    opRETURN,	/* 0D - return from a function */
    opT,   	/* 0E - load 'val' with t */
    opNIL, 	/* 0F - load 'val' with nil */
    opPUSH,	/* 10 - push the 'val' register */
    opCLOSE,	/* 11 - create a closure */
    opARGSEQ,	/* 12 - argument count == m, n extra slots */
    opARGSGE,	/* 13 - argument count >= min, n extra slots */
    opARGSBT,	/* 14 - argument count >= min, count <= max, n extra slots */
    opOPTARG,	/* 15 - check for an &optional argument */
    opREST,   	/* 16 - build a &rest list with arguments n ... */
    opDELAY,	/* 17 - create a promise */
    opATOM,	/* 18 - atom predicate */
    opEQ,  	/* 19 - eq? predicate */
    opNULL,	/* 1A - null? (or not) predicate */
    opCONS,	/* 1B - cons */
    opCAR, 	/* 1C - car */
    opCDR, 	/* 1D - cdr */
    opSETCAR,	/* 1E - set-car! */
    opSETCDR,	/* 1F - set-cdr! */
    opADD, 	/* 20 - add two numeric expressions */
    opSUB, 	/* 21 - subtract two numeric expressions */
    opMUL, 	/* 22 - multiply two numeric expressions */
    opQUO, 	/* 23 - divide two integer expressions */
    opLSS, 	/* 24 - less than */
    opEQL, 	/* 25 - equal to */
    opGTR, 	/* 26 - greater than */
    opKEYARG,	/* 27 - check for a &key argument */
    opFRAME,	/* 28 - create a new environment frame */
    opUNFRAME,	/* 29 - remove an environment frame */
    opMVFRAME,	/* 2A - create a multiple value environment frame */
    opMVPUSH,	/* 2B - push multiple values */
    opMVCALL,	/* 2C - multiple value call */
    opMVTCALL,	/* 2D - tail recursive multiple value call */
    opMETHOD,	/* 2E - mark the current frame as a method frame */
    opCATCH,	/* 2F - push a catch frame */
    opUNCATCH,	/* 30 - pop a catch frame */
    opPROTECT,	/* 31 - push a protect frame */
    opUNPROTECT,/* 32 - pop a protect frame */
    opMVPOP,	/* 33 - pop multiple values */
    opBAD,	/* 34 - (unused) */
    opBAD,	/* 35 - (unused) */
    opBAD,	/* 36 - (unused) */
    opBAD,	/* 37 - (unused) */
    opBAD,	/* 38 - (unused) */
    opBAD,	/* 39 - (unused) */
    opBAD,	/* 3A - (unused) */
    opBAD,	/* 3B - (unused) */
    opBAD,	/* 3C - (unused) */
    opBAD,	/* 3D - (unused) */
    opBAD,	/* 3E - (unused) */
    opBAD,	/* 3F - (unused) */
    opBAD,	/* 40 - (unused) */
    opBAD,	/* 41 - (unused) */
    opBAD,	/* 42 - (unused) */
    opBAD,	/* 43 - (unused) */
    opBAD,	/* 44 - (unused) */
    opBAD,	/* 45 - (unused) */
    opBAD,	/* 46 - (unused) */
    opBAD,	/* 47 - (unused) */
    opBAD,	/* 48 - (unused) */
    opBAD,	/* 49 - (unused) */
    opBAD,	/* 4A - (unused) */
    opBAD,	/* 4B - (unused) */
    opBAD,	/* 4C - (unused) */
    opBAD,	/* 4D - (unused) */
    opBAD,	/* 4E - (unused) */
    opBAD,	/* 4F - (unused) */
    opBAD,	/* 50 - (unused) */
    opBAD,	/* 51 - (unused) */
    opBAD,	/* 52 - (unused) */
    opBAD,	/* 53 - (unused) */
    opBAD,	/* 54 - (unused) */
    opBAD,	/* 55 - (unused) */
    opBAD,	/* 56 - (unused) */
    opBAD,	/* 57 - (unused) */
    opBAD,	/* 58 - (unused) */
    opBAD,	/* 59 - (unused) */
    opBAD,	/* 5A - (unused) */
    opBAD,	/* 5B - (unused) */
    opBAD,	/* 5C - (unused) */
    opBAD,	/* 5D - (unused) */
    opBAD,	/* 5E - (unused) */
    opBAD,	/* 5F - (unused) */
    opBAD,	/* 60 - (unused) */
    opBAD,	/* 61 - (unused) */
    opBAD,	/* 62 - (unused) */
    opBAD,	/* 63 - (unused) */
    opBAD,	/* 64 - (unused) */
    opBAD,	/* 65 - (unused) */
    opBAD,	/* 66 - (unused) */
    opBAD,	/* 67 - (unused) */
    opBAD,	/* 68 - (unused) */
    opBAD,	/* 69 - (unused) */
    opBAD,	/* 6A - (unused) */
    opBAD,	/* 6B - (unused) */
    opBAD,	/* 6C - (unused) */
    opBAD,	/* 6D - (unused) */
    opBAD,	/* 6E - (unused) */
    opBAD,	/* 6F - (unused) */
    opBAD,	/* 70 - (unused) */
    opBAD,	/* 71 - (unused) */
    opBAD,	/* 72 - (unused) */
    opBAD,	/* 73 - (unused) */
    opBAD,	/* 74 - (unused) */
    opBAD,	/* 75 - (unused) */
    opBAD,	/* 76 - (unused) */
    opBAD,	/* 77 - (unused) */
    opBAD,	/* 78 - (unused) */
    opBAD,	/* 79 - (unused) */
    opBAD,	/* 7A - (unused) */
    opBAD,	/* 7B - (unused) */
    opBAD,	/* 7C - (unused) */
    opBAD,	/* 7D - (unused) */
    opBAD,	/* 7E - (unused) */
    opBAD,	/* 7F - (unused) */
    opBAD,	/* 80 - (unused) */
    opBAD,	/* 81 - (unused) */
    opBAD,	/* 82 - (unused) */
    opBAD,	/* 83 - (unused) */
    opBAD,	/* 84 - (unused) */
    opBAD,	/* 85 - (unused) */
    opBAD,	/* 86 - (unused) */
    opBAD,	/* 87 - (unused) */
    opBAD,	/* 88 - (unused) */
    opBAD,	/* 89 - (unused) */
    opBAD,	/* 8A - (unused) */
    opBAD,	/* 8B - (unused) */
    opBAD,	/* 8C - (unused) */
    opBAD,	/* 8D - (unused) */
    opBAD,	/* 8E - (unused) */
    opBAD,	/* 8F - (unused) */
    opBAD,	/* 90 - (unused) */
    opBAD,	/* 91 - (unused) */
    opBAD,	/* 92 - (unused) */
    opBAD,	/* 93 - (unused) */
    opBAD,	/* 94 - (unused) */
    opBAD,	/* 95 - (unused) */
    opBAD,	/* 96 - (unused) */
    opBAD,	/* 97 - (unused) */
    opBAD,	/* 98 - (unused) */
    opBAD,	/* 99 - (unused) */
    opBAD,	/* 9A - (unused) */
    opBAD,	/* 9B - (unused) */
    opBAD,	/* 9C - (unused) */
    opBAD,	/* 9D - (unused) */
    opBAD,	/* 9E - (unused) */
    opBAD,	/* 9F - (unused) */
    opBAD,	/* A0 - (unused) */
    opBAD,	/* A1 - (unused) */
    opBAD,	/* A2 - (unused) */
    opBAD,	/* A3 - (unused) */
    opBAD,	/* A4 - (unused) */
    opBAD,	/* A5 - (unused) */
    opBAD,	/* A6 - (unused) */
    opBAD,	/* A7 - (unused) */
    opBAD,	/* A8 - (unused) */
    opBAD,	/* A9 - (unused) */
    opBAD,	/* AA - (unused) */
    opBAD,	/* AB - (unused) */
    opBAD,	/* AC - (unused) */
    opBAD,	/* AD - (unused) */
    opBAD,	/* AE - (unused) */
    opBAD,	/* AF - (unused) */
    opBAD,	/* B0 - (unused) */
    opBAD,	/* B1 - (unused) */
    opBAD,	/* B2 - (unused) */
    opBAD,	/* B3 - (unused) */
    opBAD,	/* B4 - (unused) */
    opBAD,	/* B5 - (unused) */
    opBAD,	/* B6 - (unused) */
    opBAD,	/* B7 - (unused) */
    opBAD,	/* B8 - (unused) */
    opBAD,	/* B9 - (unused) */
    opBAD,	/* BA - (unused) */
    opBAD,	/* BB - (unused) */
    opBAD,	/* BC - (unused) */
    opBAD,	/* BD - (unused) */
    opBAD,	/* BE - (unused) */
    opBAD,	/* BF - (unused) */
    opBAD,	/* C0 - (unused) */
    opBAD,	/* C1 - (unused) */
    opBAD,	/* C2 - (unused) */
    opBAD,	/* C3 - (unused) */
    opBAD,	/* C4 - (unused) */
    opBAD,	/* C5 - (unused) */
    opBAD,	/* C6 - (unused) */
    opBAD,	/* C7 - (unused) */
    opBAD,	/* C8 - (unused) */
    opBAD,	/* C9 - (unused) */
    opBAD,	/* CA - (unused) */
    opBAD,	/* CB - (unused) */
    opBAD,	/* CC - (unused) */
    opBAD,	/* CD - (unused) */
    opBAD,	/* CE - (unused) */
    opBAD,	/* CF - (unused) */
    opBAD,	/* D0 - (unused) */
    opBAD,	/* D1 - (unused) */
    opBAD,	/* D2 - (unused) */
    opBAD,	/* D3 - (unused) */
    opBAD,	/* D4 - (unused) */
    opBAD,	/* D5 - (unused) */
    opBAD,	/* D6 - (unused) */
    opBAD,	/* D7 - (unused) */
    opBAD,	/* D8 - (unused) */
    opBAD,	/* D9 - (unused) */
    opBAD,	/* DA - (unused) */
    opBAD,	/* DB - (unused) */
    opBAD,	/* DC - (unused) */
    opBAD,	/* DD - (unused) */
    opBAD,	/* DE - (unused) */
    opBAD,	/* DF - (unused) */
    opBAD,	/* E0 - (unused) */
    opBAD,	/* E1 - (unused) */
    opBAD,	/* E2 - (unused) */
    opBAD,	/* E3 - (unused) */
    opBAD,	/* E4 - (unused) */
    opBAD,	/* E5 - (unused) */
    opBAD,	/* E6 - (unused) */
    opBAD,	/* E7 - (unused) */
    opBAD,	/* E8 - (unused) */
    opBAD,	/* E9 - (unused) */
    opBAD,	/* EA - (unused) */
    opBAD,	/* EB - (unused) */
    opBAD,	/* EC - (unused) */
    opBAD,	/* ED - (unused) */
    opBAD,	/* EE - (unused) */
    opBAD,	/* EF - (unused) */
    opBAD,	/* F0 - (unused) */
    opBAD,	/* F1 - (unused) */
    opBAD,	/* F2 - (unused) */
    opBAD,	/* F3 - (unused) */
    opBAD,	/* F4 - (unused) */
    opBAD,	/* F5 - (unused) */
    opBAD,	/* F6 - (unused) */
    opBAD,	/* F7 - (unused) */
    opBAD,	/* F8 - (unused) */
    opBAD,	/* F9 - (unused) */
    opBAD,	/* FA - (unused) */
    opBAD,	/* FB - (unused) */
    opBAD,	/* FC - (unused) */
    opBAD,	/* FD - (unused) */
    opBAD,	/* FE - (unused) */
    opBAD	/* FF - (unused) */
};

/* CallFunction - call a function */
int CallFunction(LVAL *pvalue,LVAL fun,int argc,...)
{
    va_list ap;
    int valc;
    
    /* execute the function call */
    va_start(ap,argc);
    valc = InvokeInterpreter(pvalue,fun,v_nil,argc,ap);
    va_end(ap);
    
    /* return the value count */
    return valc;
}

/* InvokeInterpreter - invoke the bytecode interpreter */
int InvokeInterpreter(LVAL *pvalue,LVAL fun,LVAL sel,int argc,va_list ap)
{
    ERRORTARGET target;
    void (*next)(void);
    LVAL *save_csp,*save_sp,save_val,*p;
    void (*save_next)(void);
    int save_pcoff;
    
    /* save the interpreter state */
    check(3);
    save_next = xlnext;
    save_pcoff = pc - base;
    save_csp = xlcsp;
    push(xlval);
    push(xlfun);
    push(xlenv);
    save_sp = xlsp;
    
    /* initialize the registers */
    xlnext = NULL;

    /* push the return continuation */
    Ccheck(1);
    Cpush((LVAL)&cd_return);

    /* message sends have a negative argument count */
    if (argc < 0) {
    	argc = -argc;
	xlargc = argc;
	check(argc);
	xlsp -= argc;
	p = xlsp;
	*p++ = sel;
	--argc;
    }
    else {
    	xlargc = argc;
        check(argc);
	xlsp -= argc;
	p = xlsp;
    }
    
    /* setup the function and arguments */
    xlval = fun;
    while (--argc >= 0)
        *p++ = va_arg(ap,LVAL);
    
    /* setup a target for the error handler */
    pushtarget(&target);
    switch (setjmp(target.target)) {
    case BCD_START:
        xlapply();
        break;
    case BCD_ABORT:
        throwtoreturn();
        break;
    case BCD_RETURN:
        poptarget();
        
        /* save the return value */
        save_val = xlval;
        
        /* restore the interpreter state */
        xlnext = save_next;
        xlcsp = save_csp;
        xlsp = save_sp;
        xlenv = pop();
        xlfun = pop();
        xlval = pop();
	if (xlfun != v_nil) {
	    base = getcodestr(xlfun);
	    pc = base + save_pcoff;
	}
        
        /* return value and value count */
        *pvalue = save_val;
        return xlargc;
    }
    
    /* execute the code */
    for (;;) {

	/* check for control codes */
	if (--sample <= 0) {
	    sample = SRATE;
	    oscheck();
	}

	/* execute the next bytecode instruction */
	if (xlnext) {
	    next = xlnext;
	    xlnext = NULL;
	    (*next)();
	}
	else {
	    if (trace)
		decode_instruction(curoutput(),
				   xlfun,
	    			   (int)(pc - base),
	    		           xlenv);
	    (*optab[*pc++])();
	}
    }
}

/* xthrow - built-in function 'throw' */
void xthrow(void)
{
    throwtotag(xlgetarg());
}

/* xthrowerror - built-in function 'throw-error' */
void xthrowerror(void)
{
    xlval = xlgetarg();
    xllastarg();
    throwerror(xlval);
}

/* throwerror - throw an error */
void throwerror(LVAL type)
{
    LVAL *target;

    /* setup the arguments */
    cpush(type);
    xlargc = 1;

    /* find the throw target */
    if ((target = findmatchingcatch(s_error)) == NULL)
        jumptotarget(-1);

    /* throw to the target */
    throwtotarget(target);
}

/* throwtotag - throw to a catch tag */
static void throwtotag(LVAL tag)
{
    LVAL *target;
    
    /* find the throw target */
    if ((target = findmatchingcatch(tag)) == NULL)
        xlerror("no target for throw",tag);

    /* throw to the target */
    throwtotarget(target);
}

/* findmatchingcatch - find the catch frame matching a tag */
static LVAL *findmatchingcatch(LVAL tag)
{
    LVAL *p;
    for (p = xlcatch; p != NULL; p = (LVAL *)p[-3])
        if (tag == p[-2])
            return p;
    return NULL;
}

/* throwtoreturn - throw to a return frame */
static void throwtoreturn(void)
{
    LVAL *target,*p;
    
    /* find the return target */
    for (p = xlcsp, target = NULL; p > xlstkbase; p = cdunmark(p))
        if ((CDISPATCH *)p[-1] == &cd_return) {
            target = p;
            break;
        }
    
    /* make sure we found a target */
    if (target == NULL)
        xlfmtabort("no target for return");

    /* throw to the target */
    throwtotarget(target);
}
        
/* throwtotarget - throw to the target in xltarget */
static void throwtotarget(LVAL *target)
{        
    LVAL *dst,*p;
    int cnt;

    /* save the target */
    xltarget = target;
    
    /* save the position of the return values */
    p = xlsp + xlargc;
    
    /* unwind to the target */
    while (xlcsp > xltarget)
        cdunwind();
    cdrestore();
    
    /* move the arguments to the new stack position */
    for (dst = xlsp, cnt = xlargc; --cnt >= 0; )
	*--dst = *--p;
    xlval = *p;

    /* jump back the the bytecode interpreter */
    longjmp(xlerrtarget->target,BCD_NEXT);
}

/* jumptotarget - jump to the current target */
void jumptotarget(int sts)
{
    xlargc = sts;
    longjmp(xlerrtarget->target,BCD_ABORT);
}

/* pushtarget - push a new target */
void pushtarget(ERRORTARGET *target)
{
    target->next = xlerrtarget;
    xlerrtarget = target;
}

/* poptarget - pop a target */
void poptarget(void)
{
    xlerrtarget = xlerrtarget->next;
}

/* findvar - find a variable in an environment */
LVAL findvar(LVAL env,LVAL var,int *poff)
{
    LVAL names;
    for (; env != v_nil; env = getnextframe(env)) {
	names = getenvnames(env);
	for (*poff = FIRSTENV; names != v_nil; ++(*poff), names = cdr(names))
	    if (var == car(names))
		return env;
	if (menvp(env)) {
	    names = getivar(getenvelement(getnextframe(env),FIRSTENV),IV_IVARS);
	    for (*poff = FIRSTIVAR; names != v_nil; ++(*poff), names = cdr(names))
		if (var == car(names))
		    return getenvelement(env,FIRSTENV);
	}
    }
    return v_nil;
}

/* xlapply - apply a function to arguments */
/*	The function should be in xlval and the arguments should
	be on the stack.  The number of arguments should be in xlargc.
*/
void xlapply(void)
{
    LVAL tmp;

    /* check for null function */
    if (null(xlval))
	badfuntype(xlval);

    /* dispatch on function type */
    switch (ntype(xlval)) {
    case SUBR:
	xlfun = xlval;
	xlval = (*getsubr(xlfun))();
	svreturn();
	break;
    case XSUBR:
	xlfun = xlval;
	(*getsubr(xlfun))();
	break;
    case CLOSURE:
	xlfun = getcode(xlval);
	xlenv = getenvironment(xlval);
	base = pc = getcodestr(xlfun);
	break;
    case OBJECT:
	xlsend(xlval,xlgasymbol());
	break;
    case CONTINUATION:
	tmp = xlgetarg();
	xllastarg();
	restore_continuation();
	xlval = tmp;
	svreturn();
	break;
    default:
	badfuntype(xlval);
    }
}

/* make_continuation - make a continuation */
LVAL make_continuation(void)
{
    LVAL cont,*dst,*p;
    FIXTYPE vsize,csize;

    /* unstack all the environments saved on the control stack */
    for (p = xlcsp; p > xlstkbase; )
 	p = cdunstack(p);
 
    /* create a continuation object */
    vsize = xlstktop - xlsp;
    csize = xlcsp - xlstkbase;
    cont = newcontinuation(vsize + csize + 1);

    /* setup the destination pointer */
    dst = &cont->n_vdata[0];

    /* save the size of the value stack */
    *dst++ = cvfixnum((FIXTYPE)vsize);

    /* copy the value stack */
    for (p = xlstktop; --vsize >= 0; )
	*dst++ = *--p;

    /* copy the control stack */
    for (p = xlstkbase; --csize >= 0; )
	*dst++ = *p++;

    /* return the new continuation */
    return cont;
}

/* restore_continuation - restore a continuation to the stack */
/*	The continuation should be in xlval.
*/
static void restore_continuation(void)
{
    FIXTYPE vsize,csize;
    LVAL *src;

    /* setup the source pointer */
    src = &xlval->n_vdata[1];

    /* get the stack sizes */
    vsize = getfixnum(getelement(xlval,0));
    csize = getsize(xlval) - vsize - 1;

    /* restore the value stack */
    for (xlsp = xlstktop; --vsize >= 0; )
	*--xlsp = *src++;

    /* restore the control stack */
    for (xlcsp = xlstkbase; --csize >= 0; )
	*xlcsp++ = *src++;
}

/* unstack_environment - move stack frames to the heap */
LVAL unstack_environment(LVAL env)
{
    LVAL last,old,new,*src,*dst;
    FIXTYPE size;

    /* initialize */
    cpush(v_nil);
    last = v_nil;

    /* copy each stack environment frame to the heap */
    while (senvp(env)) {

	/* move ahead to the next frame */
	old = env;
	env = getnextframe(env);

	/* allocate a new frame and copy the data */
	size = getenvsize(old);
	new = newframe(frametype(old) - 1,v_nil,size);
	src = old->n_vdata;
	dst = new->n_vdata;
	while (--size >= 0)
	    *dst++ = *src++;

	/* link the new frame into the new environment */
	if (last == v_nil)
	    settop(new);
	else
	    setnextframe(last,new);
	last = new;
	
	/* store the forwarding address */
	setframetype(old,MSENV);
	setforwardingaddr(old,new);
    }

    /* link the first heap frame into the new environment */
    if (last == v_nil)
	settop(env);
    else
	setnextframe(last,env);

    /* return the new environment */
    return pop();
}

/* stkframe - create an environment frame on the control stack */
static void stkframe(int argc,int extra,LVAL vnames)
{
    LVAL env;
    int n;

    /* expand the argument frame if necessary */
    if (extra > 0) {
	check(extra);
	for (n = extra; --n >= 0; )
	    push(v_nil);
    }

    /* make sure we have enough space on the control stack */
    Ccheck(NODEWORDS + 1);

    /* make the environment frame */
    env = (LVAL)xlcsp;
    env->n_type = SENV;
    env->n_vsize = argc + extra;
    env->n_vdata = xlsp;
    xlcsp += NODEWORDS;

    /* store the environment variable names */
    setnextframe(env,xlenv);
    setenvnames(env,vnames);

    /* establish the new frame */
    Cpush((LVAL)&cd_frame);
    xlenv = env;
}

/* callerrorhandler - call the error handler */
void callerrorhandler(void)
{
    xlval = xlenter("*ERROR-HANDLER*");
    if ((xlval = getvalue(xlval)) != v_nil)
	CallFunction(&xlval,xlval,2,xlfun,xlenv);
    throwerror(s_error);
}

/* gc_protect - protect the state of the interpreter from the collector */
void gc_protect(void (*protected_fcn)(void))
{
    int pcoff;
    pcoff = pc - base;
    (*protected_fcn)();
    if (codep(xlfun)) {
	base = getcodestr(xlfun);
	pc = base + pcoff;
    }
}

/* show_control_stack - display some of the control stack */
void show_control_stack(int cmax)
{
    LVAL *p;
    errputstr("\nControl Stack Top:");
    for (p = xlcsp; --cmax >= 0 && p > xlstkbase; )
	p = cdprint(p);
}

/* show_value_stack - display some of the value stack */
void show_value_stack(int vmax)
{
    LVAL *p;
    errputstr("\nValue Stack Top:");
    for (p = xlsp; --vmax >= 0 && p < xlstktop; ) {
	errputstr("\n  ");
	errprint(*p++);
    }
}

/* badfuntype - bad function error */
static void badfuntype(LVAL arg)
{
    xlerror("bad function type",arg);
}

/* badargtype - bad argument type error */
static void badargtype(LVAL arg)
{
    xlbadtype(arg);
}

/* xlstkover - value stack overflow */
void xlstkover(void)
{
    xlfmtabort("value stack overflow");
}

/* xlcstkover - control stack overflow */
void xlcstkover(void)
{
    xlfmtabort("control stack overflow");
}

/* xhack - control the hacking function */
LVAL xhack(void)
{
    LVAL arg = xlgetarg();
    xllastarg();
    hacking = arg == v_nil ? FALSE : TRUE;
    return v_nil;
}

/* xltoofew - too few arguments to this function */
LVAL xltoofew(void)
{
    xlfmterror("too few arguments");
    return v_nil; /* never reached */
}

/* xltoomany - too many arguments to this function */
void xltoomany(void)
{
    xlfmterror("too many arguments");
}

/* xlbadtype - incorrect argument type */
LVAL xlbadtype(LVAL val)
{
    xlerror("incorrect type",val);
    return v_nil; /* never reached */
}
