/* xlio - xlisp i/o routines */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* global variables */
FIXTYPE xlfsize;

/* external variables */
extern LVAL s_stdin,s_stdout,s_stderr,s_unbound;

/* forward declarations */
static int fstream_getc(LVAL fptr);
static int ustream_getc(LVAL fptr);
static int ostream_getc(LVAL fptr);
static void fstream_putc(LVAL fptr,int ch);
static void ustream_putc(LVAL fptr,int ch);
static void ostream_putc(LVAL fptr,int ch);

/* xlgetc - get a character from a file or stream */
int xlgetc(LVAL fptr)
{
    int flags,ch;

    /* check for input from nil */
    if (fptr == v_nil)
	return EOF;

    /* check for a buffered character */
    if ((ch = getsavech(fptr)) != '\0') {
	setsavech(fptr,'\0');
	return ch;
    }

    /* otherwise, dispatch on stream type */
    switch (ntype(fptr)) {
    case FSTREAM:
    	ch = fstream_getc(fptr);
    	break;
    case USTREAM:
    	ch = ustream_getc(fptr);
    	break;
    case OSTREAM:
    	ch = ostream_getc(fptr);
    	break;
    default:
    	xlerror("expecting stream",fptr);
    	break;
    }

    /* set the beginning of line flag */
    if ((flags = getpflags(fptr)) & PF_TERMINAL)
	setpflags(fptr,ostatbol() ? flags | PF_BOL : flags & ~PF_BOL);
    else
	setpflags(fptr,ch == '\n' ? flags | PF_BOL : flags & ~PF_BOL);

    /* return the character */
    return ch;
}

/* fstream_getc - port getc routine */
static int fstream_getc(LVAL fptr)
{    
    int flags,ch;
    
    /* check for terminal input or file input */
    if ((flags = getpflags(fptr)) & PF_TERMINAL)
	ch = ostgetc();
    else
        ch = getc(getfile(fptr));
    
    /* return the character */
    return ch;
}

/* ustream_getc - unnamed stream getc routine */
static int ustream_getc(LVAL fptr)
{
    LVAL head = getstrhead(fptr);
    FIXTYPE iptr;
    LVAL buf;
    int ch;
    
    /* check for no buffer */
    if (head == v_nil)
        ch = EOF;
    else {
        buf = car(head);
        iptr = getfixnum(getstriptr(fptr));
        ch = getstring(buf)[iptr++];
        if (head == getstrtail(fptr)) {
            if (iptr >= getfixnum(getstroptr(fptr))) {
                setstrhead(fptr,v_nil);
                setstrtail(fptr,v_nil);
                setstroptr(fptr,cvsfixnum(0));
                iptr = 0;
            }
        }
        else if (iptr >= getslength(buf)) {
            setstrhead(fptr,cdr(head));
            iptr = 0;
        }
        setstriptr(fptr,cvfixnum(iptr));
    }
    return ch;
}

/* ostream_getc - object stream getc routine */
static int ostream_getc(LVAL fptr)
{
    LVAL val;
    CallFunction(&val,getsobject(fptr),1,xlenter("GETC"));
    if (!charp(val))
        xlerror("expecting a character",val);
    return getchcode(val);
}

/* xlungetc - unget a character */
void xlungetc(LVAL fptr,int ch)
{
    /* check for ungetc from nil */
    if (fptr == v_nil)
	return;
	
    /* otherwise, store the character */
    switch (ntype(fptr)) {
    case FSTREAM:
    case USTREAM:
    case OSTREAM:
	setsavech(fptr,ch);
	break;
    default:
        xlerror("expecting stream",fptr);
        break;
    }
}

/* xlpeek - peek at a character from a file or stream */
int xlpeek(LVAL fptr)
{
    int ch;

    /* check for input from nil */
    if (fptr == v_nil)
	return EOF;

    /* otherwise, dispatch on stream type */
    switch (ntype(fptr)) {
    case FSTREAM:
    case USTREAM:
    case OSTREAM:
	ch = xlgetc(fptr);
	setsavech(fptr,ch);
	break;
    default:
        xlerror("expecting stream",fptr);
        break;
    }

    /* return the character */
    return ch;
}

/* xliready - check for a character from a file or stream */
int xliready(LVAL fptr)
{
    int sts;

    /* check for input from nil */
    if (fptr == v_nil)
	return FALSE;

    /* otherwise, dispatch on stream type */
    switch (ntype(fptr)) {
    case FSTREAM:
    case USTREAM:
    case OSTREAM:
	if (getsavech(fptr) != '\0')
	    sts = TRUE;
	else if (getpflags(fptr) & PF_TERMINAL)
	    sts = ostcheck() != 0;
	else
	    sts = xlpeek(fptr) != EOF;
    	break;
    default:
    	xlerror("expecting stream",fptr);
    	break;
    }

    /* return the status */
    return sts;
}

/* xliflush - flush buffered input from a file or stream */
void xliflush(LVAL fptr)
{
    LVAL tmp;
    
    /* check for input from nil */
    if (fptr == v_nil)
	return;

    /* otherwise, dispatch on stream type */
    switch (ntype(fptr)) {
    case FSTREAM:
    	setsavech(fptr,'\0');
	if (getpflags(fptr) & PF_TERMINAL)
	    ostflush();
    	break;
    case USTREAM:
    	setsavech(fptr,'\0');
        setstrhead(fptr,v_nil);
        setstrtail(fptr,v_nil);
        setstriptr(fptr,cvsfixnum(0));
        setstroptr(fptr,cvsfixnum(0));
    	break;
    case OSTREAM:
     	setsavech(fptr,'\0');
	CallFunction(&tmp,getsobject(fptr),1,xlenter("FLUSH"));
   	break;
    default:
    	xlerror("expecting stream",fptr);
    	break;
    }
}

/* xlputc - put a character to a file or stream */
void xlputc(LVAL fptr,int ch)
{
    int flags;
    
    /* count the character */
    ++xlfsize;

    /* check for output to nil */
    if (fptr == v_nil)
	return;

    /* otherwise, check for output to an unnamed stream */
    switch (ntype(fptr)) {
    case FSTREAM:
        fstream_putc(fptr,ch);
        break;
    case USTREAM:
        ustream_putc(fptr,ch);
        break;
    case OSTREAM:
    	ostream_putc(fptr,ch);
	break;
    default:
        xlerror("expecting stream",fptr);
        break;
    }

    /* set the beginning of line flag */
    if ((flags = getpflags(fptr)) & PF_TERMINAL)
	setpflags(fptr,ostatbol() ? flags | PF_BOL : flags & ~PF_BOL);
    else
	setpflags(fptr,ch == '\n' ? flags | PF_BOL : flags & ~PF_BOL);
}

/* fstream_putc - port putc */
static void fstream_putc(LVAL fptr,int ch)
{
    int flags;
    if ((flags = getpflags(fptr)) & PF_TERMINAL)
	ostputc(ch);
    else
	putc(ch,getfile(fptr));
}

/* ustream_putc - unnamed stream putc */
static void ustream_putc(LVAL fptr,int ch)
{
    LVAL tail = getstrtail(fptr);
    FIXTYPE optr;
    LVAL buf;
        
    /* check for no buffer */
    if (tail == v_nil) {
        cpush(fptr);
        buf = xlnewstring(USTR_BUFSIZE);
        setstrhead(fptr,cons(buf,v_nil));
        setstrtail(fptr,getstrhead(fptr));
        setstriptr(fptr,cvsfixnum(0));
        optr = 0;
        drop(1);
    }
    else {
        buf = car(tail);
        optr = getfixnum(getstroptr(fptr));
        if (optr >= getslength(buf)) {
            cpush(fptr);
            buf = xlnewstring(USTR_BUFSIZE);
            rplacd(tail,cons(buf,v_nil));
            setstrtail(fptr,cdr(tail));
            optr = 0;
            drop(1);
        }
    }
    
    /* put the next character in the stream */
    getstring(buf)[optr++] = ch;
    setstroptr(fptr,cvfixnum(optr));    
}

/* ostream_putc - object stream putc */
static void ostream_putc(LVAL fptr,int ch)
{
    LVAL tmp;
    CallFunction(&tmp,getsobject(fptr),2,xlenter("PUTC"),cvchar(ch));
}

/* xlflush - flush the input buffer */
void xlflush(void)
{
    LVAL port = curinput();
    if (portp(port))
	setsavech(port,'\0');
    ostflush();
}

/* getstrlength - get the length of an output stream */
FIXTYPE getstrlength(LVAL stream)
{
    FIXTYPE len,bufcount;
    if ((bufcount = length(getstrhead(stream))) == 0)
        len = 0;
    else if (bufcount == 1)
        len = getfixnum(getstroptr(stream))
            - getfixnum(getstriptr(stream));
    else
        len = (USTR_BUFSIZE - getfixnum(getstriptr(stream)))
            + (bufcount - 2) * USTR_BUFSIZE
            + getfixnum(getstroptr(stream));
    if (getsavech(stream) != '\0')
        ++len;
    return len;
}

/* getstroutput - get the output stream string */
LVAL getstroutput(LVAL stream)
{
    FIXTYPE length;
    char *dst;
    LVAL val;

    /* compute the length of the stream */
    length = getstrlength(stream);
    
    /* create a new string */
    cpush(stream);
    val = xlnewstring(length);
    drop(1);
    
    /* copy the characters into the new string */
    dst = getstring(val);
    while (--length >= 0)
	*dst++ = xlgetc(stream);

    /* return the string */
    return val;
}

/* stdputstr - print a string to *standard-output* */
void stdputstr(char *str)
{
    xlputstr(curoutput(),str);
}

/* errprint - print to *error-output* */
void errprint(LVAL expr)
{
    xlprin1(expr,curerror());
}

/* errprinc - princ to *error-output* */
void errprinc(LVAL expr)
{
    xlprinc(expr,curerror());
}

/* errputstr - print a string to *error-output* */
void errputstr(char *str)
{
    xlputstr(curerror(),str);
}

