/* xlisp.c - xlisp main routine */
/*	Copyright (c) 1996, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include <stdarg.h>
#include "xlisp.h"

/* the program banner */
#define BANNER	"\
XLISP 3.0, August 1, 1996, Copyright (c) 1984-96, by David Betz"

/* global variables */
int clargc = 0;	                /* command line argument count */
char **clargv = NULL;	        /* array of command line arguments */
int xlinitialized = FALSE;      /* true if initialization is done */

/* trace file pointer */
FILE *tfp=NULL;

/* external variables */
extern LVAL s_package,s_unbound,s_stderr,s_error,s_backtrace;
extern int trace;

/* local prototypes */
static void fmterror(char *tag,char *fmt,va_list ap);
static void usage(void);

/* xlmain - the main routine */
void xlmain(int argc,char *argv[])
{
    ERRORTARGET target;
    LVAL fun;
    
    /* setup */
    if (!xlsetup(argc,argv))
	xlfatal("initialization failed");
    
    /* call the top level function */
    pushtarget(&target);
    if (setjmp(target.target) == 0) {
	fun = xlenter("*TOPLEVEL*");
        fun = (boundp(fun) ? getvalue(fun) : v_nil);
        if (!closurep(fun))
            xlfatal("*TOPLEVEL* not bound to a function");
	xlCallFunction(&xlval,fun,0);
    }
    poptarget();
    
    /* finish up and exit */
    xlwrapup();
}

/* xlsetup - the setup routine */
int xlsetup(int argc,char *argv[])
{
    extern FIXTYPE nssize,vssize;
    ERRORTARGET target;
    int src,dst;
    LVAL fun;
    char *p;
    
    /* process the arguments */
    clargc = argc == 0 ? 0 : 1;
    for (src = dst = 1, clargv = argv; src < argc; ++src) {

	/* handle options */
	if (argv[src][0] == '-') {
	    for (p = &argv[src][1]; *p != '\0'; )
	    	switch (*p++) {
		case 't':		/* root directory */
		    trace = TRUE;
		    break;
		case 'n':		/* node segment size */
		    nssize = atol(p);
		    p = "";
		    break;
		case 'v':		/* vector segment size */
		    vssize = atol(p);
		    p = "";
		    break;
		default:
	    	    usage();
		}
	}

	/* handle a filename */
	else {
	    argv[dst++] = argv[src];
	    ++clargc;
	}
    }

    /* initialize */
    osinit(BANNER);

    /* setup an initialization error handler */
    pushtarget(&target);
    if (setjmp(target.target)) {
	poptarget();
	return FALSE;
    }

    /* restore the default workspace, otherwise create a new one */
    if (!xlirestore("xlisp.wks"))
	xlinitws(STACK_SIZE);
	
    /* call the initialization function */
    fun = xlenter("*INITIALIZE*");
    fun = (boundp(fun) ? getvalue(fun) : v_nil);
    if (!closurep(fun))
        xlfatal("*INITIALIZE* not bound to a function");
    xlCallFunction(&xlval,fun,0);

    /* return successfully */
    xlinitialized = TRUE;
    poptarget();
    return TRUE;
}

/* usage - display a usage message and exit */
static void usage(void)
{
    xlinfo("usage: xlisp [-t] [-n<size>] [-v<size>]\n");
    osexit(1);
}

/* stubs for things we will implement someday */
void xlcontinue(void) {}
void xlbreak(void) { xltoplevel(); }

/* xlcleanup - cleanup after an error (leave the debugger) */
void xlcleanup(void)
{
    throwerror(xlenter("CLEANUP"));
}

/* xltoplevel - return to the top level */
void xltoplevel(void)
{
    throwerror(xlenter("RESET"));
}

/* xlerror - print an error message */
void xlerror(char *msg,LVAL arg)
{
    char fmt[256];
    sprintf(fmt,"%s - ~S",msg);
    xlfmterror(fmt,arg);
}

/* xlfmterror - report an error */
void xlfmterror(char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);
    fmterror("\nError: ",fmt,ap);
    va_end(ap);
    callerrorhandler();
}

/* xlabort - print an error message and abort */
void xlabort(char *msg,LVAL arg)
{
    char fmt[256];
    sprintf(fmt,"%s - ~S",msg);
    xlfmtabort(fmt,arg);
}

/* xlfmtabort - report an error */
void xlfmtabort(char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);
    fmterror("\nAbort: ",fmt,ap);
    va_end(ap);
    jumptotarget(-1);
}

/* fmterror - report an error */
static void fmterror(char *tag,char *fmt,va_list ap)
{
    LVAL stream = getvalue(s_stderr);
    int ch;
    
    /* an opportunity to break out */
    oscheck();
    
    /* flush the input buffer */
    xlflush();
    
    /* display the error message */
    errputstr(tag);
    
    /* process the format string */
    for (;;) {
        if ((ch = *fmt++) == '\0')
            break;
	else if (ch == '~' && *fmt != '\0') {
	    switch (*fmt++) {
	    case 'a': case 'A':
		xlprinc(va_arg(ap,LVAL),stream);
		break;
	    case 's': case 'S':
		xlprin1(va_arg(ap,LVAL),stream);
		break;
	    case '%':
		xlterpri(stream);
		break;
	    case '~':
		xlputc(stream,'~');
		break;
	    case '\n':
		while ((ch = *fmt) != '\0' && isspace(ch) && ch != '\n')
		    ++fmt;
		break;
	    default:
		xlerror("unknown format directive",cvchar(ch));
	    }
	}
	else
	    xlputc(stream,ch);
    }
    
    /* print the function where the error occurred */
    xlshowerr(xlfun);
}

/* xlshowerr - show where the error happened */
void xlshowerr(LVAL fun)
{
    LVAL levels = getvalue(s_backtrace);

    /* print the function where the error occurred */
    if (fun != v_nil) {
	errputstr("\nhappened in: ");
	errprint(fun);
    }

    /* show the call stack */
    if (fixp(levels)) {
	errputstr("\ncall stack:");
	show_call_stack((int)getfixnum(levels));
    }
}

/* xlfatal - print a fatal error message and exit */
void xlfatal(char *fmt,...)
{
    char buf[100];
    va_list ap;
    va_start(ap,fmt);
    vsprintf(buf,fmt,ap);
    va_end(ap);
    oserror(buf);
    osexit(1);
}

/* xlinfo - display debugging information */
void xlinfo(char *fmt,...)
{
    char buf[100],*p=buf;
    va_list ap;
    va_start(ap,fmt);
    vsprintf(buf,fmt,ap);
    va_end(ap);
    while (*p != '\0')
	ostputc(*p++);
}

/* xlwrapup - clean up and exit to the operating system */
void xlwrapup(void)
{
    if (tfp != NULL)
	osclose(tfp);
    osexit(0);
}
