/* xlisp.h - xlisp definitions */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#ifndef __XLISP_H__
#define __XLISP_H__

#include <stdio.h>
#include <ctype.h>
#include <setjmp.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>

/* AFMT		printf format for addresses ("%x") */
/* OFFTYPE	number the size of an address (int) */
/* FIXTYPE	data type for fixed point numbers (long) */
/* UFIXTYPE	data type for fixed point numbers (unsigned long) */
/* ICNV		fixed point input conversion routine (atol) */
/* IFMT		printf format for fixed point numbers ("%ld") */
/* FLOTYPE	data type for floating point numbers (float) */
/* FFMT		printf format for floating point numbers (%.15g) */
/* STACK_SIZE	size of control/data stack */
/* EXPORT	storage class for exported functions */

/* use zero for nil for now */
#define v_nil		(LVAL)0

/* for the Think C compiler - Macintosh */
#ifdef THINKC
#define AFMT		"%lx"
#define OFFTYPE		long
#define ADDR32
#define MACINTOSH
#endif

/* for the UNIX System V C compiler */
#ifdef UNIX
#endif

/* for the Aztec C compiler - Amiga */
#ifdef AZTEC_AMIGA
#define AFMT		"%lx"
#define OFFTYPE		long
#endif

/* for the Mark Williams C compiler - Atari ST */
#ifdef MWC
#define AFMT		"%lx"
#define OFFTYPE		long
#endif

/* for the Microsoft C 6.0 compiler */
#ifdef MSC
#ifndef MSDOS
#define MSDOS
#endif
#define SEGADDR
#endif

/* for the Turbo (Borland) C compiler */
#ifdef TURBOC
#ifndef MSDOS
#define MSDOS
#endif
#define SEGADDR
#endif

/* for the Turbo (Borland) C compiler for OS/2 */
#ifdef TURBOC_OS2
#define ADDR32
#endif

/* for the IBM CSet++ compiler */
#ifdef IBM_CSET
#define ADDR32
#endif

/* for the Zortec C++ compiler */
#ifdef ZTC
#ifndef MSDOS
#define MSDOS	1
#endif
#define SEGADDR
void free(void *);
#define STACK_SIZE	(100*1024)
#endif

/* for the Watcom C compiler */
#ifdef WTC
#define ADDR32
#ifndef MSDOS
#define MSDOS
#endif
#endif

/* for the Metaware High C compiler */
#ifdef HIGHC
#define ADDR32
#ifndef MSDOS
#define MSDOS
#endif
#endif

/* for the MS-DOS compilers */
#ifdef MSDOS
#define AFMT		"%lx"
#define OFFTYPE		long
#endif

/* for segmented addresses on Intel processors */
#ifdef SEGADDR
#define INSEGMENT(n,s)	((unsigned long)(n) >> 16 == (unsigned long)(s) >> 16)
#define VCOMPARE(f,s,t)	((s) <= (FIXTYPE)(unsigned int)((t) - (f)))
#define VSFREE(f,t)     ((t) - (f))
#endif

/* size of each type of memory segment */
#ifndef NSSIZE	/* number of nodes per node segment */
#ifdef ADDR32
#define NSSIZE	20000
#else
#define NSSIZE	6400
#endif
#endif
#ifndef VSSIZE	/* number of LVAL's per vector segment */
#ifdef ADDR32
#define VSSIZE	200000
#else
#define VSSIZE	16000
#endif
#endif

/* default important definitions */
#ifndef AFMT
#define AFMT		"%x"
#endif
#ifndef OFFTYPE
#define OFFTYPE		int
#endif
#ifndef FIXTYPE
#define FIXTYPE		long
typedef unsigned long	UFIXTYPE;
#endif
#ifndef ICNV
#define ICNV(n)		atol(n)
#endif
#ifndef IFMT
#define IFMT		"%ld"
#endif
#ifndef FLOTYPE
#define FLOTYPE		double
#endif
#ifndef FFMT
#define FFMT		"%.15g"
#endif
#ifndef SFIXMIN
#define SFIXMIN		-1073741824L
#define SFIXMAX		1073741823L
#endif
#ifndef INSEGMENT
#define INSEGMENT(n,s)	((n) >= &(s)->ns_data[0] \
                      && (n) <  &(s)->ns_data[0] + (s)->ns_size)
#endif
#ifndef VCOMPARE
#define VCOMPARE(f,s,t)	((s) <= (FIXTYPE)((t) - (f)))
#endif
#ifndef VSFREE
#define VSFREE(f,t)     ((FIXTYPE)((t) - (f)))
#endif
#ifndef STACK_SIZE
#define STACK_SIZE	8191
#endif
#ifndef EXPORT
#define EXPORT
#endif

/* useful definitions */
#ifndef TRUE
#define TRUE	1
#define FALSE	0
#endif

/* program limits */
#define STRMAX		256		/* maximum length of string constant */
#define HSIZE		199		/* symbol hash table size */
#define SRATE		1000		/* control character sample rate */

/* value stack manipulation macros */
#define check(n)	{ if (xlsp - (n) <= xlcsp) xlstkover(); }
#define cpush(v)	{ check(1); push(v); }
#define push(v)		(*--xlsp = (v))
#define pop()		(*xlsp++)
#define top()		(*xlsp)
#define settop(v)	(*xlsp = (v))
#define drop(n)		(xlsp += (n))

/* control stack manipulation macros */
#define Ccheck(n)	{ if (xlcsp + (n) >= xlsp) xlcstkover(); }
#define Cpush(v)  	(*xlcsp++ = (v))
#define Cpop()		(*--xlcsp)
#define Cdrop(n)  	(xlcsp -= (n))

/* argument list parsing macros */
#define xlgetarg()	(testarg(nextarg()))
#define xllastarg()	{ if (moreargs()) xltoomany(); }
#define xlpopargs()	drop(xlargc)
#define testarg(e)	(moreargs() ? (e) : xltoofew())
#define typearg(tp)	(tp(*xlsp) ? nextarg() : xlbadtype(*xlsp))
#define nextarg()	(--xlargc, *xlsp++)
#define moreargs()	(xlargc > 0)

/* macros to get arguments of a particular type */
#define xlgacons()	(testarg(typearg(consp)))
#define xlgalist()	(testarg(typearg(listp)))
#define xlgasymbol()	(testarg(typearg(symbolp)))
#define xlgastring()	(testarg(typearg(stringp)))
#define xlgafixnum()	(testarg(typearg(fixp)))
#define xlganumber()	(testarg(typearg(numberp)))
#define xlgachar()	(testarg(typearg(charp)))
#define xlgavector()	(testarg(typearg(vectorp)))
#define xlgaport()	(testarg(typearg(portp)))
#define xlgaiport()	(testarg(typearg(iportp)))
#define xlgaoport()	(testarg(typearg(oportp)))
#define xlgafstream()	(testarg(typearg(fstreamp)))
#define xlgaustream()	(testarg(typearg(ustreamp)))
#define xlgaclosure()	(testarg(typearg(closurep)))
#define xlgaenv()	(testarg(typearg(envp)))
#define xlgaobject()	(testarg(typearg(objectp)))
#define xlgaforeignptr() (testarg(typearg(foreignptrp)))
#define xlgatable()     (testarg(typearg(tablep)))

/* node types */
#define FREE		0
#define CONS		1
#define SYMBOL		2
#define FIXNUM		3
#define FLONUM		4
#define STRING		5
#define FSTREAM		6	/* file stream */
#define USTREAM		7	/* unnamed stream */
#define OSTREAM		8	/* object stream */
#define VECTOR		9
#define CLOSURE		10
#define CODE		11
#define SUBR		12
#define XSUBR		13
#define CONTINUATION	14
#define CHARACTER	15
#define PROMISE		16
#define ENV		17	/* heap environment */
#define SENV		18	/* stack environment (must be SENV + 1) */
#define MSENV		19	/* moved stack environment (or method environment) */
#define MENV		20	/* method environment */
#define SMENV		21	/* stack method environment (must be MENV + 1) */
#define OBJECT		22
#define PACKAGE		23
#define FOREIGNPTR	24	/* foreign pointer */
#define TABLE           25
#define MAXTYPEID	25

/* update the typename table in xlprin.c if you add more type ids to the above list */

/* node flags */
#define MARK		1
#define LEFT		2

/* port flags */
#define PF_INPUT	0x0001
#define PF_OUTPUT	0x0002
#define PF_BINARY	0x0004
#define PF_TERMINAL	0x0008
#define PF_BOL		0x0010

/* new node access macros */
#define ntype(x)	((OFFTYPE)(x) & 1 ? FIXNUM : (x)->n_type)
#define typeis(x,t)	((x) != v_nil && ntype(x) == (t))
#define ptypeis(x,t)	((x)->n_type == (t))

/* macro to determine if a non-nil value is a pointer */
#define ispointer(x)	((x) != v_nil && ((OFFTYPE)(x) & 1) == 0)

/* type predicates */			       
#define atom(x)		((x) == v_nil || ntype(x) != CONS)
#define null(x)		((x) == v_nil)
#define listp(x)	((x) == v_nil || ntype(x) == CONS)
#define numberp(x)	((x) != v_nil && (ntype(x) == FIXNUM || ntype(x) == FLONUM))
#define boundp(x)	(getvalue(x) != s_unbound)
#define portp(x)	(ispointer(x) \
                      && (ptypeis(x,FSTREAM) \
                       || ptypeis(x,USTREAM) \
                       || ptypeis(x,OSTREAM)))
#define iportp(x)	(portp(x) && (getpflags(x) & PF_INPUT) != 0)
#define oportp(x)	(portp(x) && (getpflags(x) & PF_OUTPUT) != 0)

/* basic type predicates */			       
#define consp(x)	typeis(x,CONS)
#define stringp(x)	typeis(x,STRING)
#define symbolp(x)	typeis(x,SYMBOL)
#define fstreamp(x)	typeis(x,FSTREAM)
#define ustreamp(x)	typeis(x,USTREAM)
#define ostreamp(x)	typeis(x,OSTREAM)
#define fixp(x)		typeis(x,FIXNUM)
#define floatp(x)	typeis(x,FLONUM)
#define vectorp(x)	typeis(x,VECTOR)
#define closurep(x)	typeis(x,CLOSURE)
#define continuationp(x) typeis(x,CONTINUATION)
#define codep(x)	typeis(x,CODE)
#define subrp(x)	typeis(x,SUBR)
#define xsubrp(x)	typeis(x,XSUBR)
#define charp(x)	typeis(x,CHARACTER)
#define promisep(x)	typeis(x,PROMISE)
#define envp(x)		((x) == v_nil \
                         || ntype(x) == ENV \
			 || ntype(x) == SENV \
			 || ntype(x) == MENV \
			 || ntype(x) == SMENV)
#define senvp(x)	((x) != v_nil && (ntype(x) == SENV || ntype(x) == SMENV))
#define msenvp(x)	typeis(x,MSENV)
#define menvp(x)	((x) != v_nil && (ntype(x) == MENV || ntype(x) == SMENV))
#define objectp(x)	typeis(x,OBJECT)
#define booleanp(x)	((x) == v_nil || ntype(x) == BOOLEAN)
#define packagep(x)	typeis(x,PACKAGE)
#define foreignptrp(x)	typeis(x,FOREIGNPTR)
#define tablep(x)	typeis(x,TABLE)

/* vector update macro
   This is necessary because the memory pointed to by the n_vdata field
   of a vector object can move during a garbage collection.  This macro
   guarantees that evaluation happens in the right order.
*/
#define vupdate(x,i,v)	do { \
                          LVAL vutmp=(v); \
			  (x)->n_vdata[i] = vutmp; \
			} while (0)

/* cons access macros */
#define car(x)		((x)->n_car)
#define cdr(x)		((x)->n_cdr)
#define rplaca(x,y)	((x)->n_car = (y))
#define rplacd(x,y)	((x)->n_cdr = (y))

/* package access macros */
#define getnames(x)		getelement(x,0)
#define setnames(x,v)		setelement(x,0,v)
#define getextern(x)		getelement(x,1)
#define setextern(x,v)		setelement(x,1,v)
#define getintern(x)		getelement(x,2)
#define setintern(x,v)		setelement(x,2,v)
#define getuses(x)		getelement(x,3)
#define setuses(x,v)		setelement(x,3,v)
#define getusedby(x)		getelement(x,4)
#define setusedby(x,v)		setelement(x,4,v)
#define getnextpackage(x)	getelement(x,5)
#define setnextpackage(x,v)	setelement(x,5,v)
#define PAKSIZE			6

/* symbol access macros */
#define getvalue(x)	getelement(x,0)
#define setvalue(x,v)	setelement(x,0,v)
#define getpname(x)	getelement(x,1)
#define setpname(x,v)	setelement(x,1,v)
#define getplist(x)	getelement(x,2)
#define setplist(x,v)	setelement(x,2,v)
#define getpackage(x)	getelement(x,3)
#define setpackage(x,v)	setelement(x,3,v)
#define SYMSIZE		4

/* vector access macros */
#define getsize(x)	((x)->n_vsize)
#define setsize(x,v)	((x)->n_vsize = (v))
#define getelement(x,i)	((x)->n_vdata[i])
#define setelement(x,i,v) vupdate(x,i,v)

/* environment access macros */
#define frametype(x)		((x)->n_type)
#define setframetype(x,v)	((x)->n_type = (v))
#define getenvsize(x)		getsize(x)
#define setenvsize(x,v)		((x)->n_vsize = (v))
#define getnextframe(x)		getelement(x,0)
#define setnextframe(x,v)	setelement(x,0,v)
#define getenvnames(x)		getelement(x,1)
#define setenvnames(x,v)	setelement(x,1,v)
#define getenvelement(x,i)	getelement(x,i)
#define setenvelement(x,i,v)	setelement(x,i,v)
#define FIRSTENV		2

/* moved stack environment access macros */
#define getforwardingaddr(x)	getelement(x,0)
#define setforwardingaddr(x,v)	setelement(x,0,v)

/* object access macros */
#define getclass(x)	getenvelement(x,0)
#define setclass(x,v)	setenvelement(x,0,v)
#define getivar(x,i)	getenvelement(x,i)
#define setivar(x,i,v)	setenvelement(x,i,v)
#define FIRSTIVAR	1

/* instance variable numbers for the class 'Class' */
#define IV_CLASSNAME	(FIRSTIVAR+0)	/* class name */
#define IV_MESSAGES	(FIRSTIVAR+1)	/* list of messages */
#define IV_IVARS	(FIRSTIVAR+2)	/* list of instance variable names */
#define IV_CVARS	(FIRSTIVAR+3)	/* env containing class variables */
#define IV_SUPERCLASS	(FIRSTIVAR+4)	/* pointer to the superclass */
#define IV_IVARCNT	(FIRSTIVAR+5)	/* number of class instance variables */
#define IV_IVARTOTAL	(FIRSTIVAR+6)	/* total number of instance variables */

/* number of instance variables for the class 'Class' */
#define CLASSSIZE	7

/* promise access macros */
#define getpproc(x)	car(x)
#define setpproc(x,v)	rplaca(x,v)
#define getpvalue(x)	cdr(x)
#define setpvalue(x,v)	rplacd(x,v)

/* closure access macros */
#define getcode(x)		car(x)
#define getenvironment(x)	cdr(x)

/* code access macros */
#define getbcode(x)		getelement(x,0)
#define setbcode(x,v)		setelement(x,0,v)
#define getcname(x)		getelement(x,1)
#define setcname(x,v)		setelement(x,1,v)
#define getvnames(x)		getelement(x,2)
#define setvnames(x,v)		setelement(x,2,v)
#define getcodestr(x)		((unsigned char *)getstring(getbcode(x)))
#define FIRSTLIT		3

/* fixnum/flonum/character access macros */
#define getfixnum(x)	((OFFTYPE)(x) & 1 ? getsfixnum(x) : (x)->n_int)
#define getflonum(x)	((x)->n_flonum)
#define getchcode(x)	((x)->n_chcode)

/* small fixnum access macros */
#define fastcvfixnum(x)	(sfixnump(x) ? cvsfixnum(x) : cvfixnum(x))	
#define cvsfixnum(x)	((LVAL)(((OFFTYPE)(x) << 1) | 1))
#define getsfixnum(x)	((FIXTYPE)((OFFTYPE)(x) >> 1))
#define sfixnump(x)	((x) >= SFIXMIN && (x) <= SFIXMAX)
	
/* string access macros */
#define getstring(x)	((char *)(x)->n_vdata)
#define getslength(x)	getsize(x)
#define setslength(x,v)	setsize(x,v)

/* fstream/ustream/ostream access macros */
#define getfile(x)	((FILE *)getsdata(x))
#define getustream(x)	((LVAL)getsdata(x))
#define getsobject(x)	((LVAL)getsdata(x))
#define getsdata(x)	((x)->n_sdata)
#define setsdata(x,v)	((x)->n_sdata = (void *)(v))
#define getsavech(x)	((x)->n_savech)
#define setsavech(x,v)	((x)->n_savech = (v))
#define getpflags(x)	((x)->n_pflags)
#define setpflags(x,v)	((x)->n_pflags = (v))

/* unnamed stream access macros */
#define getstrhead(x)	getelement(getustream(x),0)
#define setstrhead(x,v) setelement(getustream(x),0,v)
#define getstrtail(x)	getelement(getustream(x),1)
#define setstrtail(x,v) setelement(getustream(x),1,v)
#define getstriptr(x)   getelement(getustream(x),2)
#define setstriptr(x,v) setelement(getustream(x),2,v)
#define getstroptr(x)   getelement(getustream(x),3)
#define setstroptr(x,v) setelement(getustream(x),3,v)
#define USTRSIZE	4
#define USTR_BUFSIZE	4096

/* subr access macros */
#define getsubr(x)	((x)->n_subr)
#define getfundef(x)	((x)->n_fundef)

/* foreign pointer access macros */
#define getfptype(x)	((x)->n_fptype)
#define setfptype(x,v)	((x)->n_fptype = (v))
#define getfptr(x)	((x)->n_fptr)
#define setfptr(x,v)	((x)->n_fptr = (v))

/* list node */
#define n_car		n_info.n_xcons.xl_car
#define n_cdr		n_info.n_xcons.xl_cdr

/* integer node */
#define n_int		n_info.n_xint.xi_int

/* flonum node */
#define n_flonum	n_info.n_xflonum.xf_flonum

/* character node */
#define n_chcode	n_info.n_xchar.xc_chcode

/* stream node */
#define n_sdata		n_info.n_xstream.xst_data
#define n_savech	n_info.n_xstream.xst_savech
#define n_pflags	n_info.n_xstream.xst_pflags

/* vector node */
#define n_vsize		n_info.n_xvect.xv_size
#define n_vdata		n_info.n_xvect.xv_data

/* subr node */
#define n_subr		n_info.n_xsubr.xs_subr
#define n_fundef	n_info.n_xsubr.xs_fundef

/* foreign pointer */
#define n_fptype	n_info.n_xforeignptr.xfp_type
#define n_fptr		n_info.n_xforeignptr.xfp_ptr

/* node structure */
typedef struct node {
    char n_type;		/* type of node */
    char n_flags;		/* flag bits */
    union ninfo { 		/* value */
	struct xcons {		/* cons node */
	    struct node *xl_car;	/* the car pointer */
	    struct node *xl_cdr;	/* the cdr pointer */
	} n_xcons;
	struct xint {		/* integer node */
	    FIXTYPE xi_int;		/* integer value */
	} n_xint;
	struct xflonum {	/* flonum node */
	    FLOTYPE xf_flonum;		/* flonum value */
	} n_xflonum;
	struct xchar {		/* character node */
	    int xc_chcode;		/* character code */
	} n_xchar;
	struct xstream {	/* stream node */
	    void *xst_data;		/* the file pointer/stream/object */
	    short xst_savech;		/* lookahead character */
	    short xst_pflags;		/* port flags */
	} n_xstream;
	struct xvect {		/* vector node */
	    FIXTYPE xv_size;		/* vector size */
	    struct node **xv_data;	/* vector data */
	} n_xvect;
	struct xsubr {		/* subr/fsubr node */
	    struct node *(*xs_subr)(void);	/* function pointer */
	    struct fundef *xs_fundef;		/* function definition structure */
	} n_xsubr;
	struct xforeignptr {	/* foreign pointer node */
	    struct node *xfp_type;	/* type symbol */
	    void *xfp_ptr;		/* the pointer */
	} n_xforeignptr;
    } n_info;
} NODE,*LVAL;

/* memory allocator definitions */

/* number of pointers in a protected pointer block */
#define PPBSIZE	100

typedef struct PPBLOCK {
    struct PPBLOCK *next;
    LVAL *pointers[PPBSIZE];
    int count;
} PPBLOCK;

/* macros to compute the size of a segment */
#define nsegsize(n) (sizeof(NSEGMENT) + ((n) - 1) * sizeof(NODE))
#define vsegsize(n) (sizeof(VSEGMENT) + ((n) - 1) * sizeof(LVAL))

/* macro to convert a byte size to a word size */
#define btow_size(n)	(((n) + sizeof(LVAL) - 1) / sizeof(LVAL))

/* number of LVALs in >= the size of a NODE */
#define NODEWORDS	btow_size(sizeof(NODE))

/* node segment structure */
typedef struct nsegment {
    struct nsegment *ns_next;	/* next node segment */
    FIXTYPE ns_size;		/* number of nodes in this segment */
    NODE ns_data[1];		/* segment data */
} NSEGMENT;

/* vector segment structure */
typedef struct vsegment {
    struct vsegment *vs_next;	/* next vector segment */
    LVAL *vs_free;		/* next free location in this segment */
    LVAL *vs_top;		/* top of segment (plus one) */
    LVAL vs_data[1];		/* segment data */
} VSEGMENT;

/* function definition structure */
typedef struct fundef {
    char *fd_name;		/* function name */
    LVAL (*fd_subr)(void);	/* function entry point */
    int fd_pkgid;		/* package identifier */
} FUNDEF;

/* package identifiers */
#define PKG_XLISP	0x0001
#define PKG_SCHEME	0x0002
#define PKG_SYSTEM	0x0004

/* continuation dispatch structure */
typedef struct {
    void (*cd_restore)(void);
    LVAL *(*cd_mark)(LVAL *);
    LVAL *(*cd_unmark)(LVAL *);
    void (*cd_unwind)(void);
    LVAL *(*cd_unstack)(LVAL *);
    LVAL *(*cd_print)(LVAL *);
} CDISPATCH;

/* macros to call the continuation dispatch functions */
#define cdrestore()	((*((CDISPATCH *)Cpop())->cd_restore)())
#define cdmark(p)	(--(p), (*((CDISPATCH *)*(p))->cd_mark)(p))
#define cdunmark(p)	(--(p), (*((CDISPATCH *)*(p))->cd_unmark)(p))
#define cdunwind()	((*((CDISPATCH *)Cpop())->cd_unwind)())
#define cdunstack(p)	(--(p), (*((CDISPATCH *)*(p))->cd_unstack)(p))
#define cdprint(p)	(--(p), (*((CDISPATCH *)*(p))->cd_print)(p))

/* return a single value */
#define svreturn()	do { \
			    xlargc = 1; \
			    cdrestore(); \
			    return; \
			} while (0)
#define mvreturn(n)	do { \
			    int argcxxx = (n); \
			    xlargc = argcxxx; \
			    xlval = argcxxx > 0 ? *xlsp : v_nil; \
			    drop(xlargc); \
			    cdrestore(); \
			    return; \
			} while (0)

/* C continuation structure */
typedef struct {
    void (*cc_cont)(void);	/* continuation function */
    void (*cc_unwind)(void);	/* unwind function (or NULL) */
    int cc_cnt;			/* number of values on control stack */
    char *cc_names;		/* names of the values on the stack (comma separated) */
} CCONTINUATION;

/* error target structure */
typedef struct ERRORTARGET {
    jmp_buf target;
    struct ERRORTARGET *next;
} ERRORTARGET;

/* external variables */
extern int xlinitialized;       /* true if initialization is done */
extern LVAL *xlstkbase; 	/* base of value stack */
extern LVAL *xlstktop;		/* top of value stack */
extern LVAL *xlsp;    		/* value stack pointer */
extern LVAL *xlcsp;    		/* control stack pointer */
extern int xlargc;		/* number of arguments remaining */
extern void (*xlnext)(void);	/* pointer to the next function to call (xlapply) */

/* virtual machine registers */
extern LVAL xlval;
extern LVAL xlfun;
extern LVAL xlenv;

/* important values */
extern LVAL v_true;
extern LVAL v_false;
#ifndef v_nil
extern LVAL v_nil;
#endif

/* API status codes */
#define xlsSuccess      0
#define xlsEndOfFile    1
#define xlsError        2

/* xlisp.c */
void xlmain(int argc,char **argv);
int xlsetup(int argc,char **argv);
LVAL xl_eval(LVAL expr);
void xlcontinue(void);
void xlbreak(void);
void xlcleanup(void);
void xltoplevel(void);
void xlerror(char *msg,LVAL arg);
void xlfmterror(char *fmt,...);
void xlabort(char *msg,LVAL arg);
void xlfmtabort(char *fmt,...);
void xlfatal(char *fmt,...);
void xlinfo(char *fmt,...);
void xlwrapup(void);
void xlshowerr(LVAL fun);

/* xlapi.c */
EXPORT int xlCallFunction(LVAL *pvalue,LVAL fun,int argc,...);
EXPORT int xlCallFunctionByName(LVAL *pvalue,char *fname,int argc,...);
EXPORT int xlSendMessage(LVAL *pvalue,LVAL obj,LVAL selector,int argc,...);
EXPORT int xlSendMessageByName(LVAL *pvalue,LVAL obj,char *selector,int argc,...);
EXPORT int xlEvaluateString(char *str,FIXTYPE len,LVAL *pval);
EXPORT int xlReadFromString(char *str,FIXTYPE len,LVAL *pval);
EXPORT int xlLoadFile(char *fname);
EXPORT int xlEvaluate(LVAL expr,LVAL *pval);
EXPORT char *xlPrintToString(LVAL expr,char *buf,FIXTYPE len);
EXPORT int xlProtect(LVAL *p);

/* xlcom.c */
LVAL xlcompile(LVAL expr,LVAL ctenv);
LVAL xlmethod(LVAL fun,LVAL fargs,LVAL body,LVAL ctenv);
FIXTYPE length(LVAL list);

/* xldbg.c */
void decode_procedure(LVAL fptr,LVAL code,LVAL env);
int decode_instruction(LVAL fptr,LVAL code,FIXTYPE lc,LVAL env);

/* xldmem.c */
LVAL cons(LVAL x,LVAL y);
LVAL newframe(int type,LVAL parent,FIXTYPE size);
LVAL cvstring(char *str,FIXTYPE len);
LVAL cvcstring(char *str);
LVAL copystring(LVAL str);
LVAL cvfstream(FILE *fp,short flags);
LVAL cvustream(char *buf,FIXTYPE len);
LVAL cvostream(LVAL obj,short flags);
LVAL cvsymbol(LVAL pname);
LVAL cvfixnum(FIXTYPE n);
LVAL cvflonum(FLOTYPE n);
LVAL cvchar(int ch);
LVAL cvclosure(LVAL code,LVAL env);
LVAL cvpromise(LVAL code,LVAL env);
LVAL cvsubr(int type,FUNDEF *def);
LVAL newpackage(char *name);
LVAL newvector(FIXTYPE size);
LVAL newtable(FIXTYPE size);
LVAL xlnewstring(FIXTYPE size);
LVAL newcode(FIXTYPE nlits);
LVAL newstream(int type,short flags);
LVAL newustream(void);
LVAL newcontinuation(FIXTYPE size);
LVAL cvforeignptr(LVAL type,void *p);
int checkvmemory(FIXTYPE size);
int makevmemory(FIXTYPE size);
int nexpand(FIXTYPE size);
int vexpand(FIXTYPE size);
NSEGMENT *newnsegment(FIXTYPE n);
VSEGMENT *newvsegment(FIXTYPE n);
void gc(void);
void mark(LVAL);
void check_vector_space(void);
int protectptr(LVAL *p);
void xlminit(FIXTYPE ssize);
void xlresetstack(void);

/* xlfasl.c */
LVAL xfaslwriteprocedure(void);
LVAL xfaslreadprocedure(void);

/* xlftab.c */
void xlfinit(void);
FUNDEF *xlfindsubr(char *name,int *ptype);

/* xlfun1.c */
LVAL xcons(void);
LVAL xacons(void);
LVAL xcar(void);
LVAL xicar(void);
LVAL xcdr(void);
LVAL xicdr(void);
LVAL xcaar(void);
LVAL xcadr(void);
LVAL xcdar(void);
LVAL xcddr(void);
LVAL xcaaar(void);
LVAL xcaadr(void);
LVAL xcadar(void);
LVAL xcaddr(void);
LVAL xcdaar(void);
LVAL xcdadr(void);
LVAL xcddar(void);
LVAL xcdddr(void);
LVAL xcaaaar(void);
LVAL xcaaadr(void);
LVAL xcaadar(void);
LVAL xcaaddr(void);
LVAL xcadaar(void);
LVAL xcadadr(void);
LVAL xcaddar(void);
LVAL xcadddr(void);
LVAL xcdaaar(void);
LVAL xcdaadr(void);
LVAL xcdadar(void);
LVAL xcdaddr(void);
LVAL xcddaar(void);
LVAL xcddadr(void);
LVAL xcdddar(void);
LVAL xcddddr(void);
LVAL xsetcar(void);
LVAL xisetcar(void);
LVAL xsetcdr(void);
LVAL xisetcdr(void);
LVAL xnappend(void);
LVAL xlist(void);
LVAL xliststar(void);
LVAL xpairlis(void);
LVAL xcopylist(void);
LVAL xcopytree(void);
LVAL xcopyalist(void);
LVAL xappend(void);
LVAL xreverse(void);
LVAL xlast(void);
LVAL xlength(void);
LVAL xxmember(void);
LVAL xxmemv(void);
LVAL xxmemq(void);
LVAL xxassoc(void);
LVAL xxassv(void);
LVAL xxassq(void);
LVAL xlistref(void);
LVAL xlisttail(void);
LVAL xmkpackage(void);
LVAL xfindpackage(void);
LVAL xlistallpackages(void);
LVAL xpackagename(void);
LVAL xpkgnicknames(void);
LVAL xinpackage(void);
LVAL xusepackage(void);
LVAL xpkguselist(void);
LVAL xunusepackage(void);
LVAL xpkgusedbylist(void);
LVAL xexport(void);
LVAL xunexport(void);
LVAL ximport(void);
void xintern(void);
void xfindsymbol(void);
LVAL xunintern(void);
LVAL xmksymbol(void);
LVAL xboundp(void);
LVAL xsymname(void);
LVAL xsymvalue(void);
LVAL xsetsymvalue(void);
LVAL xsympackage(void);
LVAL xsymplist(void);
LVAL xsetsymplist(void);
LVAL xget(void);
LVAL xput(void);
LVAL xremprop(void);
LVAL xtheenvironment(void);
LVAL xprocenvironment(void);
LVAL xenvp(void);
LVAL xenvbindings(void);
LVAL xenvparent(void);
LVAL xobjectp(void);
LVAL xvector(void);
LVAL xmakevector(void);
LVAL xvlength(void);
LVAL xivlength(void);
LVAL xvref(void);
LVAL xivref(void);
LVAL xvset(void);
LVAL xivset(void);
LVAL xivbase(void);
LVAL xvectlist(void);
LVAL xlistvect(void);
LVAL xmakearray(void);
LVAL xaref(void);
LVAL xaset(void);
LVAL xmaketable(void);
LVAL xtablep(void);
LVAL xtableref(void);
LVAL xtableset(void);
LVAL xtableremove(void);
LVAL xemptytable(void);
LVAL xmapovertableentries(void);
LVAL xiaddrof(void);
LVAL xifmtaddr(void);
LVAL xnull(void);
LVAL xatom(void);
LVAL xlistp(void);
LVAL xendp(void);
LVAL xnumberp(void);
LVAL xbooleanp(void);
LVAL xpairp(void);
LVAL xsymbolp(void);
LVAL xintegerp(void);
LVAL xrealp(void);
LVAL xcharp(void);
LVAL xstringp(void);
LVAL xvectorp(void);
LVAL xprocedurep(void);
LVAL xdefaultobjectp(void);
LVAL xeq(void);
int eq(LVAL arg1,LVAL arg2);
LVAL xeqv(void);
int eqv(LVAL arg1,LVAL arg2);
LVAL xequal(void);
int equal(LVAL arg1,LVAL arg2);
LVAL xidentity(void);
LVAL xgensym(void);
int xlgetkeyarg(LVAL key,LVAL def,LVAL *pval);
int xlgkfixnum(LVAL key,FIXTYPE def,FIXTYPE *pval);
void xlgettest(LVAL def,LVAL *pfcn,LVAL *ptresult);
LVAL xlgetport(void);
LVAL xlgetiport(void);
LVAL xlgetoport(void);
LVAL xlgetenv(void);
LVAL xhack(void);

/* xlfun2.c */
LVAL xsymstr(void);
LVAL xstrsym(void);
LVAL xreadline(void);
LVAL xrdchar(void);
LVAL xunreadchar(void);
LVAL xpkchar(void);
LVAL xcharready(void);
LVAL xclearinput(void);
LVAL xrdbyte(void);
LVAL xrdshort(void);
LVAL xrdshorthf(void);
LVAL xrdshortlf(void);
LVAL xrdlong(void);
LVAL xrdlonghf(void);
LVAL xrdlonglf(void);
LVAL xeofobjectp(void);
LVAL xwrite(void);
LVAL xprint(void);
LVAL xwrchar(void);
LVAL xwrbyte(void);
LVAL xwrshort(void);
LVAL xwrshorthf(void);
LVAL xwrshortlf(void);
LVAL xwrlong(void);
LVAL xwrlonghf(void);
LVAL xwrlonglf(void);
LVAL xdisplay(void);
LVAL xnewline(void);
LVAL xfreshline(void);
LVAL xwritesize(void);
LVAL xdisplaysize(void);
LVAL xprbreadth(void);
LVAL xprdepth(void);
LVAL xfilemodtime(void);
LVAL xopeni(void);
LVAL xopeno(void);
LVAL xopena(void);
LVAL xopenu(void);
LVAL xclose(void);
LVAL xclosei(void);
LVAL xcloseo(void);
LVAL xgetfposition(void);
LVAL xsetfposition(void);
LVAL xcurinput(void);
LVAL curinput(void);
LVAL xcuroutput(void);
LVAL curoutput(void);
LVAL xcurerror(void);
LVAL curerror(void);
LVAL xportp(void);
LVAL xinputportp(void);
LVAL xoutputportp(void);
LVAL xmkstrinput(void);
LVAL xmkstroutput(void);
LVAL xgetstroutput(void);
LVAL xmkobjstream(void);
LVAL xformat(void);
LVAL xtranson(void);
LVAL xtransoff(void);
LVAL xmakestring(void);
LVAL xstrlen(void);
LVAL xstrnullp(void);
LVAL xstrappend(void);
LVAL xstrref(void);
LVAL xstrset(void);
LVAL xsubstring(void);
LVAL xstrlist(void);
LVAL xliststring(void);
LVAL xstrlss(void);
LVAL xstrleq(void);
LVAL xstreql(void);
LVAL xstrneq(void);
LVAL xstrgeq(void);
LVAL xstrgtr(void);
LVAL xstrilss(void);
LVAL xstrileq(void);
LVAL xstrieql(void);
LVAL xstrineq(void);
LVAL xstrigeq(void);
LVAL xstrigtr(void);
LVAL xstrsearch(void);
LVAL xstrisearch(void);
LVAL xnumstr(void);
LVAL xstrnum(void);
LVAL xcharint(void);
LVAL xintchar(void);
LVAL xchrlss(void);
LVAL xchrleq(void);
LVAL xchreql(void);
LVAL xchrneq(void);
LVAL xchrgeq(void);
LVAL xchrgtr(void);
LVAL xchrilss(void);
LVAL xchrileq(void);
LVAL xchrieql(void);
LVAL xchrineq(void);
LVAL xchrigeq(void);
LVAL xchrigtr(void);
LVAL xupcase(void);
LVAL xdowncase(void);
LVAL xnupcase(void);
LVAL xndowncase(void);
LVAL xtrim(void);
LVAL xlefttrim(void);
LVAL xrighttrim(void);
LVAL xchupcase(void);
LVAL xchdowncase(void);
LVAL xdigitchar(void);
LVAL xstring(void);
LVAL xchar(void);
LVAL xuppercasep(void);
LVAL xlowercasep(void);
LVAL xbothcasep(void);
LVAL xdigitp(void);
LVAL xalphanumericp(void);
LVAL xwhitespacep(void);
LVAL xcompile(void);
LVAL xdecompile(void);
LVAL xsave(void);
LVAL xrestore(void);
LVAL xgc(void);
LVAL xroom(void);
LVAL xerror(void);
LVAL xgetarg(void);
void xshowstack(void);
void xshowcontrolstack(void);
void xshowvaluestack(void);
LVAL xgettime(void);
LVAL xexit(void);
LVAL xalloccmemory(void);
LVAL xfreecmemory(void);
LVAL xforeignptrp(void);
LVAL xforeignptrtype(void);
LVAL xsetforeignptrtype(void);
LVAL xforeignptrtypep(void);
LVAL xforeignptreqp(void);
LVAL xgetcrecfield(void);
LVAL xgetcrecfieldaddr(void);
LVAL xsetcrecfield(void);
LVAL xgetcrecstring(void);
LVAL xsetcrecstring(void);
LVAL xgetcrectypesize(void);
LVAL xnullpointerp(void);

/* xlfun3.c */
void xapply(void);
void xvalues(void);
void xvalueslist(void);
void xcallcc(void);
void xcallwi(void);
void xcallwo(void);
void xload(void);
void xloadnoisily(void);
FILE *xlLoadOpen(char *name,char *mode,char *pathsym,char *rpath);
void xforce(void);

/* xlimage.c */
int xlisave(char *fname);
int xlirestore(char *fname);

/* xlinit.c */
void xlinitws(int ssize);
void xlsymbols(void);
LVAL getpackagebyid(int id);

/* xlint.c */
LVAL xtraceon(void);
LVAL xtraceoff(void);
void xthrow(void);
void xthrowerror(void);
int CallFunction(LVAL *pvalue,LVAL fun,int argc,...);
int InvokeInterpreter(LVAL *pvalue,LVAL fun,LVAL sel,int argc,va_list ap);
void jumptotarget(int sts);
void pushtarget(ERRORTARGET *target);
void poptarget(void);
LVAL findvar(LVAL env,LVAL var,int *poff);
void xlapply(void);
LVAL make_continuation(void);
LVAL unstack_environment(LVAL);
void callerrorhandler(void);
void throwerror(LVAL type);
LVAL find_top_procedure(void);
void gc_protect(void (*protected_fcn)(void));
void show_call_stack(int cmax);
void show_control_stack(int cmax);
void show_value_stack(int vmax);
void xlstkover(void);
void xlcstkover(void);
LVAL xltoofew(void);
void xltoomany(void);
LVAL xlbadtype(LVAL val);

/* xlio.c */
int xlgetc(LVAL fptr);
void xlungetc(LVAL fptr,int ch);
int xlpeek(LVAL fptr);
int xliready(LVAL fptr);
void xliflush(LVAL fptr);
void xlputc(LVAL fptr,int ch);
void xlflush(void);
FIXTYPE getstrlength(LVAL stream);
LVAL getstroutput(LVAL stream);
void stdputstr(char *str);
void errprint(LVAL expr);
void errprinc(LVAL expr);
void errputstr(char *str);

/* xliterseq.c */
void xfind(void);
void xfindif(void);
void xfindifnot(void);
void xmember(void);
void xmemberif(void);
void xmemberifnot(void);
void xassoc(void);
void xassocif(void);
void xassocifnot(void);
void xrassoc(void);
void xrassocif(void);
void xrassocifnot(void);
void xremove(void);
void xremoveif(void);
void xremoveifnot(void);
void xdelete(void);
void xdeleteif(void);
void xdeleteifnot(void);
void xcount(void);
void xcountif(void);
void xcountifnot(void);
void xposition(void);
void xpositionif(void);
void xpositionifnot(void);
void xmapcar(void);
void xmapc(void);
void xmapcan(void);
void xmaplist(void);
void xmapl(void);
void xmapcon(void);
void xsome(void);
void xevery(void);
void xnotany(void);
void xnotevery(void);

/* xlmath.c */
LVAL xexactp(void);
LVAL xinexactp(void);
LVAL xatan(void);
LVAL xfloor(void);
LVAL xceiling(void);
LVAL xround(void);
LVAL xtruncate(void);
LVAL xash(void);
LVAL xlsh(void);
LVAL xadd(void);
LVAL xmul(void);
LVAL xsub(void);
LVAL xdiv(void);
LVAL xquo(void);
LVAL xrem(void);
LVAL xmod(void);
LVAL xmin(void);
LVAL xmax(void);
LVAL xexpt(void);
LVAL xlogand(void);
LVAL xlogior(void);
LVAL xlogxor(void);
LVAL xlognot(void);
LVAL xabs(void);
LVAL xadd1(void);
LVAL xsub1(void);
LVAL xsin(void);
LVAL xcos(void);
LVAL xtan(void);
LVAL xasin(void);
LVAL xacos(void);
LVAL xxexp(void);
LVAL xsqrt(void);
LVAL xxlog(void);
LVAL xsetrandomseed(void);
LVAL xrandom(void);
LVAL xnegativep(void);
LVAL xzerop(void);
LVAL xpositivep(void);
LVAL xevenp(void);
LVAL xoddp(void);
LVAL xlss(void);
LVAL xleq(void);
LVAL xeql(void);
LVAL xneq(void);
LVAL xgeq(void);
LVAL xgtr(void);
FLOTYPE toflotype(LVAL val);

/* xlobj.c */
void xlsend(LVAL obj,LVAL sym);
void xsendsuper(void);
void clnew(void);
LVAL clmakeinstance(void);
LVAL clinitialize(void);
LVAL clanswer(void);
LVAL clshow(void);
LVAL obinitialize(void);
LVAL obclass(void);
LVAL obprint(void);
LVAL obshow(void);
LVAL obgetvariable(void);
LVAL obsetvariable(void);
void obsymbols(void);
void xloinit(void);

/* xlprint.c */
void xlprin1(LVAL expr,LVAL file);
void xlprinc(LVAL expr,LVAL file);
void xlterpri(LVAL fptr);
void xlfreshline(LVAL fptr);
void xlputstr(LVAL fptr,char *str);

/* xlread.c */
void xlrinit(void);
int xlread(LVAL fptr,LVAL  *pval);
LVAL xread(void);
LVAL xreaddelimitedlist(void);
void xrmhash(void);
void xrmquote(void);
void xrmdquote(void);
void xrmbquote(void);
void xrmcomma(void);
void xrmlparen(void);
void xrmrparen(void);
void xrmsemi(void);
LVAL chartype(int ch);
int isnumber(char *str,LVAL *pval);
int isradixnumber(char *str,int radix,LVAL *pval);

/* xlsym.c */
void xlpinit(void);
LVAL findpackage(char *name);
int usepackage(LVAL dst,LVAL src);
int unusepackage(LVAL dst,LVAL src);
LVAL intern(LVAL name,LVAL package,LVAL *pkey);
LVAL internandexport(char *name,LVAL package);
void unintern(LVAL sym,LVAL package);
int isvisible(LVAL sym,LVAL package);
int ispresent(LVAL sym,LVAL package);
LVAL findsymbol(char *name,LVAL package,LVAL *pkey);
int import(LVAL sym,LVAL package);
int importandexport(LVAL srcpack,char *name,LVAL dstpack);
int export(LVAL sym,LVAL package);
int unexport(LVAL sym,LVAL package);
void xlsubr(LVAL package,int type,FUNDEF *def);
LVAL xlenter(char *name);
LVAL xlkenter(char *name);
LVAL xlintern(char *name,LVAL package,LVAL *pkey);
LVAL xlgetprop(LVAL sym,LVAL prp);
void xlputprop(LVAL sym,LVAL val,LVAL prp);
void xlremprop(LVAL sym,LVAL prp);
LVAL findentryintable(LVAL table,LVAL key);
void addentrytotable(LVAL table,LVAL key,LVAL val);
LVAL removeentryfromtable(LVAL table,LVAL key);

/* xlansi.c */
void ossetrand(long seed);
long osrand(long n);
FILE *osaopen(char *name,char *mode);
FILE *osbopen(char *name,char *mode);
int osclose(FILE *fp);
long ostell(FILE *fp);
int osseek(FILE *fp,long offset,int whence);
void ostputs(char *str);
void oscheck(void);
void osinfo(void);
time_t ostime(void);
void *osalloc(FIXTYPE size);
void osfree(void *ptr);

/* ??stuff.c */
void osinit(char *banner);
char *osloadpath(char **pp);
void osexit(int sts);
void osenter(void);
FUNDEF *osfindsubr(char *name,int *ptype);
void oserror(char *msg);
int osfmodtime(char *fname,FIXTYPE *pModTime);
int ostgetc(void);
void ostputc(int ch);
int ostatbol(void);
void ostflush(void);
int ostcheck(void);
void osflushoutput(void);

#endif
