/* xlitersq.c - routines to iterate over a sequence */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

#define IS_X		0
#define IS_LIST		1
#define IS_KEYFCN	2
#define IS_TESTFCN	3
#define IS_TRESULT	4
#define IS_COUNT	5
#define IS_INDEX	6
#define IS_END		7
#define IS_VALUE	8
#define IS_WORK		9
#define IS_ACTION	10
#define IS_CDISPATCH	11
#define _ISSIZE		12

#define MS_FCN		0
#define MS_ARGC		1
#define MS_VALUE	2
#define MS_WORK		3
#define MS_ACTION	4
#define MS_CDISPATCH	5
#define _MSSIZE		6

/* action routine definitions */
typedef LVAL (*ACTION)(int op,LVAL val,LVAL *d);
typedef LVAL (*MACTION)(LVAL val,LVAL *d);
#define IS_FETCH	1
#define IS_UPDATE	2

/* external variables */
extern LVAL xlval,k_key,k_count,k_start,k_end;

static void iterseq1(LVAL ivalue,LVAL tresult,ACTION action);
static void iterlist1(LVAL ivalue,LVAL tresult,ACTION action);
static void do_iterseq1(void);
static void seq1key_restore(void);
static void do_seq1key(LVAL *);
static void seq1test_restore(void);
static void do_seq1test(LVAL,LVAL *);

static void iterseq2(LVAL ivalue,ACTION action);
static void iterlist2(LVAL ivalue,ACTION action);
static void do_iterseq2(void);
static void seq2key_restore(void);
static void do_seq2key(LVAL *);
static void seq2test_restore(void);
static void seq2test_check(LVAL *);
static void do_seq2test(LVAL,LVAL *);

static void seqdummy_restore(void);
static LVAL *seqkeytest_mark(LVAL *);
static LVAL *seqkeytest_skipit(LVAL *);
static void seqkeytest_unwind(void);
static LVAL *seqdummy_print(LVAL *);
static LVAL *seqkey_print(LVAL *);
static LVAL *seqtest_print(LVAL *);
static void show_seqcontinuation(LVAL *,char *);

static void mapseq(LVAL ivalue,MACTION action);
static void do_mapseq(void);
static void mapseq_restore(void);
static LVAL *mapseq_mark(LVAL *);
static LVAL *mapseq_skipit(LVAL *);
static void mapseq_unwind(void);
static LVAL *mapseq_print(LVAL *);
static void mapseqdummy_restore(void);
static LVAL *mapseqdummy_print(LVAL *);
static void show_mapseqcontinuation(LVAL *,char *);

static void maplist(LVAL ivalue,ACTION action);
static void do_maplist(LVAL *);
static void maplist_restore(void);
static LVAL *maplist_mark(LVAL *);
static LVAL *maplist_skipit(LVAL *);
static void maplist_unwind(void);
static LVAL *maplist_print(LVAL *);
static void maplistdummy_restore(void);
static LVAL *maplistdummy_print(LVAL *);
static void show_maplistcontinuation(LVAL *,char *);

static LVAL is_find(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
	if (val)
	    d[IS_VALUE] = car(d[IS_LIST]);
    	return val;
    }
}

/* xfind - built-in function 'find' */
void xfind(void)
{
    iterseq2(v_nil,is_find);
}

/* xfindif - built-in function 'find-if' */
void xfindif(void)
{
    iterseq1(v_nil,v_true,is_find);
}

/* xfindifnot - built-in function 'find-if-not' */
void xfindifnot(void)
{
    iterseq1(v_nil,v_nil,is_find);
}

static LVAL is_member(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
	if (val)
	    d[IS_VALUE] = d[IS_LIST];
    	return val;
    }
}

/* xmember - built-in function 'member' */
void xmember(void)
{
    iterlist2(v_nil,is_member);
}

/* xmemberif - built-in function 'member-if' */
void xmemberif(void)
{
    iterlist1(v_nil,v_true,is_member);
}

/* xmemberifnot - built-in function 'member-if-not' */
void xmemberifnot(void)
{
    iterlist1(v_nil,v_nil,is_member);
}

static LVAL is_assoc(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	if (!consp(val))
	    xlerror("bad association list entry",val);
	return car(val);
    case IS_UPDATE:
	if (val)
	    d[IS_VALUE] = car(d[IS_LIST]);
    	return val;
    }
}

/* xassoc - built-in function 'assoc' */
void xassoc(void)
{
    iterlist2(v_nil,is_assoc);
}

/* xassocif - built-in function 'assoc-if' */
void xassocif(void)
{
    iterlist1(v_nil,v_true,is_assoc);
}

/* xassocifnot - built-in function 'assoc-if-not' */
void xassocifnot(void)
{
    iterlist1(v_nil,v_nil,is_assoc);
}

static LVAL is_rassoc(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	if (!consp(val))
	    xlerror("bad association list entry",val);
	return cdr(val);
    case IS_UPDATE:
	if (val)
	    d[IS_VALUE] = car(d[IS_LIST]);
    	return val;
    }
}

/* xrassoc - built-in function 'rassoc' */
void xrassoc(void)
{
    iterlist2(v_nil,is_rassoc);
}

/* xrassocif - built-in function 'rassoc-if' */
void xrassocif(void)
{
    iterlist1(v_nil,v_true,is_rassoc);
}

/* xrassocifnot - built-in function 'rassoc-if-not' */
void xrassocifnot(void)
{
    iterlist1(v_nil,v_nil,is_rassoc);
}

static LVAL is_remove(int op,LVAL val,LVAL *d)
{
    FIXTYPE n;
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
    	n = d[IS_COUNT] ? getfixnum(d[IS_COUNT]) : 1;
    	if (n <= 0 || !val) {
	    if (d[IS_WORK]) {
		rplacd(d[IS_WORK],cons(car(d[IS_LIST]),v_nil));
		d[IS_WORK] = cdr(d[IS_WORK]);
	    }
	    else
		d[IS_VALUE] = d[IS_WORK] = cons(car(d[IS_LIST]),v_nil);
	}
    	if (val && d[IS_COUNT])
	    d[IS_COUNT] = cvfixnum(n - 1);
	return v_nil;
    }
}

/* xremove - built-in function 'remove' */
void xremove(void)
{
    iterseq2(v_nil,is_remove);
}

/* xremoveif - built-in function 'remove-if' */
void xremoveif(void)
{
    iterseq1(v_nil,v_true,is_remove);
}

/* xremoveifnot - built-in function 'remove-if-not' */
void xremoveifnot(void)
{
    iterseq1(v_nil,v_nil,is_remove);
}

static LVAL is_delete(int op,LVAL val,LVAL *d)
{
    FIXTYPE n;
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
    	n = d[IS_COUNT] ? getfixnum(d[IS_COUNT]) : 1;
    	if (n > 0 && val) {
 	    if (d[IS_WORK])
 		rplacd(d[IS_WORK],v_nil);
 	}
 	else {
	    if (d[IS_WORK]) {
		rplacd(d[IS_WORK],d[IS_LIST]);
		d[IS_WORK] = cdr(d[IS_WORK]);
	    }
	    else
		d[IS_VALUE] = d[IS_WORK] = d[IS_LIST];
	}
    	if (val && d[IS_COUNT])
	    d[IS_COUNT] = cvfixnum(n - 1);
    	return v_nil;
    }
}

/* xdelete - built-in function 'delete' */
void xdelete(void)
{
    iterseq2(v_nil,is_delete);
}

/* xdeleteif - built-in function 'delete-if' */
void xdeleteif(void)
{
    iterseq1(v_nil,v_true,is_delete);
}

/* xdeleteifnot - built-in function 'delete-if-not' */
void xdeleteifnot(void)
{
    iterseq1(v_nil,v_nil,is_delete);
}

static LVAL is_count(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
    	if (val)
	    d[IS_VALUE] = cvfixnum(getfixnum(d[IS_VALUE]) + 1);
	return v_nil;
    }
}

/* xcount - built-in function 'count' */
void xcount(void)
{
    iterseq2(cvfixnum((FIXTYPE)0),is_count);
}

/* xcountif - built-in function 'count-if' */
void xcountif(void)
{
    iterseq1(cvfixnum((FIXTYPE)0),v_true,is_count);
}

/* xcountifnot - built-in function 'count-if-not' */
void xcountifnot(void)
{
    iterseq1(cvfixnum((FIXTYPE)0),v_nil,is_count);
}

static LVAL is_position(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
	if (val)
	    d[IS_VALUE] = d[IS_INDEX];
	return val;
    }
}

/* xposition - built-in function 'position' */
void xposition(void)
{
    iterseq2(v_nil,is_position);
}

/* xpositionif - built-in function 'position-if' */
void xpositionif(void)
{
    iterseq1(v_nil,v_true,is_position);
}

/* xpositionifnot - built-in function 'position-if-not' */
void xpositionifnot(void)
{
    iterseq1(v_nil,v_nil,is_position);
}

static LVAL ms_mapcar(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return car(val);
    case IS_UPDATE:
	if (d[MS_WORK]) {
	    rplacd(d[MS_WORK],cons(val,v_nil));
	    d[MS_WORK] = cdr(d[MS_WORK]);
	}
	else
	    d[MS_VALUE] = d[MS_WORK] = cons(val,v_nil);
	return v_nil;
    }
}

/* xmapcar - built-in function 'mapcar' */
void xmapcar(void)
{
    maplist(v_nil,ms_mapcar);
}

static LVAL ms_mapc(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return car(val);
    case IS_UPDATE:
	return v_nil;
    }
}

/* xmapc - built-in function 'mapc' */
void xmapc(void)
{
    maplist(v_true,ms_mapc);
}

static LVAL ms_mapcan(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return car(val);
    case IS_UPDATE:
	if (consp(val)) {
	    if (d[MS_WORK])
		rplacd(d[MS_WORK],val);
	    else
		d[MS_VALUE] = d[MS_WORK] = val;
	    for (; consp(cdr(val)); val = cdr(val))
		;
	    d[MS_WORK] = val;
	}
	return v_nil;
    }
}

/* xmapcan - built-in function 'mapcan' */
void xmapcan(void)
{
    maplist(v_nil,ms_mapcan);
}

static LVAL ms_maplist(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
	if (d[MS_WORK]) {
	    rplacd(d[MS_WORK],cons(val,v_nil));
	    d[MS_WORK] = cdr(d[MS_WORK]);
	}
	else
	    d[MS_VALUE] = d[MS_WORK] = cons(val,v_nil);
	return v_nil;
    }
}

/* xmaplist - built-in function 'maplist' */
void xmaplist(void)
{
    maplist(v_nil,ms_maplist);
}

static LVAL ms_mapl(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
	return v_nil;
    }
}

/* xmapl - built-in function 'mapl' */
void xmapl(void)
{
    maplist(v_true,ms_mapl);
}

static LVAL ms_mapcon(int op,LVAL val,LVAL *d)
{
    switch (op) {
    case IS_FETCH:
	return val;
    case IS_UPDATE:
	if (consp(val)) {
	    if (d[MS_WORK])
		rplacd(d[MS_WORK],val);
	    else
		d[MS_VALUE] = d[MS_WORK] = val;
	    for (; consp(cdr(val)); val = cdr(val))
		;
	    d[MS_WORK] = val;
	}
	return v_nil;
    }
}

/* xmapcon - built-in function 'mapcon' */
void xmapcon(void)
{
    maplist(v_nil,ms_mapcon);
}

static LVAL ms_some(LVAL val,LVAL *d)
{
    return val ? (d[MS_VALUE] = val, v_true) : v_nil;
}

/* xsome - built-in function 'some' */
void xsome(void)
{
    mapseq(v_nil,ms_some);
}

static LVAL ms_every(LVAL val,LVAL *d)
{
    return !val ? (d[MS_VALUE] = v_nil, v_true) : v_nil;
}

/* xevery - built-in function 'every' */
void xevery(void)
{
    mapseq(v_true,ms_every);
}

static LVAL ms_notany(LVAL val,LVAL *d)
{
    return val ? (d[MS_VALUE] = v_nil, v_true) : v_nil;
}

/* xnotany - built-in function 'notany' */
void xnotany(void)
{
    mapseq(v_true,ms_notany);
}

static LVAL ms_notevery(LVAL val,LVAL *d)
{
    return !val ? (d[MS_VALUE] = v_true, v_true) : v_nil;
}

/* xnotevery - built-in function 'notevery' */
void xnotevery(void)
{
    mapseq(v_nil,ms_notevery);
}

/* sequence dummy continuation dispatch table */
static CDISPATCH cd_seqdummy = {
    seqdummy_restore,
    seqkeytest_mark,
    seqkeytest_skipit,
    seqkeytest_unwind,
    seqkeytest_skipit,
    seqdummy_print
};

/* do_iterseq1 - start the next iteration */
static void do_iterseq1(void)
{
    LVAL *d = xlcsp - _ISSIZE;
    if (consp(d[IS_LIST]))
	do_seq1key(d);
    else {
	xlval = d[IS_VALUE];
	Cdrop(_ISSIZE);
	svreturn();
    }
}

/* seq1key_restore - restore a sequence key continuation */
static void seq1key_restore(void)
{
    LVAL *d;
    Cpush((LVAL)&cd_seqdummy);
    d = xlcsp - _ISSIZE;
    do_seq1test(xlval,d);
}

/* sequence key continuation dispatch table */
static CDISPATCH cd_seq1key = {
    seq1key_restore,
    seqkeytest_mark,
    seqkeytest_skipit,
    seqkeytest_unwind,
    seqkeytest_skipit,
    seqkey_print
};

/* do_seq1key - setup to call the key function */
static void do_seq1key(LVAL *d)
{
    LVAL val = (*(ACTION)d[IS_ACTION])(IS_FETCH,car(d[IS_LIST]),d);
    if (d[IS_KEYFCN]) {
	d[IS_CDISPATCH] = (LVAL)&cd_seq1key;
	cpush(val);
	xlval = d[IS_KEYFCN];
	xlargc = 1;
	xlnext = xlapply;
    }
    else
	do_seq1test(val,d);
}

/* seq1test_restore - restore a sequence key continuation */
static void seq1test_restore(void)
{
    LVAL val,*d;
    Cpush((LVAL)&cd_seqdummy);
    d = xlcsp - _ISSIZE;
    val = (d[IS_TRESULT] && xlval) || (!d[IS_TRESULT] && !xlval) ? v_true : v_nil;
    if ((*(ACTION)d[IS_ACTION])(IS_UPDATE,val,d)) {
	xlval = d[IS_VALUE];
	Cdrop(_ISSIZE);
	svreturn();
    }
    else {
	d[IS_LIST] = cdr(d[IS_LIST]);
	d[IS_INDEX] = cvfixnum(getfixnum(d[IS_INDEX]) + 1);
	xlnext = do_iterseq1;
    }
}

/* sequence test continuation dispatch table */
static CDISPATCH cd_seq1test = {
    seq1test_restore,
    seqkeytest_mark,
    seqkeytest_skipit,
    seqkeytest_unwind,
    seqkeytest_skipit,
    seqtest_print
};

/* do_seq1test - setup to call the test function */
static void do_seq1test(LVAL key,LVAL *d)
{
    d[IS_CDISPATCH] = (LVAL)&cd_seq1test;
    cpush(key);
    xlval = d[IS_TESTFCN];
    xlargc = 1;
    xlnext = xlapply;
}

/* iterseq1 - iterate over a sequence (list) */
static void iterseq1(LVAL ivalue,LVAL tresult,ACTION action)
{
    LVAL seq,testfcn,keyfcn,count,start,end;
    
    /* parse the argument list */
    testfcn = xlgetarg();
    seq = xlgalist();
    xlgetkeyarg(k_key,v_nil,&keyfcn);
    xlgetkeyarg(k_count,v_nil,&count);
    xlgetkeyarg(k_start,v_nil,&start);
    xlgetkeyarg(k_end,v_nil,&end);
    xlpopargs();
    
    /* setup the continuation frame */
    Ccheck(_ISSIZE);
    Cpush(v_nil);
    Cpush(seq);
    Cpush(keyfcn);
    Cpush(testfcn);
    Cpush(tresult);
    Cpush(count);
    Cpush(cvfixnum((FIXTYPE)0));
    Cpush(end);
    Cpush(ivalue);
    Cpush(v_nil);
    Cpush((LVAL)action);
    Cpush((LVAL)&cd_seq1key);
    
    /* start the iteration */
    do_iterseq1();
}

/* iterlist1 - iterate over a list */
static void iterlist1(LVAL ivalue,LVAL tresult,ACTION action)
{
    LVAL list,testfcn,keyfcn;
    
    /* parse the argument list */
    testfcn = xlgetarg();
    list = xlgalist();
    xlgetkeyarg(k_key,v_nil,&keyfcn);
    xlpopargs();
    
    /* setup the continuation frame */
    Ccheck(_ISSIZE);
    Cpush(v_nil);
    Cpush(list);
    Cpush(keyfcn);
    Cpush(testfcn);
    Cpush(tresult);
    Cpush(v_nil);
    Cpush(cvfixnum((FIXTYPE)0));
    Cpush(v_nil);
    Cpush(ivalue);
    Cpush(v_nil);
    Cpush((LVAL)action);
    Cpush((LVAL)&cd_seq1key);
    
    /* start the iteration */
    do_iterseq1();
}

/* do_iterseq2 - start the next iteration */
static void do_iterseq2(void)
{
    LVAL *d = xlcsp - _ISSIZE;
    if (consp(d[IS_LIST]))
	do_seq2key(d);
    else {
	xlval = d[IS_VALUE];
	Cdrop(_ISSIZE);
	svreturn();
    }
}

/* seq2key_restore - restore a sequence key continuation */
static void seq2key_restore(void)
{
    LVAL *d;
    Cpush((LVAL)&cd_seqdummy);
    d = xlcsp - _ISSIZE;
    do_seq2test(xlval,d);
}

/* sequence key continuation dispatch table */
static CDISPATCH cd_seq2key = {
    seq2key_restore,
    seqkeytest_mark,
    seqkeytest_skipit,
    seqkeytest_unwind,
    seqkeytest_skipit,
    seqkey_print
};

/* do_seq2key - setup to call the key function */
static void do_seq2key(LVAL *d)
{
    LVAL val = (*(ACTION)d[IS_ACTION])(IS_FETCH,car(d[IS_LIST]),d);
    if (d[IS_KEYFCN]) {
	d[IS_CDISPATCH] = (LVAL)&cd_seq2key;
	cpush(val);
	xlval = d[IS_KEYFCN];
	xlargc = 1;
	xlnext = xlapply;
    }
    else
	do_seq2test(val,d);
}

/* seq2test_restore - restore a sequence key continuation */
static void seq2test_restore(void)
{
    LVAL *d;
    Cpush((LVAL)&cd_seqdummy);
    d = xlcsp - _ISSIZE;
    seq2test_check(d);
}

/* seq2test_check - check for termination */
static void seq2test_check(LVAL *d)
{
    LVAL val;
    val = (d[IS_TRESULT] && xlval) || (!d[IS_TRESULT] && !xlval) ? v_true : v_nil;
    if ((*(ACTION)d[IS_ACTION])(IS_UPDATE,val,d)) {
	xlval = d[IS_VALUE];
	Cdrop(_ISSIZE);
	svreturn();
    }
    else {
	d[IS_LIST] = cdr(d[IS_LIST]);
	d[IS_INDEX] = cvfixnum(getfixnum(d[IS_INDEX]) + 1);
	xlnext = do_iterseq2;
    }
}

/* sequence test continuation dispatch table */
static CDISPATCH cd_seq2test = {
    seq2test_restore,
    seqkeytest_mark,
    seqkeytest_skipit,
    seqkeytest_unwind,
    seqkeytest_skipit,
    seqtest_print
};

/* do_seq2test - setup to call the test function */
static void do_seq2test(LVAL key,LVAL *d)
{
    if (d[IS_TESTFCN]) {
	d[IS_CDISPATCH] = (LVAL)&cd_seq2test;
	check(2);
	push(key);
	push(d[IS_X]);
	xlval = d[IS_TESTFCN];
	xlargc = 2;
	xlnext = xlapply;
    }
    else {
	xlargc = 1;
	xlval = eqv(d[IS_X],key) ? v_true : v_nil;
	seq2test_check(d);
    }
}

/* iterseq2 - iterate over a sequence (list) */
static void iterseq2(LVAL ivalue,ACTION action)
{
    LVAL arg,seq,keyfcn,testfcn,tresult,count,start,end;
    
    /* parse the argument list */
    arg = xlgetarg();
    seq = xlgalist();
    xlgetkeyarg(k_key,v_nil,&keyfcn);
    xlgetkeyarg(k_count,v_nil,&count);
    xlgetkeyarg(k_start,v_nil,&start);
    xlgetkeyarg(k_end,v_nil,&end);
    xlgettest(v_nil,&testfcn,&tresult);
    xlpopargs();
    
    /* setup the continuation frame */
    Ccheck(_ISSIZE);
    Cpush(arg);
    Cpush(seq);
    Cpush(keyfcn);
    Cpush(testfcn);
    Cpush(tresult);
    Cpush(count);
    Cpush(cvfixnum((FIXTYPE)0));
    Cpush(end);
    Cpush(ivalue);
    Cpush(v_nil);
    Cpush((LVAL)action);
    Cpush((LVAL)&cd_seq2key);
    
    /* start the iteration */
    do_iterseq2();
}

/* iterlist2 - iterate over a list */
static void iterlist2(LVAL ivalue,ACTION action)
{
    LVAL fcn,tresult;
    Ccheck(_ISSIZE);
    Cpush(xlgetarg());
    Cpush(xlgalist());
    xlgetkeyarg(k_key,v_nil,&fcn);
    Cpush(fcn);
    xlgettest(v_nil,&fcn,&tresult);
    Cpush(fcn);
    Cpush(tresult);
    xlpopargs();
    Cpush(v_nil);
    Cpush(cvfixnum((FIXTYPE)0));
    Cpush(v_nil);
    Cpush(ivalue);
    Cpush(v_nil);
    Cpush((LVAL)action);
    Cpush((LVAL)&cd_seq2key);
    do_iterseq2();
}

/* seqdummy_restore - restore a dummy frame (an error) */
static void seqdummy_restore(void)
{
    xlfmterror("shouldn't happen -- seqdummy_restore");
}

/* seqkeytest_mark - mark a sequence key/test continuation */
static LVAL *seqkeytest_mark(LVAL *p)
{
    --p;	/* action */
    mark(*--p);	/* work */
    mark(*--p);	/* value */
    mark(*--p);	/* end */
    mark(*--p);	/* index */
    mark(*--p);	/* count */
    mark(*--p);	/* tresult */
    mark(*--p);	/* testfcn */
    mark(*--p);	/* keyfcn */
    mark(*--p);	/* list */
    mark(*--p);	/* x */
    return p;
}

/* seqkeytest_skipit - unmark/unstack a sequence key/test continuation (just skip over it) */
static LVAL *seqkeytest_skipit(LVAL *p)
{
    return p - _ISSIZE + 1;
}

/* seqkeytest_unwind - unwind past a sequence key/test continuation (just skip over it) */
static void seqkeytest_unwind(void)
{
    Cdrop(_ISSIZE - 1);
}

/* seqdummy_print - print a sequence dummy continuation */
static LVAL *seqdummy_print(LVAL *p)
{
    p -= _ISSIZE - 1;
    show_seqcontinuation(p,"SeqDummy");
    return p;
}

/* seqkey_print - print a sequence key continuation */
static LVAL *seqkey_print(LVAL *p)
{
    p -= _ISSIZE - 1;
    show_seqcontinuation(p,"SeqKey");
    return p;
}

/* seqtest_print - print a sequence key continuation */
static LVAL *seqtest_print(LVAL *p)
{
    p -= _ISSIZE - 1;
    show_seqcontinuation(p,"SeqTest");
    return p;
}

/* show_seqcontinuation - show a sequence key or test continuation */
static void show_seqcontinuation(LVAL *d,char *tag)
{
    errputstr("\n ");
    errputstr(tag);
    errputstr("\n  x: ");
    errprint(d[IS_X]);
    errputstr("\n  list: ");
    errprint(d[IS_LIST]);
    errputstr("\n  keyfcn: ");
    errprint(d[IS_KEYFCN]);
    errputstr("\n  testfcn: ");
    errprint(d[IS_TESTFCN]);
    errputstr("\n  tresult: ");
    errprint(d[IS_TRESULT]);
    errputstr("\n  count: ");
    errprint(d[IS_COUNT]);
    errputstr("\n  index: ");
    errprint(d[IS_INDEX]);
    errputstr("\n  end: ");
    errprint(d[IS_END]);
    errputstr("\n  value: ");
    errprint(d[IS_VALUE]);
    errputstr("\n  work: ");
    errprint(d[IS_WORK]);
}

/* map dummy continuation dispatch table */
static CDISPATCH cd_mapseqdummy = {
    mapseqdummy_restore,
    mapseq_mark,
    mapseq_skipit,
    mapseq_unwind,
    mapseq_skipit,
    mapseqdummy_print
};

/* mapseq_restore - restore a sequence key continuation */
static void mapseq_restore(void)
{
    LVAL *d;
    Cpush((LVAL)&cd_mapseqdummy);
    d = xlcsp - _MSSIZE;
    if ((*(MACTION)d[MS_ACTION])(xlval,d)) {
	drop((int)getfixnum(d[MS_ARGC]));
	xlval = d[MS_VALUE];
	Cdrop(_MSSIZE);
	svreturn();
    }
    else
	xlnext = do_mapseq;
}

/* mapseqdummy_restore - restore a dummy frame (an error) */
static void mapseqdummy_restore(void)
{
    xlfmterror("shouldn't happen -- mapdummy_restore");
}

/* mapseq_mark - mark a sequence key/test continuation */
static LVAL *mapseq_mark(LVAL *p)
{
    --p;	/* action */
    mark(*--p);	/* work */
    mark(*--p);	/* value */
    mark(*--p);	/* argc */
    mark(*--p);	/* fcn */
    return p;
}

/* mapseq_skipit - unmark/unstack a sequence key/test continuation (just skip over it) */
static LVAL *mapseq_skipit(LVAL *p)
{
    return p - _MSSIZE + 1;
}

/* mapseq_unwind - unwind past a sequence key/test continuation (just skip over it) */
static void mapseq_unwind(void)
{
    Cdrop(_MSSIZE - 1);
}

/* mapseq_print - print a map continuation */
static LVAL *mapseq_print(LVAL *p)
{
    p -= _MSSIZE - 1;
    show_mapseqcontinuation(p,"MapSeq");
    return p;
}

/* mapseqdummy_print - print a dummy map continuation */
static LVAL *mapseqdummy_print(LVAL *p)
{
    p -= _MSSIZE - 1;
    show_mapseqcontinuation(p,"MapSeqDummy");
    return p;
}

/* show_mapseqcontinuation - print a map continuation */
static void show_mapseqcontinuation(LVAL *d,char *tag)
{
    errputstr("\n ");
    errputstr(tag);
    errputstr("\n  fcn: ");
    errprint(d[MS_FCN]);
    errputstr("\n  argc: ");
    errprint(d[MS_ARGC]);
    errputstr("\n  value: ");
    errprint(d[MS_VALUE]);
    errputstr("\n  work: ");
    errprint(d[MS_WORK]);
}

/* sequence key continuation dispatch table */
static CDISPATCH cd_mapseq = {
    mapseq_restore,
    mapseq_mark,
    mapseq_skipit,
    mapseq_unwind,
    mapseq_skipit,
    mapseq_print
};

/* do_mapseq - start the next iteration */
static void do_mapseq(void)
{
    LVAL *d = xlcsp - _MSSIZE;
    LVAL *finalsp,*argp,x;
    int argc;

    /* initialize */
    xlargc = argc = (int)getfixnum(d[MS_ARGC]);
    finalsp = argp = xlsp + argc;

    /* get the function to apply */
    xlval = d[MS_FCN];

    /* build the argument list for the next application */
    check(argc);
    while (--argc >= 0) {
	x = *--argp;
	if (consp(x)) {
	    push(car(x));
	    *argp = cdr(x);
	}
	else {
	    xlsp = finalsp;
	    xlval = d[MS_VALUE];
	    Cdrop(_MSSIZE);
	    svreturn();
	    return;
	}
    }

    /* apply the function to the argument list */
    d[MS_CDISPATCH] = (LVAL)&cd_mapseq;
    xlnext = xlapply;
}

/* mapseq - iterate over a sequence (list) */
static void mapseq(LVAL ivalue,MACTION action)
{
    LVAL fcn,*d = xlcsp;
    fcn = xlgetarg();
    Ccheck(_MSSIZE);
    Cpush(fcn);
    Cpush(cvfixnum((FIXTYPE)xlargc));
    Cpush(ivalue);
    Cpush(v_nil);
    Cpush((LVAL)action);
    Cpush((LVAL)&cd_mapseq);
    do_mapseq();
}

/* map list dummy continuation dispatch table */
static CDISPATCH cd_maplistdummy = {
    maplistdummy_restore,
    maplist_mark,
    maplist_skipit,
    maplist_unwind,
    maplist_skipit,
    maplistdummy_print
};

/* maplist_restore - restore a map list continuation */
static void maplist_restore(void)
{
    LVAL *d;
    Cpush((LVAL)&cd_maplistdummy);
    d = xlcsp - _MSSIZE;
    if ((*(ACTION)d[MS_ACTION])(IS_UPDATE,xlval,d)) {
	drop((int)getfixnum(d[MS_ARGC]));
	xlval = d[MS_VALUE];
	Cdrop(_MSSIZE);
	svreturn();
    }
    else
	do_maplist(d);
}

/* maplistdummy_restore - restore a dummy frame (an error) */
static void maplistdummy_restore(void)
{
    xlfmterror("shouldn't happen -- maplistdummy_restore");
}

/* maplist_mark - mark a map list continuation */
static LVAL *maplist_mark(LVAL *p)
{
    --p;	/* action */
    mark(*--p);	/* work */
    mark(*--p);	/* value */
    mark(*--p);	/* argc */
    mark(*--p);	/* fcn */
    return p;
}

/* maplist_skipit - unmark/unstack a map list continuation (just skip over it) */
static LVAL *maplist_skipit(LVAL *p)
{
    return p - _MSSIZE + 1;
}

/* maplist_unwind - unwind past a map list continuation (just skip over it) */
static void maplist_unwind(void)
{
    Cdrop(_MSSIZE - 1);
}

/* maplist_print - print a map list continuation */
static LVAL *maplist_print(LVAL *p)
{
    p -= _MSSIZE - 1;
    show_maplistcontinuation(p,"MapList");
    return p;
}

/* maplistdummy_print - print a dummy map list continuation */
static LVAL *maplistdummy_print(LVAL *p)
{
    p -= _MSSIZE - 1;
    show_maplistcontinuation(p,"MapListDummy");
    return p;
}

/* show_maplistcontinuation - print a map list continuation */
static void show_maplistcontinuation(LVAL *d,char *tag)
{
    errputstr("\n ");
    errputstr(tag);
    errputstr("\n  fcn: ");
    errprint(d[MS_FCN]);
    errputstr("\n  argc: ");
    errprint(d[MS_ARGC]);
    errputstr("\n  value: ");
    errprint(d[MS_VALUE]);
    errputstr("\n  work: ");
    errprint(d[MS_WORK]);
}

/* map list continuation dispatch table */
static CDISPATCH cd_maplist = {
    maplist_restore,
    maplist_mark,
    maplist_skipit,
    maplist_unwind,
    maplist_skipit,
    maplist_print
};

/* do_maplist - start the next iteration */
static void do_maplist(LVAL *d)
{
    LVAL *finalsp,*argp,x;
    int argc;

    /* initialize */
    xlargc = argc = (int)getfixnum(d[MS_ARGC]);
    finalsp = argp = xlsp + argc;

    /* get the function to apply */
    xlval = d[MS_FCN];

    /* build the argument list for the next application */
    check(argc);
    while (--argc >= 0) {
	x = *--argp;
	if (consp(x)) {
	    push((*(ACTION)d[MS_ACTION])(IS_FETCH,x,d));
	    *argp = cdr(x);
	}
	else {
	    xlsp = finalsp;
	    xlval = d[MS_VALUE];
	    Cdrop(_MSSIZE);
	    svreturn();
	    return;
	}
    }

    /* apply the function to the argument list */
    d[MS_CDISPATCH] = (LVAL)&cd_maplist;
    xlnext = xlapply;
}

/* maplist - iterate over a list */
static void maplist(LVAL ivalue,ACTION action)
{
    LVAL fcn,*d = xlcsp;
    fcn = xlgetarg();
    if (xlargc == 0)
	xltoofew();
    Ccheck(_MSSIZE);
    Cpush(fcn);
    Cpush(cvfixnum((FIXTYPE)xlargc));
    Cpush(ivalue ? *xlsp : v_nil);
    Cpush(v_nil);
    Cpush((LVAL)action);
    Cpush((LVAL)&cd_maplist);
    do_maplist(d);
}
