/* xlmath.c - xlisp built-in arithmetic functions */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"
#include <math.h>
#include <limits.h>

/* forward declarations */
static LVAL binary(int fcn);
static FIXTYPE remain(FIXTYPE num,FIXTYPE den);
static FIXTYPE modulo(FIXTYPE num,FIXTYPE den);
static LVAL unary(int fcn);
static LVAL predicate(int fcn);
static LVAL compare(int fcn);
static void checkizero(FIXTYPE iarg);
static void checkineg(FIXTYPE iarg);
static void checkfzero(FLOTYPE farg);
static void checkfneg(FLOTYPE farg);
static void badiop(void);
static void badfop(void);

/* xexactp - built-in function 'exact?' */
/**** THIS IS REALLY JUST A STUB FOR NOW ****/
LVAL xexactp(void)
{
    xlval = xlganumber();
    xllastarg();
    return fixp(xlval) ? v_true : v_false;
}

/* xinexactp - built-in function 'inexact?' */
/**** THIS IS REALLY JUST A STUB FOR NOW ****/
LVAL xinexactp(void)
{
    xlval = xlganumber();
    xllastarg();
    return fixp(xlval) ? v_false : v_true;
}

/* xatan - built-in function 'atan' */
LVAL xatan(void)
{
    LVAL arg,arg2;
    FLOTYPE val;
    
    /* get the first argument */
    arg = xlganumber();
    
    /* handle two argument (atan y x) */
    if (moreargs()) {
	arg2 = xlganumber();
	xllastarg();
	val = atan2(toflotype(arg),toflotype(arg2));
    }
    
    /* handle one argument (atan x) */
    else
	val = atan(toflotype(arg));

    /* return the resulting flonum */
    return cvflonum(val);
}

/* xfloor - built-in function 'floor' */
LVAL xfloor(void)
{
    LVAL arg;

    /* get the argument */
    arg = xlgetarg();
    xllastarg();

    /* check its type */
    if (fixp(arg))
	return arg;
    else if (floatp(arg))
	return cvfixnum((FIXTYPE)floor(getflonum(arg)));
    xlbadtype(arg);
    return v_nil; /* never reached */
}

/* xceiling - built-in function 'ceiling' */
LVAL xceiling(void)
{
    LVAL arg;

    /* get the argument */
    arg = xlgetarg();
    xllastarg();

    /* check its type */
    if (fixp(arg))
	return arg;
    else if (floatp(arg))
	return cvfixnum((FIXTYPE)ceil(getflonum(arg)));
    xlbadtype(arg);
    return v_nil; /* never reached */
}

/* xround - built-in function 'round' */
LVAL xround(void)
{
    FLOTYPE x,y,z;
    LVAL arg;

    /* get the argument */
    arg = xlgetarg();
    xllastarg();

    /* check its type */
    if (fixp(arg))
	return arg;
    else if (floatp(arg)) {
	x = getflonum(arg);
	y = floor(x);
	z = x - y;
	if (z == 0.5) {
	    if (((FIXTYPE)y & 1) == 1)
		y += 1.0;
	    return cvfixnum((FIXTYPE)y);
	}
	else if (z < 0.5)
	    return cvfixnum((FIXTYPE)y);
	else
	    return cvfixnum((FIXTYPE)(y + 1.0));
    }
    xlbadtype(arg);
    return v_nil; /* never reached */
}

/* xtruncate - built-in function 'truncate' */
LVAL xtruncate(void)
{
    LVAL arg;

    /* get the argument */
    arg = xlgetarg();
    xllastarg();

    /* check its type */
    if (fixp(arg))
	return arg;
    else if (floatp(arg))
	return cvfixnum((FIXTYPE)(getflonum(arg)));
    xlbadtype(arg);
    return v_nil; /* never reached */
}

#define FIXBITS ((int)(sizeof(FIXTYPE) * CHAR_BIT))

/* xash - arithmetic shift */
LVAL xash(void)
{
    FIXTYPE val,shift;
    xlval = xlgafixnum(); val = getfixnum(xlval);
    xlval = xlgafixnum(); shift = getfixnum(xlval); 
    xllastarg();
    if (shift <= -FIXBITS)
        return cvfixnum(val >> (FIXBITS - 1));
    else if (shift >= FIXBITS)
        return cvfixnum(0);
    return cvfixnum(shift > 0 ? val << shift : val >> -shift);
}

/* xlsh - logical shift */
LVAL xlsh(void)
{
    FIXTYPE val,shift;
    xlval = xlgafixnum(); val = getfixnum(xlval);
    xlval = xlgafixnum(); shift = getfixnum(xlval); 
    xllastarg();
    if (shift <= -FIXBITS || shift >= FIXBITS)
        return cvfixnum(0);
    return cvfixnum(shift > 0 ? val << shift : (UFIXTYPE)val >> -shift);
}

/* binary functions */
LVAL xadd(void)				/* + */
{
    if (!moreargs())
	return cvfixnum((FIXTYPE)0);
    return binary('+');
}
LVAL xmul(void)				/* * */
{
    if (!moreargs())
	return cvfixnum((FIXTYPE)1);
    return binary('*');
}
LVAL xsub(void)    { return binary('-'); } /* - */
LVAL xdiv(void)    { return binary('/'); } /* / */
LVAL xquo(void)    { return binary('Q'); } /* quotient */
LVAL xrem(void)    { return binary('R'); } /* remainder */
LVAL xmod(void)	   { return binary('r'); } /* modulo */
LVAL xmin(void)    { return binary('m'); } /* min */
LVAL xmax(void)    { return binary('M'); } /* max */
LVAL xexpt(void)   { return binary('E'); } /* expt */
LVAL xlogand(void) { return binary('&'); } /* logand */
LVAL xlogior(void) { return binary('|'); } /* logior */
LVAL xlogxor(void) { return binary('^'); } /* logxor */

/* binary - handle binary operations */
static LVAL binary(int fcn)
{
    FIXTYPE ival,iarg;
    FLOTYPE fval,farg;
    LVAL arg;
    int mode;

    /* get the first argument */
    arg = xlgetarg();

    /* set the type of the first argument */
    if (fixp(arg)) {
	ival = getfixnum(arg);
	mode = 'I';
    }
    else if (floatp(arg)) {
	fval = getflonum(arg);
	mode = 'F';
    }
    else
	xlbadtype(arg);

    /* treat a single argument as a special case */
    if (!moreargs()) {
	switch (fcn) {
	case '-':
	    switch (mode) {
	    case 'I':
		ival = -ival;
		break;
	    case 'F':
		fval = -fval;
		break;
	    }
	    break;
	case '/':
	    switch (mode) {
	    case 'I':
		checkizero(ival);
		if (ival != 1) {
		    fval = 1.0 / (FLOTYPE)ival;
		    mode = 'F';
		}
		break;
	    case 'F':
		checkfzero(fval);
		fval = 1.0 / fval;
		break;
	    }
	}
    }

    /* handle each remaining argument */
    while (moreargs()) {

	/* get the next argument */
	arg = xlgetarg();

	/* check its type */
	if (fixp(arg)) {
	    switch (mode) {
	    case 'I':
	        iarg = getfixnum(arg);
	        break;
	    case 'F':
	        farg = (FLOTYPE)getfixnum(arg);
		break;
	    }
	}
	else if (floatp(arg)) {
	    switch (mode) {
	    case 'I':
	        fval = (FLOTYPE)ival;
		farg = getflonum(arg);
		mode = 'F';
		break;
	    case 'F':
	        farg = getflonum(arg);
		break;
	    }
	}
	else
	    xlbadtype(arg);

	/* accumulate the result value */
	switch (mode) {
	case 'I':
	    switch (fcn) {
	    case '+':	ival += iarg; break;
	    case '-':	ival -= iarg; break;
	    case '*':	ival *= iarg; break;
	    case '/':	checkizero(iarg);
			if ((ival % iarg) == 0)	    
			    ival /= iarg;
			else {
			    fval = (FLOTYPE)ival;
			    farg = (FLOTYPE)iarg;
			    fval /= farg;
			    mode = 'F';
			}
			break;
	    case 'Q':	checkizero(iarg); ival /= iarg; break;
	    case 'R':	ival = remain(ival,iarg); break;
	    case 'r':	ival = modulo(ival,iarg); break;
	    case 'M':	if (iarg > ival) ival = iarg; break;
	    case 'm':	if (iarg < ival) ival = iarg; break;
	    case 'E':	return cvflonum((FLOTYPE)pow((FLOTYPE)ival,(FLOTYPE)iarg));
	    case '&':	ival &= iarg; break;
	    case '|':	ival |= iarg; break;
	    case '^':	ival ^= iarg; break;
	    default:	badiop();
	    }
	    break;
	case 'F':
	    switch (fcn) {
	    case '+':	fval += farg; break;
	    case '-':	fval -= farg; break;
	    case '*':	fval *= farg; break;
	    case '/':	checkfzero(farg); fval /= farg; break;
	    case 'M':	if (farg > fval) fval = farg; break;
	    case 'm':	if (farg < fval) fval = farg; break;
	    case 'E':	fval = pow(fval,farg); break;
	    default:	badfop();
	    }
    	    break;
	}
    }

    /* return the result */
    return mode == 'I' ? cvfixnum(ival) : cvflonum(fval);
}

/* remain - divide two numbers returning the remainder */
static FIXTYPE remain(FIXTYPE num,FIXTYPE den)
{
    FIXTYPE result;
    checkizero(den);
    result = num % den;
    return result;
}

/* modulo - divide two numbers returning the modulo */
static FIXTYPE modulo(FIXTYPE num,FIXTYPE den)
{
    FIXTYPE result;
    checkizero(den);
    result = num % den;
    return result != 0  && (num < 0) != (den < 0) ? result + den : result;
}

/* unary functions */
LVAL xlognot(void)   { return unary('~'); } /* lognot */
LVAL xabs(void)      { return unary('A'); } /* abs */
LVAL xadd1(void)     { return unary('+'); } /* 1+ */
LVAL xsub1(void)     { return unary('-'); } /* -1+ */
LVAL xsin(void)      { return unary('S'); } /* sin */
LVAL xcos(void)      { return unary('C'); } /* cos */
LVAL xtan(void)      { return unary('T'); } /* tan */
LVAL xasin(void)     { return unary('s'); } /* asin */
LVAL xacos(void)     { return unary('c'); } /* acos */
LVAL xxexp(void)     { return unary('E'); } /* exp */
LVAL xsqrt(void)     { return unary('R'); } /* sqrt */
LVAL xxlog(void)     { return unary('L'); } /* log */
LVAL xrandom(void)   { return unary('?'); } /* random */

/* xsetrandomseed - set random number generator seed */
LVAL xsetrandomseed(void)
{
    xlval = xlgafixnum();
    xllastarg();
    ossetrand(getfixnum(xlval));
    return xlval;
}

/* unary - handle unary operations */
static LVAL unary(int fcn)
{
    FLOTYPE fval;
    FIXTYPE ival;
    LVAL arg;

    /* get the argument */
    arg = xlgetarg();
    xllastarg();

    /* check its type */
    if (fixp(arg)) {
	ival = getfixnum(arg);
	switch (fcn) {
	case '~':	ival = ~ival; break;
	case 'A':	ival = (ival < 0 ? -ival : ival); break;
	case '+':	ival++; break;
	case '-':	ival--; break;
	case 'S':	return cvflonum((FLOTYPE)sin((FLOTYPE)ival));
	case 'C':	return cvflonum((FLOTYPE)cos((FLOTYPE)ival));
	case 'T':	return cvflonum((FLOTYPE)tan((FLOTYPE)ival));
	case 's':	return cvflonum((FLOTYPE)asin((FLOTYPE)ival));
	case 'c':	return cvflonum((FLOTYPE)acos((FLOTYPE)ival));
	case 't':	return cvflonum((FLOTYPE)atan((FLOTYPE)ival));
	case 'E':	return cvflonum((FLOTYPE)exp((FLOTYPE)ival));
	case 'L':	return cvflonum((FLOTYPE)log((FLOTYPE)ival));
	case 'R':	checkineg(ival);
			return cvflonum((FLOTYPE)sqrt((FLOTYPE)ival));
	case '?':	ival = (FIXTYPE)osrand((int)ival); break;
	default:	badiop();
	}
	return cvfixnum(ival);
    }
    else if (floatp(arg)) {
	fval = getflonum(arg);
	switch (fcn) {
	case 'A':	fval = (fval < 0.0 ? -fval : fval); break;
	case '+':	fval += 1.0; break;
	case '-':	fval -= 1.0; break;
	case 'S':	fval = sin(fval); break;
	case 'C':	fval = cos(fval); break;
	case 'T':	fval = tan(fval); break;
	case 's':	fval = asin(fval); break;
	case 'c':	fval = acos(fval); break;
	case 't':	fval = atan(fval); break;
	case 'E':	fval = exp(fval); break;
	case 'L':	fval = log(fval); break;
	case 'R':	checkfneg(fval);
			fval = sqrt(fval); break;
	default:	badfop();
	}
	return cvflonum(fval);
    }
    xlbadtype(arg);
    return v_nil; /* never reached */
}

/* unary predicates */
LVAL xnegativep(void) { return predicate('-'); } /* negative? */
LVAL xzerop(void)     { return predicate('Z'); } /* zero? */
LVAL xpositivep(void) { return predicate('+'); } /* positive? */
LVAL xevenp(void)     { return predicate('E'); } /* even? */
LVAL xoddp(void)      { return predicate('O'); } /* odd? */

/* predicate - handle a predicate function */
static LVAL predicate(int fcn)
{
    FLOTYPE fval;
    FIXTYPE ival;
    LVAL arg;

    /* get the argument */
    arg = xlgetarg();
    xllastarg();

    /* check the argument type */
    if (fixp(arg)) {
	ival = getfixnum(arg);
	switch (fcn) {
	case '-':	ival = (ival < 0); break;
	case 'Z':	ival = (ival == 0); break;
	case '+':	ival = (ival > 0); break;
	case 'E':	ival = ((ival & 1) == 0); break;
	case 'O':	ival = ((ival & 1) != 0); break;
	default:	badiop();
	}
    }
    else if (floatp(arg)) {
	fval = getflonum(arg);
	switch (fcn) {
	case '-':	ival = (fval < 0); break;
	case 'Z':	ival = (fval == 0); break;
	case '+':	ival = (fval > 0); break;
	default:	badfop();
	}
    }
    else
	xlbadtype(arg);

    /* return the result value */
    return ival ? v_true : v_false;
}

/* comparison functions */
LVAL xlss(void) { return compare('<'); } /* < */
LVAL xleq(void) { return compare('L'); } /* <= */
LVAL xeql(void) { return compare('='); } /* = */
LVAL xneq(void) { return compare('#'); } /* = */
LVAL xgeq(void) { return compare('G'); } /* >= */
LVAL xgtr(void) { return compare('>'); } /* > */

/* compare - common compare function */
static LVAL compare(int fcn)
{
    FIXTYPE icmp,ival,iarg;
    FLOTYPE fcmp,fval,farg;
    LVAL arg;
    int mode;

    /* get the first argument */
    arg = xlgetarg();

    /* set the type of the first argument */
    if (fixp(arg)) {
	ival = getfixnum(arg);
	mode = 'I';
    }
    else if (floatp(arg)) {
	fval = getflonum(arg);
	mode = 'F';
    }
    else
	xlbadtype(arg);

    /* handle each remaining argument */
    for (icmp = TRUE; icmp && moreargs(); ival = iarg, fval = farg) {

	/* get the next argument */
	arg = xlgetarg();

	/* check its type */
	if (fixp(arg)) {
	    switch (mode) {
	    case 'I':
	        iarg = getfixnum(arg);
	        break;
	    case 'F':
	        farg = (FLOTYPE)getfixnum(arg);
		break;
	    }
	}
	else if (floatp(arg)) {
	    switch (mode) {
	    case 'I':
	        fval = (FLOTYPE)ival;
		farg = getflonum(arg);
		mode = 'F';
		break;
	    case 'F':
	        farg = getflonum(arg);
		break;
	    }
	}
	else
	    xlbadtype(arg);

	/* compute result of the compare */
	switch (mode) {
	case 'I':
	    icmp = ival - iarg;
	    switch (fcn) {
	    case '<':	icmp = (icmp < 0); break;
	    case 'L':	icmp = (icmp <= 0); break;
	    case '=':	icmp = (icmp == 0); break;
	    case '#':	icmp = (icmp != 0); break;
	    case 'G':	icmp = (icmp >= 0); break;
	    case '>':	icmp = (icmp > 0); break;
	    }
	    break;
	case 'F':
	    fcmp = fval - farg;
	    switch (fcn) {
	    case '<':	icmp = (fcmp < 0.0); break;
	    case 'L':	icmp = (fcmp <= 0.0); break;
	    case '=':	icmp = (fcmp == 0.0); break;
	    case '#':	icmp = (fcmp != 0.0); break;
	    case 'G':	icmp = (fcmp >= 0.0); break;
	    case '>':	icmp = (fcmp > 0.0); break;
	    }
	    break;
	}
    }
    xlpopargs();

    /* return the result */
    return icmp ? v_true : v_false;
}

/* toflotype - convert a lisp value to a floating point number */
FLOTYPE toflotype(LVAL val)
{
    /* must be a number for this to work */
    return ntype(val) == FIXNUM ? (FLOTYPE)getfixnum(val)
				: getflonum(val);
}

/* checkizero - check for integer division by zero */
static void checkizero(FIXTYPE iarg)
{
    if (iarg == 0)
	xlfmterror("division by zero");
}

/* checkineg - check for square root of a negative number */
static void checkineg(FIXTYPE iarg)
{
    if (iarg < 0)
	xlfmterror("square root of a negative number");
}

/* checkfzero - check for floating point division by zero */
static void checkfzero(FLOTYPE farg)
{
    if (farg == 0.0)
	xlfmterror("division by zero");
}

/* checkfneg - check for square root of a negative number */
static void checkfneg(FLOTYPE farg)
{
    if (farg < 0.0)
	xlfmterror("square root of a negative number");
}

/* badiop - bad integer operation */
static void badiop(void)
{
    xlfmterror("bad integer operation");
}

/* badfop - bad floating point operation */
static void badfop(void)
{
    xlfmterror("bad floating point operation");
}
