/* xlobj.c - xlisp object-oriented programming support */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* external variables */
extern LVAL xlenv,xlval;
extern LVAL s_stdout;

/* local variables */
static LVAL k_initialize;
static LVAL class,object;

/* local prototypes */
static void showobject(LVAL obj,LVAL fptr);
static void parse_cvars(LVAL frame,LVAL defs);
static void addivar(LVAL cls,char *var);
static void addmsg(LVAL cls,char *msg,char *fname);
static LVAL entermsg(LVAL cls,LVAL msg);
static int getivcnt(LVAL cls,int ivar);
static LVAL copylists(LVAL list1,LVAL list2);

/* newobject - allocate and initialize a new object */
static LVAL newobject(LVAL cls,int size)
{
    LVAL val;
    cpush(cls);
    val = newvector(size + 1); /* class, ivars */
    val->n_type = OBJECT;
    setclass(val,pop());
    return val;
}

/* xlsend - send a message to an object */
void xlsend(LVAL obj,LVAL sym)
{
    LVAL msg,cls,p;

    /* look for the message in the class or superclasses */
    for (cls = getclass(obj); cls; cls = getivar(cls,IV_SUPERCLASS))
	for (p = getivar(cls,IV_MESSAGES); p; p = cdr(p))
	    if ((msg = car(p)) != v_nil && car(msg) == sym) {
		push(obj); ++xlargc; /* insert 'self' argument */
		xlval = cdr(msg);    /* get the method */
		xlnext = xlapply;    /* invoke the method */
		return;
	    }

    /* message not found */
    xlfmterror("~S has no method for the message ~S",obj,sym);
}

/* xsendsuper - built-in function 'send-super' */
void xsendsuper(void)
{
    LVAL sym,msg,cls,p;

    /* get the method class and the message selector */
    cls = xlgaobject();
    sym = xlgasymbol();
    
    /* look for the message in the class or superclasses */
    for (cls = getivar(cls,IV_SUPERCLASS); cls; cls = getivar(cls,IV_SUPERCLASS))
	for (p = getivar(cls,IV_MESSAGES); p; p = cdr(p))
	    if ((msg = car(p)) != v_nil && car(msg) == sym) {
		xlval = cdr(msg);    /* get the method */
		xlnext = xlapply;    /* invoke the method */
		return;
	    }

    /* message not found */
    xlerror("no method for this message",sym);
}

/* obinitialize - default 'initialize' method */
LVAL obinitialize(void)
{
    LVAL self;
    self = xlgaobject();
    xllastarg();
    return (self);
}

/* obclass - get the class of an object */
LVAL obclass(void)
{
    LVAL self;
    self = xlgaobject();
    xllastarg();
    return (getclass(self));
}

/* obgetvariable - get the value of an instance variable */
LVAL obgetvariable(void)
{
    LVAL class,ivars,self,sym;
    int offset = FIRSTIVAR;
    
    /* parse the argument list */
    self = xlgaobject();
    sym = xlgasymbol();
    xllastarg();

    /* get the instance variable offset */
    class = getclass(self);
    for (ivars = getivar(class,IV_IVARS); consp(ivars); ivars = cdr(ivars)) {
        if (sym == car(ivars))
	    break;
        ++offset;
    }

    /* make sure we found it */
    if (!consp(ivars))
        xlfmterror("no instance variable ~S",sym);

    /* return the value */
    return getivar(self,offset);
}

/* obsetvariable - set the value of an instance variable */
LVAL obsetvariable(void)
{
    LVAL class,ivars,self,sym,val;
    int offset = FIRSTIVAR;
    
    /* parse the argument list */
    self = xlgaobject();
    sym = xlgasymbol();
    val = xlgetarg();
    xllastarg();

    /* get the instance variable offset */
    class = getclass(self);
    for (ivars = getivar(class,IV_IVARS); consp(ivars); ivars = cdr(ivars)) {
        if (sym == car(ivars))
	    break;
        ++offset;
    }

    /* make sure we found it */
    if (!consp(ivars))
        xlfmterror("no instance variable ~S",sym);

    /* set and return the value */
    setivar(self,offset,val);
    return val;
}

/* obprint - print an object */
LVAL obprint(void)
{
    LVAL self,class,fptr;
    char buf[256];
    
    /* get self and the file pointer */
    self = xlgaobject();
    fptr = (moreargs() ? xlgetoport() : curoutput());
    xllastarg();
    
    /* print the object */
    if ((class = getclass(self)) != v_nil
    &&  (class = getivar(class,IV_CLASSNAME)) != v_nil) {
        xlputstr(fptr,"#<Object:");
        xlprin1(class,fptr);
	xlputstr(fptr," #");
    }
    else
        xlputstr(fptr,"#<Object #x");
    sprintf(buf,AFMT,self);
    strcat(buf,">");
    xlputstr(fptr,buf);
    return self;
}

/* obshow - show an object */
LVAL obshow(void)
{
    LVAL self,fptr;

    /* get self and the file pointer */
    self = xlgaobject();
    fptr = (moreargs() ? xlgetoport() : curoutput());
    xllastarg();
    
    /* show the object */
    showobject(self,fptr);
    return self;
}

/* showobject - show an object */
static void showobject(LVAL obj,LVAL fptr)
{
    LVAL cls,names;
    int maxi,i;

    /* get the object's class */
    cls = getclass(obj);

    /* print the object and class */
    xlputstr(fptr,"\nObject is ");
    xlprin1(obj,fptr);
    xlputstr(fptr,", Class is ");
    xlprin1(cls,fptr);

    /* print the object's instance variables */
    names = getivar(cls,IV_IVARS);
    maxi = FIRSTIVAR + getivcnt(cls,IV_IVARTOTAL);
    for (i = FIRSTIVAR; i < maxi; ++i) {
	if (i == FIRSTIVAR)
	    xlputstr(fptr,"\nInstance variables:");
	xlputstr(fptr,"\n  ");
	xlprin1(car(names),fptr);
	xlputstr(fptr," = ");
	xlprin1(getivar(obj,i),fptr);
	names = cdr(names);
    }
}

/* clmakeinstance - create a new object instance */
LVAL clmakeinstance(void)
{
    LVAL self;
    self = xlgaobject();
    xllastarg();
    return newobject(self,getivcnt(self,IV_IVARTOTAL));
}

/* clnew - create a new object instance and initialize */
void clnew(void)
{
    LVAL self;

    /* create a new object */
    self = xlgaobject();
    xlval = newobject(self,getivcnt(self,IV_IVARTOTAL));

    /* send the 'initialize' message */
    xlsend(xlval,k_initialize);
}

/* clinitialize - initialize a new class */
LVAL clinitialize(void)
{
    LVAL self,ivars,cvars,super,name;
    FIXTYPE n;

    /* get self, the ivars, cvars and superclass */
    self = xlgaobject();
    ivars = xlgalist();
    cvars = (moreargs() ? xlgalist() : v_nil);
    super = (moreargs() ? xlgaobject() : object);
    name = (moreargs() ? xlgasymbol() : v_nil);
    xllastarg();

    /* create the class variable environment */
    check(5);
    push(self);
    push(name);
    push(super);
    push(ivars);
    push(cvars);
    xlval = newframe(ENV,getivar(super,IV_CVARS),FIRSTENV + length(top()) + 1);
    parse_cvars(xlval,pop());
    setenvelement(xlval,FIRSTENV,self);

    /* store the instance and class variable lists and the superclass */
    setivar(self,IV_CLASSNAME,name);
    setivar(self,IV_IVARS,copylists(getivar(super,IV_IVARS),ivars));
    setivar(self,IV_CVARS,xlval);
    setivar(self,IV_SUPERCLASS,super);
    drop(4);
    
    /* compute the instance variable count */
    n = length(ivars);
    setivar(self,IV_IVARCNT,cvfixnum(n));
    n += getivcnt(super,IV_IVARTOTAL);
    setivar(self,IV_IVARTOTAL,cvfixnum(n));

    /* return the new class object */
    return (self);
}

/* parse_cvars - parse class variable declarations */
static void parse_cvars(LVAL frame,LVAL defs)
{
    int i = FIRSTENV; /* leave space for %%CLASS */
    LVAL this,last;
    check(2);
    push(frame);
    push(defs);
    last = cons(xlenter("%%CLASS"),v_nil);
    setenvnames(frame,last);
    for (; consp(defs); defs = cdr(defs)) {
	LVAL def = car(defs),sym,val;
	if (symbolp(def)) {
	    sym = def;
	    val = v_nil;
	}
	else if (consp(def) && symbolp(car(def))) {
	    sym = car(def);
	    val = consp(cdr(def)) ? car(cdr(def)) : v_nil;
	}
	else
	    xlerror("expecting a class variable definition",def);
	this = cons(sym,v_nil);
	rplacd(last,this);
	setelement(frame,++i,val);
	last = this;
    }
    drop(2);
}

/* clanswer - define a method for answering a message */
LVAL clanswer(void)
{
    LVAL self,msg,fargs,code,mptr;

    /* message symbol, formal argument list and code */
    self = xlgaobject();
    msg = xlgasymbol();
    fargs = xlgetarg();
    code = xlgalist();
    xllastarg();

    /* protect our working structures */
    check(4);
    push(self);
    push(msg);
    push(fargs);
    push(code);
    
    /* make a new message list entry */
    mptr = entermsg(self,msg);

    /* compile and store the method */
    xlval = xlmethod(msg,fargs,code,getivar(self,IV_CVARS));
    rplacd(mptr,cvclosure(xlval,getivar(self,IV_CVARS)));
    drop(4);
    
    /* return the object */
    return (self);
}

/* clshow - show a class */
LVAL clshow(void)
{
    LVAL self,fptr,env;
    int first = TRUE;
    
    /* get self and the file pointer */
    self = xlgaobject();
    fptr = (moreargs() ? xlgetoport() : curoutput());
    xllastarg();
    
    /* show the object */
    showobject(self,fptr);

    /* print the object's class variables */
    for (env = getivar(self,IV_CVARS); env != v_nil; env = getnextframe(env)) {
        LVAL names = cdr(getenvnames(env)); /* skip the %%class variable */
        FIXTYPE i,maxi = getenvsize(env);
	for (i = FIRSTENV + 1; i < maxi; ++i) {
	    if (first) {
	        xlputstr(fptr,"\nClass variables:");
	        first = FALSE;
	    }
	    xlputstr(fptr,"\n  ");
	    xlprin1(car(names),fptr);
	    xlputstr(fptr," = ");
	    xlprin1(getenvelement(env,i),fptr);
	    names = cdr(names);
	}
    }
    return self;
}

/* addivar - enter an instance variable */
static void addivar(LVAL cls,char *var)
{
    setivar(cls,IV_IVARS,cons(xlenter(var),getivar(cls,IV_IVARS)));
}

/* addmsg - add a message to a class */
static void addmsg(LVAL cls,char *msg,char *fname)
{
    LVAL mptr;

    /* enter the message selector */
    mptr = entermsg(cls,xlenter(msg));

    /* store the method for this message */
    rplacd(mptr,getvalue(xlenter(fname)));
}

/* entermsg - add a message to a class */
static LVAL entermsg(LVAL cls,LVAL msg)
{
    LVAL lptr,mptr;

    /* lookup the message */
    for (lptr = getivar(cls,IV_MESSAGES); lptr; lptr = cdr(lptr))
	if (car(mptr = car(lptr)) == msg)
	    return (mptr);

    /* allocate a new message entry if one wasn't found */
    cpush(cons(msg,v_nil));
    setivar(cls,IV_MESSAGES,cons(top(),getivar(cls,IV_MESSAGES)));

    /* return the symbol node */
    return (pop());
}

/* getivcnt - get the number of instance variables for a class */
static int getivcnt(LVAL cls,int ivar)
{
    LVAL cnt;
    if ((cnt = getivar(cls,ivar)) == v_nil || !fixp(cnt))
	xlerror("bad value for instance variable count",cnt);
    return ((int)getfixnum(cnt));
}

/* copylists - make a copy of two lists */
static LVAL copylists(LVAL list1,LVAL list2)
{
    LVAL last,next;
    
    /* initialize */
    cpush(v_nil); last = v_nil;
    
    /* copy the first list */
    for (; consp(list1); list1 = cdr(list1)) {
	next = cons(car(list1),v_nil);
	if (last) rplacd(last,next);
	else settop(next);
	last = next;
    }
    
    /* append the second list */
    for (; consp(list2); list2 = cdr(list2)) {
	next = cons(car(list2),v_nil);
	if (last) rplacd(last,next);
	else settop(next);
	last = next;
    }
    return pop();
}

/* obsymbols - initialize symbols */
void obsymbols(void)
{
    /* enter the object related symbols */
    k_initialize = xlenter("INITIALIZE");

    /* get the Object and Class symbol values */
    object = getvalue(xlenter("OBJECT"));
    class  = getvalue(xlenter("CLASS"));
}

/* xloinit - object function initialization routine */
void xloinit(void)
{
    LVAL sym;
    
    /* create the 'Object' object */
    sym = xlenter("OBJECT");
    object = newobject(v_nil,CLASSSIZE);
    setvalue(sym,object);
    setivar(object,IV_CLASSNAME,sym);
    setivar(object,IV_IVARTOTAL,cvfixnum((FIXTYPE)0));
    setivar(object,IV_IVARCNT,cvfixnum((FIXTYPE)0));
    setivar(object,IV_CVARS,newframe(ENV,v_nil,FIRSTENV + 1));
    setenvnames(getivar(object,IV_CVARS),cons(xlenter("%%CLASS"),v_nil));
    setenvelement(getivar(object,IV_CVARS),FIRSTENV,object);
    addmsg(object,"INITIALIZE","%OBJECT-INITIALIZE");
    addmsg(object,"CLASS","%OBJECT-CLASS");
    addmsg(object,"GET-VARIABLE","%OBJECT-GET-VARIABLE");
    addmsg(object,"SET-VARIABLE!","%OBJECT-SET-VARIABLE!");
    addmsg(object,"PRINT","%OBJECT-PRINT");
    addmsg(object,"SHOW","%OBJECT-SHOW");
    
    /* create the 'Class' object */
    sym = xlenter("CLASS");
    class = newobject(v_nil,CLASSSIZE);
    setvalue(sym,class);
    addivar(class,"IVARTOTAL");	/* ivar number 6 */
    addivar(class,"IVARCNT");	/* ivar number 5 */
    addivar(class,"SUPERCLASS");/* ivar number 4 */
    addivar(class,"CVARS");	/* ivar number 3 */
    addivar(class,"IVARS");	/* ivar number 2 */
    addivar(class,"MESSAGES");	/* ivar number 1 */
    addivar(class,"NAME");	/* ivar number 0 */
    setivar(class,IV_CLASSNAME,sym);
    setivar(class,IV_IVARTOTAL,cvfixnum((FIXTYPE)CLASSSIZE));
    setivar(class,IV_IVARCNT,cvfixnum((FIXTYPE)CLASSSIZE));
    setivar(class,IV_CVARS,newframe(ENV,v_nil,FIRSTENV + 1));
    setenvnames(getivar(class,IV_CVARS),cons(xlenter("%%CLASS"),v_nil));
    setenvelement(getivar(class,IV_CVARS),FIRSTENV,class);
    setivar(class,IV_SUPERCLASS,object);
    addmsg(class,"MAKE-INSTANCE","%CLASS-MAKE-INSTANCE");
    addmsg(class,"NEW","%CLASS-NEW");
    addmsg(class,"INITIALIZE","%CLASS-INITIALIZE");
    addmsg(class,"ANSWER","%CLASS-ANSWER");
    addmsg(class,"SHOW","%CLASS-SHOW");

    /* patch the class into 'object' and 'class' */
    setclass(object,class);
    setclass(class,class);
}
