/* xlprint.c - xlisp print routine */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* global variables */
int prbreadth = -1;
int prdepth = -1;

/* local variables */
static char buf[200];

/* external variables */
extern LVAL s_printcase,k_downcase;
extern LVAL s_fixfmt,s_flofmt,s_unbound;

static void print(LVAL fptr,LVAL vptr,int escflag,int depth);
static void putatm(LVAL fptr,char *tag,LVAL val);
static void putfstream(LVAL fptr,LVAL val);
static void putstring(LVAL fptr,LVAL str);
static void putqstring(LVAL fptr,LVAL str);
static void putsymbol(LVAL fptr,LVAL sym,int escflag);
static void putsympname(LVAL fptr,LVAL pname,int escflag);
static void putidentifier(LVAL fptr,char *name,FIXTYPE len,int escflag);
static void putpackage(LVAL fptr,LVAL val);
static void putsubr(LVAL fptr,char *tag,LVAL val);
static void putclosure(LVAL fptr,char *tag,LVAL val);
static void putcode(LVAL fptr,char *tag,LVAL val);
static void putnumber(LVAL fptr,FIXTYPE n);
static void putoct(LVAL fptr,int n);
static void putflonum(LVAL fptr,FLOTYPE n);
static void putcharacter(LVAL fptr,int ch);
static void putobject(LVAL fptr,LVAL val);
static void putforeignptr(LVAL fptr,LVAL val);
static void putfree(LVAL fptr,LVAL val);

/* xlprin1 - print an expression with quoting */
void xlprin1(LVAL expr,LVAL file)
{
    check(2);
    push(file);
    push(expr);
    print(file,expr,TRUE,0);
    drop(2);
}

/* xlprinc - print an expression without quoting */
void xlprinc(LVAL expr,LVAL file)
{
    check(2);
    push(file);
    push(expr);
    print(file,expr,FALSE,0);
    drop(2);
}

/* xlterpri - terminate the current print line */
void xlterpri(LVAL fptr)
{
    xlputc(fptr,'\n');
}

/* xlfreshline - terminate the current print line if necessary */
void xlfreshline(LVAL fptr)
{
    if ((getpflags(fptr) & PF_BOL) == 0)
    	xlterpri(fptr);
}

/* xlputstr - output a string */
void xlputstr(LVAL fptr,char *str)
{
    while (*str)
	xlputc(fptr,*str++);
}

/* print - internal print routine */
static void print(LVAL fptr,LVAL vptr,int escflag,int depth)
{
    extern FUNDEF subrtab[],xsubrtab[];
    FIXTYPE size,i;
    LVAL nptr,next;
    int breadth;

    /* print #t */
    if (vptr == v_true) {
        putidentifier(fptr,"#T",2,escflag);
        return;
    }
    
    /* print nil */
    else if (vptr == v_nil) {
	xlputstr(fptr,"()");
	return;
    }

    /* check value type */
    switch (ntype(vptr)) {
    case SUBR:
	    putsubr(fptr,"Subr",vptr);
	    break;
    case XSUBR:
	    putsubr(fptr,"XSubr",vptr);
	    break;
    case CONS:
	    if (prdepth >= 0 && depth >= prdepth) {
		xlputstr(fptr,"(...)");
		break;
	    }
	    xlputc(fptr,'(');
	    breadth = 0;
	    for (nptr = vptr; nptr != v_nil; nptr = next) {
		if (prbreadth >= 0 && breadth++ >= prbreadth) {
		    xlputstr(fptr,"...");
		    break;
		}
	        print(fptr,car(nptr),escflag,depth+1);
		if ((next = cdr(nptr)) != v_nil)
		    if (consp(next))
			xlputc(fptr,' ');
		    else {
			xlputstr(fptr," . ");
			print(fptr,next,escflag,depth+1);
			break;
		    }
	    }
	    xlputc(fptr,')');
	    break;
    case VECTOR:
	    xlputstr(fptr,"#(");
	    for (i = 0, size = getsize(vptr); i < size; ++i) {
		if (i != 0) xlputc(fptr,' ');
		print(fptr,getelement(vptr,i),escflag,depth+1);
	    }
	    xlputc(fptr,')');
	    break;
    case SYMBOL:
	    putsymbol(fptr,vptr,escflag);
	    break;
    case PACKAGE:
	    putpackage(fptr,vptr);
	    break;
    case PROMISE:
	    if (getpproc(vptr) != v_nil)
		putatm(fptr,"Promise",vptr);
	    else
		putatm(fptr,"Forced-promise",vptr);
	    break;
    case CLOSURE:
	    putclosure(fptr,"Procedure",vptr);
	    break;
    case FIXNUM:
	    putnumber(fptr,getfixnum(vptr));
	    break;
    case FLONUM:
	    putflonum(fptr,getflonum(vptr));
	    break;
    case CHARACTER:
	    if (escflag)
		putcharacter(fptr,getchcode(vptr));
	    else
		xlputc(fptr,getchcode(vptr));
	    break;
    case STRING:
	    if (escflag)
	        putqstring(fptr,vptr);
	    else
	        putstring(fptr,vptr);
	    break;
    case FSTREAM:
	    putfstream(fptr,vptr);
	    break;
    case USTREAM:
	    putatm(fptr,"Unnamed-stream",vptr);
	    break;
    case OSTREAM:
	    putatm(fptr,"Object-stream",vptr);
	    break;
    case CODE:
	    putcode(fptr,"Code",vptr);
	    break;
    case CONTINUATION:
	    putatm(fptr,"Escape-procedure",vptr);
	    break;
    case ENV:
	    putatm(fptr,"Environment",vptr);
	    break;
    case SENV:
	    putatm(fptr,"Stack-environment",vptr);
	    break;
    case MSENV:
	    putatm(fptr,"Moved-stack-environment",vptr);
	    break;
    case MENV:
	    putatm(fptr,"Method-environment",vptr);
	    break;
    case SMENV:
	    putatm(fptr,"Stack-method-environment",vptr);
	    break;
    case OBJECT:
	    putobject(fptr,vptr);
	    break;
    case FOREIGNPTR:
	    putforeignptr(fptr,vptr);
	    break;
    case TABLE:
	    putatm(fptr,"Table",vptr);
	    break;
    case FREE:
	    putfree(fptr,vptr);
	    break;
    default:
	    putatm(fptr,"Foo",vptr);
	    break;
    }
}

/* putatm - output an atom */
static void putatm(LVAL fptr,char *tag,LVAL val)
{
    sprintf(buf,"#<%s #x",tag); xlputstr(fptr,buf);
    sprintf(buf,AFMT,val); xlputstr(fptr,buf);
    xlputc(fptr,'>');
}

/* putfstream - output a file stream */
static void putfstream(LVAL fptr,LVAL val)
{
    xlputstr(fptr,"#<File-stream #x");
    sprintf(buf,AFMT,val); xlputstr(fptr,buf);
    xlputc(fptr,':');
    sprintf(buf,AFMT,getfile(val)); xlputstr(fptr,buf);
    xlputc(fptr,'>');
}

/* putstring - output a string */
static void putstring(LVAL fptr,LVAL str)
{
    FIXTYPE len;
    char *p;

    /* get the pointer and count */
    p = getstring(str);
    len = getslength(str);

    /* output each character in the string */
    while (--len >= 0)
	xlputc(fptr,*p++);
}

/* putqstring - output a quoted string */
static void putqstring(LVAL fptr,LVAL str)
{
    FIXTYPE len;
    char *p;
    int ch;

    /* use an unsigned pointer and length */
    p = getstring(str);
    len = getslength(str);

    /* output the initial quote */
    xlputc(fptr,'"');

    /* output each character in the string */
    while (--len >= 0)

	/* check for a control character */
	if ((ch = *p++) < 040 || ch == '\\' || ch == '"') {
	    xlputc(fptr,'\\');
	    switch (ch) {
	    case '\033':
		    xlputc(fptr,'e');
		    break;
	    case '\n':
		    xlputc(fptr,'n');
		    break;
	    case '\r':
		    xlputc(fptr,'r');
		    break;
	    case '\t':
		    xlputc(fptr,'t');
		    break;
	    case '\\':
	    case '"':
		    xlputc(fptr,ch);
		    break;
	    default:
		    putoct(fptr,ch);
		    break;
	    }
	}

	/* output a normal character */
	else
	    xlputc(fptr,ch);

    /* output the terminating quote */
    xlputc(fptr,'"');
}

/* putsymbol - output a symbol */
static void putsymbol(LVAL fptr,LVAL sym,int escflag)
{
    extern LVAL s_package,keywordpackage,k_internal;
    LVAL package,key;
    if ((package = getpackage(sym)) == v_nil)
	xlputstr(fptr,"#:");
    else if (package == keywordpackage)
	xlputc(fptr,':');
    else if (!isvisible(sym,getvalue(s_package))) {
	findsymbol(getstring(getpname(sym)),package,&key);
	putsympname(fptr,car(getnames(getpackage(sym))),escflag);
	if (key == k_internal) xlputc(fptr,':');
	xlputc(fptr,':');
    }
    putsympname(fptr,getpname(sym),escflag);
}

/* putsympname - output a symbol print name */
static void putsympname(LVAL fptr,LVAL pname,int escflag)
{
    putidentifier(fptr,getstring(pname),getslength(pname),escflag);
}

/* putidentifier - output an identifier */
static void putidentifier(LVAL fptr,char *name,FIXTYPE len,int escflag)
{
   if (escflag) {
	if (getvalue(s_printcase) == k_downcase)
	    while (--len >= 0) {
	        int ch = *name++;
		xlputc(fptr,isupper(ch) ? tolower(ch) : ch);
	    }
	else
	    while (--len >= 0) {
	        int ch = *name++;
		xlputc(fptr,islower(ch) ? toupper(ch) : ch);
	    }
    }
    else {
        while (--len >= 0)
	    xlputc(fptr,*name++);
    }
}

/* putpackage - output a package */
static void putpackage(LVAL fptr,LVAL val)
{
    xlputstr(fptr,"#<Package ");
    putstring(fptr,car(getnames(val)));
    xlputstr(fptr,">");
}

/* putsubr - output a subr/fsubr */
static void putsubr(LVAL fptr,char *tag,LVAL val)
{
    sprintf(buf,"#<%s %s>",tag,getfundef(val)->fd_name);
    xlputstr(fptr,buf);
}

/* putclosure - output a closure */
static void putclosure(LVAL fptr,char *tag,LVAL val)
{
    putcode(fptr,tag,getcode(val));
}

/* putcode - output a code object */
static void putcode(LVAL fptr,char *tag,LVAL val)
{
    LVAL name;
    if ((name = getelement(val,1)) != v_nil) {
	xlputstr(fptr,"#<");
	xlputstr(fptr,tag);
	xlputstr(fptr," ");
	putstring(fptr,getpname(name));
	xlputstr(fptr,">");
    }
    else
	putatm(fptr,tag,val);
}

/* putnumber - output a number */
static void putnumber(LVAL fptr,FIXTYPE n)
{
    LVAL fmt = getvalue(s_fixfmt);
    sprintf(buf,stringp(fmt) ? getstring(fmt) : IFMT,n);
    xlputstr(fptr,buf);
}

/* putoct - output an octal byte value */
static void putoct(LVAL fptr,int n)
{
    sprintf(buf,"%03o",n);
    xlputstr(fptr,buf);
}

/* putflonum - output a flonum */
static void putflonum(LVAL fptr,FLOTYPE n)
{
    LVAL fmt = getvalue(s_flofmt);
    sprintf(buf,stringp(fmt) ? getstring(fmt) : FFMT,n);
    xlputstr(fptr,buf);
}

/* putcharacter - output a character value */
static void putcharacter(LVAL fptr,int ch)
{
    switch (ch) {
    case '\n':
	xlputstr(fptr,"#\\Newline");
	break;
    case ' ':
	xlputstr(fptr,"#\\Space");
	break;
    default:
	sprintf(buf,"#\\%c",ch);
	xlputstr(fptr,buf);
	break;
    }
}

/* putobject - output an object value */
static void putobject(LVAL fptr,LVAL obj)
{
    extern LVAL s_print;
    CallFunction(&obj,obj,2,s_print,fptr);
}

/* putforeignptr - output a foreign pointer value */
static void putforeignptr(LVAL fptr,LVAL val)
{
    char buf[100];
    xlputstr(fptr,"#<FP:");
    xlprin1(getfptype(val),fptr);
    xlputstr(fptr," #");
    sprintf(buf,AFMT,val);
    strcat(buf,":");
    sprintf(&buf[strlen(buf)],AFMT,getfptr(val));
    strcat(buf,">");
    xlputstr(fptr,buf);
}

/* must be in type id order */
static char *typenames[] = {
"FREE",
"CONS",
"SYMBOL",
"FIXNUM",
"FLONUM",
"STRING",
"FSTREAM",
"USTREAM",
"OSTREAM",
"VECTOR",
"CLOSURE",
"CODE",
"SUBR",
"XSUBR",
"CONTINUATION",
"CHARACTER",
"PROMISE",
"ENV",
"SENV",
"MSENV",
"MENV",
"SMENV",
"OBJECT",
"PACKAGE",
"FOREIGNPTR"
};

/* putfree - output a free value */
static void putfree(LVAL fptr,LVAL val)
{
    char buf[100];
    int typeid;
    xlputstr(fptr,"#<Free #");
    sprintf(buf,AFMT,val);
    strcat(buf," was ");
    typeid = (int)(long)car(val);
    if (typeid >= 0 && typeid <= MAXTYPEID)
        sprintf(&buf[strlen(buf)],"%s",typenames[typeid]);
    else
        sprintf(&buf[strlen(buf)],"unknown type (%d)",typeid);
    strcat(buf,">");
    xlputstr(fptr,buf);
}


