/* xlsym.c - symbol handling routines */
/*	Copyright (c) 1995, by David Michael Betz
	All Rights Reserved
	Permission is granted for unrestricted non-commercial use	*/

#include "xlisp.h"

/* global variables */
LVAL schemepackage,xlisppackage,keywordpackage,systempackage;

/* external variables */
extern LVAL s_package,packages,k_internal,k_external,k_inherited;

/* forward declarations */
static LVAL addtolist(LVAL list,LVAL val);
static LVAL removefromlist(LVAL list,LVAL val);
static LVAL findinlist(LVAL list,LVAL val);
static void addtotable(LVAL table,LVAL sym);
static void removefromtable(LVAL table,LVAL sym);
static LVAL findintable(LVAL table,LVAL sym);
static LVAL findnameintable(LVAL array,char *name);
static void entersymbol(LVAL sym,LVAL table);
static LVAL findprop(LVAL sym,LVAL prp);
static int comparestr(char *cstr,LVAL str);
static FIXTYPE hash(LVAL val,FIXTYPE size);
static FIXTYPE hashstr(char *str,FIXTYPE len,FIXTYPE size);

/* xlpinit - initialize the packages */
void xlpinit(void)
{
    LVAL package,key;
    
    /* create the XLISP package */
    xlisppackage = newpackage("XLISP");
    s_package = xlintern("*PACKAGE*",xlisppackage,&key);
    export(s_package,xlisppackage);
    
    /* create the rest of the packages */
    schemepackage  = newpackage("SCHEME");
    keywordpackage = newpackage("KEYWORD");
    systempackage  = newpackage("SYSTEM");
    
    /* create the USER package */
    package = newpackage("USER");
    setvalue(s_package,package);
    usepackage(xlisppackage,package);
    usepackage(schemepackage,package);
    usepackage(systempackage,package);
}

/* xlsubr - define a builtin function */
void xlsubr(LVAL package,int type,FUNDEF *def)
{
    LVAL sym,key;
    sym = xlintern(def->fd_name,package,&key);
    export(sym,package);
    setvalue(sym,cvsubr(type,def));
}

/* xlenter - enter a symbol in the current package */
LVAL xlenter(char *name)
{
    LVAL key;
    return xlintern(name,getvalue(s_package),&key);
}

/* xlkenter - enter a keyword */
LVAL xlkenter(char *name)
{
    return internandexport(name,keywordpackage);
}

/* xlintern - intern a symbol in a package */
LVAL xlintern(char *name,LVAL package,LVAL *pkey)
{
    LVAL sym;
    if ((sym = findsymbol(name,package,pkey)) == v_nil) {
    	cpush(cvsymbol(cvcstring(name)));
	entersymbol(top(),getintern(package));
	if (package == keywordpackage)
	    setvalue(top(),top());
	setpackage(top(),package);
	sym = pop();
    }
    return sym;
}

/* findpackage - find a package by name */
LVAL findpackage(char *name)
{
    LVAL pack,p;
    for (pack = packages; packagep(pack); pack = getnextpackage(pack))
	for (p = getnames(pack); consp(p); p = cdr(p))
	    if (comparestr(name,car(p)))
		return pack;
    return v_nil;
}

/* usepackage - add a package to another package usedby list */
int usepackage(LVAL src,LVAL dst)
{
    setuses(dst,addtolist(getuses(dst),src));
    setusedby(src,addtolist(getusedby(src),dst));
    return TRUE;
}

/* unusepackage - remove a package from another package usedby list */
int unusepackage(LVAL src,LVAL dst)
{
    setuses(dst,removefromlist(getuses(dst),src));
    setusedby(src,removefromlist(getusedby(src),dst));
    return TRUE;
}

/* import - import a symbol into a package */
int import(LVAL sym,LVAL package)
{
    if (!ispresent(sym,package)) {
	addtotable(getintern(package),sym);
	if (getpackage(sym) == v_nil)
	    setpackage(sym,package);
    }
    return TRUE;
}

/* export - export a symbol from a package */
int export(LVAL sym,LVAL package)
{
    if (ispresent(sym,package)) {
	removefromtable(getintern(package),sym);
	addtotable(getextern(package),sym);
    }
    else
	xlerror("symbol is not present",sym);
    return TRUE;
}

/* unexport - unexport a symbol from a package */
int unexport(LVAL sym,LVAL package)
{
    if (ispresent(sym,package)) {
	removefromtable(getextern(package),sym);
	addtotable(getintern(package),sym);
    }
    else
	xlerror("symbol is not present",sym);
    return TRUE;
}

/* isvisible - determine if a symbol is visible in a package */
int isvisible(LVAL sym,LVAL package)
{
    LVAL list;
    if (ispresent(sym,package))
	return TRUE;
    for (list = getuses(package); consp(list); list = cdr(list))
	if (findintable(getextern(car(list)),sym) != v_nil)
	    return TRUE;
    return FALSE;
}

/* ispresent - determine if a symbol is present in a package */
int ispresent(LVAL sym,LVAL package)
{
    return findintable(getextern(package),sym) != v_nil
    ||     findintable(getintern(package),sym) != v_nil;
}

/* findsymbol - find a symbol in a package */
LVAL findsymbol(char *name,LVAL package,LVAL *pkey)
{
    LVAL list,sym;
    if ((sym = findnameintable(getextern(package),name)) != v_nil) {
	*pkey = k_external;
	return sym;
    }
    else if ((sym = findnameintable(getintern(package),name)) != v_nil) {
	*pkey = k_internal;
	return sym;
    }
    for (list = getuses(package); consp(list); list = cdr(list))
	if ((sym = findnameintable(getextern(car(list)),name)) != v_nil) {
	    *pkey = k_inherited;
	    return sym;
	}
    *pkey = v_nil;
    return v_nil;
}

/* intern - intern a symbol in a package */
LVAL intern(LVAL name,LVAL package,LVAL *pkey)
{
    LVAL sym;
    if ((sym = findsymbol(getstring(name),package,pkey)) == v_nil) {
    	cpush(cvsymbol(copystring(name)));
	entersymbol(top(),getintern(package));
	if (package == keywordpackage)
	    setvalue(top(),top());
	setpackage(top(),package);
	sym = pop();
    }
    return sym;
}

/* internandexport - intern a symbol in a package and make it external */
LVAL internandexport(char *name,LVAL package)
{
    LVAL sym,key;
    sym = xlintern(name,package,&key);
    export(sym,package);
    return sym;
}

/* unintern - remove a symbol from a package */
void unintern(LVAL sym,LVAL package)
{
    removefromtable(getextern(package),sym);
    removefromtable(getintern(package),sym);
    if (getpackage(sym) == package)
	setpackage(sym,v_nil);
}

/* addtolist - add a value to a list if it isn't already there */
static LVAL addtolist(LVAL list,LVAL val)
{
    LVAL this;
    for (this = list; consp(this); this = cdr(this))
	if (val == car(this))
	    return list;
    return cons(val,list);
}

/* removefromlist - remove an entry from a list */
static LVAL removefromlist(LVAL list,LVAL val)
{
    LVAL prev,this;
    for (prev = v_nil, this = list; consp(this); prev = this, this = cdr(this))
	if (val == car(this)) {
	    if (prev != v_nil) rplacd(prev,cdr(this));
	    else list = cdr(this);
	    break;
	}
    return list;
}

/* findinlist - find a value in a list */
static LVAL findinlist(LVAL list,LVAL val)
{
    LVAL this;
    for (this = list; consp(this); this = cdr(this))
	if (val == car(this))
	    return val;
    return v_nil;
}

/* addtotable - add a symbol to a hash table */
static void addtotable(LVAL table,LVAL sym)
{
    LVAL pname = getpname(sym);
    FIXTYPE i = hashstr(getstring(pname),getslength(pname),getsize(table));
    setelement(table,i,addtolist(getelement(table,i),sym));
}

/* removefromtable - remove a symbol from a hash table */
static void removefromtable(LVAL table,LVAL sym)
{
    LVAL pname = getpname(sym);
    FIXTYPE i = hashstr(getstring(pname),getslength(pname),getsize(table));
    setelement(table,i,removefromlist(getelement(table,i),sym));
}

/* findintable - find a symbol in a hash table */
static LVAL findintable(LVAL table,LVAL sym)
{
    LVAL pname = getpname(sym);
    FIXTYPE i = hashstr(getstring(pname),getslength(pname),getsize(table));
    return findinlist(getelement(table,i),sym);
}

/* findnameintable - find a symbol by name in a hash table */
static LVAL findnameintable(LVAL table,char *name)
{
    FIXTYPE i = hashstr(name,strlen(name),getsize(table));
    LVAL sym;
    for (sym = getelement(table,i); sym != v_nil; sym = cdr(sym))
	if (comparestr(name,getpname(car(sym))))
	    return car(sym);
    return v_nil;
}

/* entersymbol - enter a symbol into a hash table */
static void entersymbol(LVAL sym,LVAL table)
{
    LVAL pname = getpname(sym);
    FIXTYPE i = hashstr(getstring(pname),getslength(pname),getsize(table));
    setelement(table,i,cons(sym,getelement(table,i)));
}

/* xlgetprop - get the value of a property */
LVAL xlgetprop(LVAL sym,LVAL prp)
{
    LVAL p;
    return (p = findprop(sym,prp)) != v_nil ? car(p) : v_nil;
}

/* xlputprop - put a property value onto the property list */
void xlputprop(LVAL sym,LVAL val,LVAL prp)
{
    LVAL pair;
    if ((pair = findprop(sym,prp)) != v_nil)
	rplaca(pair,val);
    else
	setplist(sym,cons(prp,cons(val,getplist(sym))));
}

/* xlremprop - remove a property from a property list */
void xlremprop(LVAL sym,LVAL prp)
{
    LVAL last,p;
    last = v_nil;
    for (p = getplist(sym); consp(p) && consp(cdr(p)); p = cdr(last)) {
	if (car(p) == prp)
	    if (last != v_nil)
		rplacd(last,cdr(cdr(p)));
	    else
		setplist(sym,cdr(cdr(p)));
	last = cdr(p);
    }
}

/* findprop - find a property pair */
static LVAL findprop(LVAL sym,LVAL prp)
{
    LVAL p;
    for (p = getplist(sym); consp(p) && consp(cdr(p)); p = cdr(cdr(p)))
	if (car(p) == prp)
	    return cdr(p);
    return v_nil;
}

/* comparestr - compare a c string with a lisp string */
static int comparestr(char *cstr,LVAL str)
{
    char *lstr = getstring(str);
    FIXTYPE len = getslength(str);
    for (; --len >= 0 && *cstr != '\0'; ++cstr, ++lstr)
        if (*cstr != *lstr)
            return FALSE;
    return len == -1 && *cstr == '\0';
}

/* findentryintable - find an entry in a hash table by key */
LVAL findentryintable(LVAL table,LVAL key)
{
    FIXTYPE i = hash(key,getsize(table));
    LVAL list = getelement(table,i);
    for (; list != v_nil; list = cdr(list)) {
        LVAL entry = car(list);
	if (eqv(key,car(entry)))
	    return entry;
    }
    return v_nil;
}

/* addentrytotable - add an entry to a hash table */
void addentrytotable(LVAL table,LVAL key,LVAL val)
{
    FIXTYPE i = hash(key,getsize(table));
    LVAL list = getelement(table,i);
    for (; list != v_nil; list = cdr(list)) {
        LVAL entry = car(list);
	if (eqv(key,car(entry))) {
            rplacd(entry,val);
	    return;
        }
    }
    setelement(table,i,cons(cons(key,val),getelement(table,i)));
}

/* removeentryfromtable - remove an entry from a hash table by key */
LVAL removeentryfromtable(LVAL table,LVAL key)
{
    FIXTYPE i = hash(key,getsize(table));
    LVAL list = getelement(table,i);
    LVAL prev = v_nil;
    for (; list != v_nil; prev = list, list = cdr(list)) {
        LVAL entry = car(list);
	if (eqv(key,car(entry))) {
            if (prev == v_nil)
                setelement(table,i,cdr(list));
            else
                rplacd(prev,cdr(list));
	    return entry;
        }
    }
    return v_nil;
}

/* hash - hash a lisp value */
static FIXTYPE hash(LVAL val,FIXTYPE size)
{
    switch (ntype(val)) {
    case STRING:
        return hashstr(getstring(val),getslength(val),size);
    case FIXNUM:
        return (FIXTYPE)getfixnum(val) % size;
    case FLONUM:
        return (FIXTYPE)getflonum(val) % size;
    default:
        return (FIXTYPE)val % size;
    }
}

/* hashstr - hash a symbol name string */
static FIXTYPE hashstr(char *str,FIXTYPE len,FIXTYPE size)
{
    unsigned long i;
    for (i = 0; --len >= 0; )
	i = (i << 2) ^ *str++;
    return (FIXTYPE)(i %= size);
}
