/****************************************************************************

    PROGRAM: lzhuf.c

    PURPOSE: Test program for HUFFMAN DLL routines

    ORIGINS: This program, and accompanying DLL, are derived from DDJ's
	          NELSON.ARC and Microsoft's Windows Developers Toolkit.
			   The original preambles follow this babble.
			   Performance has not been a consideration in the adapting of
			   this module or the DLL - simply functionality.

    EXECUTION: Run LZHUF.EXE from the Program Manager "File" pull down.
	            Select "Open" from the LZHUF menu bar.
				 Choose a file to be test input - LZHUF produces output
				 as defined below.
				 Accompanying HUFFMAN.DLL must be dynamically available.

    FUNCTIONS: To take any file and run it through a variety of 
	            compression/expansion functions.
				 File size limit of 64K.
				 Function tested are:
				     CompressFileToBuffer
				     ExpandBufferToBuffer
				     CompressFileToBuffer
				     ExpandBufferToFile
				 Output files produced are:
				     LZH_OUT.CFB - CompressFileToBuffer (copy of buffer)
				     LZH_OUT.EBB - ExpandBufferToBuffer (copy of buffer)
				     LZH_OUT.CBB - CompressBufferToBuffer (copy of buffer)
				     LZH_OUT.EBF - Expand Buffer To File
	        

****************************************************************************/
/**************** ORIGINAL DDJ NELSON.ARC PREAMBLE **************************
    lzhuf.c
    written by Haruyasu Yoshizaki 1988/11/20
    some minor changes 1989/04/06
    comments translated by Haruhiko Okumura 1989/04/07
    getbit and getbyte modified 1990/03/23 by Paul Edwards
      so that they would work on machines where integers are
      not necessarily 16 bits (although ANSI guarantees a
      minimum of 16).  This program has compiled and run with
      no errors under Turbo C 2.0, Power C, and SAS/C 4.5
      (running on an IBM mainframe under MVS/XA 2.2).  Could
      people please use YYYY/MM/DD date format so that everyone
      in the world can know what format the date is in?
    external storage of filesize changed 1990/04/18 by Paul Edwards to
      Intel's "little endian" rather than a machine-dependant style so
      that files produced on one machine with lzhuf can be decoded on
      any other.  "little endian" style was chosen since lzhuf
      originated on PC's, and therefore they should dictate the
      standard.
    initialization of something predicting spaces changed 1990/04/22 by
      Paul Edwards so that when the compressed file is taken somewhere
      else, it will decode properly, without changing ascii spaces to
      ebcdic spaces.  This was done by changing the ' ' (space literal)
      to 0x20 (which is the far most likely character to occur, if you
      don't know what environment it will be running on.
    storage of filesize modified 1990/06/02 by Mark Nelson.
      When reading in the file size, I was getting sign extension
      when reading in bytes greater than or equal to 0x80, which
      messed everything up.
*****************************************************************************/
/*************** ORIGINAL MICROSOFT PREAMBLE ********************************

    PROGRAM: Fileopen.c

    PURPOSE: Loads, saves, and edits text files

    FUNCTIONS:

        WinMain() - calls initialization function, processes message loop
        InitApplication() - initializes window data and registers window
        InitInstance() - saves instance handle and creates main window
        MainWndProc() - processes messages
        About() - processes messages for "About" dialog box
        OpenDlg() - let user select a file, and open it.
        UpdateListBox() - Update the list box of OpenDlg
        ChangeDefExt() - Change the default extension
        SeparateFile() - Separate filename and pathname
        AddExt() - Add default extension

****************************************************************************/

#include "windows.h"
#include "sys\types.h"
#include "sys\stat.h"
#include "lzhuf.h"

#include "huffman.h"

HANDLE hInst;

HANDLE hAccTable;                                /* handle to accelerator table */
HWND hEditWnd;                                      /* handle to edit window */
HWND hwnd;                                      /* handle to main window */

char FileName[128];
char PathName[128];
char OpenName[128];
char DefPath[128];
char DefSpec[13] = "*.*";
char DefExt[] = ".txt";
char str[255];

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
HANDLE hPrevInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
    MSG msg;

    if (!hPrevInstance)
        if (!InitApplication(hInstance))
            return (FALSE);

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    while (GetMessage(&msg, NULL, NULL, NULL)) {

    /* Only translate message if it is not an accelerator message */

        if (!TranslateAccelerator(hwnd, hAccTable, &msg)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg); 
        }
    }
    return (msg.wParam);
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;
{
    WNDCLASS  wc;

    wc.style = NULL;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName =  "LzhufMenu";
    wc.lpszClassName = "LzhufWClass";

    return (RegisterClass(&wc));
}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;
    int             nCmdShow;
{
    RECT            Rect;

    hInst = hInstance;

    hwnd = CreateWindow(
        "LzhufWClass",
        "Huffman Compress/Expand Application",
        WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        NULL,
        NULL,
        hInstance,
        NULL
    );

    if (!hwnd)
        return (FALSE);

    GetClientRect(hwnd, (LPRECT) &Rect);

    /* Create a child window */

    hEditWnd = CreateWindow("Edit",
        NULL,
        WS_CHILD | WS_VISIBLE |
        ES_MULTILINE |
        WS_VSCROLL | WS_HSCROLL |
        ES_AUTOHSCROLL | ES_AUTOVSCROLL,
        0,
        0,
        (Rect.right-Rect.left),
        (Rect.bottom-Rect.top),
        hwnd,
        IDC_EDIT,                          /* Child control i.d. */
        hInst,
        NULL);

    if (!hEditWnd) {
        DestroyWindow(hwnd);
        return (NULL);
    }

    ShowWindow(hwnd, nCmdShow);
    UpdateWindow(hwnd);
    return (TRUE);

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

    MESSAGES:

        WM_COMMAND    - application menu (About dialog box)
        WM_DESTROY    - destroy window

    COMMENTS:

        WM_COMMAND processing:

            IDM_OPEN - query to save current file if there is one and it
                       has been changed, open a new file.

            IDM_ABOUT - display "About" box.

****************************************************************************/

long FAR PASCAL MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;
unsigned message;
WORD wParam;
LONG lParam;
{
    FARPROC lpProcAbout, lpOpenDlg, lpSaveAsDlg;

    int Success;                            /* return value from SaveAsDlg() */
    int IOStatus;                           /* result of file i/o      */
    int Return;

    switch (message) {
        case WM_COMMAND:
            switch (wParam) {
                case IDM_ABOUT:
                    lpProcAbout = MakeProcInstance(About, hInst);
                    DialogBox(hInst, "AboutBox", hWnd, lpProcAbout);
                    FreeProcInstance(lpProcAbout);
                    break;

                case IDM_OPEN:
                    /* Call OpenDlg() to get the filename */

                    lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                    Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                    FreeProcInstance(lpOpenDlg);

                    if (Return)
					      HuffmanTest(OpenName, hWnd);

                    break;

                case IDM_EXIT:
                    DestroyWindow(hWnd);
                    break;
    
            } 
            break;

        case WM_SETFOCUS:
            SetFocus (hEditWnd);
            break;

        case WM_SIZE:
            MoveWindow(hEditWnd, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
            break;

        case WM_DESTROY:
            PostQuitMessage(0);
            break;

        default:
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}

/****************************************************************************

    FUNCTION: OpenDlg(HWND, unsigned, WORD, LONG)

    PURPOSE: Let user select a file, and return.  Open code not provided.

****************************************************************************/

HANDLE FAR PASCAL OpenDlg(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
    WORD index;
    PSTR pTptr;
    HANDLE hFile=1;     /* Temp value for return */

    switch (message) {
        case WM_COMMAND:
            switch (wParam) {

                case IDC_LISTBOX:
                    switch (HIWORD(lParam)) {

                        case LBN_SELCHANGE:
                            /* If item is a directory name, append "*.*" */
                            if (DlgDirSelect(hDlg, str, IDC_LISTBOX)) 
                                strcat(str, DefSpec);

                            SetDlgItemText(hDlg, IDC_EDIT, str);
                            SendDlgItemMessage(hDlg,
                                IDC_EDIT,
                                EM_SETSEL,
                                NULL,
                                MAKELONG(0, 0x7fff));
                            break;

                        case LBN_DBLCLK:
                            goto openfile;
                    }
                    return (TRUE);

                case IDOK:
openfile:
                    GetDlgItemText(hDlg, IDC_PATH, PathName, 128);
                    GetDlgItemText(hDlg, IDC_EDIT, OpenName, 128);
                    if (strchr(OpenName, '*') || strchr(OpenName, '?')) {
                        SeparateFile(hDlg, (LPSTR) str, (LPSTR) DefSpec,
                            (LPSTR) OpenName);
                        if (str[0])
                            strcpy(DefPath, str);
                        ChangeDefExt(DefExt, DefSpec);
                        UpdateListBox(hDlg);
                        return (TRUE);
                    }

                    if (!OpenName[0]) {
                        MessageBox(hDlg, "No filename specified.",
                            NULL, MB_OK | MB_ICONHAND);
                        return (TRUE);
                    }

                    AddExt(OpenName, DefExt);

                    /* The routine to open the file would go here, and the */
                    /* file handle would be returned instead of NULL.           */
                    EndDialog(hDlg, hFile);
                    return (TRUE);

                case IDCANCEL:
                    EndDialog(hDlg, NULL);
                    return (FALSE);
            }
            break;

        case WM_INITDIALOG:                        /* message: initialize    */
            UpdateListBox(hDlg);
            SetDlgItemText(hDlg, IDC_EDIT, DefSpec);
            SendDlgItemMessage(hDlg,               /* dialog handle      */
                IDC_EDIT,                          /* where to send message  */
                EM_SETSEL,                         /* select characters      */
                NULL,                              /* additional information */
                MAKELONG(0, 0x7fff));              /* entire contents      */
            SetFocus(GetDlgItem(hDlg, IDC_EDIT));
            return (FALSE); /* Indicates the focus is set to a control */
    }
    return FALSE;
}

/****************************************************************************

    FUNCTION: UpdateListBox(HWND);

    PURPOSE: Update the list box of OpenDlg

****************************************************************************/

void UpdateListBox(hDlg)
HWND hDlg;
{
    strcpy(str, DefPath);
    strcat(str, DefSpec);
    DlgDirList(hDlg, str, IDC_LISTBOX, IDC_PATH, 0x4010);

    /* To ensure that the listing is made for a subdir. of
     * current drive dir...
     */
    if (!strchr (DefPath, ':'))
	DlgDirList(hDlg, DefSpec, IDC_LISTBOX, IDC_PATH, 0x4010);

    /* Remove the '..' character from path if it exists, since this
     * will make DlgDirList move us up an additional level in the tree
     * when UpdateListBox() is called again.
     */
    if (strstr (DefPath, ".."))
	DefPath[0] = '\0';

    SetDlgItemText(hDlg, IDC_EDIT, DefSpec);
}

/****************************************************************************

    FUNCTION: ChangeDefExt(PSTR, PSTR);

    PURPOSE: Change the default extension

****************************************************************************/

void ChangeDefExt(Ext, Name)
PSTR Ext, Name;
{
    PSTR pTptr;

    pTptr = Name;
    while (*pTptr && *pTptr != '.')
        pTptr++;
    if (*pTptr)
        if (!strchr(pTptr, '*') && !strchr(pTptr, '?'))
            strcpy(Ext, pTptr);
}

/****************************************************************************

    FUNCTION: SeparateFile(HWND, LPSTR, LPSTR, LPSTR)

    PURPOSE: Separate filename and pathname

****************************************************************************/

void SeparateFile(hDlg, lpDestPath, lpDestFileName, lpSrcFileName)
HWND hDlg;
LPSTR lpDestPath, lpDestFileName, lpSrcFileName;
{
    LPSTR lpTmp;
    char  cTmp;

    lpTmp = lpSrcFileName + (long) lstrlen(lpSrcFileName);
    while (*lpTmp != ':' && *lpTmp != '\\' && lpTmp > lpSrcFileName)
        lpTmp = AnsiPrev(lpSrcFileName, lpTmp);
    if (*lpTmp != ':' && *lpTmp != '\\') {
        lstrcpy(lpDestFileName, lpSrcFileName);
        lpDestPath[0] = 0;
        return;
    }
    lstrcpy(lpDestFileName, lpTmp + 1);
    cTmp = *(lpTmp + 1);
    lstrcpy(lpDestPath, lpSrcFileName);
     *(lpTmp + 1) = cTmp;
    lpDestPath[(lpTmp - lpSrcFileName) + 1] = 0;
}

/****************************************************************************

    FUNCTION: AddExt(PSTR, PSTR);

    PURPOSE: Add default extension

/***************************************************************************/

void AddExt(Name, Ext)
PSTR Name, Ext;
{
    PSTR pTptr;

    pTptr = Name;
    while (*pTptr && *pTptr != '.')
        pTptr++;
    if (*pTptr != '.')
        strcat(Name, Ext);
}

/****************************************************************************

    FUNCTION: HuffmanTest(PSTR, HWND);

    PURPOSE: Perform Huffman Compress/Expand Tests

/***************************************************************************/

void HuffmanTest(Name, hWnd)
PSTR Name;
HWND hWnd;
{
HDC      hDC;
struct stat statbuf;
short    rc;
int      hFile;
OFSTRUCT OfStruct;
HANDLE   hBufferA;
HANDLE   hBufferB;
LPSTR    BufferA;
LPSTR    BufferB;
unsigned long ulBuffSize;
unsigned long ulTextSize;
char     OutFileName[128];

     hDC = GetDC(hWnd);
     TextOut(hDC, 160, 15, Name, strlen(Name)); 
     if (hFile = OpenFile(Name, &OfStruct, OF_READ))
     {
        fstat(hFile, &statbuf);
        ulBuffSize = statbuf.st_size;
     }
     else{
        ReleaseDC(hWnd, hDC);
        return;
        }
     if (ulBuffSize > 65535){
        _lclose(hFile);
        sprintf(str, "%s opened, %12ld bytes -- Sorry TOO LARGE", Name, ulBuffSize);
        TextOut(hDC, 160, 15, str, strlen(str)); 
        ReleaseDC(hWnd, hDC);
        return;
        }
     sprintf(str, "%s opened, %12ld bytes", Name, ulBuffSize);
     TextOut(hDC, 160, 15, str, strlen(str)); 
     sprintf(str, "Default path is %s", PathName);
     TextOut(hDC, 160, 30, str, strlen(str)); 
	  hBufferA = GlobalAlloc(GMEM_MOVEABLE, ulBuffSize);
     if (hBufferA == NULL){
        _lclose(hFile);
        sprintf(str, "Couldn't Allocate Buffer A - %12ld", ulBuffSize);
        TextOut(hDC, 160, 45, str, strlen(str)); 
        ReleaseDC(hWnd, hDC);
        return;
     }
	  BufferA = (LPSTR)GlobalLock(hBufferA);
	  hBufferB = GlobalAlloc(GMEM_MOVEABLE, ulBuffSize);
     if (hBufferB == NULL){
        _lclose(hFile);
        sprintf(str, "Couldn't Allocate Buffer B - %12ld", ulBuffSize);
        TextOut(hDC, 160, 45, str, strlen(str)); 
        ReleaseDC(hWnd, hDC);
        return;
     }
	  BufferB = (LPSTR)GlobalLock(hBufferB);
     rc = CompressFileToBuffer(hFile, BufferA, ulBuffSize,
	                            &ulTextSize);
     _lclose(hFile);
     sprintf(str, "Compressed Buffer Size - %12ld", ulTextSize);
     TextOut(hDC, 160, 45, str, strlen(str)); 
     strcpy(OutFileName, PathName);
     strcat(OutFileName, "\\LZH_OUT.CFB");
     hFile = OpenFile(OutFileName, &OfStruct, OF_CREATE);
     if (!hFile){
        sprintf(str, "Unable to open file - %s", OutFileName);
        TextOut(hDC, 160, 60, str, strlen(str)); 
        ReleaseDC(hWnd, hDC);
        return;
        }
     _lwrite(hFile, BufferA, (USHORT)ulTextSize);
     _lclose(hFile);
     ulTextSize = BufferA[0];
     ulTextSize <<= 8;
     ulTextSize |= BufferA[1];
     ulTextSize <<= 8;
     ulTextSize |= BufferA[2];
     ulTextSize <<= 8;
     ulTextSize |= BufferA[3];
     rc = ExpandBufferToBuffer(BufferA, ulTextSize,
	                            BufferB, ulBuffSize,
	                            &ulTextSize);
     sprintf(str, "Expanded Buffer Size - %12ld", ulTextSize);
     TextOut(hDC, 160, 60, str, strlen(str)); 
     strcpy(OutFileName, PathName);
     strcat(OutFileName, "\\LZH_OUT.EBB");
     hFile = OpenFile(OutFileName, &OfStruct, OF_CREATE);
     if (!hFile){
        sprintf(str, "Unable to open file - %s", OutFileName);
        TextOut(hDC, 160, 60, str, strlen(str)); 
        ReleaseDC(hWnd, hDC);
        return;
        }
     _lwrite(hFile, BufferB, (USHORT)ulTextSize);
     _lclose(hFile);
     rc = CompressBufferToBuffer(BufferB, ulTextSize,
	                              BufferA, ulBuffSize,
	                              &ulTextSize);
     sprintf(str, "Re-compressed Buffer Size - %12ld", ulTextSize);
     TextOut(hDC, 160, 75, str, strlen(str)); 
     ulTextSize = BufferA[0];
     ulTextSize <<= 8;
     ulTextSize |= BufferA[1];
     ulTextSize <<= 8;
     ulTextSize |= BufferA[2];
     ulTextSize <<= 8;
     ulTextSize |= BufferA[3];
     strcpy(OutFileName, PathName);
     strcat(OutFileName, "\\LZH_OUT.EBF");
     hFile = OpenFile(OutFileName, &OfStruct, OF_CREATE);
     if (!hFile){
        sprintf(str, "Unable to open file - %s", OutFileName);
        TextOut(hDC, 160, 60, str, strlen(str)); 
        ReleaseDC(hWnd, hDC);
        return;
        }
     rc = ExpandBufferToFile(BufferA, ulTextSize, hFile);
     _lclose(hFile);

     while (GlobalFlags(hBufferA) & GMEM_LOCKCOUNT)
            GlobalUnlock(hBufferA);
     GlobalFree(hBufferA);
     while (GlobalFlags(hBufferB) & GMEM_LOCKCOUNT)
            GlobalUnlock(hBufferB);
     GlobalFree(hBufferB);
     ReleaseDC(hWnd, hDC);
}

/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

        WM_INITDIALOG - initialize dialog box
        WM_COMMAND    - Input received

****************************************************************************/

BOOL FAR PASCAL About(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
    switch (message) {
        case WM_INITDIALOG:
            return (TRUE);

        case WM_COMMAND:
	    if (wParam == IDOK
                || wParam == IDCANCEL) {
                EndDialog(hDlg, TRUE);
                return (TRUE);
            }
            break;
    }
    return (FALSE);
}

