/*
 * dezip1.c - sample code to deflate first file in a .zip archive
 *
 * This code demonstrates how to use inflate.h and crc.h to
 * process data in a .zip file.
 *
 * The .zip file header is described below.  It consists of
 * 30 fixed bytes, followed by two variable length fields
 * whose length is contained in the first 30 bytes.  After this
 * header, the data is stored (in deflate format if the compression
 * method is 8).
 *
 * .zip file header:
 *
 *      local file header signature     4 bytes  (0x04034b50)
 *      version needed to extract       2 bytes
 *      general purpose bit flag        2 bytes
 *      compression method              2 bytes
 *      last mod file time              2 bytes
 *      last mod file date              2 bytes
 *      crc-32                          4 bytes
 *      compressed size                 4 bytes
 *      uncompressed size               4 bytes
 *      filename length                 2 bytes
 *      extra field length              2 bytes
 *
 *      filename (variable size)
 *      extra field (variable size)
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef BENCHMARK
#include <time.h>
#endif

#include "inflate.h"
#include "crc.h"

#define GETUINT4(ptr, i4)                                               \
  {                                                                     \
    i4 = (((unsigned long) *(((unsigned char *) (ptr)) + 0))      ) |   \
	 (((unsigned long) *(((unsigned char *) (ptr)) + 1)) <<  8) |   \
	 (((unsigned long) *(((unsigned char *) (ptr)) + 2)) << 16) |   \
	 (((unsigned long) *(((unsigned char *) (ptr)) + 3)) << 24)   ; \
  }

#define GETUINT2(ptr, i2)                                               \
  {                                                                     \
    i2 = (((unsigned  int) *(((unsigned char *) (ptr)) + 0))      ) |   \
	 (((unsigned  int) *(((unsigned char *) (ptr)) + 1)) <<  8)   ; \
  }

/* Variable to accumulate crc into */
static unsigned long crc;

#ifdef BENCHMARK
  clock_t t1, t2;
  long total;
#endif

int inflate_putbuffer(                    /* returns 0 on success       */
    void *AppState,                       /* opaque ptr from Initialize */
    unsigned char *buffer,                /* buffer to put              */
    long length                           /* length of buffer           */
)
{
  size_t ret;

  crc = CrcUpdate(crc, buffer, length);

#ifdef BENCHMARK
  total += length;
  return 0;
#else
  ret = fwrite(buffer, 1, (size_t) length, (FILE *) AppState);

  if ((size_t) length != ret)
    return 1;
  else
    return 0;
#endif
}

void *inflate_malloc(long length)
{
  void *ret;

  ret = malloc((size_t) length);

  if (!ret) fprintf(stderr, "Error allocating %ld bytes",length);

  return ret;
}

void inflate_free(void *buffer)
{
  free(buffer);
}

#define BUFFSIZE 16384

static unsigned char buff[BUFFSIZE];

int main(int argc, char **argv)
{
  unsigned long sign;     /* local file header signature (0x04034b50) */
  unsigned int  vers;     /* version needed to extract       2 bytes  */
  unsigned int  flag;     /* general purpose bit flag        2 bytes  */
  unsigned int  comp;     /* compression method              2 bytes  */
  unsigned int  mtim;     /* last mod file time              2 bytes  */
  unsigned int  mdat;     /* last mod file date              2 bytes  */
  unsigned long crc3;     /* crc-32                          4 bytes  */
  unsigned long csiz;     /* compressed size                 4 bytes  */
  unsigned long usiz;     /* uncompressed size               4 bytes  */
  unsigned int  flen;     /* filename length                 2 bytes  */
  unsigned int  elen;     /* extra field length              2 bytes  */
  
  FILE *inp, *out;
  
  int inplen;

  void *inflatestate;

  /* check number of arguments */
  if (argc != 3)
  {
    fprintf(stderr, "Usage: dezip input.zip output.dat\n");
    return 1;
  }

  /* open the input file */
  inp = fopen(argv[1], "rb");

  if (!inp)
  {
    fprintf(stderr, "Error return from fopen(%s)\n", argv[1]);
    return 1;
  }

  /* read and decode the header */
  if ((inplen = fread(buff, 1, (size_t) BUFFSIZE, inp)) < 30)
  {
    fprintf(stderr, "Error reading zip header\n");
    return 1;
  }
  
  GETUINT4(buff+ 0, sign);
  GETUINT2(buff+ 4, vers);
  GETUINT2(buff+ 6, flag);
  GETUINT2(buff+ 8, comp);
  GETUINT2(buff+10, mtim);
  GETUINT2(buff+12, mdat);
  GETUINT4(buff+14, crc3);
  GETUINT4(buff+18, csiz);
  GETUINT4(buff+22, usiz);
  GETUINT2(buff+26, flen);
  GETUINT2(buff+28, elen);
  
  fprintf(stderr, "local file header signature  hex %8lx\n", sign);
  fprintf(stderr, "version needed to extract        %8d\n" , vers);
  fprintf(stderr, "general purpose bit flag     hex %8x\n" , flag);
  fprintf(stderr, "compression method               %8d\n" , comp);
  fprintf(stderr, "last mod file time               %8d\n" , mtim);
  fprintf(stderr, "last mod file date               %8d\n" , mdat);
  fprintf(stderr, "crc-32                       hex %8lx\n", crc3);
  fprintf(stderr, "compressed size                  %8ld\n", csiz);
  fprintf(stderr, "uncompressed size                %8ld\n", usiz);
  fprintf(stderr, "filename length                  %8d\n" , flen);
  fprintf(stderr, "extra field length               %8d\n" , elen);

  /* Check the signature on the .zip file */
  if (sign != 0x04034b50L)
  {
    fprintf(stderr, "Invalid .zip header\n");
    return 1;
  }

  /* Check that the file isn't encrypted */
  if (flag & 1)
  {
    fprintf(stderr, "File is encrypted\n");
    return 1;
  }

  /* Check that the compression type is 8 (deflate) */
  if (comp != 8)
  {
    fprintf(stderr, "File is not compressed with deflate\n");
    return 1;
  }

  /* Check that there's data in the compressed area */
  if (inplen - 30 - flen - elen <= 0)
  {
    fprintf(stderr, "File is too short\n");
    return 1;
  }

  /* open the output file */
  out = fopen(argv[2], "wb");

  if (!out)
  {
    fprintf(stderr, "Error return from fopen(%s)\n", argv[2]);
    return 1;
  }

  /* Initialize the crc */
  crc = 0xffffffffL;

  /* Initialize the decompression routines */
  inflatestate = InflateInitialize(
		   (void *) out,
		   inflate_putbuffer,
		   inflate_malloc,
		   inflate_free
		 );
  
  if (!inflatestate)
  {
    fprintf(stderr, "Error return from InflateInitialize\n");
    return 1;
  }

#ifdef BENCHMARK  
  t1 = clock();
  total = 0;
#endif

  /* Feed the bytes into the decompressor */
  if (InflatePutBuffer(inflatestate, buff  +30+flen+elen, 
				     inplen-30-flen-elen))
  {
    fprintf(stderr, "Error return from InflatePutBuffer\n");
    return 1;
  }
  csiz -= inplen-30-flen-elen;

  while (csiz > 0)
  {
    inplen = fread(buff, 1, (size_t) ((csiz > BUFFSIZE) ? BUFFSIZE : csiz), inp);
    if (inplen == 0)
    {
      fprintf(stderr, "Error return from fread\n");
      return 1;
    }
    
    csiz -= inplen;

    if (InflatePutBuffer(inflatestate, buff, inplen))
    {
      fprintf(stderr, "Error return from InflatePutBuffer\n");
      return 1;
    }
  }

  if (InflateTerminate(inflatestate))
  {
    fprintf(stderr, "Error return from InflateTerminate\n");
    return 1;
  }

  /* Check that the CRC is OK */
  if (crc != (crc3 ^ 0xffffffffL))
  {
    fprintf(stderr, "Error - uncompressed data has bad CRC\n");
    return 1;
  }

#ifdef BENCHMARK
  t2 = clock();
  printf("Transfer rate %ld bytes per second\n",total*((long) CLK_TCK)/(t2-t1));
#endif

  return 0;
}
