/* ziplib.h -- interface of the zip compression library
 * Copyright (C) 1995 Jean-loup Gailly.
 * For conditions of distribution and use, see the accompanying README file.
 * version 0.2  March 2nd, 1995.
 */

#ifndef _ZIPLIB_H
#define _ZIPLIB_H

#include "zipconf.h"

/* 
     The zip compression library provides in-memory compression and
  decompression functions, including integrity checks of the uncompressed
  data.  This version of the library supports only one compression method
  (deflation) but other algorithms may be added later and will have the same
  stream interface.

     For both compression and decompression, the application can provide
  the input and output buffers. Compression can be done in a single step
  if the buffers are large enough (for example if an input file is mmap'ed),
  or can be done by repeated calls of the compression function.
  In the latter case, the application must provide more input and/or
  consume the output (providing more output space) before each call.
*/

typedef struct zip_stream {
    Byte     *next_in;  /* next input byte */
    uInt;    avail_in;  /* number of bytes available at next_in */
    uLong    total_in;  /* total nb of input bytes read so far */

    Byte     *next_out; /* next output byte should be put there */
    uInt     avail_out; /* remaining free space at next_out */
    uLong    total_out; /* total nb of input bytes output so far */

    char     *msg;      /* last error message, NULL if no error */
    voidp     state;    /* internal state, not visible by clients */

    voidp    (*zalloc)__P((uInt)); /* used to allocate the internal state */
    /* If uInt == 16 bits, zalloc(0) must allocate 65536 bytes */

    void     (*zfree)__P((voidp)); /* used to free the internal state */

} zip_stream;

/* The application must update next_in and avail_in when avail_in
   has dropped to zero. It must update next_out and avail_out when
   avail_out has dropped to zero. The application must initialize
   zalloc and zfree before calling the init function. All other fields
   are set by the compression library and must not be updated by the
   application.

   The fields total_in and total_out can be used for statistics or
   progress reports. After compression, total_in holds the total size of
   the uncompressed data and may be saved for use in the decompressor
   (particularly if the decompressor wants to decompress everything in
   a single step).
 */

			/* constants */

#define NO_FLUSH      0
#define PARTIAL_FLUSH 1
#define BLOCK_FLUSH   2
#define FULL_FLUSH    3
/* See deflate() below for the usage of these constants */

#define STREAM_OK     0
#define STREAM_END    1
#define STREAM_ERROR (-1)
#define DATA_ERROR   (-2)
#define MEM_ERROR    (-3)
#define BUF_ERROR    (-4)
/* error codes for the compression/decompression functions */

#define BEST_SPEED             1
#define BEST_COMPRESSION       9
#define DEFAULT_COMPRESSION  (-1)
/* compression levels */

			/* basic functions */

extern int deflateInit __P((zip_stream *strm, int level));
/* 
     Initializes the internal stream state for compression. The fields
   next_in, zalloc and zfree must be initialized before by the caller.

     If next_in is not null, the library will use this buffer for the
   sliding window; the buffer must either hold the entire input data,
   or have at least WSIZE bytes and be writable.  If next_in is null,
   the library will allocate its own buffer (and leave next_in null).

     If the sliding window is provided by the application, next_in must
   must never be changed by the application since the compressor maintains
   history information inside this buffer from call to call; the application
   must provide more input only by increasing avail_in. next_in is always
   reset by the library in this case.

     The compression level must be -1, or between 1 and 9: 1 gives best speed,
   9 gives best compression. A level of -1 requests a default compromise
   between speed and compression  (currently equivalent to level 6).

     deflateInit returns STREAM_OK if success, MEM_ERROR if there was not
   enough memory, STREAM_ERROR if the stream state was inconsistent (such
   as zalloc being NULL). msg is set to null if there is no error message.
   deflateInit does not perform any compression: this will be done by
   deflate() or deflateEnd().
*/


extern int deflateEnd __P((zip_stream *strm));
/*
     Terminates processing of all pending input and flush all pending output.
   All dynamically allocated data structures are freed.

     deflateEnd can be called immediately after deflateInit if all
   the compression is to be done in a single step. In this case, avail_out
   must be at least 0.1% larger than avail_in plus 8 bytes.

     deflateEnd returns STREAM_OK if success, BUF_ERROR if there was not
   enough room in the output buffer, STREAM_ERROR if the stream state was
   inconsistent. In the error case, msg may be set but then points to a
   static string (which must not be deallocated).
*/


extern int deflate __P((zip_stream *strm, int flush));
/*
  Performs one or both of the following actions:

  - Compress more input starting at next_in and update next_in and avail_in
    accordingly. If not all input can be processed (because there is not
    enough room in the output buffer), next_in is updated and processing
    will resume at this point for the next call of deflate().

  - Provide more output starting at next_out and update next_out and avail_out
    accordingly. This action is forced if the parameter flush is non zero.
    Forcing flush frequently degrades the compression ratio, so this parameter
    should be set only when necessary (in interactive applications).
    Some output may be provided even if flush is not set.

  Before the call of deflate(), the application should ensure that at least
  one of the actions is possible, by providing more input and/or consuming
  more output, and updating avail_in or avail_out accordingly.
  The application can consume the compressed output when the output
  buffer is full (avail_out == 0), or after each call of deflate().

    If the parameter flush is set to PARTIAL_FLUSH, the current compression
  block is byte aligned and flushed to the output buffer so that the
  decompressor can get all input data available so far. If flush is set to
  BLOCK_FLUSH, the current block is terminated; this is option is provided
  only for compatibility with old decompressors which do not accept partial
  block flushes. If flush is set to FULL_FLUSH, the compression block is
  terminated, a special marker is output and the compression dictionary is
  discarded; this is useful to allow the decompressor to synchronize if one
  compressed block has been damaged. Using FULL_FLUSH too often can seriously
  degrade the compression.

    deflate() return STREAM_OK if some progress has been made (more input
  processed or more output produced), BUF_ERROR if no progress is
  possible and STREAM_ERROR if the stream state was inconsistent.
*/


extern int inflateInit __P((zip_stream *strm));
/* 
     Initializes the internal stream state for decompression. The fields
   next_out, zalloc and zfree must be initialized before by the caller.

     If next_out is not null, the library will use this buffer for the
   sliding window; the buffer must either be large enough to hold the
   entire output data, or have at least WSIZE bytes and be writable.
   If next_out is null, the library will allocate its own buffer
   (and leave next_out null).

   If the sliding window is provided by the application, next_out must
   must never be changed by the application since the decompressor maintains
   history information inside this buffer from call to call; the application
   can only reset next_out to the beginning of the window when avail_out
   is zero and all output has been consumed.

     inflateInit returns STREAM_OK if success, MEM_ERROR if there was not
   enough memory, STREAM_ERROR if the stream state was inconsistent (such
   as zalloc being NULL). msg is set to null if there is no error message.
   inflateInit does not perform any decompression: this will be done by
   inflate() or inflateEnd().
*/


extern int inflateEnd __P((zip_stream *strm);
/*
     Terminates processing of all pending input and flush all pending
   output.  All dynamically allocated data structures are then freed.

     inflateEnd can be called immediately after inflateInit if all
   the decompression is to be done in a single step. In this case, avail_out
   must be large enough to hold all the uncompressed data. (The size of the
   uncompressed data may have been saved by the compressor for this purpose.)

     inflateEnd returns STREAM_OK if success, DATA_ERROR if the input data was
   corrupted, BUF_ERROR if there was not enough room in the output buffer,
   STREAM_ERROR if the stream structure was inconsistent. (In the
   DATA_ERROR case, inflateEnd does not provide the boundaries between the
   recovered portions. You can use inflate() and inflateSync() to get them
   and use inflateEnd() only to deallocate all data structures.)  In the
   error cases, msg may be set but then points to a static string (which
   must not be deallocated).
 */

extern int inflate __P((zip_stream *strm));
/*
  Performs one or both of the following actions:

  - Decompress more input starting at next_in and update next_in and avail_in
    accordingly. If not all input can be processed (because there is not
    enough room in the output buffer), next_in is updated and processing
    will resume at this point for the next call of inflate().

  - Provide more output starting at next_out and update next_out and avail_out
    accordingly.  inflate() always provides as much output as possible
    (until no more input data or no more space in the output buffer).

  Before the call of inflate(), the application should ensure that at least
  one of the actions is possible, by providing more input and/or consuming
  more output, and updating the next_* and avail_* values accordingly.
  The application can consume the uncompressed output when the output
  buffer is full (avail_out == 0), or after each call of inflate().

    inflate() return STREAM_OK if some progress has been made (more
  input processed or more output produced), STREAM_END if the end of
  the compressed data has been reached, BUF_ERROR if no progress is
  possible, DATA_ERROR if the input data was corrupted, STREAM_ERROR
  if the stream structure was inconsistent.  In the DATA_ERROR case,
  the application may then call inflateSync to look for a good
  compression block.
*/

                        /* advanced functions */

/*
    The following functions are needed only in some special applications.
*/

extern int inflateSync __P((zip_stream *strm));
/* 
    Skips invalid compressed data until the special marker and a valid block
  can be found, or until all available input is skipped. No output is provided.

    inflateSync returns STREAM_OK if a valid block has been found, BUF_ERROR
  if no more input was provided, DATA_ERROR if not valid block has been found,
  STREAM_ERROR if the stream structure was inconsistent. In the success
  case, the application may save the current current value of total_in
  which indicates where valid compressed data was found. In the error case,
  the application may repeatedly call inflateSync, providing more input each
  time, until success or end of the input data.
 */

extern int deflateOptions __P((zip_stream *strm,
                               int  windowBits,
                               int  memLevel,
                               int  filter));
/*   
     Sets some special compression options. This function must be called
   before deflateInit.
     The windowBits parameter is the base two logarithm of the window
   size; it should be in the range 9..15 for this version of the library
   (the value 16 will be allowed soon). Larger values of this parameter
   result in better compression at the expense of memory usage. The default
   value is 15 if deflateOptions is not called.
    The memLevel parameter specifies how much memory should be allocated
   for the internal compression state. memLevel=1 uses minimum memory but
   is slow and reduces compression ratio; memLevel=9 uses maximum memory
   for optimal speed. The default value is 8.
     The filter parameter should be set to 0 for normal data, non-zero for
   data produced by a filter (or predictor). Such data consists mostly of
   small values with a somewhat random distribution. In this case, the
   compression algorithm is tuned to compress them better. This parameter
   only affects the compression ratio but not the correctness of the
   compressed output even if it is not set appropriately. The default value
   is zero. [Note: this filter parameter may go away if I find a cheap
   heuristic allowing its determination from the input data itself.]

      deflateOptions returns STREAM_OK if success, MEM_ERROR if there was not
   enough memory, STREAM_ERROR if the stream state was inconsistent (such
   as zalloc being NULL). msg is set to null if there is no error message.
   deflateOptions does not perform any compression: this will be done by
   deflate() or deflateEnd().
*/
                            
extern int deflateCopy __P((zip_stream *dest,
                            zip_stream *source));
/*
     Sets the destination stream as a complete copy of the source stream.
   This function is useful when several compression strategies will
   be tried, for example when there are several ways of pre-processing
   the input data with a filter. The streams that will be discarded
   should then be freed by setting avail_out to zero, calling deflateEnd
   and ignoring the result code. Note that deflateCopy duplicates the
   internal compression state which can be quite large, so this strategy
   is slow and can consume lots of memory.
      deflateCopy returns STREAM_OK if success, MEM_ERROR if there was not
   enough memory, STREAM_ERROR if the source stream state was inconsistent
   (such as zalloc being NULL). msg is left unchanged in both source and
   destination.
*/

                        /* utility functions */

/*
     The following utility functions are implemented on top of the
   basic stream-oriented functions. To simplify the interface, some
   default options are assumed (compression level, window size,
   standard memory allocation functions). The source code of these
   utility functions can easily be modified if you need special options.
*/

extern int zip __P((Byte *dest,   uLong *destLen,
                    Byte *source, uLong sourceLen));
/*
     Compresses the source buffer into the destination buffer.  sourceLen is
   the byte length of the source buffer. Upon entry, destLen is the total
   size of the destination buffer, which must be at least 0.1% larger than
   sourceLen plus 8 bytes. Upon exit, destLen is the actual size of the
   compressed buffer.
     This function can be used to compress a whole file at once if the
   input file is mmap'ed.
     zip returns STREAM_OK if success, MEM_ERROR if there was not enough
   memory, BUF_ERROR if there was not enough room in the output buffer.
*/

extern int unzip __P((Byte *dest,   uLong *destLen,
                      Byte *source, uLong sourceLen));
/*
     Decompresses the source buffer into the destination buffer.  sourceLen is
   the byte length of the source buffer. Upon entry, destLen is the total
   size of the destination buffer, which must be large enough to hold the
   entire uncompressed data. (The size of the uncompressed data must have
   been saved previously by the compressor and transmitted to the decompressor
   by some mechanism outside the scope of this compression library.)
   Upon exit, destLen is the actual size of the compressed buffer.
     This function can be used to compress a whole file at once if the
   input file is mmap'ed.

     unzip returns STREAM_OK if success, MEM_ERROR if there was not enough
   memory, BUF_ERROR if there was not enough room in the output buffer, or
   DATA_ERROR if the input data was corrupted.
*/


typedef voidp gzFile;

extern gzFile gzopen  __P((char *path, char *mode));
/*
     Opens a gzip (.gz) file for reading or writing. The mode parameter
   is as in fopen ("rb" or "wb").
     gzopen return NULL if the file could not be opened or if there was
   insufficient memory to allocate the (de)compression state.
*/

extern int    gzread  __P((gzFile file, voidp buf, unsigned len));
/*
     Reads the given number of uncompressed bytes from the compressed file.
   gzread returns the number of bytes actually read (0 for end of file).
*/

extern int    gzwrite __P((gzFile file, voidp buf, unsigned len));
/*
     Writes the given number of uncompressed bytes into the compressed file.
   gzwrite returns the number of bytes actually written (0 in case of error).
*/

extern int    gzflush __P((gzFile file));
/*
     Flushes all pending output into the compressed file. This should be
   done only when strictly necessary because it can degrade compression.
*/

extern int    gzclose __P((gzFile file));
/*
     Flushes all pending output if necessary, closes the compressed file
   and deallocates all the (de)compression state
*/

#endif /* _ZIPLIB_H */
