#include "common.h"
#include "date.h"

//
//	Note: the internal representation counts the days in a year from March 1st 1200
//

//
//	Length of 1,4,100 and 400 years, in days
//
const DATE LY_1 = 365;
const DATE LY_4 = 4 * LY_1 + 1;
const DATE LY_100 = 25 * LY_4 - 1;
const DATE LY_400 = 4 * LY_100 + 1;

//
// constant set to make a Sunday or Monday divisible by 7, 1 March 1200 would have been a Wednesday
//

const USHORT EPOCH_YEAR = 1200;	// must be divisible by 400.
const DATE DOFFSET = WEDNESDAY; 

const USHORT FIRST_YEAR = 1582; // introduction of Gregorian calendar by Conc. of Nicae.
const USHORT LAST_YEAR = 9366;	// to prevent overflow in TIMEDATE


//
//	Lookup table: length of a month
//
static const USHORT monthlen [12] = 
{
	31, 30, 31, 30, 31, 31, 30, 31, 30, 31, 31, 29
};

//
//	number of days before a month
//
static const USHORT daysbeforemonth[13] = 
{
	0,31,61,92,122,153,184,214,245,275,306,337,366
}; 

//
//	lookup table from day to month
//
static const unsigned char day2month[366] =
{
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
	5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,
	10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,
		10,10,10,10,10,10,10,10,10,10,
	11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,
		11,11,11,11,11,11,11,11
};

//
//	Leap year test
//
inline BOOL IsLeapYear (USHORT year)
{
	if(year % 100 != 0)
		return (year % 4 == 0);
	else
		return (year % 400 == 0);
}

//
//	The Gregorian calendar was introduced in 1582
//
BOOL SetDate(DATE &d, USHORT year, USHORT month, USHORT day)
{
	if(year < FIRST_YEAR || year > LAST_YEAR)	// 18 March 28277 is the last representable day
		return FALSE;

	year -= EPOCH_YEAR;

	if(month < 1 || month > 12)
		return FALSE;

	//	switch to year starting in March
	if(month >= 3){
		month -= 3;
	}else{
		year--;
		month += 9;
	};

	//	check whether day is within month
	if(day < 1 || day > monthlen[month])
		return FALSE;

	// very special case: the leap day
	if(month == 11 && day >= 29 && !IsLeapYear(year+1))
		return FALSE; 

	// calculate day number
	d = DATE(365) * year + (year / 400) + (year / 4) - (year / 100) + daysbeforemonth[month] + 
			(day-1) + DOFFSET;

	return TRUE;
}

BOOL GetDate(DATE d, USHORT &year, USHORT &month, USHORT &day)
{
	//
	//	adjust for March 1st, 0, being on a Wednesday
	//
	if(d < DOFFSET)
		return FALSE;

	d -= DOFFSET;

	//	subtract 400-year periods
	year = 400 * (USHORT)(d / LY_400);
	d %= LY_400;

	//	check for leap day once every 400 years
	if(d == LY_400-1){
		year += (EPOCH_YEAR + 400);
		month = 2;
		day = 29;
		return TRUE;
	};

	//	subtract centuries
	year += 100 * (USHORT)(d / LY_100);
	d %= LY_100;

	//	subtract periods of four years
	year += 4 * (USHORT)(d / LY_4);
	d %= LY_4;

	// check for normal leap day
	if(d == LY_4-1){
		year += (EPOCH_YEAR + 4);
		month = 2;
		day = 29;
		return TRUE;
	};

	// subtract all full years
	year += (USHORT)(d / LY_1);
	d %= LY_1;

	//	look up month
	unsigned i = day2month[d];

	//	adjust month and year to external representation
	month = i+3;
	if(month > 12){
		year++;
		month -= 12;
	};

	year += EPOCH_YEAR;

	// calculate day
	d -= daysbeforemonth[i];
	day = (USHORT)(d+1);

	return TRUE;
}

//
//	Calculate date for first 'wday' _after_ today.
//
inline DATE NextWeekDay (DATE d, USHORT wday)
{
	return (d + 7 - wday)/ 7 * 7 + wday;
}


//
//	Lookup table for Metonic cycle, the year mod 19 determines the date of the
//	paschal full moon.
//

// M converts dates in offsets from the first of March

#define	M(mo, da)	(USHORT)(31 * (mo - 3) + (da - 1))

static const USHORT m_cycle[19] = {
   M(4, 14), M(4, 3),  M(3, 23), M(4, 11), M(3, 31), M(4, 18),
   M(4, 8),  M(3, 28), M(4, 16), M(4,  5), M(3, 25), M(4, 13),
   M(4, 2),  M(3, 22), M(4, 10), M(3, 30), M(4, 17), M(4, 7),
   M(3, 27)
};

#undef M

//
//	Calculate date of easter, the first Sunday after the paschal full moon.
//
BOOL CalcEaster(USHORT year, DATE &d)
{
	DATE firstofmarch;
	if(!SetDate(firstofmarch, year, 3, 1))
		return FALSE;
	d = NextWeekDay(firstofmarch + m_cycle[year % 19], SUNDAY);
	return TRUE;
}

//
//	TIMEDATE routines
//

// 	some rather obvious constants
const TIMEDATE MINUTESPERHOUR = 60;
const TIMEDATE HOURSPERDAY = 24;
const TIMEDATE MINUTESPERDAY = HOURSPERDAY * MINUTESPERHOUR;

//
//	Convert DATE and time into TIMEDATE
//
BOOL SetTimeDate(TIMEDATE &d, DATE date, USHORT hh, USHORT mmm)
{
	if(hh >= HOURSPERDAY || mmm >= MINUTESPERHOUR)
		return FALSE;
	d = mmm + MINUTESPERHOUR * (hh + HOURSPERDAY * (TIMEDATE)date);
	return TRUE;
}

//
//	Convert a date and time into TIMEDATE
//
BOOL SetTimeDate(TIMEDATE &d, USHORT yy, USHORT mm, USHORT dd, USHORT hh, USHORT mmm)
{
	DATE date;
	if(!SetDate(date, yy, mm, dd))
		return FALSE;
	return SetTimeDate(d, date, hh, mm);
}

//
//	Extract DATE and time from a TIMEDATE
//
BOOL GetTimeDate(TIMEDATE d, DATE &date, USHORT &hh, USHORT &mmm)
{
	date = d / MINUTESPERDAY;
	USHORT hhmm = (USHORT)(d % MINUTESPERDAY);
	hh = hhmm / (USHORT)MINUTESPERHOUR;
	mmm = hhmm % (USHORT)MINUTESPERHOUR;
	return TRUE;
}

//
//	Extract date and time from TIMEDATE
//
BOOL GetTimeDate(TIMEDATE d, USHORT &yy, USHORT &mm, USHORT &dd, USHORT &hh, USHORT &mmm)
{
	DATE date;
	return (GetTimeDate(d, date, hh, mm) && GetDate(date, yy, mm, dd));
}


//
//	Get the current date (needs system-dependent GetCurrentTimeDate function)
//
DATE GetCurrentDate ()
{
	DATE date;
	USHORT hh, mm;
	GetTimeDate(GetCurrentTimeDate(), date, hh, mm);
	return date;
}

//
//	Week #1 is the first week of the year that contains a working day, i.e. Jan. 2 is a Mon-Fri.
//
BOOL CalcWeekNumber (DATE date, USHORT &weeknum)
{
	USHORT yy, mm, dd;

	// use last day of this week (Sat/Sun, depending you're using US or ISO calendar)
	date += (6 - DayOfWeek(date));

	// we need the current year
	if(!GetDate(date, yy, mm, dd))
		return FALSE;

	for(;;){
		// figure out what day Jan. 2 is.
		DATE d;
		if(!SetDate(d, yy, 1, 2))
			return FALSE;
		USHORT dow = DayOfWeek(d);

		// Calculate the date of the start of week 1
		switch(dow){
		case MONDAY: case TUESDAY: case WEDNESDAY: case THURSDAY: case FRIDAY:
			d -= dow;
			break;
		default:
			d += (7 - dow);
		};

		if(date >= d){
			weeknum = (USHORT)((date - d)/7 + 1);
			return TRUE;
		};

		// we get here if the date _precedes_ the start of week 1, i.e. it's week 52/53 of the previous year
		yy--;
	}
}
