/*\t*******************************************************************/
/*    Creation Date .......  Fri  06-04-1993  07:37:52                */
/*    Filename  ...........  fiufile.cpp                              */
/*    Project .............  File Utilities                           */
/*    Author  .............  Matthew J. W. Ratcliff                   */
/*    Language  ...........  C++                                      */
/*    Operating System  ...  DOS/Windows                              */
/*    Processor  ..........  FIU - File Utilities                     */
/*    Function: Support code for file related operations.             */
/*                                                                    */
/*\t*******************************************************************/

/*\r********************************************************************
**                         Revision History
***********************************************************************/
/*

   Date    By           Change Description
dd-mmm-yy  nnn          text
---------  ----         -----------------------------------------------
04-JUN-93  MJWR         Build file support functions.
30-JUN-92  MJWR         Resolve differences between Borland & MSC compilers
14-Sep-93  MJWR         Add FIUfopen, FIUfclose, and FIUnumFilesOpened.
                        This will allow us to check if any files are
                        left open before exiting a program - debugging
                        tool.
21-Oct-93  MJWR         Allow user to get to file date, size, and
                        time with one procedure call.  Date and
                        Time are returned as INT32 values, not decoded.
                        This format is useful for checking if a file
                        has changed since the last time you looked at
                        it.

25-Mar-94  MJWR         Add BOOLEAN FIUisFloppyFile( const char *pasFileName );
06-Jun-94  MJWR         Add FIUlocateFile - find a file in the current
                        working directory, or in the default path directory,
                        or in any of the search path's subdirectories.  If
                        it's out there, I'll find it!
07-Jul-94  MJWR         Return BOOLEAN not int for "FIUxxx" test functions.
                        Also modify fiuFindFirst to accept a parameter
                        allowing it to search for any file type, not just
                        file names.  Used by FIUpathExists function.
**\r*/

/*\i********************************************************************
**                       Module Include Files
***********************************************************************/

/*********************** System Include Files *************************/
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#include <direct.h>
#ifdef _MSC_VER
#pragma message("Compiling for Microsoft C")
#else              
#include <dir.h>
#pragma message("Compiling for Borland C")
#endif

/********************** Constant Include Files ************************/
#include "liidSys.h"
#include "liuStrg.h"
#include "erapdefs.h"
#include "figfile.h"
#include "fipfile.h"

/***************** External Variable Include Files ********************/


/***************** External Procedure Include Files *******************/
#include "eraprpt.h"
#include "fiufile.h"

/*\i*/

/*\m********************************************************************
**                       Module Declarations
***********************************************************************/

/************************* Module Constants ***************************/

/************************* Module Variables ***************************/

static FIG_DOS_FBLK_TY fimFblk;
static int             fimFilesOpened = 0;

/************************* Module Procedures **************************/

// in FIUFIND.CPP

/*\m*/

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUcreateUniqueFile

    PURPOSE:

        A unique file name is created with 1 - 5
        character prefix defined in 'pas1_5CharPrefix'.
        If null or zero length, the prefix is set to
        '$'. If 'pasAltPathOrNull' is NULL, no
        alternate path is given for a temporary file.
        If not null, and the system variables TMP and
        TEMP are NOT defined, then the temporary file
        is created at 'pasAltPathOrNull'.  If all else
        fails the temporary file is created in the
        current working directory. Pass in the desired
        wrige mode string in 'pasWriteMode'.  It should
        be the same as the last parameter passed to the
        normal 'fopen' function.  The new unique file
        name, with full drive and path specifiers, is
        copied to the string 'refNewUniqueName' of
        'pasCbNameLen' bytesz or less.  The opened file
        handle is returned. If we fail to open the file
        a NULL is returned. The alternate path should
        NOT be a network drive. This function is
        suitable for DOS or Windows, because Windows is
        non-preemptive multi-tasking, and we don't
        relinquish control to Windows until after the
        file name is created and the file opened.

        If the entire file name cannot fit in the destination
        string, a NULL is returned and no attempt is made to
        create the file.  (You must get the correct path and
        name to the file or you can't delete it - so we won't
        create it on such an error condition.)

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filFilePtr         FNC    FILE *, open file handle or NULL on err**
**\p*******************************************************************/

FILE *FIUcreateUniqueFile( const char *pas1_5CharPrefix,
                           const char *pasAltPathOrNull,
                           const char *pasWriteMode, // "wb" or "w" for example
                           int         pasCbNameLen,
                           char       *refNewUniqueName )

{ /* FIUcreateUniqueFile procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
CHARPTR FIL_PROC_NAME = "FIUcreateUniqueFile";

#define FIL_NOT_FOUND -1
#define FIL_MAX_TRIES 1000

/******************* Local Variable Declarations **********************/
FILE                   *filFilePtr;

char                   *filPrefix;
char                   *filPath;
char                   *filFilePath;

int                     filInx;
BOOLEAN                 filExists;

/************************* Procedure Body *****************************/
filPrefix       = new char[FIP_MAX_PREFIX_LEN+4];
filPath         = new char[FIP_MAX_NAME_LEN];
filFilePath     = new char[2*FIP_MAX_NAME_LEN];

filFilePtr      = NULL;

if ((pasCbNameLen < FIP_MIN_NAME_LEN)   ||
    (refNewUniqueName == NULL)          ||
    (pasWriteMode     == NULL))
  {
  ERAparameterError( FIL_PROC_NAME );
  }
else
  {
  *refNewUniqueName = '\0';
  *filPrefix = '\0';
  if (pas1_5CharPrefix)
    {
    LIUstrncpy(filPrefix, (char *)pas1_5CharPrefix, FIP_MAX_PREFIX_LEN);
    }
  if (!filPrefix[0])
    {
    strcpy(filPrefix, FIP_DEFAULT_PREFIX );
    }
  strcpy(filPath,"C:\\TMP");
  if (access(filPath,0) == FIL_NOT_FOUND)
    {
    strcpy(filPath,"C:\\TEMP");
    if (access(filPath,0) == FIL_NOT_FOUND)
      {
      if (pasAltPathOrNull)
        {
        LIUstrncpy(filPath,pasAltPathOrNull,FIP_MAX_NAME_LEN);
        }
      else
        {
        if (mkdir(filPath) == FIL_NOT_FOUND)
          {
          strcpy(filPath,"C:");
          if (access(filPath,0) == FIL_NOT_FOUND)
            {
            *filPath = '\0'; // forget it!
            }
          }
        }
      }
    }

  LIUstrncpy(filPrefix, pas1_5CharPrefix, FIP_MAX_PREFIX_LEN);
  filInx = 0;
  do
    {
    sprintf(filFilePath,"%s\\%s%03d.$$$", filPath, filPrefix, filInx );
    filExists = FIUfileExists( filFilePath );
    if (filExists) filInx++;
    }
  while ((filInx < FIL_MAX_TRIES) && (filExists));
  if (filExists)
    {
    ERAerrorLog( FIL_PROC_NAME,
      "Failed to make unique file in path '%s' for prefix '%s'",
      filPath, filPrefix );
    }
  else
    {
    LIUstrncpy( refNewUniqueName, filFilePath, pasCbNameLen );
    if ((int)strlen(filFilePath) > pasCbNameLen)
      {
      ERAerrorLog( FIL_PROC_NAME,
        "Destination string %d bytes, too short for '%s'",
        pasCbNameLen, filFilePath );
      }
    else
      {
      filFilePtr = FIUfopen( refNewUniqueName, pasWriteMode );
      if (filFilePtr == NULL)
        {
        ERAerrorLog(FIL_PROC_NAME,
          "Failed to open file '%s' in write mode '%s'",
          refNewUniqueName, pasWriteMode );
        }
      }
    }
  }

LIMDELA(filPath);
LIMDELA(filFilePath);
LIMDELA(filPrefix);

return(filFilePtr);
} /* FIUcreateUniqueFile end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUsetExtender

    PURPOSE:  to set the file extender.  If extender pointer is
      NULL, set file extender to none.  Else, delete
      existing extension from file and add the one passed in.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE FIUsetExtender
( const char    *pasExtender,
  char          *refFilePath )

{ /* FIUsetExtender procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
CHARPTR FIL_PROC_NAME = "FIUsetExtender";

#define FIL_EXT_LEN 3

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               filErr;
char                   *filExtPtr;

/************************* Procedure Body *****************************/

filErr = FAILED_;

if (refFilePath == NULL)
  {
  ERAerrorLog( FIL_PROC_NAME,
    "Invalid parameter");
  }
else
  {
  /****************************************
  * Find the last 'dot' in the file
  * name.  For example, the path could
  * be:  A:\PATH.EXT\FILE.EXT
  *
  * That is, directory names can have extenders,
  * but we want to modify the extension of
  * the file only.
  */
  filExtPtr = LIUstrchrLast(refFilePath, '.');
  filErr = SUCCEEDED_;
  if (filExtPtr)
    {
    *filExtPtr = '\0';
    }
  else
    {
    filExtPtr = refFilePath;
    filExtPtr += strlen(refFilePath);
    }
  if (pasExtender)
    {
    if (*pasExtender == '.')
      {
      pasExtender++; // User gave us ".EXT", not "EXT"
      }
    *filExtPtr = '.';
    filExtPtr++;
    strncpy(filExtPtr, pasExtender, FIL_EXT_LEN );
    filExtPtr += FIL_EXT_LEN;
    *filExtPtr = '\0';
    }
  }
return(filErr);
} /* FIUsetExtender end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUextractFname

    PURPOSE:  to extract file name only from full path/file specification.
      The file will contain filename.ext only.


**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUextractFname( const char *pasFilePath,
                          int         pasDestLen,
                          char       *refFileOnly )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
char                   *filDs;
/************************* Procedure Body *****************************/

if ((pasFilePath == NULL) ||
    (pasDestLen <= 0)     ||
    (refFileOnly == NULL))
  {
  filErr = FAILED_;
  }
else
  {
  filDs = LIUstrchrLast( pasFilePath, FIP_PATH_SEP );
  if (filDs)
    {
    filDs++;
    }
  else
    {
    filDs = strchr(pasFilePath, FIP_DRIVE_SEP );
    if (filDs) filDs++;
    }
  if (!filDs)
    {
    filDs = (char *)pasFilePath;
    }
  LIUstrncpy( refFileOnly, (char *)filDs, pasDestLen );
  }
return(filErr);
} /* FIUextractFname end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUextractExt

    PURPOSE:  to extract file name extension from the filename.


**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUextractExt ( const char *pasFilePath,
                          int         pasDestLen,
                          char       *refFileExt,
                          BOOLEAN     pasMakeWcSpec )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
char                   *filDs;
char                   *filWcs;
/************************* Procedure Body *****************************/

if ((pasFilePath == NULL) ||
    (pasDestLen  <= 0)    ||
    (refFileExt  == NULL))
  {
  filErr = FAILED_;
  }
else
  {
  filDs = LIUstrchrLast( pasFilePath, '.');
  if (pasMakeWcSpec)
    {
    filWcs = new char[40];
    if (filDs)
      {
      sprintf(filWcs,"*%s", filDs);
      LIUstrncpy( refFileExt, filWcs, pasDestLen );
      }
    else
      {
      LIUstrncpy( refFileExt, "*.*", pasDestLen );
      }
    }
  else
    {
    if (filDs)
      {
      filDs++;
      LIUstrncpy( refFileExt, filDs, pasDestLen );
      }
    else
      {
      *refFileExt = '\0';
      }
    }
  }
return(filErr);
} /* FIUextractExt end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUdeleteFile

    PURPOSE:  to delete the specified file.  If file not
      found, return an error.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE FIUdeleteFile( const char *pasFilePath )

{ /* FIUdeleteFile procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
CHARPTR FIL_PROC_NAME = "FIUdeleteFile";

/******************* Local Variable Declarations **********************/
/* error return flag       */
int                     filUnlink;
STAT_TYPE               filErr;
int                     filFileFound;

/************************* Procedure Body *****************************/
/* Initialize return val   */
filErr = FAILED_;

filFileFound = FIUfileExists( pasFilePath );
if (filFileFound)
  {
  filUnlink = unlink( pasFilePath );
  if (filUnlink != 0)
    {
    ERAerrorLog( FIL_PROC_NAME,
      "Failed to delete file '%s'", pasFilePath );
    }
  else
    {
    filErr = SUCCEEDED_;
    }
  }
else
  {
  ERAerrorLog( FIL_PROC_NAME,
    "File '%s' not found, can't delete",
    pasFilePath );
  }

return(filErr);
} /* FIUdeleteFile end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUdeleteWCfiles

    PURPOSE:  to delete one or more files matching file template
      with wild card characters.  WARNING: Use at your own risk.
      If you call FIUdeleteWCfiles( "*.*" ); then all files
      in the current directory (not write protected) will be
      deleted without the usual DOS prompts to be SURE.  Return
      a count of the number of files actually deleted.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filDelCnt          int, Number of files matching template and    **
**                      deleted successfully. 0 on any error          **
**\p*******************************************************************/

int FIUdeleteWCfiles( const char *pasNukemFileSpec )
{
/*********** Local Constant & Variable Declarations *******************/
CHARPTR FIL_PROC_NAME="FIUdeleteWCfiles";
int                     filDelCnt        = 0;
int                     filUnlink;
const char             *filNukeMe;

/************************* Procedure Body *****************************/
filNukeMe = FIUgetFirstFilename( pasNukemFileSpec );

while (filNukeMe)
  {
  filUnlink = unlink( filNukeMe );
  if (filUnlink != 0)
    {
    ERAerrorLog( FIL_PROC_NAME,
      "Failed to delete file '%s'", filNukeMe );
    }
  else
    {
    filDelCnt++;
    }
  filNukeMe = FIUgetNextFilename();
  }

return(filDelCnt);
} /* FIUdeleteWCfiles end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUbuildDefaultFname

    PURPOSE:  User specifies path, a name, and an extension. A desirable
     default path/filename is created.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUbuildDefaultFname
( const char    *pasPath,
  const char    *pasName,
  const char    *pasExt,
  int            pasLen,
  char          *refDefName,
  int            pasDefaultChar )

{
/*********** Local Constant & Variable Declarations *******************/
#define FIL_FNAME_LEN   9
#define FIL_MAXPATH     120

CHAR                   *FIL_PROC_NAME = "FIUbuildDefaultFname";
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
char                    filName[FIL_FNAME_LEN+1];
char                    filPath[FIL_MAXPATH+1];

/************************* Procedure Body *****************************/

if ((pasName == NULL)           ||
    (pasLen  <= 0)              ||
    (refDefName == NULL))
  {
  ERAerrorLog( FIL_PROC_NAME, "Bad param");
  }
else
  {
  LIUstrncpy( filName, (char *)pasName, FIL_FNAME_LEN );
  FIUmakeFnameChars( filName, pasDefaultChar );
  if (pasPath)
    {
    FIUconstructFilePath( pasPath, filName, FIL_MAXPATH, filPath );
    }
  else
    {
    strcpy( filPath, filName );
    }
  if (pasExt)
    {
    FIUsetExtender( pasExt, filPath );
    }
  LIUstrncpy( refDefName, filPath, pasLen );
  }

return(filErr);
} /* FIUbuildDefaultFname end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUmakeFnameChars

    PURPOSE:  to convert all characters which are NOT VALID
      file name characters into pasSubstChar characters, so that
      the string may be used as a filename.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUmakeFnameChars( char *refMessyName,
                             int   pasSubstChar )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
char                   *filCptr;
int                     filCh;

/************************* Procedure Body *****************************/

if ((refMessyName == NULL)  ||
    (*refMessyName == '\0') ||
    (pasSubstChar <= ' ')   ||
    (pasSubstChar > 'z'))
  {
  filErr = FAILED_;
  }
else
  {
  filCptr = refMessyName;
  while (*filCptr)
    {
    filCh = *filCptr;
    if ((filCh < '0')  ||
        (filCh > 'Z'))
      {
      if ((filCh != '$') &&
          (filCh != '@'))
        {
        filCh = pasSubstChar;
        }
      }
    *filCptr = filCh;
    filCptr++;
    }
  }

return(filErr);
} /* FIUmakeFnameChars end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUfopen

    PURPOSE:  to open a file, increasing registered file open count.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filFptr            FILE *, file pointer, NULL on error           **
**\p*******************************************************************/

FILE *FIUfopen( const char *pasFilePath, const char *pasFileMode )
{
/*********** Local Constant & Variable Declarations *******************/
FILE *filFptr = NULL;
/************************* Procedure Body *****************************/

if ((pasFilePath == NULL) || (pasFileMode == NULL))
  {
  ERAparameterError("FIUfopen");
  }
else
  {
  filFptr = fopen(pasFilePath,pasFileMode);
  if (filFptr)
    {
    fimFilesOpened++;
    }
  }

return(filFptr);
} /* FIUfopen end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUfclose

    PURPOSE:  to close an opened file and decrease the file open
      counter.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUfclose( FILE **refFptr )
{
/*********** Local Constant & Variable Declarations *******************/
FILE    *filFptr;
STAT_TYPE filErr = FAILED_;

/************************* Procedure Body *****************************/

if ((refFptr) && (*refFptr))
  {
  filFptr = *refFptr;
  if (fclose(filFptr) == 0)
    {
    filErr = SUCCEEDED_;
    fimFilesOpened--;
    }
  *refFptr = NULL;
  }

return(filErr);
} /* FIUfclose end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUnumFilesOpened

    PURPOSE:  to report number of files now opened via FIUfopen().

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   fimFilesOpened     int, number of files currently opened         **
**\p*******************************************************************/

int FIUnumFilesOpened()
{
/************************* Procedure Body *****************************/

return(fimFilesOpened);
} /* FIUnumFilesOpened end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUnewline

    PURPOSE:  to write a newline to the file

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   fimFilesOpened     int, number of files currently opened         **
**\p*******************************************************************/

STAT_TYPE FIUnewline( FILE *pasFptr, int pasNum )
{
/************************* Procedure Body *****************************/
STAT_TYPE filErr = SUCCEEDED_;

if (pasFptr)
  {
  while (pasNum > 0)
    {
    fprintf(pasFptr,"\n");
    pasNum--;
    }
  }
else
  {
  filErr = FAILED_;
  }

return(filErr);
} /* FIUnewline end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  fiuCopy

    PURPOSE:  to copy from the specified source file name
      (including full path) to the destination file
      (also including full path in its name).

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE fiuCopy
( const char    *pasSourceFile,
  const char    *pasDestFile,
  const char    *pasWriteMode )

{ /* fiuCopy procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
CHARPTR FIL_PROC_NAME = "fiuCopy";

#define FIL_MIN_BUF      512
#define FIL_DEF_BUF    32000

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               filErr;
FILE                   *filSFptr;
FILE                   *filDFptr;
CHAR                   *filMinBuffer;
CHAR                   *filBuffer;
unsigned                filBufferLen;
unsigned                filFread;
unsigned                filFwrite;


/************************* Procedure Body *****************************/
/* Initialize return val   */
filMinBuffer = new CHAR[FIL_MIN_BUF];
filErr = SUCCEEDED_;

filSFptr = FIUfopen( pasSourceFile, "rb");
if (filSFptr == NULL)
  {
  filErr = FAILED_;
  ERAerrorReport( ER_WARNING, FIL_PROC_NAME,
    "Failed to open source file %s to copy",
    pasSourceFile );
  }
else
  {
  filDFptr = FIUfopen( pasDestFile, pasWriteMode );
  if (filDFptr == NULL)
    {
    filErr = FAILED_;
    ERAerrorReport( ER_WARNING, FIL_PROC_NAME,
      "Failed to open file %s for destination copy",
      pasDestFile );
    }
  else
    {
    filBufferLen    = FIL_DEF_BUF;
    filBuffer       = new CHAR[filBufferLen]; // equivalent to 'malloc'
    if (filBuffer  == NULL)
      { // Couldn't allocate buffer, so use default local buffer
      filBuffer     = filMinBuffer;
      filBufferLen  = FIL_MIN_BUF;
      }
    do
      {
      filFread = fread( filBuffer, sizeof(CHAR), filBufferLen, filSFptr );
      if (filFread)
        {
        filFwrite = fwrite( filBuffer, sizeof(CHAR), filFread, filDFptr );
        if (filFwrite != filFread)
          {
          ERAerrorReport( ER_WARNING, FIL_PROC_NAME,
            "Only wrote %u of %u bytes to disk, out of disk space?",
            filFwrite, filFread );
          filFread = 0;
          filErr   = FAILED_;
          }
        }
      }
    while (filFread == filBufferLen);
    FIUfclose( &filSFptr );
    FIUfclose( &filDFptr );
    if (filBuffer != filMinBuffer)
      { // then it was created with 'new', we must delete it
      LIMDELA(filBuffer);
      }
    }
  }

LIMDELA(filMinBuffer);
return(filErr);
} /* fiuCopy end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUcopyFile

    PURPOSE:  to copy from the specified source file name
      (including full path) to the destination file
      (also including full path in its name).

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE FIUcopyFile
( const char    *pasSourceFile,
  const char    *pasDestFile )

{ /* FIUcopyFile procedure */
/************************* Procedure Body *****************************/
return(fiuCopy(pasSourceFile,pasDestFile,"wb"));
} /* FIUcopyFile end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUappendFile

    PURPOSE:  to copy from the specified source file name
      (including full path) to the end of destination file
      (also including full path in its name).

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE FIUappendFile
( const char    *pasSourceFile,
  const char    *pasDestFile )

{ /* FIUappendFile procedure */

/******************* Local Constant Declarations **********************/

return(fiuCopy(pasSourceFile,pasDestFile,"ab"));

} /* FIUappendFile end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUfileInfo

    PURPOSE:  to return the size, date, and time of last modification.

        Use "findfirst" function.  This gives a structure
        including file size info.  This allows us to
        find the size of a file without opening it (the
        information comes from the File Allocation Table).
        Thus, you can check size on a file that you may
        not have permission to open.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filFound           FNC    STAT_TYPE, SUCCEEDED_ if file was found**
**\p*******************************************************************/

STAT_TYPE FIUfileInfo( const char *pasFilePath,
                       INT32      *refSize,
                       INT32      *refRevDate,
                       INT32      *refRevTime )

{ /* FIUfileInfo procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
CHARPTR FIL_PROC_NAME = "FIUfileInfo";

#define FIL_FOUND_FILE 0

/******************* Local Variable Declarations **********************/
STAT_TYPE               filFound;

/************************* Procedure Body *****************************/

filFound = fiuFindFirst( pasFilePath, &fimFblk );
if (filFound == FAILED_)
  {
  filFound = FAILED_;
  ERAerrorLog( FIL_PROC_NAME,
    "File '%s' not found to check file size on",
    pasFilePath );
  }
else
  {
  if (refSize)    *refSize    = (INT32)fimFblk.fsize;
  if (refRevDate) *refRevDate = (INT32)fimFblk.fdate;
  if (refRevTime) *refRevTime = (INT32)fimFblk.ftime;
  }

return(filFound);
} /* FIUfileInfo end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUlocateNewestFile

    PURPOSE:  to locate the most recent incarnation of a file
      that fits the file spec. template.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUlocateNewestFile( const char *pasFileSpec,
                               int         pasStrLen,
                               char       *refFilePath )

{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
STAT_TYPE               filFound;
char                   *filFullPath;
char                   *filPathOnly;
INT32                   filDate;
INT32                   filTime;
INT32                   filSize;

/************************* Procedure Body *****************************/

if ((pasFileSpec == NULL)               ||
    (pasStrLen <= FIP_MIN_NAME_LEN)     ||
    (refFilePath == NULL))
  {
  ERAparameterError( "FIUlocateNewestFile" );
  filErr = FAILED_;
  }
else
  {
  filFullPath = new char[FIP_MAX_NAME_LEN+1];
  filPathOnly = new char[FIP_MAX_NAME_LEN+1];
  filFound = fiuFindFirst( pasFileSpec, &fimFblk  );
  if (filFound == FAILED_)
    {
    filErr = FAILED_;
    *refFilePath = '\0';
    }
  else
    {
    filSize     = (INT32)fimFblk.fsize;
    filDate     = (INT32)fimFblk.fdate;
    filTime     = (INT32)fimFblk.ftime;
    FIUextractPath( pasFileSpec, FIP_MAX_NAME_LEN, filPathOnly );
    FIUconstructFilePath( filPathOnly,
                          fimFblk.name,
                          FIP_MAX_NAME_LEN,
                          filFullPath );
    while (filFound == SUCCEEDED_)
      {
      filFound = fiuFindNext( &fimFblk );
      if (filFound == SUCCEEDED_)
        {
        // Is this file newer than the one we
        // already have saved?  Or equal date, but
        // larger?
        if (((INT32)fimFblk.fdate > filDate) ||
            (((INT32)fimFblk.fdate == filDate) && ((INT32)fimFblk.ftime > filTime)) ||
            (((INT32)fimFblk.fdate == filDate) && ((INT32)fimFblk.ftime == filTime) &&
              ((INT32)fimFblk.fsize > filSize)))
          {
          filSize     = (INT32)fimFblk.fsize;
          filDate     = (INT32)fimFblk.fdate;
          filTime     = (INT32)fimFblk.ftime;
          FIUconstructFilePath( filPathOnly,
                                fimFblk.name,
                                FIP_MAX_NAME_LEN,
                                filFullPath );
          }
        }
      }
    LIUstrncpy( refFilePath, filFullPath, pasStrLen );
    LIMDELA(filFullPath);
    LIMDELA(filPathOnly);
    }
  }

return(filErr);
} /* FIUlocateNewestFile end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUrenameFile

    PURPOSE:  to rename a file

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUrenameFile
( const char    *pasOldFname,
  const char    *pasNewFname )

{
/*********** Local Constant & Variable Declarations *******************/
//CHARPTR                 FIL_PROC_NAME = "FIUrenameFile";
STAT_TYPE               filErr;
int                     filRez;

/************************* Procedure Body *****************************/
filErr = FAILED_;
if ((pasOldFname != NULL) && (pasNewFname != NULL))
  {
  if ((!FIUfileExists(pasNewFname)) && (FIUfileExists(pasOldFname)))
    {
    filRez = rename(pasOldFname,pasNewFname);
    if (filRez != -1)
      {
      filErr = SUCCEEDED_;
      }
    }
  }

return(filErr);
} /* FIUrenameFile end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUsetFileDate

    PURPOSE:  to set the date stamp for the specified file to
      the month, day, year indicated

        Note this routine does VERY LITTLE VALIDITY CHECKING ON THE DATE.
        If you want to make certain it's a valid date, use the
        functions in TIUdt class first.  (Didn't want to put TIUdt
        class here and create an interdependancy between FIU and TIU
        code.)

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

static STAT_TYPE fiuSetFileDT( const char *pasFileName,
                               int         pasMMorHH,
                               int         pasDDorMM,
                               int         pasYYorSS,
                               BOOLEAN     pasTsetDate )
{
/*********** Local Constant & Variable Declarations *******************/
#define FLL_OPENOK 0
STAT_TYPE               filErr;
int                     fHandle;
unsigned                fStat;
unsigned                fDate;
unsigned                fTime;

/************************* Procedure Body *****************************/

fStat = _dos_open( pasFileName, O_RDONLY, &fHandle );

if (fStat == FLL_OPENOK)
  {
  filErr = SUCCEEDED_;
  _dos_getftime( fHandle, &fDate, &fTime );
  if (pasTsetDate)
    {
    /****************************************
    * Massage year into proper form
    */
    if ((pasYYorSS >= 1980) && (pasYYorSS <= 2099))
      {
      pasYYorSS -= 1980; // magic number from IBM OS
      }
    else
      {
      if ((pasYYorSS >= 80) && (pasYYorSS <= 99))
        {
        pasYYorSS -= 80;
        }
      else
        {
        if ((pasYYorSS >= 0) && (pasYYorSS <= 79))
          {
          pasYYorSS += 20;
          }
        }
      }
    if ((pasYYorSS <= 0) || (pasYYorSS > 128)) // only 7 bits for year
      {
      pasYYorSS = 0;
      }
    if ((pasMMorHH <= 0) || (pasMMorHH > 12))
      {
      pasMMorHH = 1;
      }
    if (pasDDorMM > 31) pasDDorMM = 1;
    fDate = (pasYYorSS << 9) + (pasMMorHH << 5) + pasDDorMM;
    }
  else
    {
    if (pasMMorHH > 24) pasMMorHH = 0;
    if (pasDDorMM > 60) pasDDorMM = 0;
    if (pasYYorSS > 60) pasYYorSS = 0;
    fTime = (pasMMorHH << 11) + (pasDDorMM << 5) + (pasYYorSS >> 1);
    }
  if (_dos_setftime( fHandle, fDate, fTime ) == 0)
    {
    filErr = SUCCEEDED_;
    }
  else
    {
    filErr = FAILED_;
    }
  _dos_close( fHandle );
  }
else
  {
  filErr = FAILED_;
  }
return(filErr);
} /* fiuSetFileDT end */

STAT_TYPE FIUsetFileDate( const char *pasFileName,
                          int         pasMM,
                          int         pasDD,
                          int         pasYY )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr;
/************************* Procedure Body *****************************/

filErr = fiuSetFileDT( pasFileName, pasMM, pasDD, pasYY, TRUE_ );
return(filErr);
} /* FIUsetFileDate end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUsetFileTime

    PURPOSE:  to set the specified time stamp for the file indicated

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUsetFileTime( const char *pasFileName,
                          int         pasHH,
                          int         pasMM,
                          int         pasSS )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr;
/************************* Procedure Body *****************************/

filErr = fiuSetFileDT( pasFileName, pasHH, pasMM, pasSS, FALSE_ );

return(filErr);
} /* FIUsetFileTime end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetFileDate

    PURPOSE:  to get the date stamp for the specified file

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

static STAT_TYPE fiuGetFileDT( const char *pasFileName,
                               int        *refMMorHH,
                               int        *refDDorMM,
                               int        *refYYorSS,
                               BOOLEAN     pasTgetDate )
{
/*********** Local Constant & Variable Declarations *******************/
#define FLL_OPENOK 0
STAT_TYPE               filErr;
int                     fHandle;
unsigned                fStat;
unsigned                fDate;
unsigned                fTime;

/************************* Procedure Body *****************************/
fStat = _dos_open(pasFileName, O_RDONLY, &fHandle );

if (fStat == FLL_OPENOK)
  {
  filErr = SUCCEEDED_;
  _dos_getftime( fHandle, &fDate, &fTime );
  _dos_close( fHandle );
  if (pasTgetDate)
    {
    if (refYYorSS) *refYYorSS = ((fDate >> 9) & 0x7f ) + 1980;
    if (refDDorMM) *refDDorMM = fDate & 0x1f;
    if (refMMorHH) *refMMorHH = (fDate >> 5) & 0x0f;
    }
  else
    {
    if (refYYorSS) *refYYorSS = (fTime & 0x01f) << 1;
    if (refDDorMM) *refDDorMM = (fTime >> 5) & 0x2f;
    if (refMMorHH) *refMMorHH = (fTime >> 11) & 0x1f;
    }
  }
else
  {
  filErr = FAILED_;
  }

return(filErr);
} /* fiuGetFileDT end */

STAT_TYPE FIUgetFileDate( const char *pasFileName,
                          int        *refMM,
                          int        *refDD,
                          int        *refYY )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr;
/************************* Procedure Body *****************************/

filErr = fiuGetFileDT(pasFileName,refMM,refDD,refYY, TRUE_ );
return(filErr);
} /* FIUgetFileDate end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetFileTime

    PURPOSE:  to get the specified time stamp for the file indicated

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUgetFileTime( const char *pasFileName,
                          int        *refHH,
                          int        *refMM,
                          int        *refSS )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr;
/************************* Procedure Body *****************************/

filErr = fiuGetFileDT(pasFileName,refHH,refMM,refSS, FALSE_ );

return(filErr);
} /* FIUgetFileTime end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUisFloppyFile

    PURPOSE:  to check if the specified file refers to a floppy disk.
      If no floppy specifier is found in the file name, then get
      the current working directory and check it.


**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filFlp             BOOLEAN, TRUE if file refers to a floppy disk **
** By Matthew J. W. Ratcliff                                          **
**\p*******************************************************************/

BOOLEAN FIUisFloppyFile( const char *pasFname )
{
/*********** Local Constant & Variable Declarations *******************/
BOOLEAN                 filFlp;
int                     filDrive;
/************************* Procedure Body *****************************/
filFlp = FALSE_;

if (pasFname)
  {
  if (*pasFname)
    {
    if (pasFname[1] == ':')
      {
      if ((toupper(*pasFname) == 'A') ||
          (toupper(*pasFname) == 'B'))
        {
        filFlp = TRUE_;
        }
      }
    else // get current working...
      {
      filDrive = getdisk() + 'A';
      if ((filDrive == 'A') || (filDrive == 'B'))
        {
        filFlp = TRUE_;
        }
      }
    }
  }

return(filFlp);
} /* FIUisFloppyFile end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUlocateFile

    PURPOSE:  to locate a file.  See if it's in the current directory.
      If not, see if it's in the default directory the caller passes
      in.  If not, see if it is in ANY ONE of the system search
      path directories.  When found, build a full drive/path file
      name and return that to the caller dude(ette).

** By Matthew J. W. Ratcliff                                          **
**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE       FIUlocateFile( const char *pasArgvZeroOrDefaultPath,
                               const char *pasFileToFind,
                               int         pasFullPathLen,
                               char       *refFullPathToFile )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
char                   *filPath;
char                   *filFile;
BOOLEAN                 filExists;
char                   *filEnv;
char                   *filPenv;
char                   *filNextPath;

/************************* Procedure Body *****************************/
filExists = FALSE_;


if ((pasFileToFind) && (pasFullPathLen > 0) && (refFullPathToFile))
  {
  filPath = new char[FIP_MAX_NAME_LEN+1];
  filFile = new char[FIP_MAX_NAME_LEN+1];

  filErr = FIUgetCwd( FIP_MAX_NAME_LEN, filPath );
  }
else // parameter error
  {
  filErr = FAILED_;
  }

if (filErr == SUCCEEDED_)
  {
  FIUconstructFilePath( filPath, pasFileToFind, FIP_MAX_NAME_LEN, filFile );
  /****************************************
  * Does it exist in the CWD?
  */
  filExists = FIUfileExists( filFile );
  if (!filExists && pasArgvZeroOrDefaultPath != NULL)
    {
    /****************************************
    * Try default path from passed command line.
    * For DOS it should be argv[0]. For
    * Windows this should be the default path. That
    * is, we want the caller to pass in the value
    * returned by the Windows API call:
    *  int GetModuleFileName(HINSTANCE hinst,
    *                        LPSTR     lpszfilename,
    *                        int       cbFileName );
    * Returns zero on error.
    */
    STAT_TYPE FIUextractPath( const char *pasFilePath,
                          int         pasDestLen,
                          char       *refFullPath );

    FIUextractPath( pasArgvZeroOrDefaultPath, FIP_MAX_NAME_LEN, filPath );
    
    FIUconstructFilePath( filPath,
                          pasFileToFind,
                          FIP_MAX_NAME_LEN,
                          filFile );
    filExists = FIUfileExists( filFile );
    }
  if (!filExists)
    {
    /****************************************
    * Still not found.
    * Try system search path.
    */
    filPenv = getenv("PATH");
    if (filPenv)
      {
      filEnv = new char[strlen(filPenv)+1];
      strcpy(filEnv,filPenv);
      filPenv     = filEnv;
      filNextPath = filPenv;
      do
        {
        filNextPath = strchr(filNextPath, ';'); // find semicolon separator
                                                // in path list
        if (filNextPath)
          {
          *filNextPath = '\0';
          filNextPath++;
          }
        LIUstrncpy(filPath, filPenv, FIP_MAX_NAME_LEN );
        filPenv = filNextPath;
        FIUconstructFilePath( filPath,
                              pasFileToFind,
                              FIP_MAX_NAME_LEN,
                              filFile );
        filExists = FIUfileExists( filFile );
        }
      while ((filExists == FALSE_) && (filNextPath != NULL));
      }
    }
  }
if (filExists)
  {
  LIUstrncpy( refFullPathToFile, filFile, pasFullPathLen ); 
  }

LIMDELA(filPath);
LIMDELA(filFile);
return(filErr);
} /* FIUlocateFile end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetCwd

    PURPOSE:  to get a copy of the current working directory

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE       FIUgetCwd( int          pasPathLen,
                           char        *refPath )
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
/************************* Procedure Body *****************************/
if ((pasPathLen <= 0) || (refPath == NULL))
  {
  filErr = FAILED_;
  }
else
  {
#ifdef _MSC_VER
  _getcwd( refPath, pasPathLen );
#else
  getcwd( refPath, pasPathLen );
#endif
  }

return(filErr);
} /* FIUgetCwd end */


/*\p********************************************************************
**                                                                    **
    NAME:  FIUchangeDrive

    PURPOSE:  to set default disk drive

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUchangeDrive( int pasDriveLetter )
{
/*********** Local Constant & Variable Declarations *******************/
#define FIL_OLD_DRIVE_A	0
#define FIL_DRIVE_A     1

STAT_TYPE filErr;
int       filResult;
/************************* Procedure Body *****************************/
  
pasDriveLetter = toupper( pasDriveLetter );

if (pasDriveLetter >= 'A') // 1=A, 2=B, etc.
  {
  pasDriveLetter = pasDriveLetter - (int)'A' + 1;
  }                                      
// else it is assumed the user passed in a 1 for 'A, etc.
// if user passed in a 0 this means the user is using some
// OLD DOS mentality, where some DOS functions assumed 0 = A and
// others assumed 1 = A.  So, if we see a 0, don't flag it as an
// error, just set it to 1 for 'A'
else if (pasDriveLetter == FIL_OLD_DRIVE_A)
  {
  pasDriveLetter = FIL_DRIVE_A;
  }  

filErr = SUCCEEDED_;                                      

// Both microsoft and borland implement _chdrive
// the same way.
filResult = _chdrive( pasDriveLetter );
if (filResult < 0) filErr = FAILED_;

return( filErr );
}


/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetDrive

    PURPOSE:  to get default disk drive letter

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filDriveLetter	int, current drive letter, 'A', 'B', etc.     **
**\p*******************************************************************/

int FIUgetDrive( )
{
/*********** Local Constant & Variable Declarations *******************/
int       filDriveLetter;
/************************* Procedure Body *****************************/

// Both microsoft and borland implement _getdrive
// the same way.
filDriveLetter = _getdrive() + (int)'A' - 1;

return( filDriveLetter );
}
