/*\t*******************************************************************/
/*    Creation Date .......  Fri  06-04-1993  07:39:01                */
/*    Filename  ...........  fiufile.h                                */
/*    Project .............  File Utilities                           */
/*    Author  .............  Matthew J. W. Ratcliff                   */
/*    Language  ...........  C++                                      */
/*    Operating System  ...  DOS/Windows                              */
/*    Processor  ..........  FIU - File utilities.                    */
/*    Function:                                                       */
/*                                                                    */
/*\t*******************************************************************/

/*\r********************************************************************
**                         Revision History
***********************************************************************/
/*

   Date    By           Change Description
dd-mmm-yy  nnn          text
---------  ----         -----------------------------------------------
04-JUN-93  MJWR         Build file support utilities.
**\r*/

/*\m********************************************************************
**                       Definitions
***********************************************************************/

#ifndef _FIUFILE_H_
#define _FIUFILE_H_

#include <stdio.h>

#include "liidsys.h"

#include "fipfile.h"


/****************************************
* A unique file name is created with
* 1 - 5 character prefix defined in
* 'pas1_5CharPrefix'.  If null or
* zero length, the prefix is set to '$'.
* If 'pasAltPathOrNull' is NULL, no alternate
* path is given for a temporary file. If not
* null, and the system variables TMP and TEMP
* are NOT defined, then the temporary file
* is created at 'pasAltPathOrNull'.  If
* all else fails the temporary file is created
* in the current working directory.
* Pass in the desired wrige mode string
* in 'pasWriteMode'.  It should be the same as
* the last parameter passed to the normal 'fopen'
* function.  The new unique file name, with full
* drive and path specifiers, is copied to the
* string 'refNewUniqueName' of 'pasCbNameLen' bytesz
* or less.  The opened file handle is returned. If
* we fail to open the file a NULL is returned.
* The alternate path should NOT be a network drive.
* This function is suitable for DOS or Windows, because
* Windows is non-preemptive multi-tasking, and we
* don't relinquish control to Windows until after
* the file name is created and the file opened.
*
*/

FILE *FIUcreateUniqueFile( const char *pas1_5CharPrefix,
                           const char *pasAltPathOrNull,
                           const char *pasWriteMode, // "wb" or "w" for example
                           int         pasCbNameLen,
                           char       *refNewUniqueName );


/****************************************
* Does the file exist? If not,
* return FALSE_, else return TRUE_.
*/

BOOLEAN FIUfileExists( const char *pasFileName );

/****************************************
* Does the path exist? If not,
* return FALSE_, else return TRUE_.
*/
BOOLEAN FIUpathExists( const char *pasPathName );



/****************************************
* Copy the file from the source to the
* destination.  Both names MUST specify
* the FULL PATH to the file.
*/

STAT_TYPE FIUcopyFile
( const char    *pasSourceFile,
  const char    *pasDestFile );

STAT_TYPE FIUrenameFile
( const char    *pasOldFname,
  const char    *pasNewFname );

// Append pasSourceFile to tail
// end of pasDestFile.  If pasDestFile
// doesn't exist, it's the same as
// FIUcopyFile
STAT_TYPE FIUappendFile
( const char    *pasSourceFile,
  const char    *pasDestFile );


/****************************************
* to combine the filename and path name into
* a complete path\file specification.  If
* trailing '\' is missing from path, add it
* before appending file name.  Keep careful
* watch on length of destination string so
* that we don't overrun its length.  If
* destination string is too short, return
* FAILED_, else return SUCCEEDED_.
*/

STAT_TYPE FIUconstructFilePath
( const char    *pasPath,
  const char    *pasFileName,
  int            pasNameLen,
  char          *refFilePath );

/****************************************
* Set the file extender.  If extender
* pointer is NULL, set file extender to none.
* Else, delete existing extension from file
* and add the one passed in.
*/

STAT_TYPE FIUsetExtender
( const char    *pasExtender,
  char          *refFilePath );

STAT_TYPE FIUdeleteFile
( const char *pasFilePath );

INT32 FIUfileSize
( const char *pasFilePath );

STAT_TYPE FIUfileInfo( const char *pasFilePath,
                       INT32    *refSize,
                       INT32    *refRevDate,
                       INT32    *refRevTime );

/****************************************
* Delete files with wild card character
* specifications, such as FIUdeleteWCfiles( "*.bak" );
* Returns a count of the number of files deleted.
*/

int FIUdeleteWCfiles( const char *pasNukemFileSpec );

/****************************************
* Extract just the path name from a
* full path/file specification - examples:
*
* Input:                Output:
* ---------------------+-----------------
* c:\code\fun\file.c    c:\code\fun\
* c:junk                c:
* file.exe              "\0"
*/

STAT_TYPE FIUextractPath( const char *pasFilePath,
                          int         pasDestLen,
                          char       *refFullPath );

STAT_TYPE FIUextractFname( const char *pasFilePath,
                           int         pasDestLen,
                           char       *refFnameOnly );

/****************************************
* Extract just the file extension
* from the file spec - examples:
*
* Input:                Output:
*             pasWcSpec= TRUE     FALSE              
* ---------------------+-----------------
* c:\code\fun\file.c    "C"       "*.C"       
* c:junk                "\0"      "*.*"
* c:junk.               "\0"      "*."
* file.exe              "EXE"     "*.EXE"
*
* 
*/

STAT_TYPE FIUextractExt(  const char *pasFilePath,
                          int         pasDestLen,
                          char       *refFileExt,
                          BOOLEAN     pasMakeWcSpec = FALSE_ );

/****************************************
* Use FIUgetFirstFilename to find the
* first file matching a template, including
* optional PATH, with wildcard character(s)
* in the file specifier, such as:
*
* FIUgetFirstFilename( "c:\temp\*.$$$" );
*
* FIUgetNextFilename() will get the NEXT FILE
* matching the previous template. A NULL is
* returned when no more matches are found, else
* a constant character pointer to a formatted
* path & file specification to the file found
* is returned.
*/


/* If pasReturnFullPath is FALSE then file NAME ONLY is
   returned.  Else full path and file name are returned.
   For example, we have files ERR0.$$$ and ERR1.$$$ in
   directory C:\TEMP.  Below are the calls and return
   values:

   Call                                                  Returned
   FIUgetFirstFilename( "c:\temp\*.$$$", TRUE_ );        "c:\temp\err0.$$$"
   FIUgetFirstFilename( "c:\temp\*.$$$", FALSE_);        "err0.$$$"
   FIUgetNextFilename( TRUE_ );                          "c:\temp\err1.$$$"
   FIUgetNextFilename( FALSE_ );                         "err1.$$$"

   Note that a DEFAULT value of TRUE_ is provided for the pasReturnFullPath
   parameter.
*/


const char *FIUgetFirstFilename( const char *pasFileSpec,
                                 BOOLEAN     pasReturnFullPath = TRUE_ );
const char *FIUgetNextFilename(  BOOLEAN     pasReturnFullPath = TRUE_ );

STAT_TYPE FIUbuildDefaultFname
( const char    *pasPath,
  const char    *pasName,
  const char    *pasExt,
  int            pasLen,
  char          *refDefName,
  int            pasDefaultChar = '_' );

STAT_TYPE FIUmakeFnameChars( char *refMessyName,
                             int   pasSubstChar = '_' );


FILE *FIUfopen( const char *pasFilePath,
                const char *pasFileMode );

// Pass the address of your file pointer!
// That way we set to NULL for you after
// closing.

STAT_TYPE FIUfclose( FILE **refFptr );

int       FIUnumFilesOpened();

STAT_TYPE FIUnewline( FILE *pasFptr, int count = 1 );

/****************************************
* Given a file search specification, such
* as "C:\MYFILES\C*.DAT", search all the
* files that fit this file spec.  Find
* the newest one of these files and return
* it by copying it into refFilePath.
* If NONE matches file spec, set refFilePath
* to zero length and return failed.
*/

STAT_TYPE FIUlocateNewestFile( const char *pasFileSpec,
                               int         pasStrLen,
                               char       *refFilePath );


STAT_TYPE FIUsetFileDate( const char *pasFileName,
                          int         pasMM,
                          int         pasDD,
                          int         pasYY );

STAT_TYPE FIUsetFileTime( const char *pasFileName,
                          int         pasHH,
                          int         pasMM,
                          int         pasSS );

STAT_TYPE FIUgetFileDate( const char *pasFileName,
                          int        *refMM,
                          int        *refDD,
                          int        *refYY );

STAT_TYPE FIUgetFileTime( const char *pasFileName,
                          int        *refHH,
                          int        *refMM,
                          int        *refSS );

/****************************************
* Get the name of the disk.  You specify
* a drive letter, such as 'A' or 'B', etc.
* for "A:", "B:", etc.
* The disk's volume name is read into the
* referenced string.
*/

STAT_TYPE FIUgetDiskName( int         pasDriveLetter,
                          int         pasStrLen,
                          char       *refDiskName );

/****************************************
* Is the specified file name on a floppy
* disk or a hard drive?  If the file
* name has no path specifier, then the
* current working directory is assumed...
* which may be A: or B:, a floppy.
*/

BOOLEAN FIUisFloppyFile( const char *pasFileName );

/****************************************
* Locate a file. If the file is in the
* current working directory, return it.
* If the file is in the same directory
* as the executable (or whatever default
* path is passed in) find it there.  Else
* if the file is in the system search path
* find the file in one of those.  If not found
* in any of these possible locations, return
* FAILED_.  Fill the user specified file name
* with the full (fopen ready) path to the
* file.
******************************************/

STAT_TYPE       FIUlocateFile( const char *pasArgvZeroOrDefaultPath,
                               const char *pasFileToFind,
                               int         pasFullPathLen,
                               char       *refFullPathToFile );

/****************************************
* Get a copy of the current working
* directory into the destination path
* name.
*/
STAT_TYPE       FIUgetCwd( int          pasPathLen,
                           char        *refPath );
/****************************************
* Set the current working directory to
* the path specified.  If the path does
* not exist, do not change working directory,
* and return failed.  Else switch to it
* and return succeeded.
*/

STAT_TYPE       FIUsetCwd( char        *pasPath );

/****************************************
* Change the default drive to 'A' or 'B' or 'C'
* etc.  You may also pass a drive number
* of 1=A, 2=B, etc.  You may also pass 'a', 'b',
* or 'c'.  All of the above refer to drive
* A:, B:, and C: respectively.  If fail to change
* default drive, FAILED_ is returned. Else
* SUCCEEDED_ is returned.
*/

STAT_TYPE FIUchangeDrive( int pasDriveLetter );

/****************************************
* Get the current default disk drive
* letter. 'A' for A:, 'B' for B:, etc.
*/

int FIUgetDrive( );

#endif
