/*\t*******************************************************************/
/*    Creation Date .......  Fri  06-04-1993  07:37:52                */
/*    Filename  ...........  fiufile.cpp                              */
/*    Project .............  File Utilities                           */
/*    Author  .............  Matthew J. W. Ratcliff                   */
/*    Language  ...........  C++                                      */
/*    Operating System  ...  DOS/Windows                              */
/*    Processor  ..........  FIU - File Utilities                     */
/*    Function: Support code for file related operations.             */
/*                                                                    */
/*                                                                    */
/*\t*******************************************************************/

/*\r********************************************************************
**                         Revision History
***********************************************************************/
/*

   Date    By           Change Description
dd-mmm-yy  nnn          text
---------  ----         -----------------------------------------------
17-Jun-94  MJWR         Carved from FIUFILE.CPP.  This allows you to
                        use a subset of the file processing without
                        the added burden of ERAxxx.
07-Jul-94  MJWR         Return BOOLEAN not int for "FIUxxx" test functions.
                        Also modify fiuFindFirst to accept a parameter
                        allowing it to search for any file type, not just
                        file names.  Used by FIUpathExists function.

**\r*/

/*\i********************************************************************
**                       Module Include Files
***********************************************************************/

/*********************** System Include Files *************************/
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#ifdef _MSC_VER
#include <direct.h>
#pragma message("Compiling for Microsoft C")
#else              
#include <dir.h>
#pragma message("Compiling for Borland C")
#endif

/********************** Constant Include Files ************************/
#include "liidSys.h"
#include "figfile.h"
#include "fipfile.h"
#include "liustrg.h"

/***************** External Variable Include Files ********************/


/***************** External Procedure Include Files *******************/
#include "fiufile.h"

/*\i*/

/*\m********************************************************************
**                       Module Declarations
***********************************************************************/

/************************* Module Constants ***************************/


/************************* Module Variables ***************************/

static FIG_DOS_FBLK_TY fimFblk;

/************************* Module Procedures **************************/

/*\m*/

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  fiuFindFirst
    NAME:  fiuFindNext

    PURPOSE:  Setup fimFblk structure with first or next
      file information based on the DOS findfirst/findnext
      functions.  Code is setup to compile and run under
      Borland or Microsoft C.  Both functions return a
      SUCCEEDED_ or FAILED_ condition, so it's easy to tell
      the result.
**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filFound           FNC    STAT_TYPE, SUCCEEDED_ or FAILED_       **
**\p*******************************************************************/

STAT_TYPE       fiuFindFirst( const char *pasFilePath,
                              FIG_DOS_FBLK_TY *refFblk,
                              unsigned         pasSeaMode ) // = FIG_NORMAL );

{
/******************* Local Variable Declarations **********************/
#define FIL_FOUND_FILE  0
STAT_TYPE               filFound;
int                     filStatus;
/************************* Procedure Body *****************************/

#ifdef _MSC_VER
filStatus       = _dos_findfirst( pasFilePath, pasSeaMode,
                                  (struct _find_t *)refFblk );
#else
filStatus       = findfirst( pasFilePath,
                             (struct ffblk *)refFblk, pasSeaMode );
#endif

filFound = FAILED_;
if (filStatus == FIL_FOUND_FILE)
  {
  filFound = SUCCEEDED_;
  }
return(filFound);
}

STAT_TYPE       fiuFindNext( FIG_DOS_FBLK_TY *refFblk )

{
/******************* Local Variable Declarations **********************/
#define FIL_FOUND_FILE  0
STAT_TYPE               filFound;
int                     filStatus;
/************************* Procedure Body *****************************/

#ifdef _MSC_VER
filStatus       = _dos_findnext( (struct _find_t *)refFblk );
#else
filStatus       = findnext( (struct ffblk *)refFblk );
#endif

filFound = FAILED_;
if (filStatus == FIL_FOUND_FILE)
  {
  filFound = SUCCEEDED_;
  }
return(filFound);
}
/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUfileSize

    PURPOSE:  to return the size, in bytes, of the specified
      file.  Return 0 if file not found.

        Use "findfirst" function.  This gives a structure
        including file size info.  This allows us to
        find the size of a file without opening it (the
        information comes from the File Allocation Table).
        Thus, you can check size on a file that you may
        not have permission to open.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filSiz             FNC    INT32, size of file in bytes, 0=error  **
**\p*******************************************************************/

INT32 FIUfileSize( const char *pasFilePath )

{ /* FIUfileSize procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
#define FIL_FOUND_FILE 0

/******************* Local Variable Declarations **********************/
/* error return flag       */
INT32                   filSiz;
STAT_TYPE               filFound;

/************************* Procedure Body *****************************/

filFound = fiuFindFirst( pasFilePath, &fimFblk );
if (filFound == FAILED_)
  {
  filSiz          = 0L;
  }
else
  {
  filSiz          = fimFblk.fsize;
  }

return(filSiz);
} /* FIUfileSize end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUfileExists

    PURPOSE:  to attempt to open the file for read. If successful,
      return TRUE_, else return FALSE_.


        NOTE: This algorithm doesn't distinguish between
         the conditions "file does not exist" and
         "file access denied".  In the latter case, it doesn't
         really matter if the file exists or not - you can't
         open it anyway.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filFindFlg         FNC    BOOLEAN, TRUE_ if exists, else FALSE_  **
**\p*******************************************************************/

BOOLEAN FIUfileExists( const char *pasFilePath )

{ /* FIUfileExists procedure */
/******************* Local Constant Declarations **********************/
#define FIL_FOUND_FILE 0

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               filFound;
BOOLEAN                 filFindFlag;

/************************* Procedure Body *****************************/

filFound = fiuFindFirst( pasFilePath, &fimFblk  );

filFindFlag = FALSE_;
if (filFound == SUCCEEDED_)
  {
  filFindFlag = TRUE_;
  }

return(filFindFlag);
} /* FIUfileExists end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetFirstFilename

    PURPOSE:  to get the first file name based on the passed
      name string, with or without wildcards.  Return a
      character pointer to the filename.  If no file found
      return NULL.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

static char    fimFindFile[FIP_MAX_NAME_LEN+1];
static char    fimFindPath[FIP_MAX_NAME_LEN+1];

const char *FIUgetFirstFilename( const char *pasFileSpec,
                                 BOOLEAN     pasReturnFullPath)
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
const char             *filFile;
STAT_TYPE               filStat;

/************************* Procedure Body *****************************/

// Save the full path and return it with file name(s)
// found.

filFile = NULL;

filStat = FIUextractPath( pasFileSpec, FIP_MAX_NAME_LEN, fimFindPath );

if (filStat == SUCCEEDED_)
  {
  filErr = fiuFindFirst( pasFileSpec, &fimFblk  );
  if (filErr == SUCCEEDED_)
    {
    if (pasReturnFullPath)
      {
      FIUconstructFilePath( fimFindPath,
                            fimFblk.name,
                            FIP_MAX_NAME_LEN,
                            fimFindFile );
      }
    else
      {
      strcpy(fimFindFile, fimFblk.name );
      }
    filFile = (const char *)fimFindFile;
    }
  }

return( filFile );
} /* FIUgetFirstFilename end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetNextFilename

    PURPOSE:  to get the next file name based on the results
      a previous call to FIUgetFirstFilename.
      Return a character pointer to the filename.  If no file found
      return NULL.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

const char *FIUgetNextFilename( BOOLEAN     pasReturnFullPath)
{
/*********** Local Constant & Variable Declarations *******************/
STAT_TYPE        filErr;
const char      *filFile;
/************************* Procedure Body *****************************/
filFile    =  NULL;
filErr     =  fiuFindNext( &fimFblk );
if (filErr == SUCCEEDED_)
  {
  if (pasReturnFullPath)
    {
    FIUconstructFilePath( fimFindPath,
                          fimFblk.name,
                          FIP_MAX_NAME_LEN,
                          fimFindFile );
    }
  else
    {
    strcpy( fimFindFile, fimFblk.name );
    }
  filFile = (const char *)fimFindFile;
  }
return(filFile);
} /* FIUgetNextFilename end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUextractPath

    PURPOSE:  to extract full path from full path/file specification.
      The outputted path will include the drive specifier, if any
      is given, as well.


**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUextractPath( const char *pasFilePath,
                          int         pasDestLen,
                          char       *refFullPath )
{
/*********** Local Constant & Variable Declarations *******************/
//CHAR                   *FIL_PROC_NAME = "FIUextractPath";
STAT_TYPE               filErr        = SUCCEEDED_; /* Iniz ret val   */
char                   *filDs;
/************************* Procedure Body *****************************/

if ((pasFilePath == NULL) ||
    (pasDestLen <= 0)     ||
    (refFullPath == NULL))
  {
  filErr = FAILED_;
  }
else
  {
  LIUstrncpy( refFullPath, (char *)pasFilePath, pasDestLen );
  filDs = LIUstrchrLast( refFullPath, FIP_PATH_SEP );
  if (filDs)
    {
    filDs++;
    *filDs = '\0';
    }
  else
    {
    filDs = strchr( refFullPath, FIP_DRIVE_SEP );
    if (filDs)
      {
      filDs++;
      *filDs = '\0';
      }
    else
      {
      *refFullPath = '\0';
      }
    }
  }
return(filErr);
} /* FIUextractPath end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  FIUconstructFilePath

    PURPOSE:  to combine the filename and path name into a
      complete path\file specification.  If trailing '\'
      is missing from path, add it before appending file
      name.  Keep careful watch on length of destination string
      so that we don't overrun its length.  If destination
      string is too short, return FAILED_, else return
      SUCCEEDED_.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   filErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE FIUconstructFilePath
( const char    *pasPath,
  const char    *pasFileName,
  int            pasNameLen,
  char          *refFilePath )

{ /* FIUconstructFilePath procedure */

/******************* Local Constant Declarations **********************/
/* Proc name for error log */
#define FIL_MIN_LEN 3

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               filErr;
int                     filLen;
int                     filCatLen;

/************************* Procedure Body *****************************/

filErr = FAILED_;

if (pasNameLen > FIL_MIN_LEN || refFilePath != NULL)
  {
  LIUstrncpy( refFilePath, (char *)pasPath, pasNameLen-1 );
  filLen = strlen(refFilePath);
  if ( (filLen > 0) &&
       (refFilePath[filLen-1] != FIP_PATH_SEP) &&
       (refFilePath[filLen-1] != FIP_DRIVE_SEP) )
    {
    refFilePath[filLen]   = FIP_PATH_SEP;
    refFilePath[filLen+1] = '\0';
    filLen++;
    }
  filCatLen = pasNameLen - filLen;
  if (filCatLen > 0)
    {
    strncat(refFilePath, pasFileName, filCatLen );
    refFilePath[pasNameLen-1] = '\0';
    filErr = SUCCEEDED_;
    }
  }
return(filErr);
} /* FIUconstructFilePath end */

/*\p********************************************************************
**                                                                    **
    NAME:  BOOLEAN FIUpathExists

    PURPOSE:  to see if the specified path exists

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filFnd             BOOLEAN, true if found                        **
**\p*******************************************************************/

BOOLEAN FIUpathExists( const char *pasPath )
{
/*********** Local Constant & Variable Declarations *******************/
BOOLEAN filFnd;
STAT_TYPE filErr;

/************************* Procedure Body *****************************/

filErr = fiuFindFirst( pasPath, &fimFblk, FIG_SUBDIR );

if (filErr == SUCCEEDED_) filFnd = TRUE_;
else                      filFnd = FALSE_;

return(filFnd);
} /* FIUpathExists end */

/*\p********************************************************************
**                                                                    **
    NAME:  FIUgetDiskName

    PURPOSE:  to read the disk label name for the specified drive.

** By Matthew J. W. Ratcliff                                          **
**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   filErr             STAT_TYPE, SUCCEEDED_ or FAILED_              **
**\p*******************************************************************/

STAT_TYPE FIUgetDiskName( int         pasDriveLetter,
                          int         pasStrLen,
                          char       *refDiskName )
{
/*********** Local Constant & Variable Declarations *******************/
#define FIL_MIN_LABEL_LEN       11 // Minimum disk label name length

STAT_TYPE               filErr;
char                    filDrv[10];
/************************* Procedure Body *****************************/

if ((!refDiskName) || (pasStrLen < FIL_MIN_LABEL_LEN))
  {
  filErr = FAILED_;
  }
else
  {
  strcpy(filDrv,"X:\\*.*");
  // If caller passed drive NUMBER convert to letter
  if (pasDriveLetter < 'A')
    {
    pasDriveLetter += (int)'A';
    }
  filDrv[0] = (char)pasDriveLetter;
  filErr    = fiuFindFirst( filDrv, &fimFblk, FIG_VOLID );
  if (filErr == SUCCEEDED_)
    {
    LIUstrncpy( refDiskName, fimFblk.name, pasStrLen );
    }
  else
    {
    *refDiskName = '\0';
    }
  }

return(filErr);
} /* FIUgetDiskName end */

